/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#ifndef _ASYNCBUF_H_
#define _ASYNCBUF_H_

#include <glib-object.h>
#include <glib.h>

#ifdef __cplusplus
extern "C" {
#endif


#define DEBUG_ASYNC_BUFFER	0
#define DEBUG_ASYNC_BUFFER_MOST	0


#define ASYNC_BUFFER_TYPE			(async_buffer_get_type())
#define ASYNC_BUFFER(obj)			(G_TYPE_CHECK_INSTANCE_CAST((obj), ASYNC_BUFFER_TYPE, AsyncBuffer))
#define ASYNC_BUFFER_CLASS(klass)		(G_TYPE_CHECK_CLASS_CAST((klass), ASYNC_BUFFER_TYPE, AsyncBufferClass))
#define IS_ASYNC_BUFFER(obj)			(G_TYPE_CHECK_INSTANCE_TYPE((obj), ASYNC_BUFFER_TYPE))
#define IS_ASYNC_BUFFER_CLASS(klass)		(G_TYPE_CHECK_CLASS_TYPE((klass), ASYNC_BUFFER_TYPE))
#define ASYNC_BUFFER_GET_CLASS(obj)		(G_TYPE_INSTANCE_GET_CLASS((obj), ASYNC_BUFFER_TYPE, AsyncBufferClass))


typedef struct _AsyncBuffer AsyncBuffer;
typedef struct _AsyncBufferClass AsyncBufferClass;

#define INITIAL_ASYNC_BUFFER_SIZE	4096

typedef struct _PThreadSyncObject PThreadSyncObject;
typedef void SyncFunc(gpointer sync_object);
typedef void DestructFunc(AsyncBuffer *buffer, gpointer user_data);


typedef enum
{
  ASYNC_BUFFER_OK,
  ASYNC_BUFFER_SUSPENDED,
  ASYNC_BUFFER_TERMINATED,
} AsyncBufferStatus;


struct _AsyncBuffer
{
  GObject parent_object;

  volatile gboolean fixed;
  char volatile *buffer;
  volatile int length;		/* bufferͭʥǡĹ */

  /* ʲprivate */
  int buffer_length;
  volatile int number_of_active_users;
  volatile int number_of_suspended_users;
  volatile AsyncBufferStatus state;

  PThreadSyncObject *sync_object;
  gpointer user_data;
  DestructFunc *destructor;
};


struct _AsyncBufferClass
{
  GObjectClass parent_class;
};


GType async_buffer_get_type(void);

AsyncBuffer *async_buffer_new(char *buffer, int length, gpointer user_data,
			      DestructFunc *destructor);

gboolean async_buffer_active_ref(AsyncBuffer *buffer, const char *user);
void async_buffer_active_unref(AsyncBuffer *buffer, const char *user);

void async_buffer_suspend_all(void);
void async_buffer_resume_all(void);

void async_buffer_suspend(AsyncBuffer *buffer);
void async_buffer_resume(AsyncBuffer *buffer);
void async_buffer_terminate(AsyncBuffer *buffer);


gboolean async_buffer_fix(AsyncBuffer *buffer, const char *user);
gboolean async_buffer_update_length(AsyncBuffer *buffer, int length,
				    const char *user);
gboolean async_buffer_resize(AsyncBuffer *buffer, int length,
			     const char *user);
gboolean async_buffer_ensure_free_space(AsyncBuffer *buffer, int length,
					const char *user);

gboolean async_buffer_append_data(AsyncBuffer *buffer, const char *data,
				  int length, const char *user);

void async_buffer_lock(AsyncBuffer *buffer);
void async_buffer_unlock(AsyncBuffer *buffer);

gboolean async_buffer_check_active(AsyncBuffer *buffer, const char *user);
gboolean async_buffer_signal(AsyncBuffer *buffer, const char *user);
gboolean async_buffer_broadcast(AsyncBuffer *buffer, const char *user);
gboolean async_buffer_wait(AsyncBuffer *buffer, const char *user);

void free_when_finished(AsyncBuffer *buffer, gpointer unused);


#ifdef __cplusplus
} // extern "C"
#endif

#endif	/* _ASYNCBUF_H_ */
