/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#ifndef _OCHUSHA_H_
#define _OCHUSHA_H_

#include "config.h"

#include "asyncbuf.h"
#include "utils.h"

#include <glib.h>
#include <iconv.h>

#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif

#define OCHUSHA_BOARDLIST_XML		"boardlist.xml"
#define OCHUSHA_CONFIG_XML		"config.xml"
#define OCHUSHA_THREADLIST_XML		"threadlist.xml"

#define OCHUSHA_USER_AGENT		"Monazilla/1.00 (Ochusha/" VERSION ")"

#define OCHUSHA_DEFAULT_BBSMENU_URL	"http://www.ff.iij4u.or.jp/~ch2/bbsmenu.html"

#define OCHUSHA_SUBJECT_TXT		"subject.txt"


typedef struct _OchushaConfig OchushaConfig;
typedef struct _BBSTable BBSTable;
typedef struct _BoardCategory BoardCategory;
typedef struct _BulletinBoard BulletinBoard;
typedef struct _BBSThread BBSThread;
typedef struct _BBSResponse BBSResponse;


/*
 * Ū˴ؤΤɤ
 */
struct _OchushaConfig
{
  /* ochushaΥ桼ۡǥ쥯ȥؤPATH̾(ɸǤ$HOME/.ochusha) */
  char *home;


  /* İμѤURL */
  char *bbsmenu_url;


  /* network˥åȤ̤ */
  gboolean offline;
};


/*
 * İ˴ؤΤɤ
 */

struct _BBSTable
{
  GSList *category_list;
  GHashTable *category_table;	/* name=>(BoardCategory *)ʥơ֥ */
  GHashTable *board_table;	/* base_url=>(BulletinBoard *)ʥơ֥ */
};


BBSTable *ochusha_bbs_table_new(void);
void ochusha_bbs_table_free(BBSTable *table);
void ochusha_bbs_table_add_category(BBSTable *table, BoardCategory *category);
void ochusha_bbs_table_add_board(BBSTable *table, BulletinBoard *board);

typedef gboolean EachCategoryCallback(BoardCategory *category,
				      gpointer user_data);
typedef gboolean EachBoardCallback(BulletinBoard *board, gpointer user_data);

gboolean ochusha_bbs_table_analyze(BBSTable *table, AsyncBuffer *buffer,
				   iconv_t converter,
				   EachCategoryCallback *each_category_cb,
				   EachBoardCallback *each_board_cb,
				   gpointer user_data);

gboolean ochusha_bbs_table_parse_url_2ch(BBSTable *table, const char *url,
					 BulletinBoard **board,
					 char **thread_part);


/*
 * İˤ륫ƥɽ¤
 */
struct _BoardCategory
{
  gchar *name;

  GSList *board_list;

  gpointer user_data;
};


/*
 * ġηǼĤɽ¤
 */
struct _BulletinBoard
{
  gchar *name;
  char *base_url;
  char *server;
  char *base_path;

  /* from SETTING.TXTΤȤȤäƤʤ */
  gchar *title_picture;
  gchar *title_color;
  gchar *title_link;
  gchar *bg_color;
  gchar *bg_picture;
  gchar *noname_name;
  gchar *makethread_color;
  gchar *menu_color;
  gchar *thread_color;
  gchar *text_color;
  gchar *name_color;
  gchar *link_color;
  gchar *alink_color;
  gchar *vlink_color;
  gchar *subject_color;

  GSList *thread_list;		/* subject.txt˸줿(BBSThread *) */
  GHashTable *thread_table;	/* DATե̾(BBSThread *) */

  GSList *dropped_list;		/* DATåɤФ(BBSThread *) */

  gpointer user_data;
};


BoardCategory *ochusha_board_category_new(gchar *name);
void ochusha_board_category_free(BoardCategory *category);

BulletinBoard *ochusha_bulletin_board_new(gchar *name, char *url);
void ochusha_bulletin_board_free(BulletinBoard *board);

typedef gboolean EachThreadCallback(BBSThread *thread, gpointer user_data);
gboolean ochusha_analyze_subject_txt(BulletinBoard *board, AsyncBuffer *buffer,
				     iconv_t converter,
				     EachThreadCallback *each_thread_cb,
				     gpointer user_data);


/*
 * åդκ쥹
 */
#define MAX_RESPONSE	1024


/* ƥåɤɽ¤ */
struct _BBSThread
{
  BulletinBoard *board;

  gchar *title;
  char *dat_filename;
  char *dat_url;

  char *pseudo_path;	/* board->base_url餳ΥؤβŪʥѥ */

  /* оΥ쥹(subject.txt) */
  int number_of_responses_on_server;

  /* ɥ쥹(ɤ߹쥹) */
  int number_of_responses_read;

  /* responses[n]: nܤΥ쥹DAToffset */
  unsigned int *responses;

  /* Ĥ˴ؤäȤ(BBSThreadStateORȤä) */
  /* XXX: ΤȤ̤ */
  int flags;

  gboolean alive;

  gpointer user_data;
};


typedef enum
{
  BBS_THREAD_DAT_DROPPED = 1 << 0,
} BBSThreadState;


BBSThread *ochusha_bbs_thread_new(BulletinBoard *board, char *dat_filename,
				  gchar *title);
void ochusha_bbs_thread_free(BBSThread *thread);
const char *ochusha_bbs_thread_get_dat_url(BBSThread *thread);


struct _BBSResponse
{
  char *name;
  char *mailto;
  char *date_id;
  char *content;
};


/*
 * DATϴϢ
 */
typedef gboolean StartThreadCallback(BBSThread *thread, const char *title,
				     gpointer user_data);
typedef gboolean EachResponseCallback(BBSThread *thread, int number,
				      const BBSResponse *response,
				      gpointer user_data);
typedef gboolean EndThreadCallback(BBSThread *thread, gboolean finished,
				   gpointer user_data);

gboolean ochusha_analyze_dat(BBSThread *thread, AsyncBuffer *buffer,
			     int start, int number,
			     StartThreadCallback *start_thread_cb,
			     EachResponseCallback *each_response_cb,
			     EndThreadCallback *end_thread_cb,
			     gpointer user_data);


/*
 * OCHUSHA_HOMEϢ
 */
void ochusha_prepare_home(OchushaConfig *conf);
char *ochusha_find_file(OchushaConfig *conf, const char *filename);
int ochusha_cache_open_file(OchushaConfig *conf, const char *url, int flags);
void ochusha_cache_unlink_file(OchushaConfig *conf, const char *url);


/*
 * ReadConfigPrefFunc, ReadCategoryElementFunc, ReadBoardElementFunc,
 * ReadThreadElementFuncutilityؿ
 */
gboolean get_attribute_boolean(GHashTable *hash_table, char *attr_name);
int get_attribute_int(GHashTable *hash_table, char *attr_name);
char *get_attribute_string(GHashTable *hash_table, char *attr_name);


/*
 * config.xmlϢ
 */
typedef void WritePreferenceElementFunc(FILE *config_xml, gpointer user_data);
typedef void ReadPreferenceElementFunc(GHashTable *pref_attrs,
				       gpointer user_data);

gboolean ochusha_write_config_xml(OchushaConfig *conf,
				  WritePreferenceElementFunc *optional,
				  gpointer user_data);
void ochusha_read_config_xml(OchushaConfig *conf,
			     ReadPreferenceElementFunc *optional,
			     gpointer user_data);


/*
 * boardlist.xmlϢ
 */
typedef void WriteCategoryElementFunc(FILE *boardlist_xml,
				      BoardCategory *category,
				      gpointer user_data);
typedef void ReadCategoryElementFunc(BoardCategory *category,
				     GHashTable *cat_attrs,
				     gpointer user_data);

typedef void WriteBoardElementFunc(FILE *boardlist_xml, BulletinBoard *board,
				   gpointer user_data);
typedef void ReadBoardElementFunc(BulletinBoard *board,
				  GHashTable *board_attrs, gpointer user_data);

gboolean ochusha_write_boardlist_xml(OchushaConfig *conf, BBSTable *table,
				     WriteCategoryElementFunc *cat_optional,
				     WriteBoardElementFunc *board_optional,
				     gpointer user_data);
BBSTable *ochusha_read_boardlist_xml(OchushaConfig *conf,
				     ReadCategoryElementFunc *cat_optional,
				     ReadBoardElementFunc *board_optional,
				     gpointer user_data);


/*
 * threadlist.xmlϢ
 */
typedef void WriteThreadElementFunc(FILE *threadlist_xml, BBSThread *thread,
				    gpointer user_data);
typedef void ReadThreadElementFunc(BBSThread *thread, GHashTable *hash_table,
				   gpointer user_data);

gboolean ochusha_write_threadlist_xml(OchushaConfig *conf,
				      BulletinBoard *board,
				      WriteThreadElementFunc *optional,
				      gpointer user_data);
void ochusha_read_threadlist_xml(OchushaConfig *conf, BulletinBoard *board,
				 ReadThreadElementFunc *optional,
				 gpointer user_data);




/*
 * ͥåȥϢ
 */
typedef void NetCallbackFunc(gpointer callback_data);

typedef struct _OchushaNetworkCallbacks
{
  NetCallbackFunc *access_started;
  NetCallbackFunc *access_progressed;
  NetCallbackFunc *access_completed;
  NetCallbackFunc *access_terminated;
  NetCallbackFunc *access_failed;
} OchushaNetworkCallbacks;


typedef enum
{
  OCHUSHA_UNKNOWN_STATE,
  OCHUSHA_DIRECT_ACCESS,
  OCHUSHA_CACHE_HIT,
  OCHUSHA_CACHE_IS_OK,
  OCHUSHA_CACHE_IS_DIRTY,
  OCHUSHA_ACCESS_OK,
  OCHUSHA_ACCESS_FAILED
} OchushaNetworkState;


typedef struct _OchushaNetworkStatus
{
  volatile OchushaNetworkState state;

  int status_code;
  char *last_modified;
} OchushaNetworkStatus;


typedef enum
{
  OCHUSHA_CACHE_IGNORE,
  OCHUSHA_CACHE_ONLY,
  OCHUSHA_CACHE_AS_IS,
  OCHUSHA_CACHE_TRY_UPDATE,
  OCHUSHA_CACHE_TRY_REFRESH
} OchushaCacheMode;


AsyncBuffer *ochusha_read_from_url(OchushaConfig *application, const char *url,
				   const char *last_modified,
				   OchushaCacheMode mode,
				   OchushaNetworkCallbacks *callbacks,
				   gpointer callback_data);
AsyncBuffer *ochusha_read_subject_txt(OchushaConfig *application,
				      BulletinBoard *board,
				      const char *last_modified,
				      OchushaNetworkCallbacks *callbacks,
				      gpointer callback_data);

#ifdef __cplusplus
} // extern "C"
#endif

#endif	/* _OCHUSHA_H_ */
