/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_private.h,v 1.2 2003/06/06 19:38:04 fuyu Exp $
 */

#ifndef _OCHUSHA_PRIVATE_H_
#define _OCHUSHA_PRIVATE_H_

#include "config.h"

#define TRACE_MEMORY_USAGE	0
#include "utils.h"

#include <glib.h>
#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif

/* for gettext */
#define N_(String) String
#ifdef ENABLE_NLS
# include <libintl.h>
# define _(String) gettext(String)
#else
# define _(String) (String)
# define gettext(String) (String)
# define textdomain(Domain)
# define bindtextdomain(Package, Directory)
# define bind_textdomain_codeset(Package, CodeSet)
#endif

#if TRACE_MEMORY_USAGE
#include "utils.h"

static inline gpointer
trace_g_malloc(gulong size, char *file, int line)
{
  char where[1024];
  gpointer result = g_malloc(size);
  snprintf(where, 1024, "g_malloc(%ld) returns %p at %s, line %d",
	   size, result, file, line);
  register_pointer(result, where);
  return result;
}
#define G_MALLOC(size)		trace_g_malloc(size, __FILE__, __LINE__)

static inline gpointer
trace_g_malloc0(gulong size, char *file, int line)
{
  char where[1024];
  gpointer result = g_malloc0(size);
  snprintf(where, 1024, "g_malloc0(%ld) returns %p at %s, line %d",
	   size, result, file, line);
  register_pointer(result, where);
  return result;
}
#define G_MALLOC0(size)		trace_g_malloc0(size, __FILE__, __LINE__)

static inline gpointer
trace_g_realloc(gpointer pointer, gulong size, char *file, int line)
{
  char where[1024];
  gpointer result = g_realloc(pointer, size);
  snprintf(where, 1024, "g_realloc(%p, %ld) returns %p at %s, line %d",
	   pointer, size, result, file, line);
  unregister_pointer(pointer, where);
  register_pointer(result, where);
  return result;
}
#define G_REALLOC(pointer, size)	trace_g_realloc(pointer, size, __FILE__, __LINE__)

#define G_NEW(struct_type, n_structs)	((struct_type *)trace_g_malloc (((gsize) sizeof (struct_type)) * ((gsize) (n_structs)), __FILE__, __LINE__))
#define G_NEW0(struct_type, n_structs)	((struct_type *)trace_g_malloc0(((gsize) sizeof (struct_type)) * ((gsize) (n_structs)), __FILE__, __LINE__))

static inline gchar *
trace_g_strdup(const gchar *string, char *file, int line)
{
  char where[1024];
  gpointer result = g_strdup(string);
  snprintf(where, 1024, "g_strdup(%p) returns %p at %s, line %d",
	   string, result, file, line);
  register_pointer(result, where);
  return result;
}
#define G_STRDUP(string)	trace_g_strdup(string, __FILE__, __LINE__)


static inline gchar *
trace_g_strndup(const gchar *string, int len, char *file, int line)
{
  char where[1024];
  gpointer result = g_strndup(string, len);
  snprintf(where, 1024, "g_strndup(%p, %d) returns %p at %s, line %d",
	   string, len, result, file, line);
  register_pointer(result, where);
  return result;
}
#define G_STRNDUP(string, len)	trace_g_strndup(string, len, __FILE__, __LINE__)


static inline void
trace_g_free(gpointer pointer, char *file, int line)
{
  char where[1024];
  snprintf(where, 1024, "g_free(%p) at %s, line %d",
	   pointer, file, line);
  unregister_pointer(pointer, where);
  g_free(pointer);
}
#define G_FREE(pointer)		trace_g_free(pointer, __FILE__, __LINE__)

#else
#define G_MALLOC(size)			g_malloc(size)
#define G_MALLOC0(size)			g_malloc0(size)
#define G_REALLOC(pointer, size)	g_realloc(pointer, size)
#define G_NEW(struct_type, n_structs)	g_new(struct_type, n_structs)
#define G_NEW0(struct_type, n_structs)	g_new0(struct_type, n_structs)
#define G_STRDUP(string)		g_strdup(string)
#define G_STRNDUP(string, len)		g_strndup(string, len)
#define G_FREE(pointer)			g_free(pointer)
#endif

#ifdef __cplusplus
} // extern "C"
#endif

#endif	/* _OCHUSHA_PRIVATE_H_ */
