/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: threadlist_view.c,v 1.16 2003/06/21 16:23:53 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_bbs_thread.h"

#include "ochusha_ui.h"
#include "threadlist_view.h"

#include "marshal.h"

#include <gdk/gdkkeysyms.h>
#include <glib.h>
#include <gtk/gtk.h>

#if DEBUG_WIDGET || DEBUG_WIDGET_MOST
#include <stdio.h>
#endif
#include <stdlib.h>
#include <string.h>


static void threadlist_view_class_init(ThreadlistViewClass *klass);
static void popup_menu_callback(gpointer data, guint action,
				GtkWidget *widget);
static void popup_menu_destructed_cb(gpointer data);
static void threadlist_view_init(ThreadlistView *view);

static void threadlist_view_finalize(GObject *object);
static void threadlist_view_destroy(GtkObject *object);

static gboolean threadlist_view_title_not_equals(GtkTreeModel *model,
						 gint column,
						 const gchar *key,
						 GtkTreeIter *iter,
						 gpointer search_data);

static gboolean threadlist_view_move_cursor(ThreadlistView *view,
					    GtkMovementStep step,
					    gint count);

static void threadlist_view_do_on_selection(ThreadlistView *view,
					    gint what_to_do);
static void threadlist_view_do_on_thread_at_cursor(ThreadlistView *view,
						   gint what_to_do);

static gboolean tree_view_motion_notify_event_cb(GtkWidget *widget,
						 GdkEventMotion *event,
						 ThreadlistView *view);
static gboolean tree_view_leave_notify_event_cb(GtkWidget *widget,
						GdkEventCrossing *event,
						ThreadlistView *view);
static gboolean tree_view_button_press_event_cb(GtkWidget *widget,
						GdkEventButton *event,
						ThreadlistView *view);
static gboolean tree_view_button_release_event_cb(GtkWidget *widget,
						  GdkEventButton *event,
						  ThreadlistView *view);

static void decorate_threadlist_entry(GtkTreeViewColumn *tree_column,
				       GtkCellRenderer *renderer,
				       GtkTreeModel *tree_model,
				       GtkTreeIter *iter, gpointer unused);
static void active_set_cell_data_flags(GtkTreeViewColumn *tree_column,
				       GtkCellRenderer *renderer,
				       GtkTreeModel *tree_model,
				       GtkTreeIter *iter,
				       gpointer unused);
static void text_set_cell_data_rank_diff(GtkTreeViewColumn *tree_column,
					 GtkCellRenderer *renderer,
					 GtkTreeModel *tree_model,
					 GtkTreeIter *iter,
					 gpointer unused);


GType
threadlist_view_get_type(void)
{
  static GType tlv_type = 0;

  if (tlv_type == 0)
    {
      static const GTypeInfo tlv_info =
	{
	  sizeof(ThreadlistViewClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)threadlist_view_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(ThreadlistView),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)threadlist_view_init,
	};

      tlv_type = g_type_register_static(GTK_TYPE_SCROLLED_WINDOW,
					"ThreadlistView", &tlv_info, 0);
    }

  return tlv_type;
}


enum {
  OPEN_VIEW_SIGNAL,

  THREAD_MOUSE_OVER_SIGNAL,
  THREAD_MOUSE_OUT_SIGNAL,

  TOGGLE_MARK_SIGNAL,
  TOGGLE_HIDE_SIGNAL,
  MARK_THREAD_SIGNAL,
  ADVANCE_VIEW_SIGNAL,
  BACK_VIEW_SIGNAL,

  MOVE_CURSOR_SIGNAL,
  INTERACTIVE_SEARCH_SIGNAL,

  DO_ON_SELECTION_SIGNAL,
  DO_ON_THREAD_AT_CURSOR_SIGNAL,

  LAST_SIGNAL
};


enum {
  MARK_THREAD,
  UNMARK_THREAD,
  TOGGLE_HIDDEN_STATE,
  ADVANCE_VIEW,
  BACK_VIEW,
};


enum {
  TREE_VIEW_MOVE_CURSOR_SIGNAL,
  TREE_VIEW_UNSELECT_ALL_SIGNAL,
  TREE_VIEW_SELECT_CURSOR_ROW_SIGNAL,
  TREE_VIEW_START_INTERACTIVE_SEARCH_SIGNAL,
  TREE_VIEW_LAST_SIGNAL
};


enum {
  POPUP_MENU_OPEN_SELECTION = 1,
  POPUP_MENU_OPEN_SELECTION_IN_TAB,
  POPUP_MENU_MARK_SELECTION,
  POPUP_MENU_UNMARK_SELECTION,
  POPUP_MENU_TOGGLE_HIDE_SELECTION
};


struct _ThreadlistInfo
{
  GtkTreeIter iter;
  ThreadlistView *threadlist_view;
};


static GtkScrolledWindowClass *parent_class = NULL;
static gint threadlist_view_signals[LAST_SIGNAL] =
  { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
static gint tree_view_signals[TREE_VIEW_LAST_SIGNAL] = { 0, 0, 0, 0 };
static GtkItemFactory *popup_menu_item_factory = NULL;
static GQuark list_entry_info_id;


static const char *list_entry_foreground_color[16] =
{
  NULL,	/* åȤdefault */
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
};


static const char *list_entry_background_color[16] =
{
  NULL,	/* åȤdefault */
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
};


static void
threadlist_view_class_init(ThreadlistViewClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
  GtkBindingSet *binding_set;

  /* ݥåץåץ˥塼ν */
  GtkItemFactoryEntry menu_items[] =
    {
      {
	_("/_Open Selected Threads"),		/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/Open Selected Threads In _Tab"),	/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_OPEN_SELECTION_IN_TAB,	/* act */
	NULL					/* type */
      },
      {
	_("/-------"),
	NULL,
	NULL,
	0,
	"<Separator>"
      },
      {
	_("/Mark Selected Threads(_*)"),	/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_MARK_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/_Unmark Selected Threads"),		/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_UNMARK_SELECTION,		/* act */
	NULL					/* type */
      },
      {
	_("/Toggle Hide Selected Threads(_D)"),	/* menu path */
	NULL,					/* accelerator */
	popup_menu_callback,			/* callback_func */
	POPUP_MENU_TOGGLE_HIDE_SELECTION,	/* act */
	NULL					/* type */
      },
    };
  gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);

  popup_menu_item_factory = gtk_item_factory_new(GTK_TYPE_MENU,
						 "<ThreadlistViewPopup>",
						 NULL);
  gtk_item_factory_create_items(popup_menu_item_factory,
				nmenu_items, menu_items, NULL);

  parent_class = g_type_class_peek_parent(klass);
  binding_set = gtk_binding_set_by_class(klass);


  /* GObject signals */
  o_class->finalize = threadlist_view_finalize;

  /* GtkObject signals */
  object_class->destroy = threadlist_view_destroy;

  threadlist_view_signals[OPEN_VIEW_SIGNAL] =
    g_signal_new("open_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, open_view),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_BOOLEAN,
		 G_TYPE_NONE, 2,
		 G_TYPE_POINTER,
		 G_TYPE_BOOLEAN);

  threadlist_view_signals[THREAD_MOUSE_OVER_SIGNAL] =
    g_signal_new("thread_mouse_over",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, thread_mouse_over),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  threadlist_view_signals[THREAD_MOUSE_OUT_SIGNAL] =
    g_signal_new("thread_mouse_out",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, thread_mouse_out),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);

  threadlist_view_signals[TOGGLE_MARK_SIGNAL] =
    g_signal_new("toggle_mark",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, toggle_mark),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  threadlist_view_signals[TOGGLE_HIDE_SIGNAL] =
    g_signal_new("toggle_hide",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, toggle_hide),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  threadlist_view_signals[MARK_THREAD_SIGNAL] =
    g_signal_new("mark_thread",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, mark_thread),
		 NULL, NULL,
		 ochusha_marshal_VOID__POINTER_BOOLEAN,
		 G_TYPE_NONE, 2,
		 G_TYPE_POINTER,
		 G_TYPE_BOOLEAN);
  threadlist_view_signals[ADVANCE_VIEW_SIGNAL] =
    g_signal_new("advance_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, advance_view),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_BBS_THREAD);
  threadlist_view_signals[BACK_VIEW_SIGNAL] =
    g_signal_new("back_view",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(ThreadlistViewClass, back_view),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_BBS_THREAD);

  threadlist_view_signals[MOVE_CURSOR_SIGNAL] =
    g_signal_new("wrapped_move_cursor",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, move_cursor),
		 NULL, NULL,
		 ochusha_marshal_BOOLEAN__ENUM_INT,
		 G_TYPE_BOOLEAN, 2,
		 GTK_TYPE_MOVEMENT_STEP,
		 G_TYPE_INT);
  threadlist_view_signals[INTERACTIVE_SEARCH_SIGNAL] =
    g_signal_new("interactive_search",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, interactive_search),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);

  threadlist_view_signals[DO_ON_SELECTION_SIGNAL] =
    g_signal_new("do_on_selection",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, do_on_selection),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);
  threadlist_view_signals[DO_ON_THREAD_AT_CURSOR_SIGNAL] =
    g_signal_new("do_on_thread_at_cursor",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		 G_STRUCT_OFFSET(ThreadlistViewClass, do_on_thread_at_cursor),
		 NULL, NULL,
		 ochusha_marshal_VOID__INT,
		 G_TYPE_NONE, 1,
		 G_TYPE_INT);

  tree_view_signals[TREE_VIEW_MOVE_CURSOR_SIGNAL] =
    g_signal_lookup("move_cursor", GTK_TYPE_TREE_VIEW);
  tree_view_signals[TREE_VIEW_UNSELECT_ALL_SIGNAL] =
    g_signal_lookup("unselect_all", GTK_TYPE_TREE_VIEW);
  tree_view_signals[TREE_VIEW_SELECT_CURSOR_ROW_SIGNAL] =
    g_signal_lookup("select_cursor_row", GTK_TYPE_TREE_VIEW);
  tree_view_signals[TREE_VIEW_START_INTERACTIVE_SEARCH_SIGNAL] =
    g_signal_lookup("start_interactive_search", GTK_TYPE_TREE_VIEW);

  /* Key bindings */
  gtk_binding_entry_add_signal(binding_set, GDK_o, 0, "do_on_selection", 1,
			       G_TYPE_INT, POPUP_MENU_OPEN_SELECTION);
  gtk_binding_entry_add_signal(binding_set, GDK_t, 0, "do_on_selection", 1,
			       G_TYPE_INT, POPUP_MENU_OPEN_SELECTION_IN_TAB);
  gtk_binding_entry_add_signal(binding_set, GDK_asterisk, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, MARK_THREAD);
  gtk_binding_entry_add_signal(binding_set, GDK_u, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, UNMARK_THREAD);
  gtk_binding_entry_add_signal(binding_set, GDK_d, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, TOGGLE_HIDDEN_STATE);
  gtk_binding_entry_add_signal(binding_set, GDK_space, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, ADVANCE_VIEW);
  gtk_binding_entry_add_signal(binding_set, GDK_BackSpace, 0,
			       "do_on_thread_at_cursor", 1,
			       G_TYPE_INT, BACK_VIEW);

  gtk_binding_entry_add_signal(binding_set, GDK_Up, 0,
			       "wrapped_move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, -1);
  gtk_binding_entry_add_signal(binding_set, GDK_Down, 0,
			       "wrapped_move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, 1);

  gtk_binding_entry_add_signal(binding_set, GDK_p, GDK_CONTROL_MASK,
			       "wrapped_move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, -1);
  gtk_binding_entry_add_signal(binding_set, GDK_n, GDK_CONTROL_MASK,
			       "wrapped_move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_DISPLAY_LINES,
			       G_TYPE_INT, 1);

  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_CONTROL_MASK,
			       "wrapped_move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, 1);
  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_MOD1_MASK,
			       "wrapped_move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, -1);
  gtk_binding_entry_add_signal(binding_set, GDK_v, GDK_MOD4_MASK,
			       "wrapped_move_cursor", 2,
			       G_TYPE_ENUM, GTK_MOVEMENT_PAGES,
			       G_TYPE_INT, -1);

  gtk_binding_entry_add_signal(binding_set, GDK_f, GDK_CONTROL_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT, THREADLIST_SEARCH_ACTION_FORWARD);

  gtk_binding_entry_add_signal(binding_set, GDK_s, GDK_CONTROL_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT, THREADLIST_SEARCH_ACTION_FORWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_r, GDK_CONTROL_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT, THREADLIST_SEARCH_ACTION_BACKWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_S,
			       GDK_CONTROL_MASK | GDK_SHIFT_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT,
			       THREADLIST_SEARCH_ACTION_REGEXP_FORWARD);
  gtk_binding_entry_add_signal(binding_set, GDK_R,
			       GDK_CONTROL_MASK | GDK_SHIFT_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT,
			       THREADLIST_SEARCH_ACTION_REGEXP_BACKWARD);

  gtk_binding_entry_add_signal(binding_set, GDK_Escape, 0,
			       "interactive_search", 1,
			       G_TYPE_INT,
			       THREADLIST_SEARCH_ACTION_TERMINATE);
  gtk_binding_entry_add_signal(binding_set, GDK_g, GDK_CONTROL_MASK,
			       "interactive_search", 1,
			       G_TYPE_INT,
			       THREADLIST_SEARCH_ACTION_TERMINATE);

  klass->open_view = NULL;

  klass->thread_mouse_over = NULL;
  klass->thread_mouse_out = NULL;

  klass->toggle_mark = NULL;
  klass->toggle_hide = NULL;
  klass->mark_thread = NULL;
  klass->advance_view = NULL;
  klass->back_view = NULL;

  klass->move_cursor = threadlist_view_move_cursor;
  klass->interactive_search = NULL;

  klass->do_on_selection = threadlist_view_do_on_selection;
  klass->do_on_thread_at_cursor = threadlist_view_do_on_thread_at_cursor;

  list_entry_info_id
    = g_quark_from_static_string("ThreadlistView::ListEntryInfo");

  /*
   * XXX: ʲñʤǥեȤȤ
   *      桼ˤ񤭴ǽˤ٤
   */
  list_entry_foreground_color[1] = g_strdup("blue");
  list_entry_foreground_color[2] = g_strdup("red");
  list_entry_foreground_color[3] = g_strdup("green");
  list_entry_foreground_color[4] = g_strdup("gray50");
  list_entry_foreground_color[5] = g_strdup("gray75");

  list_entry_background_color[1] = g_strdup("lavender");
  list_entry_background_color[2] = g_strdup("#ffe0e0");
  list_entry_background_color[3] = g_strdup("DarkSeaGreen1");
  list_entry_background_color[4] = g_strdup("gray75");
  list_entry_background_color[5] = g_strdup("gray90");
}


typedef enum
{
  THREADLIST_RANK = 0,
  THREADLIST_VISUAL_PARAMETER,
  THREADLIST_TITLE,
  THREADLIST_N_RESPONSES,
  THREADLIST_N_RESPONSES_READ,
  THREADLIST_N_RESPONSES_UNREAD,
  THREADLIST_RANK_DIFFERENCE,
  THREADLIST_DATA,
  THREADLIST_N_COLUMNS
} OchushaThreadlistItem;


static void
threadlist_view_init(ThreadlistView *view)
{
  GtkWidget *tree_view;
#if 0
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
#endif

  GTK_WIDGET_SET_FLAGS(view, GTK_CAN_FOCUS | GTK_RECEIVES_DEFAULT);

  gtk_scrolled_window_set_policy(&view->container,
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  tree_view = gtk_tree_view_new();
  view->view = GTK_TREE_VIEW(tree_view);

  gtk_scrolled_window_set_hadjustment(&view->container, GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 0.0, 0.0, 0.0, 0.0)));
  gtk_scrolled_window_set_vadjustment(&view->container, GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 0.0, 0.0, 0.0, 0.0)));

  /* MEMO: Τ褦GtkAdjustmentͿƤʤassertion˰óݤ:
   * Gtk-CRITICAL **: file gtkrange.c: line 438 (gtk_range_get_adjustment):
   * assertion 'GTK_IS_RANGE(range)' failed
   * 2Ф롣
   */
  gtk_container_add(GTK_CONTAINER(&view->container), tree_view);

  gtk_widget_show(tree_view);

  view->selection = gtk_tree_view_get_selection(view->view);
  gtk_tree_selection_set_mode(view->selection, GTK_SELECTION_MULTIPLE);

  g_signal_connect(G_OBJECT(view->view), "motion_notify_event",
		   G_CALLBACK(tree_view_motion_notify_event_cb), view);
  g_signal_connect(G_OBJECT(view->view), "leave_notify_event",
		   G_CALLBACK(tree_view_leave_notify_event_cb), view);
  g_signal_connect(G_OBJECT(view->view), "button_press_event",
		   G_CALLBACK(tree_view_button_press_event_cb), view);
  g_signal_connect(G_OBJECT(view->view), "button_release_event",
		   G_CALLBACK(tree_view_button_release_event_cb), view);

  view->model = NULL;
  view->model_is_opened = FALSE;

  gtk_tree_view_set_search_column(view->view, THREADLIST_TITLE);
  gtk_tree_view_set_search_equal_func(view->view,
				      threadlist_view_title_not_equals,
				      NULL, NULL);
  gtk_tree_view_set_enable_search(view->view, TRUE);
  view->use_native_search = TRUE;
  view->being_searched = FALSE;
}


static void
threadlist_view_finalize(GObject *object)
{
  ThreadlistView *view = (ThreadlistView *)object;

#if DEBUG_WIDGET_MOST
  fprintf(stderr, "threadlist_view_finalize: view->ref_count=%d\n",
	  object->ref_count);
  fprintf(stderr, "threadlist_view_finalize: model->ref_count=%d\n",
	  G_OBJECT(view->model)->ref_count);
#endif
  if (view->model != NULL)
    {
      GObject *model = G_OBJECT(view->model);
      view->model = NULL;
      g_object_unref(model);
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
threadlist_view_destroy(GtkObject *object)
{
  ThreadlistView *view = THREADLIST_VIEW(object);
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "threadlist_view_destroy: object->ref_count=%d\n",
	  G_OBJECT(object)->ref_count);
#endif

  if (view->last_path != NULL)
    {
      gtk_tree_path_free(view->last_path);
      view->last_path = NULL;
    }

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    (*GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


static ThreadlistInfo *
ensure_threadlist_info(OchushaBBSThread *thread)
{
  ThreadlistInfo *info = g_object_get_qdata(G_OBJECT(thread),
					    list_entry_info_id);
  if (info == NULL)
    {
      info = G_NEW0(ThreadlistInfo, 1);
      g_object_set_qdata_full(G_OBJECT(thread), list_entry_info_id,
			      info, (GDestroyNotify)TRACE_FREE);
    }

  return info;
}


static gboolean
threadlist_view_title_not_equals(GtkTreeModel *model, gint column,
				 const gchar *key, GtkTreeIter *iter,
				 gpointer search_data)
{
  const gchar *title;
  gchar *normalized_key;
  gchar *case_normalized_key;
  gchar *normalized_title;
  gchar *case_normalized_title;
  gboolean result;
#if DEBUG_SEARCH
  char *native_string;
#endif

#if DEBUG_SEARCH
  if (key != NULL)
    {
      native_string = convert_string(utf8_to_native, key, -1);
      fprintf(stderr,
	      "threadlist_view_title_not_equals(): column=%d, key=\"%s\":",
	      column, key);
      G_FREE(native_string);
    }
#endif
  if (key == NULL)
    return FALSE;

  gtk_tree_model_get(model, iter, THREADLIST_TITLE, &title, -1);
#if DEBUG_SEARCH
  if (title != NULL)
    {
      native_string = convert_string(utf8_to_native, title, -1);
      fprintf(stderr, " title at iter=\"%s\"\n", native_string);
      G_FREE(native_string);
    }
  else
    fprintf(stderr, " title at iter=NULL\n");
#endif
  if (title == NULL)
    return FALSE;

  normalized_key = g_utf8_normalize(key, -1, G_NORMALIZE_ALL);
  case_normalized_key = g_utf8_casefold(normalized_key, -1);

  normalized_title = g_utf8_normalize(title, -1, G_NORMALIZE_ALL);
  case_normalized_title = g_utf8_casefold(normalized_title, -1);

  result = (strstr(case_normalized_title, case_normalized_key) == NULL);

  g_free(normalized_key);
  g_free(case_normalized_key);
  g_free(normalized_title);
  g_free(case_normalized_title);

  return result;
}


static gboolean
threadlist_view_move_cursor(ThreadlistView *view, GtkMovementStep step,
			    gint count)
{
  gboolean result = FALSE;

  /*
   * use_native_searchǤʤ硢ġʤ饫ưѤ
   * ʥФ
   */
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "view->being_searched=%d\n", view->being_searched);
#endif
  if (!view->use_native_search && view->being_searched
      && step == GTK_MOVEMENT_DISPLAY_LINES)
    {
      g_signal_emit(G_OBJECT(view),
		    threadlist_view_signals[INTERACTIVE_SEARCH_SIGNAL],
		    0,
		    (count == 1
		     ? THREADLIST_SEARCH_ACTION_FORWARD
		     : THREADLIST_SEARCH_ACTION_BACKWARD),
		    &result);
      return TRUE;
    }

  g_signal_emit(G_OBJECT(view->view),
		tree_view_signals[TREE_VIEW_MOVE_CURSOR_SIGNAL],
		0,
		step, count,
		&result);

  return result;
}


static void
find_initial_path_helper(GtkTreeModel *model, GtkTreePath *path,
			 GtkTreeIter *iter, gpointer data)
{
  GtkTreePath **initial_path = (GtkTreePath **)data;
  if (*initial_path != NULL)
    return;

  *initial_path = gtk_tree_path_copy(path);
  return;
}


static GtkTreePath *
threadlist_view_get_current_path(ThreadlistView *view)
{
  GtkTreePath *initial_path = NULL;
  GtkTreeIter iter;

  gtk_tree_view_get_cursor(view->view, &initial_path, NULL);
  if (initial_path != NULL)
    return initial_path;

  gtk_widget_grab_focus(GTK_WIDGET(view));
  gtk_tree_view_get_cursor(view->view, &initial_path, NULL);
  if (initial_path != NULL)
    return initial_path;

  gtk_tree_selection_selected_foreach(view->selection,
				      find_initial_path_helper, &initial_path);
  if (initial_path != NULL)
    return initial_path;

  if (!gtk_tree_model_get_iter_first(view->model, &iter))
    return NULL;

  return gtk_tree_model_get_path(view->model, &iter);
}


void
threadlist_view_set_being_searched(ThreadlistView *view, gboolean being)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view));

  view->being_searched = (being ? 1 : 0);
}


gboolean
threadlist_view_find_next(ThreadlistView *view,
			  ThreadlistSearchDirection direction,
			  gboolean enable_wrap, gboolean include_current_pos,
			  gpointer search_data)
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreePath *initial_path;
  GtkTreePath *path = NULL;
  gboolean advance_iter = FALSE;

  g_return_val_if_fail(IS_THREADLIST_VIEW(view), FALSE);
  g_return_val_if_fail(search_data != NULL, FALSE);
  g_return_val_if_fail(view->title_match_func != NULL
		       && !view->use_native_search, FALSE);

  view->being_searched = TRUE;

  selection = view->selection;
  initial_path = threadlist_view_get_current_path(view);
  if (initial_path == NULL)
    return FALSE;

  gtk_tree_selection_unselect_all(selection);

  if (!gtk_tree_model_get_iter(view->model, &iter, initial_path))
    {
#if DEBUG_WIDGET_MOST
      fprintf(stderr, "Couldn't get iter.\n");
#endif
      goto no_match_found;
    }

  if (!include_current_pos)
    advance_iter = TRUE;

  while (TRUE)
    {
      const gchar *title;
      if (advance_iter)
	{
	  if (direction == THREADLIST_SEARCH_DIRECTION_FORWARD)
	    {
	      if (!gtk_tree_model_iter_next(view->model, &iter))
		{
		  if (view->last_search_position == 1 && enable_wrap)
		    {
		      view->last_search_position = 0;
		      if (!gtk_tree_model_get_iter_first(view->model, &iter))
			{
#if DEBUG_WIDGET_MOST
			  fprintf(stderr, "Couldn't get first iter.\n");
#endif
			  goto no_match_found;
			}
		    }
		  else
		    {
		      {
			view->last_search_position = 1;
			goto no_match_found;
		      }
		    }
		}
	    }
	  else
	    {
	      if (path == NULL)
		path = gtk_tree_path_copy(initial_path);

	      if (path == NULL)
		goto no_match_found;

	      if (!gtk_tree_path_prev(path))
		{
		  if (view->last_search_position == -1 && enable_wrap)
		    {
		      view->last_search_position = 0;
		      if (view->last_path == NULL)
			{
#if DEBUG_WIDGET_MOST
			  fprintf(stderr, "Couldn't get last iter.\n");
#endif
			  goto no_match_found;
			}
		      if (path != NULL)
			gtk_tree_path_free(path);
		      path = gtk_tree_path_copy(view->last_path);
		    }
		  else
		    {
		      view->last_search_position = -1;
		      goto no_match_found;
		    }
		}

	      if (!gtk_tree_model_get_iter(view->model, &iter, path))
		{
#if DEBUG_WIDGET_MOST
		  fprintf(stderr, "Couldn't get prev iter\n");
#endif
		  goto no_match_found;
		}
	    }
	}

      gtk_tree_model_get(view->model, &iter, THREADLIST_TITLE, &title, -1);
      if (title == NULL)
	{
#if DEBUG_WIDGET_MOST
	  fprintf(stderr, "title is NULL\n");
#endif
	  goto no_match_found;
	}

      if ((*view->title_match_func)(title, search_data))
	{
	  /* found */
	  gtk_tree_path_free(initial_path);
	  if (path != NULL)
	    gtk_tree_path_free(path);
	  path = gtk_tree_model_get_path(view->model, &iter);
	  gtk_tree_view_scroll_to_cell(view->view, path, NULL,
				       TRUE, 0.5, 0.0);
	  gtk_tree_selection_select_iter(selection, &iter);
	  gtk_tree_view_set_cursor(view->view, path, NULL, FALSE);
	  gtk_tree_path_free(path);

	  return TRUE;
	}

      advance_iter = TRUE;
    }

 no_match_found:
  if (path != NULL)
    gtk_tree_path_free(path);

  gtk_tree_view_set_cursor(view->view, initial_path, NULL, FALSE);
  gtk_tree_path_free(initial_path);

  return FALSE;
}


void
threadlist_view_set_enable_native_search(ThreadlistView *view,
					 gboolean enable)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view));

  gtk_tree_view_set_enable_search(view->view, enable);
  view->use_native_search = enable;
}


gboolean
threadlist_view_get_enable_native_search(ThreadlistView *view)
{
  g_return_val_if_fail(IS_THREADLIST_VIEW(view), FALSE);

  return gtk_tree_view_get_enable_search(view->view);
}


static gboolean
threadlist_view_interactive_search(ThreadlistView *view,
				   ThreadlistSearchAction search_action)
{
  gboolean result;

  if (view->use_native_search)
    {
      gtk_widget_grab_focus(GTK_WIDGET(view->view));

      g_signal_emit(G_OBJECT(view->view),
		tree_view_signals[TREE_VIEW_START_INTERACTIVE_SEARCH_SIGNAL],
		0,
		&result);
    }
  else
    {
      view->being_searched = TRUE;
      g_signal_emit(G_OBJECT(view),
		    threadlist_view_signals[INTERACTIVE_SEARCH_SIGNAL],
		    0,
		    search_action,
		    &result);
    }

  return result;
}


gboolean
threadlist_view_start_interactive_search(ThreadlistView *view)
{
  g_return_val_if_fail(IS_THREADLIST_VIEW(view) && view->view != NULL, FALSE);

  return threadlist_view_interactive_search(view,
					THREADLIST_SEARCH_DIRECTION_FORWARD);
}


void
threadlist_view_set_title_match_func(ThreadlistView *view,
				     ThreadlistViewTitleMatchFunc *func)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view));
  view->title_match_func = func;
}


ThreadlistViewTitleMatchFunc *
threadlist_view_get_title_match_func(ThreadlistView *view)
{
  g_return_val_if_fail(IS_THREADLIST_VIEW(view), NULL);
  return view->title_match_func;
}


typedef struct _DoOnSelectionArgs
{
  ThreadlistView *view;
  gint what_to_do;
} DoOnSelectionArgs;


static void
do_on_selection_helper(GtkTreeModel *model, GtkTreePath *path,
		       GtkTreeIter *iter, DoOnSelectionArgs *args)
{
  OchushaBBSThread *thread;
  gint what_to_do = args->what_to_do;

  gtk_tree_model_get(model, iter,
		     THREADLIST_DATA, &thread, -1);
  if (thread == NULL)
    return;	/* ͭʤǰΤ*/

  switch (what_to_do)
    {
    case POPUP_MENU_OPEN_SELECTION:
      /* 2ܰʹߤ϶Ū˥֤ǳ */
      args->what_to_do = POPUP_MENU_OPEN_SELECTION_IN_TAB;

      /* FALL THROUGH */
    case POPUP_MENU_OPEN_SELECTION_IN_TAB:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[OPEN_VIEW_SIGNAL],
		    0,
		    thread,
		    what_to_do == POPUP_MENU_OPEN_SELECTION_IN_TAB);
      break;

    case POPUP_MENU_MARK_SELECTION:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[MARK_THREAD_SIGNAL],
		    0,
		    thread,
		    TRUE);
	  break;

    case POPUP_MENU_UNMARK_SELECTION:
      g_signal_emit(G_OBJECT(args->view),
		    threadlist_view_signals[MARK_THREAD_SIGNAL],
		    0,
		    thread,
		    FALSE);
	  break;

    case POPUP_MENU_TOGGLE_HIDE_SELECTION:
	  g_signal_emit(G_OBJECT(args->view),
			threadlist_view_signals[TOGGLE_HIDE_SIGNAL],
			0,
			thread);
	  break;
    }

}


static void
threadlist_view_do_on_selection(ThreadlistView *view, gint what_to_do)
{
  DoOnSelectionArgs args = { view, what_to_do };
  gtk_tree_selection_selected_foreach(view->selection,
			(GtkTreeSelectionForeachFunc)do_on_selection_helper,
			&args);
}


static void
threadlist_view_do_on_thread_at_cursor(ThreadlistView *view, gint what_to_do)
{
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  gpointer data = NULL;

  gtk_tree_view_get_cursor(view->view, &path, NULL);
  if (path == NULL)
    return;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter, THREADLIST_DATA, &data, -1);

  if (data != NULL)
    {
      gboolean result = FALSE;
      g_signal_emit(G_OBJECT(view->view),
		    tree_view_signals[TREE_VIEW_UNSELECT_ALL_SIGNAL],
		    0,
		    &result);
      g_signal_emit(G_OBJECT(view->view),
		    tree_view_signals[TREE_VIEW_SELECT_CURSOR_ROW_SIGNAL],
		    0,
		    TRUE,
		    &result);

      switch (what_to_do)
	{
	case MARK_THREAD:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[MARK_THREAD_SIGNAL],
			0,
			data,
			TRUE);
	  break;

	case UNMARK_THREAD:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[MARK_THREAD_SIGNAL],
			0,
			data,
			FALSE);
	  break;

	case TOGGLE_HIDDEN_STATE:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[TOGGLE_HIDE_SIGNAL],
			0,
			data);
	  break;

	case ADVANCE_VIEW:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[ADVANCE_VIEW_SIGNAL],
			0,
			data,
			&result);
	  break;

	case BACK_VIEW:
	  g_signal_emit(G_OBJECT(view),
			threadlist_view_signals[BACK_VIEW_SIGNAL],
			0,
			data,
			&result);
	  break;
	}
    }

 done:
  gtk_tree_path_free(path);
}


static void
popup_menu_callback(gpointer data, guint action, GtkWidget *widget)
{
  ThreadlistView *view = gtk_item_factory_popup_data(popup_menu_item_factory);
#if DEBUG_GUI_MOST
  fprintf(stderr, "popup_menu_callback: view=%p, action=%d\n",
	  view, action);
#endif
  threadlist_view_do_on_selection(view, action);
}


static void
popup_menu_destructed_cb(gpointer data)
{
#if DEBUG_GUI_MOST
  ThreadlistView *view = THREADLIST_VIEW(data);
  fprintf(stderr, "popup_menu_destructed_cb: data=%p\n", view);
#endif
}


GtkWidget *
threadlist_view_new(void)
{
  return GTK_WIDGET(g_object_new(THREADLIST_VIEW_TYPE, NULL));
}


static gboolean
tree_view_motion_notify_event_cb(GtkWidget *widget, GdkEventMotion *event,
				 ThreadlistView *view)
{
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  GtkTreeViewColumn *column;
  gpointer data = NULL;

  if (!gtk_tree_view_get_path_at_pos(view->view, event->x, event->y, &path,
				     &column, NULL, NULL))
    goto done;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter, THREADLIST_DATA, &data, -1);

  if (view->recently_pointed_thread != data)
    {
      OchushaBBSThread *thread = (OchushaBBSThread *)data;

      if (view->recently_pointed_thread != NULL)
	g_signal_emit(G_OBJECT(view),
		      threadlist_view_signals[THREAD_MOUSE_OUT_SIGNAL],
		      0,
		      view->recently_pointed_thread);

      view->recently_pointed_thread = thread;

      if (thread != NULL)
	g_signal_emit(G_OBJECT(view),
		      threadlist_view_signals[THREAD_MOUSE_OVER_SIGNAL],
		      0,
		      thread);
    }

 done:
  if (path != NULL)
    gtk_tree_path_free(path);

  return FALSE;
}


static gboolean
tree_view_leave_notify_event_cb(GtkWidget *widget, GdkEventCrossing *event,
				ThreadlistView *view)
{
  if (view->recently_pointed_thread != NULL)
    {
      g_signal_emit(G_OBJECT(view),
		    threadlist_view_signals[THREAD_MOUSE_OUT_SIGNAL],
		    0,
		    view->recently_pointed_thread);

      view->recently_pointed_thread = NULL;
    }

  return FALSE;
}


static gboolean
tree_view_button_press_event_cb(GtkWidget *widget, GdkEventButton *event,
				ThreadlistView *view)
{
  GtkTreeIter iter;
  GtkTreePath *path = NULL;
  GtkTreeViewColumn *column;
  gpointer data = NULL;
  gboolean result = FALSE;

  if (!gtk_tree_view_get_path_at_pos(view->view, event->x, event->y, &path,
				     &column, NULL, NULL))
    goto done;

  if (!gtk_tree_model_get_iter(view->model, &iter, path))
    goto done;

  gtk_tree_model_get(view->model, &iter, THREADLIST_DATA, &data, -1);

  if (data != NULL)
    {
      if (event->button == 3)
	{
	  gtk_item_factory_popup_with_data(popup_menu_item_factory,
					   view, popup_menu_destructed_cb,
					   event->x_root, event->y_root,
					   event->button,
					   gtk_get_current_event_time());
	  if (gtk_tree_selection_iter_is_selected(view->selection, &iter))
	    result = TRUE;
	}
      else if ((event->state & (GDK_CONTROL_MASK | GDK_SHIFT_MASK)) == 0)
	{
	  if (column == view->mark_column)
	    {
	      g_signal_emit(G_OBJECT(view),
			    threadlist_view_signals[TOGGLE_MARK_SIGNAL],
			    0,
			    data);
	    }
	  else
	    {
	      g_signal_emit(G_OBJECT(view),
			    threadlist_view_signals[OPEN_VIEW_SIGNAL],
			    0,
			    data,
			    event->button == 2);
	    }
	}
    }

 done:
  if (path != NULL)
    gtk_tree_path_free(path);

  return result;
}


static gboolean
tree_view_button_release_event_cb(GtkWidget *widget, GdkEventButton *event,
				  ThreadlistView *view)
{
  return FALSE;
}


#define RANK_UNSPECIFIED		-1
#define RANK_DAT_DROPPED		-2

#define RANK_NOT_CHANGED		0
#define RANK_NEW_COMER_MAGIC		0x7fffffff
#define RANK_SAME_RANK_DOWN_MAGIC	0x7ffffffe


static void
set_thread_values(ThreadlistView *view, GtkListStore *store,
		  int last_rank, int new_rank, OchushaBBSThread *thread)
{
  ThreadlistInfo *info = ensure_threadlist_info(thread);
  int num_unread
    = thread->number_of_responses_on_server - thread->number_of_responses_read;
  num_unread = (num_unread >= 0 ? num_unread : 0);

  if (new_rank != RANK_UNSPECIFIED)
    {
      gint rank_diff;
      if (new_rank == RANK_DAT_DROPPED)
	rank_diff = RANK_NOT_CHANGED;
      else
	rank_diff = (last_rank == 0
		     ? RANK_NEW_COMER_MAGIC : last_rank - new_rank);
      if (view->last_rank_diff != RANK_NEW_COMER_MAGIC
	  && view->last_rank_diff != RANK_NOT_CHANGED
	  && view->last_rank_diff == rank_diff
	  && rank_diff < 0)
	rank_diff = RANK_SAME_RANK_DOWN_MAGIC;
      else
	view->last_rank_diff = rank_diff;

      gtk_list_store_set(store, &info->iter,
		THREADLIST_RANK, new_rank,
		THREADLIST_VISUAL_PARAMETER, 0,
		THREADLIST_TITLE, thread->title,
		THREADLIST_N_RESPONSES, thread->number_of_responses_on_server,
		THREADLIST_N_RESPONSES_READ, thread->number_of_responses_read,
		THREADLIST_N_RESPONSES_UNREAD, num_unread,
		THREADLIST_RANK_DIFFERENCE, rank_diff,
		THREADLIST_DATA, thread,
		-1);
    }
  else
    {
      gtk_list_store_set(store, &info->iter,
		THREADLIST_TITLE, thread->title,
		THREADLIST_N_RESPONSES, thread->number_of_responses_on_server,
		THREADLIST_N_RESPONSES_READ, thread->number_of_responses_read,
		THREADLIST_N_RESPONSES_UNREAD, num_unread,
		THREADLIST_DATA, thread,
		-1);
    }
}


static guint
threadlist_view_encode_visual_params(ThreadlistView *view,
				     ThreadlistEntryVisualParameter *parameter)
{
  guint result = 0;

  result |= (parameter->flags & 3);
  result |= (parameter->foreground_color & 0xf)
    << (THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH);
  result |= (parameter->background_color & 0xf)
    << (THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH
	+ THREADLIST_ENTRY_VISUAL_FOREGROUND_COLOR_WIDTH);
#if DEBUG_WIDGET_MOST
  fprintf(stderr, "parameter{ flags=%d, foreground_color=%d, background_color=%d}, result=%d\n", parameter->flags, parameter->foreground_color, parameter->background_color, result);
#endif

  return result;
}


static guint
threadlist_view_decode_visual_param_flags(ThreadlistView *view,
					  guint parameter)
{
  return parameter & 3;
}


static guint
threadlist_view_decode_visual_param_foreground_color(ThreadlistView *view,
						     guint parameter)
{
  return (parameter >> THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH) & 0xf;
}


static guint
threadlist_view_decode_visual_param_background_color(ThreadlistView *view,
						     guint parameter)
{
  return (parameter
	  >> (THREADLIST_ENTRY_VISUAL_FLAGS_WIDTH
	      + THREADLIST_ENTRY_VISUAL_FOREGROUND_COLOR_WIDTH)) & 0xf;
}


void
threadlist_view_update_thread_visual(ThreadlistView *view,
				     OchushaBBSThread *thread,
				     ThreadlistEntryVisualParameter *parameter)
{
  gpointer data = NULL;
  ThreadlistInfo *info;
  g_return_if_fail(IS_THREADLIST_VIEW(view) && thread != NULL);

  info = ensure_threadlist_info(thread);

  if (info->threadlist_view != view)
    {
#if DEBUG_WIDGET
      fprintf(stderr, "info->threadlist_view=%p, view=%p\n",
	      info->threadlist_view, view);
#endif
      return;
    }

  if (!gtk_list_store_iter_is_valid(GTK_LIST_STORE(view->model), &info->iter))
    {
      /* 줿ե륿ξ֤ѤäơiterŤʤäȤ롣 */
      return;
    }

  gtk_tree_model_get(view->model, &info->iter,
		     THREADLIST_DATA, &data, -1);
  g_return_if_fail(thread == data);

  gtk_list_store_set(GTK_LIST_STORE(view->model), &info->iter,
		     THREADLIST_VISUAL_PARAMETER,
		     threadlist_view_encode_visual_params(view, parameter),
		     -1);
}


static void
decorate_threadlist_entry(GtkTreeViewColumn *tree_column,
			  GtkCellRenderer *renderer, GtkTreeModel *tree_model,
			  GtkTreeIter *iter, gpointer view)
{
  guint flags;
  guint color_code;
  GValue fg = { 0, };
  GValue bg = { 0, };

  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));
  gtk_tree_model_get(tree_model, iter,
		     THREADLIST_VISUAL_PARAMETER, &flags, -1);

  if (!GTK_IS_CELL_RENDERER_TOGGLE(renderer))
    {
      color_code
	= threadlist_view_decode_visual_param_foreground_color(view, flags);
      g_value_init(&fg, G_TYPE_STRING);
      g_value_set_static_string(&fg, list_entry_foreground_color[color_code]);
      g_object_set_property((GObject *)renderer, "foreground", &fg);
    }

  color_code =
    threadlist_view_decode_visual_param_background_color(view, flags);
  g_value_init(&bg, G_TYPE_STRING);
  g_value_set_static_string(&bg, list_entry_background_color[color_code]);
  g_object_set_property((GObject *)renderer, "cell_background", &bg);

  if (!GTK_IS_CELL_RENDERER_TEXT(renderer))
    return;

  if (threadlist_view_decode_visual_param_flags(view, flags)
      & THREADLIST_ENTRY_FLAG_STRIKE)
    {
      GValue st = { 0, };
      g_value_init(&st, G_TYPE_BOOLEAN);
      g_value_set_boolean(&st, TRUE);
      g_object_set_property((GObject *)renderer, "strikethrough", &st);
      g_object_set_property((GObject *)renderer, "strikethrough_set", &st);
    }
  else
    {
      GValue st = { 0, };
      g_value_init(&st, G_TYPE_BOOLEAN);
      g_value_set_boolean(&st, FALSE);
      g_object_set_property((GObject *)renderer, "strikethrough", &st);
      g_object_set_property((GObject *)renderer, "strikethrough_set", &st);
    }
}


static void
active_set_cell_data_flags(GtkTreeViewColumn *tree_column,
			   GtkCellRenderer *renderer, GtkTreeModel *tree_model,
			   GtkTreeIter *iter, gpointer view)
{
  guint flags;
  GtkCellRendererToggle *toggle_renderer = GTK_CELL_RENDERER_TOGGLE(renderer);
  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));

  gtk_tree_model_get(tree_model, iter,
		     THREADLIST_VISUAL_PARAMETER, &flags, -1);

  flags = threadlist_view_decode_visual_param_flags(view, flags);

  gtk_cell_renderer_toggle_set_active(toggle_renderer,
				      flags & THREADLIST_ENTRY_FLAG_MARK);
  decorate_threadlist_entry(tree_column, renderer, tree_model, iter, view);
}


static void
text_set_cell_data_rank_diff(GtkTreeViewColumn *tree_column,
			     GtkCellRenderer *renderer,
			     GtkTreeModel *tree_model, GtkTreeIter *iter,
			     gpointer view)
{
  GValue src = { 0, };
  GValue value = { 0, };
  int diff;
  gchar buffer[512];

  g_return_if_fail(gtk_list_store_iter_is_valid(GTK_LIST_STORE(tree_model),
						iter));

  gtk_tree_model_get_value(tree_model, iter, THREADLIST_RANK_DIFFERENCE, &src);
  diff = g_value_get_int(&src);

  g_value_init(&value, G_TYPE_STRING);
  if (diff == RANK_NOT_CHANGED)
    {
      g_value_set_static_string(&value, "");
    }
  else if (diff == RANK_NEW_COMER_MAGIC)
    {
      /* оξ */
      g_value_set_static_string(&value, _("New Face!"));
    }
  else if (diff == RANK_SAME_RANK_DOWN_MAGIC)
    {
      /* ľΥȥƱ̥ξ */
      g_value_set_static_string(&value, _("down"));
    }
  else if (diff > 0)
    {
      /* ̥åפξ */
      snprintf(buffer, 512, _("%d Rank Up"), diff);
      g_value_set_string(&value, buffer);
    }
  else
    {
      /* ̥ξ */
      snprintf(buffer, 512, _("%d Rank Down"), -diff);
      g_value_set_string(&value, buffer);
    }

  g_object_set_property((GObject *)renderer, "text", &value);
  decorate_threadlist_entry(tree_column, renderer, tree_model, iter, view);
}


void
threadlist_view_append_rank(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->rank_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Rank"), renderer,
						    "text", THREADLIST_RANK,
						    NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->rank_column = column;

  view->view_is_initialized = TRUE;
}


void
threadlist_view_append_mark(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->mark_column == NULL);

  renderer = gtk_cell_renderer_toggle_new();
  gtk_cell_renderer_toggle_set_radio(GTK_CELL_RENDERER_TOGGLE(renderer),
				     FALSE);
  column = gtk_tree_view_column_new_with_attributes(_("Mark"), renderer,
					"active", THREADLIST_VISUAL_PARAMETER,
					NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  active_set_cell_data_flags,
					  view, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->mark_column = column;

  view->view_is_initialized = TRUE;
}


void
threadlist_view_append_title(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->title_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Thread Title"),
						    renderer, "text",
						    THREADLIST_TITLE, NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->title_column = column;

  view->view_is_initialized = TRUE;
}


void
threadlist_view_append_number_of_responses(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->n_res_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("#Res"),
						    renderer, "text",
						    THREADLIST_N_RESPONSES,
						    NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->n_res_column = column;

  view->view_is_initialized = TRUE;
}


void
threadlist_view_append_number_of_responses_read(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->n_read_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("#Read"),
						renderer, "text",
						THREADLIST_N_RESPONSES_READ,
						NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->n_read_column = column;

  view->view_is_initialized = TRUE;
}


void
threadlist_view_append_number_of_responses_unread(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->n_unread_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("#Unread"),
						renderer, "text",
						THREADLIST_N_RESPONSES_UNREAD,
						NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  decorate_threadlist_entry,
					  view, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->n_unread_column = column;

  view->view_is_initialized = TRUE;
}


void
threadlist_view_append_rank_difference(ThreadlistView *view)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;

  g_return_if_fail(IS_THREADLIST_VIEW(view) && view->rank_diff_column == NULL);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(_("Rank Difference"),
						    renderer, "text",
						    THREADLIST_RANK_DIFFERENCE,
						    NULL);
  gtk_tree_view_column_set_cell_data_func(column, renderer,
					  text_set_cell_data_rank_diff,
					  view, NULL);
  gtk_tree_view_append_column(view->view, column);
  view->rank_diff_column = column;

  view->view_is_initialized = TRUE;
}


void
threadlist_view_initialize_default_view(ThreadlistView *view)
{
  g_return_if_fail(IS_THREADLIST_VIEW(view) && !view->view_is_initialized);

  threadlist_view_append_rank(view);
  threadlist_view_append_mark(view);
  threadlist_view_append_title(view);
  threadlist_view_append_number_of_responses(view);
  threadlist_view_append_number_of_responses_read(view);
  threadlist_view_append_number_of_responses_unread(view);
  threadlist_view_append_rank_difference(view);
}


gboolean
threadlist_view_open(ThreadlistView *view)
{
  GtkListStore *store;

  g_return_val_if_fail(IS_THREADLIST_VIEW(view)
		       && !view->model_is_opened, FALSE);

  if (!view->view_is_initialized)
    threadlist_view_initialize_default_view(view);

  if (view->model != NULL)
    g_object_unref(G_OBJECT(view->model));

  store = gtk_list_store_new(THREADLIST_N_COLUMNS,
			     G_TYPE_INT,	/* rank */
			     G_TYPE_INT,	/* flags */
			     G_TYPE_STRING,	/* thread title */
			     G_TYPE_INT,	/* number of responses read */
			     G_TYPE_INT,	/* number of responses read */
			     G_TYPE_INT,	/* number of responses unread*/
			     G_TYPE_INT,	/* rank difference */
			     G_TYPE_POINTER);	/* data (OchushaBBSThread *) */

  gtk_tree_view_set_model(view->view, GTK_TREE_MODEL(store));
  view->model = GTK_TREE_MODEL(store);

  view->last_rank_diff = 0;

  return TRUE;
}


void
threadlist_view_close(ThreadlistView *view, OchushaBBSThread *last_thread)
{
  ThreadlistInfo *info;
  g_return_if_fail(IS_THREADLIST_VIEW(view));

  view->model_is_opened = FALSE;

  if (last_thread == NULL)
    return;

  info = ensure_threadlist_info(last_thread);
  g_return_if_fail(info != NULL
		   && gtk_list_store_iter_is_valid(GTK_LIST_STORE(view->model),
						   &info->iter));
  if (view->last_path != NULL)
    gtk_tree_path_free(view->last_path);
  view->last_path = gtk_tree_model_get_path(view->model, &info->iter);
}


void
threadlist_view_append_thread(ThreadlistView *view, OchushaBBSThread *thread,
			      gint last_rank, gint new_rank)
{
  ThreadlistInfo *info;
  g_return_if_fail(IS_THREADLIST_VIEW(view));
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  info = ensure_threadlist_info(thread);

  g_assert(view->model != NULL);

  gtk_list_store_append(GTK_LIST_STORE(view->model), &info->iter);
  info->threadlist_view = view;
  set_thread_values(view, GTK_LIST_STORE(view->model),
		    last_rank, new_rank, thread);
}


void
threadlist_view_update_thread(ThreadlistView *view, OchushaBBSThread *thread)
{
  gpointer data = NULL;
  ThreadlistInfo *info;
  g_return_if_fail(IS_THREADLIST_VIEW(view));
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  info = ensure_threadlist_info(thread);

  if (info->threadlist_view != view)
    {
#if DEBUG_WIDGET	/* ΥƥȼΤĹ */
      fprintf(stderr, "info->threadlist_view=%p, view=%p\n",
	      info->threadlist_view, view);
#endif
      return;
    }

  if (!gtk_list_store_iter_is_valid(GTK_LIST_STORE(view->model), &info->iter))
    {
      /* 줿ե륿ξ֤ѤäơiterŤʤäȤ롣 */
      return;
    }

  gtk_tree_model_get(view->model, &info->iter,
		     THREADLIST_DATA, &data, -1);

#if DEBUG_WIDGET
  if (thread != data)
    fprintf(stderr, "inconsistent ThreadlistView?\n");
#endif
  g_return_if_fail(thread == data);

  set_thread_values(view, GTK_LIST_STORE(view->model), 0, RANK_UNSPECIFIED,
		    thread);
}


void
threadlist_view_scroll_to_thread(ThreadlistView *view,
				 OchushaBBSThread *thread)
{
  gpointer data = NULL;
  GtkTreePath *path;
  ThreadlistInfo *info;
  g_return_if_fail(IS_THREADLIST_VIEW(view));
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  info = ensure_threadlist_info(thread);

  if (info->threadlist_view != view
      || !gtk_list_store_iter_is_valid(GTK_LIST_STORE(view->model),
				       &info->iter))
    return;

  gtk_tree_model_get(view->model, &info->iter,
		     THREADLIST_DATA, &data, -1);
  g_return_if_fail(thread == data);

  path = gtk_tree_model_get_path(view->model, &info->iter);
  gtk_tree_view_scroll_to_cell(view->view, path, NULL, FALSE, 0.0f, 0.0f);
  gtk_tree_view_set_cursor_on_cell(view->view, path, NULL, NULL, FALSE);
  gtk_tree_path_free(path);
}

