/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_utils_2ch.c,v 1.12 2003/11/21 21:30:03 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_bbs_table.h"
#include "ochusha_board_2ch.h"
#include "ochusha_thread_2ch.h"
#include "ochusha_utils_2ch.h"

#include "utils.h"

#include <glib.h>

#if ENABLE_REGEXP
# if HAVE_ONIGURUMA
#  if HAVE_ONIG_ONIGPOSIX_H
#    include <onig/onigposix.h>
#  elif HAVE_ONIGPOSIX_H
#    include <onigposix.h>
#  else
#    error "Use of oniguruma without onig/onigposix.h header isn't considered."
#  endif
# else
#  if HAVE_POSIX_REGEX
#    if HAVE_SYS_TYPES_H
#      include <sys/types.h>
#    endif
#    if HAVE_REGEX_H
#      include <regex.h>
#    else
#      warning "I don't know suitable header file of your posix regex library."
#    endif
#  endif
# endif
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define GUESS_2CH_BOARD_URL	1


static gboolean
parse_thread_path(char *path, char **thread_id_p,
		  unsigned int *from_p, unsigned int *to_p)
{
  char *thread_id = NULL;
  unsigned int from = 0;
  unsigned int to = 0;
  char *slash_pos = strchr(path + 1, '/');

#if 0
  fprintf(stderr, "parse_thread_path()\n");
  fprintf(stderr, "  path: %s\n", path);
#endif

  if (slash_pos != NULL)
    {
      if (thread_id_p != NULL)
	thread_id = G_STRNDUP(path, slash_pos - path);
      else
	thread_id = path;
      path = slash_pos + 1;

      if (*path != '\0')
	{
	  char tmp_char;
	  char *hyphen_pos = strchr(path, '-');
	  if (hyphen_pos != NULL)
	    {
	      tmp_char = *hyphen_pos;
	      *hyphen_pos = '\0';
	    }
	  else
	    tmp_char = '\0';

	  if (hyphen_pos != path)
	    {
	      if (sscanf(path, "%d", &from) != 1)
		from = 0;
	    }

	  if (hyphen_pos != NULL && hyphen_pos[1] != '\0')
	    {
	      if (sscanf(hyphen_pos + 1, "%d", &to) != 1)
		to = from;
	    }
	}
    }
  else
    {
      if (thread_id_p != NULL && *path != '\0')
	thread_id = G_STRDUP(path);
      else
	thread_id = path;
    }

  if (thread_id_p != NULL)
    *thread_id_p = thread_id;

  if (from_p != NULL)
    *from_p = from;

  if (to_p != NULL)
    *to_p = to;

#if 0
  fprintf(stderr, "  thread_id: %s\n", thread_id);
  fprintf(stderr, "  from=%d, to=%d\n", from, to);
#endif

  return thread_id != NULL;
}


#if ENABLE_REGEXP
static gboolean
parse_cgi_query(const char *query, char **board_id_p, char **thread_id_p,
		unsigned int *from_p, unsigned int *to_p)
{
  char board_id[PATH_MAX];
  char thread_id[PATH_MAX];
  char tmp_buf[PATH_MAX];
  unsigned int from = 0;
  unsigned int to = 0;
  regex_t cgi_query_pattern;
  regmatch_t match[3];
  char *tmp_pos;

  if (regcomp(&cgi_query_pattern, "&?([^=]+)=([^&]*)", REG_EXTENDED))
    {
      fprintf(stderr, "invalid_regular expression\n");
      return FALSE;
    }
#if 0
  fprintf(stderr, "parse_cgi_query()\n");
  fprintf(stderr, "  query: %s\n", query);
#endif

  board_id[0] = '\0';
  thread_id[0] = '\0';

  tmp_pos = strchr(query, '?');
  if (tmp_pos != NULL)
    query = tmp_pos + 1;	/* ?ޤǤФ */

  while (regexec(&cgi_query_pattern, query, 2, match, 0) == 0)
    {
      int match_len = match[0].rm_eo - match[0].rm_so;
      const char *name_head = query + match[1].rm_so;
      int name_len = match[1].rm_eo - match[1].rm_so;
      const char *value_head = query + match[2].rm_so;
      int value_len = match[2].rm_eo - match[2].rm_so;

      if (g_ascii_strncasecmp("BBS", name_head, name_len) == 0)
	{
	  if (value_len < (PATH_MAX - 1))
	    {
	      memcpy(board_id, value_head, value_len);
	      board_id[value_len] = '\0';
	    }
	}
      else if (g_ascii_strncasecmp("KEY", name_head, name_len) == 0)
	{
	  if (value_len < (PATH_MAX - 1))
	    {
	      memcpy(thread_id, value_head, value_len);
	      thread_id[value_len] = '\0';
	    }
	}
      else if (g_ascii_strncasecmp("START", name_head, name_len) == 0
	       || g_ascii_strncasecmp("ST", name_head, name_len) == 0)
	{
	  if (value_len > 0 && value_len < (PATH_MAX - 1))
	    {
	      memcpy(tmp_buf, value_head, value_len);
	      tmp_buf[value_len] = '\0';
	      if (sscanf(tmp_buf, "%d", &from) != 1)
		from = 0;
	    }
	}
      else if (g_ascii_strncasecmp("END", name_head, name_len) == 0
	       || g_ascii_strncasecmp("TO", name_head, name_len) == 0)
	{
	  if (value_len > 0 && value_len < (PATH_MAX - 1))
	    {
	      memcpy(tmp_buf, value_head, value_len);
	      tmp_buf[value_len] = '\0';
	      if (sscanf(tmp_buf, "%d", &to) != 1)
		to = 0;
	    }
	}

      query += match_len;
    }

  if (from == 0 && to != 0)
    from = to;
  else if (to == 0 && from != 0)
    to = from;
#if 0
  else if (from == 0 && to == 0)
    {
      from = 1;
      to = 1;
    }
#endif

  if (from_p != NULL)
    *from_p = from;

  if (to_p != NULL)
    *to_p = to;

  if (board_id[0] == '\0')
    thread_id[0] = '\0';

  if (board_id_p != NULL)
    {
      if (board_id[0] != '\0')
	*board_id_p = G_STRDUP(board_id);
      else
	*board_id_p = NULL;
    }

  if (thread_id_p != NULL)
    {
      if (thread_id[0] != '\0')
	*thread_id_p = G_STRDUP(thread_id);
      else
	*thread_id_p = NULL;
    }

#if 0
  fprintf(stderr, "  board_id: %s\n", board_id);
  fprintf(stderr, "  thread_id: %s\n", thread_id);
  fprintf(stderr, "  from=%d, to=%d\n", from, to);
#endif

  regfree(&cgi_query_pattern);

  return board_id[0] != '\0';
}
#endif


/*
 * Ϳ줿URLϤбƤʷǼĤΤΤɤ
 * Ƚꤹ롣
 *
 * Ƚ˲äʲΤ褦Ѥ򵯤
 *
 * ȤͿ줿board_url_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ
 *   tableˤ̤Ͽʾ硢ĤURLȤ餷ʸΥԡ
 *   board_url_p˳Ǽ롣
 * ȤͿ줿bbs_type_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ
 *   tableˤ̤Ͽʾ硢ļ̤Ȥ餷ͤ
 *   bbs_type_p˳Ǽ롣
 * ȤͿ줿thread_id_pfrom_pto_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ硢
 *   ID(thread_id)䡢쥹ϰϻƬ(from)(to)򤽤줾
 *   thread_id_pfrom_pto_p˳Ǽ롣thread_id_p˳ǼΤ
 *   IDʸΥԡ
 */
gboolean
ochusha_utils_2ch_check_url(const char *url, char **board_url_p,
			    OchushaBBSType *bbs_type_p,
			    char **board_id_p, char **thread_id_p,
			    unsigned int *from_p, unsigned int *to_p)
{
  char *server;
  char *abs_path;
  char *path_pos;
  OchushaBBSType bbs_type = OCHUSHA_BBS_TYPE_UNKNOWN;
  char *thread_id = NULL;
  char *board_id = NULL;
  int from = 0;
  int to = 0;
  char board_url[PATH_MAX];
  char tmp_buf[PATH_MAX];

  g_return_val_if_fail(url != NULL, FALSE);

  board_url[0] = '\0';

  server = ochusha_utils_url_extract_http_server(url);
  if (server != NULL)
    abs_path = ochusha_utils_url_extract_http_absolute_path(url);
  else if (board_url_p == NULL)
    {
      const char *tmp_pos = url;
      server = G_STRDUP("localhost");	/*  */
      while (TRUE)
	{
	  if (strncmp(tmp_pos, "../", 3) == 0)
	    {
	      tmp_pos += 2;
	      continue;
	    }
	  if (strncmp(tmp_pos, "/../", 4) == 0)
	    {
	      tmp_pos += 3;
	      continue;
	    }
	  if (strncmp(tmp_pos, "./", 2) == 0)
	    {
	      tmp_pos ++;
	      continue;
	    }
	  if (strncmp(tmp_pos, "/./", 3) == 0)
	    {
	      tmp_pos += 2;
	      continue;
	    }
	  break;
	}
      abs_path = G_STRDUP(tmp_pos);
    }
  else
    abs_path = NULL;

#if 0
  fprintf(stderr, "ochusha_utils_2ch_check_url()\n");
  fprintf(stderr, "  url: %s\n", url);
  fprintf(stderr, "  server: %s\n", server);
  fprintf(stderr, "  abs_path: %s\n", abs_path);
#endif

  if (server == NULL || abs_path == NULL || *abs_path != '/')
    goto done;

  path_pos = abs_path + 1;
  if (strncmp(path_pos, "test/read.cgi", 13) == 0)
    {
      bbs_type = OCHUSHA_BBS_TYPE_2CH;
      if (path_pos[13] == '/')
	{
	  char *slash_pos = strchr(path_pos + 14, '/');
	  if (slash_pos != NULL)
	    {
	      char tmp_char = slash_pos[1];
	      slash_pos[1] = '\0';
	      if (snprintf(board_url, PATH_MAX, "http://%s%s", server,
			   path_pos + 13) < PATH_MAX)
		{
#if 0
		  fprintf(stderr, "  board_url: %s\n", board_url);
#endif

		  slash_pos[1] = tmp_char;
		  tmp_char = slash_pos[0];
		  slash_pos[0] = '\0';
		  board_id = G_STRDUP(path_pos + 14);
		  slash_pos[0] = tmp_char;

		  parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
		}
	      else
		board_url[0] = '\0';
	    }
	}
#if ENABLE_REGEXP
      else if (path_pos[13] == '?')
	{
	  if (parse_cgi_query(path_pos + 14, &board_id, &thread_id,
			      &from, &to))
	    {
	      if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server,
			   board_id) >= PATH_MAX)
		board_url[0] = '\0';
#if 0
	      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	    }
	}
#endif
    }
  else if (g_ascii_strcasecmp(server, "jbbs.shitaraba.com") == 0)
    {
      /* JBBSФ̯ʴ*/
      char *tmp_pos = strstr(path_pos, "bbs/read.cgi");
      bbs_type = OCHUSHA_BBS_TYPE_JBBS_SHITARABA;
      if (tmp_pos == NULL)
	{
	  /* URLȤƤ
	   * http://jbbs.shitaraba.com/computer/351/
	   * ߤʤΤ⤢롣
	   */
	  tmp_pos = strchr(path_pos, '/');
	  if (tmp_pos != NULL)
	    {
	      char *board_id_head = tmp_pos + 1;
	      tmp_pos = strchr(board_id_head, '/');
	      if (tmp_pos != NULL)
		{
		  char tmp_char = tmp_pos[1];
		  tmp_pos[1] = '\0';
		  if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server,
			       path_pos) < PATH_MAX)
		    board_id = G_STRDUP(board_id_head);
		  else
		    board_url[0] = '\0';
		  tmp_pos[1] = tmp_char;
		}
	    }
	}
      else
	{
	  /*
	   * URLȤƤϰʲ2ĤηΤʤ¾ˤ⤢뤫⡣
	   */
	  if (tmp_pos[12] == '/')
	    {
	      /*
	       * http://jbbs.shitaraba.com/bbs/read.cgi/computer/351/1040452877/109
	       * URL
	       */
	      char *slash_pos = strchr(tmp_pos + 1, '/');
	      if (slash_pos != NULL)
		{
		  char *board_id_head = slash_pos + 1;
		  slash_pos = strchr(board_id_head, '/'); /* 2ܤ'/'õ*/
		  if (slash_pos != NULL)
		    {
		      /* (tmp_pos + 12)slash_posޤǤʸ
		       * "/computer/351/"ʬˤʤ
		       */
		      *slash_pos = '\0';
		      if (snprintf(board_url, PATH_MAX, "http://%s%s/", server,
				   tmp_pos + 12) < PATH_MAX)
			{
			  board_id = G_STRDUP(board_id_head);
			  *slash_pos = '/';
			  parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
			}
		      else
			board_url[0] = '\0';
		    }
		}
	    }
#if ENABLE_REGEXP
	  else if (tmp_pos[12] == '?')
	    {
	      /*
	       * http://jbbs.shitaraba.com/computer/bbs/read.cgi?BBS=351&KEY=1040452877&START=109&END=109&NOFIRST=TRUE
	       * URL
	       */
	      if (parse_cgi_query(tmp_pos + 13, &board_id, &thread_id,
				  &from, &to))
		{
		  *tmp_pos = '\0';
		  if (snprintf(board_url, PATH_MAX, "http://%s%s%s/",
			       server, abs_path, board_id) >= PATH_MAX)
		    board_url[0] = '\0';
		}
	    }
#endif
	}
    }
#if ENABLE_REGEXP
  else if (strncmp(path_pos, "bbs/read.pl?", 12) == 0)
    {
      bbs_type = OCHUSHA_BBS_TYPE_MACHIBBS;
      if (parse_cgi_query(path_pos + 12, &board_id, &thread_id,
			  &from, &to))
	{
	  if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server, board_id)
	      >= PATH_MAX)
	    board_url[0] = '\0';
#if 0
	  fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	}
    }
  else if (strncmp(path_pos, "read.cgi?", 9) == 0)
    {
      bbs_type = OCHUSHA_BBS_TYPE_MITINOKU;
      if (parse_cgi_query(path_pos + 9, &board_id, &thread_id,
			  &from, &to))
	{
	  if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server, board_id)
	      >= PATH_MAX)
	    board_url[0] = '\0';
#if 0
	  fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	}
    }
#endif
  else if (strncmp(path_pos, "bbs/read.cgi", 12) == 0)
    {
      if (path_pos[12] == '/')
	{
	  char *slash_pos = strchr(path_pos + 13, '/');
	  if (slash_pos != NULL)
	    {
	      int len = slash_pos - (path_pos + 12) -1;
	      memcpy(tmp_buf, path_pos + 12, len);
	      tmp_buf[len] = '\0';
	      if (snprintf(board_url, PATH_MAX, "http://%s%s/", server,
			   tmp_buf) < PATH_MAX)
		{
		  board_id = G_STRDUP(tmp_buf);
		  parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
		  bbs_type = OCHUSHA_BBS_TYPE_JBBS;	/* ̤ƥ */
		}
	      else
		board_url[0] = '\0';
#if 0
	      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	    }
	}
#if ENABLE_REGEXP
      else if (path_pos[12] == '?')
	{
	  if (parse_cgi_query(path_pos + 13, &board_id, &thread_id,
			      &from, &to))
	    {
	      if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server,
			   board_id) < PATH_MAX)
		bbs_type = OCHUSHA_BBS_TYPE_JBBS;
	      else
		board_url[0] = '\0';
#if 0
	      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	    }
	}
#endif
    }
  else
    {
      /* 2chäݤġġ*/
      char *slash_pos = strchr(path_pos, '/');
      if (slash_pos != NULL)
	{
	  slash_pos[0] = '\0';
	  snprintf(board_url, PATH_MAX, "http://%s%s", server, abs_path);
	  board_id = G_STRDUP(abs_path + 1);
	  slash_pos[0] = '/';
	  parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
	  bbs_type = OCHUSHA_BBS_TYPE_2CH;
	}
#if 0
      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
    }

 done:
  if (server != NULL)
    G_FREE(server);

  if (abs_path != NULL)
    G_FREE(abs_path);

  if (board_url[0] == '\0')
    {
      /*  */
      if (board_id != NULL)
	{
	  G_FREE(board_id);
	  board_id = NULL;
	}
      if (thread_id != NULL)
	{
	  G_FREE(thread_id);
	  thread_id = NULL;
	}
      bbs_type = OCHUSHA_BBS_TYPE_UNKNOWN;
    }

  if (board_url_p != NULL)
    {
      if (board_url[0] != '\0')
	*board_url_p = G_STRDUP(board_url);
      else
	*board_url_p = NULL;
    }

  if (bbs_type_p != NULL)
    *bbs_type_p = bbs_type;

  if (board_id_p != NULL)
    *board_id_p = board_id;
  else if (board_id != NULL)
    G_FREE(board_id);

  if (thread_id_p != NULL)
    *thread_id_p = thread_id;
  else if (thread_id != NULL)
    G_FREE(thread_id);

  if (from_p != NULL)
    *from_p = from;

  if (to_p != NULL)
    *to_p = to;

  return board_url[0] != '\0';
}


OchushaAsyncBuffer *
ochusha_utils_2ch_get_bbsmenu(OchushaNetworkBroker *broker)
{
  return ochusha_network_broker_read_from_url(broker,
				broker->config->bbsmenu_url, NULL,
				OCHUSHA_NETWORK_BROKER_CACHE_IGNORE);
}


static gboolean
is_2ch_board_url(char *url)
{
  if (strstr(url, ".html") != NULL)
    return FALSE;
  if (strstr(url, "del_2ch") != NULL)
    return FALSE;
  return TRUE;
}


static gboolean
extract_boards(OchushaBBSTable *table, OchushaBoardCategory *category,
	       iconv_t converter,
	       char *head, char *tail,
	       EachBoardCallback *each_board_callback,
	       BoardMovedCallback *board_moved_callback,
	       gpointer callback_data)
{
  /* ǽ(HTMLA)õ*/
  char *cur_pos = g_strstr_len(head, tail - head, "<A HREF=http");
  if (cur_pos == NULL)
    return FALSE;
  cur_pos += 8;	/* skip "<A HREF=" */

  while (cur_pos != NULL && cur_pos < tail)
    {
      char *tag_tail = memchr(cur_pos, '>', tail - cur_pos);
      char *url_tail;
      gchar *name = NULL;
      char *url = NULL;
      char *close_tag;
      OchushaBulletinBoard *board;
      OchushaBulletinBoard *board_by_name;
      gboolean is_machibbs = FALSE;

      if (tag_tail == NULL)
	return FALSE;	/* Ƥ롩 */

      url_tail = strpbrk(cur_pos, " \t\r\n>");
      /* tag_tail != NULLʤΤurl_tail != NULL*/

      close_tag = g_strstr_len(url_tail, tail - url_tail, "</A>");
      if (close_tag == NULL)
	return FALSE;	/* Ƥ롩 */

#if GUESS_2CH_BOARD_URL
      {
	/* 2chİʳؤΥ󥫡ˤTARGE°դƤ̣*/
	char *target = g_strstr_len(cur_pos, close_tag - cur_pos, "TARGET=");
	if (target != NULL)
	  {
	    /* ޤBBS⥵ݡȤͽ */
	    if (g_strstr_len(cur_pos, close_tag - cur_pos, "machi.to") != NULL)
	      is_machibbs = TRUE;
	    else
	      goto search_next;
	  }
      }
#endif
      name = convert_string(converter, cp932_to_utf8_helper,
			    tag_tail + 1, close_tag - tag_tail - 1);
      url = G_STRNDUP(cur_pos, url_tail - cur_pos);
#if GUESS_2CH_BOARD_URL
      if (!is_2ch_board_url(url))
	goto search_next;
#endif
      board = ochusha_bbs_table_lookup_board_by_url(table, url);
      board_by_name = ochusha_bbs_table_lookup_board_by_name(table, name);

      if (board == NULL)
	{
#if GUESS_2CH_BOARD_URL
	  const char *base_path;
#endif
	  if (name == NULL)
	    {
	      G_FREE(url);
	      return FALSE;	/* Out of memory */
	    }

	  board = ochusha_board_2ch_new(name, url);
	  if (is_machibbs)
	    board->bbs_type = OCHUSHA_BBS_TYPE_MACHIBBS;

#if GUESS_2CH_BOARD_URL
	  base_path = ochusha_bulletin_board_get_base_path(board);
	  /* ФΥ롼ľηǼĤϸʤġġ*/
	  if (base_path[0] == '\0'
	      || (base_path[0] == '/' && base_path[1] == '\0'))
	    {
	      g_object_unref(G_OBJECT(board));
	      board = NULL;
	      goto search_next;
	    }
#endif

	  if (board_by_name != NULL && board_moved_callback != NULL)
	    {
	      OchushaBulletinBoard *winner
		= (*board_moved_callback)(board_by_name, board, callback_data);
	      /* XXX:
	       *   winnerNULLʤ()boardɲá
	       *   winnerboard_by_nameʤ()boardΤƤƼء
	       *   winnerboardʤ(Ť)board_by_nameΤƤơ
	       *   ()boardɲäפˡƥ꤫board_by_name
	       *   
	       */
	      if (winner == board_by_name)
		{
		  /* boardϥơ֥ɲäʤ */
		  g_object_unref(G_OBJECT(board));
		  board = NULL;
		  goto search_next;
		}

	      if (winner != NULL)
		{
		  /* (winner == board)ȸʤ */
		  ochusha_board_category_remove_board(category, board_by_name);
		  ochusha_bbs_table_remove_board(table, board_by_name);
		}
	    }

	  if (board != NULL)
	    {
	      if (each_board_callback != NULL)
		{
		  gboolean result = (*each_board_callback)(board,
							   callback_data);
		  if (!result)
		    {
		      g_object_unref(G_OBJECT(board));
		      return FALSE;
		    }
		}
	      ochusha_bbs_table_add_board(table, board);
	      g_object_unref(G_OBJECT(board));
	    }
	}

      board->killed = FALSE;
      ochusha_board_category_add_board(category, board);

    search_next:
      if (url != NULL)
	G_FREE(url);
      if (name != NULL)
	G_FREE(name);
      /* (HTMLA)õ*/
      cur_pos = g_strstr_len(close_tag + 4, tail - close_tag, "<A HREF=http");
      if (cur_pos == NULL)
	return TRUE;
      cur_pos += 8;	/* skip "<A HREF=" */
    }
  return TRUE;
}


/*
 * ochusha_utils_2ch_analyze_bbsmenu
 *
 * Ϳ줿bufferbbsmenu.htmlǤȸʤƲϤ˴ޤޤ
 * ƥƥбOchushaBoardCategoryȳĤб
 * OchushaBulletinBoard(Υ֥饹Υ֥OchushaBoard2ch)ۤ
 * Ϳ줿OchushaBBSTable򹹿롣
 *
 * ޤ˥ХåؿͿ줿硢ƥƥꡢĤˤĤ
 * б륳ХåؿƤ֡
 * ХåؿFALSE֤硢ǲϤλ롣
 *
 * OchushaBBSTableURLOchushaBulletinBoardʤäˤؤ餺
 * ̾Ǥϰ褦ʾ硢İžεǻʤΤǤ˴ؤƤ⥳
 * Хåؿ(ͿƤ)Ƥ֡ХåؿϰȤƥݥ
 * Ϳ뿷ĤOchushaBulletinBoard⡢OchushaBBSTable˻Ĥ
 * ؤΥݥ󥿤֤ޤNULL֤ˤϿξĤ롣
 *
 * Ϥ˽λTRUE֤
 */
gboolean
ochusha_utils_2ch_analyze_bbsmenu(OchushaBBSTable *table,
				  OchushaAsyncBuffer *buffer,
				  EachCategoryCallback *category_callback,
				  EachBoardCallback *board_callback,
				  BoardMovedCallback *moved_callback,
				  gpointer callback_data)
{
  gboolean result = TRUE;
  iconv_t converter;
  OchushaNetworkBrokerBufferStatus *status;

  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table)
		       && OCHUSHA_IS_ASYNC_BUFFER(buffer), FALSE);

  status = g_object_get_data(G_OBJECT(buffer),
			     "OchushaNetworkBroker::BufferStatus");

  g_return_val_if_fail(status != NULL, FALSE);

  converter = iconv_open("UTF-8//IGNORE", "CP932");  /* ϡɥǥ */
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  if (!ochusha_async_buffer_active_ref(buffer, "ochusha_utils_2ch.c: ochusha_utils_2ch_analyze_bbsmenu"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      iconv_close(converter);
      return FALSE;
    }

  ochusha_async_buffer_lock(buffer);
  {
    unsigned int offset = 0;

    while (result)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	unsigned int length = buffer->length;
	unsigned int rest_of_data = length - offset;

	while (rest_of_data > 0)
	  {
	    char *end_name_pos;
	    gchar *category_name;
	    char *end_category_pos;
	    OchushaBoardCategory *category;

	    cur_pos = g_strstr_len(cur_pos, rest_of_data, "<B>");

	    if (cur_pos == NULL)
	      break;	/* ǡ­ʤ⤦ƥ꤬ʤ */
	    cur_pos += 3;	/* skip "<B>" */
	    rest_of_data -= 3;

	    end_name_pos = g_strstr_len(cur_pos, rest_of_data, "</B>");

	    if (end_name_pos == NULL)
	      break;	/* ǡ­ʤ */

	    /* ߤΥƥϼΥƥľޤ */
	    end_category_pos = g_strstr_len(end_name_pos + 4,
					    rest_of_data - (end_name_pos
							    - cur_pos),
					    "<B>");
	    /* ǸΥƥ</BODY>ľޤ */
	    if (end_category_pos == NULL)
	      end_category_pos = g_strstr_len(end_name_pos + 4,
					      rest_of_data - (end_name_pos
							      - cur_pos),
					      "</BODY>");
	    if (end_category_pos == NULL)
	      break;

	    category_name = convert_string(converter, cp932_to_utf8_helper,
					   cur_pos, end_name_pos - cur_pos);

	    category = ochusha_bbs_table_lookup_category(table, category_name);
	    if (category == NULL)
	      category = ochusha_board_category_new(category_name);
	    G_FREE(category_name);

	    cur_pos = end_name_pos + 4;	/* skip "</B>" */
	    rest_of_data -= 4;
	      
	    result = extract_boards(table, category, converter,
				    cur_pos, end_category_pos,
				    board_callback, moved_callback,
				    callback_data);

	    if (!result || category->board_list == NULL)
	      g_object_unref(G_OBJECT(category));
	    else
	      {
		category->killed = FALSE;
		ochusha_bbs_table_add_category(table, category);
		if (category_callback != NULL
		    && !(*category_callback)(category, callback_data))
		  {
		    result = FALSE;
		    break;
		  }
	      }

	    cur_pos = end_category_pos;
	    offset = cur_pos - buffer_top;
	    rest_of_data = (length - offset);
	  }

	if (buffer->fixed)
	  break;

	if (!ochusha_async_buffer_wait(buffer, "ochusha_utils_2ch.c: ochusha_utils_2ch_analyze_bbsmenu"))
	  {
	    /* bufferϤ޲٤callerˤޤ */
#if DEBUG_ASYNC_BUFFER_MOST
	    fprintf(stderr, "ochusha_utils_2ch_analyze_bbsmenu(): buffer has been terminated.\n");
#endif
	    result = FALSE;
	    break;
	  }

	if (status->state
	    == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
	  {
	    result = FALSE;
	    break;
	  }
      }
  }
  ochusha_async_buffer_unlock(buffer);

  ochusha_async_buffer_active_unref(buffer, "ochusha_utils_2ch.c: ochusha_utils_2ch_analyze_bbs_menu");

  iconv_close(converter);

  return result;
}


OchushaUtils2chPostResult
ochusha_utils_2ch_try_post(OchushaBBSThread *thread, const char *message)
{
  char uri[PATH_MAX];
  ghttp_request *request;
  const char *cookie;
  const char *new_cookie = NULL;
  int status_code;
  OchushaBoard2ch *board_2ch;
  gboolean post_error = FALSE;

  g_return_val_if_fail(g_snprintf(uri, PATH_MAX, "http://%s/test/bbs.cgi",
				  thread->board->server) < PATH_MAX,
		       OCHUSHA_UTILS_2CH_POST_FAILURE);
  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread),
		       OCHUSHA_UTILS_2CH_POST_FAILURE);
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(thread->board),
		       OCHUSHA_UTILS_2CH_POST_FAILURE);
  board_2ch = OCHUSHA_BOARD_2CH(thread->board);

  request = ghttp_request_new();

  ghttp_set_uri(request, uri);
  ghttp_set_type(request, ghttp_type_post);

  ghttp_set_header(request, http_hdr_Host, thread->board->server);
  ghttp_set_header(request, http_hdr_User_Agent, OCHUSHA_USER_AGENT);

  /* ɬפʤΤ */
  ghttp_set_header(request, http_hdr_Accept, "text/xml,application/xml,application/xhtml+xml,text/html;q=0.9,text/plain;q=0.8,video/x-mng,image/png,image/jpeg,image/gif;q=0.2,text/css,*/*;q=0.1");
  ghttp_set_header(request, http_hdr_Accept_Language, "ja,en-us;q=0.66,en;q=0.33");
  ghttp_set_header(request, http_hdr_Accept_Charset, "EUC-JP,utf-8;q=0.66,*;q=0.66");
  ghttp_set_header(request, http_hdr_Connection, "close");
  ghttp_set_header(request, http_hdr_Referrer,
		   ochusha_bbs_thread_get_url_to_post_response(thread));
  ghttp_set_header(request, "Referer",
		   ochusha_bbs_thread_get_url_to_post_response(thread));

  cookie = ochusha_board_2ch_get_cookie(board_2ch);
  if (cookie != NULL)
    {
#if DEBUG_POST_MOST
      fprintf(stderr, "Cookie: %s\n", cookie);
#endif
      ghttp_set_header(request, "Cookie", cookie);
    }

  ghttp_set_body(request, (char *)message, strlen(message));
#if DEBUG_POST_MOST
  fprintf(stderr, "Post-Body: %s\n", message);
#endif

  ghttp_prepare(request);
  if (ghttp_process(request) == ghttp_error)
    {
#if DEBUG_POST
      fprintf(stderr, "ghttp_process() returns ghttp_error\n");
#endif
      ghttp_request_destroy(request);
      ochusha_board_2ch_set_cookie(board_2ch, NULL);
    }

  status_code = ghttp_status_code(request);

  if (status_code == 200)
    {
      const char *body = ghttp_get_body(request);
      int len = ghttp_get_body_len(request);
      const char *x_2ch = g_strstr_len(body, len, "2ch_X:");
#if DEBUG_POST_MOST
      char *native_body = g_convert(body, len, "EUC-JP", "CP932",
				    NULL, NULL, NULL);
      fprintf(stderr, "Set-Cookie: %s\n",
	      ghttp_get_header(request, http_hdr_Set_Cookie));
      fprintf(stderr, "Result-Body: %s\n", native_body);
      g_free(native_body);
#endif

      if (x_2ch != NULL)
	{
	  if (g_strstr_len(x_2ch, len, "2ch_X:cookie") != NULL)
	    {
	      new_cookie = ghttp_get_header(request, http_hdr_Set_Cookie);
	      if (new_cookie != NULL && *new_cookie != '\0')
		{
		  char *pos = strchr(new_cookie, ';');
		  if (pos != NULL)
		    {
		      char *tmp_cookie = g_strndup(new_cookie,
						   pos - new_cookie);
		      ochusha_board_2ch_set_cookie(board_2ch, tmp_cookie);
		      g_free(tmp_cookie);
		    }
		}
	    }
	  else
	    {
#if DEBUG_POST
	      char *native_body = g_convert(body, len, "EUC-JP", "CP932",
					    NULL, NULL, NULL);
	      fprintf(stderr, "Set-Cookie: %s\n",
		      ghttp_get_header(request, http_hdr_Set_Cookie));
	      fprintf(stderr, "Result-Body: %s\n", native_body);
	      g_free(native_body);
#endif
	      post_error = TRUE;
	    }
	}
    }
#if DEBUG_POST
  else
    {
      const char *body = ghttp_get_body(request);
      int len = ghttp_get_body_len(request);
      char *native_body = g_convert(body, len, "EUC-JP", "CP932",
				    NULL, NULL, NULL);
      fprintf(stderr, "status_code = %d\n", status_code);
      fprintf(stderr, "Set-Cookie: %s\n",
	      ghttp_get_header(request, http_hdr_Set_Cookie));
      fprintf(stderr, "Result-Body: %s\n", native_body);
      g_free(native_body);
    }
#endif

  ghttp_request_destroy(request);

  if (new_cookie != NULL)
    return OCHUSHA_UTILS_2CH_POST_NO_COOKIE;

  return post_error ? OCHUSHA_UTILS_2CH_POST_FAILURE : OCHUSHA_UTILS_2CH_POST_SUCCESS;
}
