/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: virtual_board.c,v 1.9 2004/01/13 01:23:08 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_bbs_table.h"
#include "ochusha_bulletin_board.h"
#include "ochusha_network_broker.h"

#include "utils.h"

#include "ochusha_ui.h"
#include "ochusha_ui_private.h"
#include "bulletin_board_ui.h"
#include "thread_proxy.h"
#include "virtual_board.h"

#include <glib-object.h>
#include <glib.h>

#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include <zlib.h>


static OchushaBulletinBoardClass *parent_class = NULL;
static GQuark broker_id;
static GQuark cache_mode_id;


static void virtual_board_class_init(VirtualBoardClass *klass);
static void virtual_board_init(VirtualBoard *board);
static void virtual_board_finalize(GObject *object);
static char *virtual_board_generate_base_path(OchushaBulletinBoard *board,
					      const char *url);
static char *virtual_board_generate_board_id(OchushaBulletinBoard *board,
					     const char *url);
static OchushaBBSThread *virtual_board_bbs_thread_new(
					OchushaBulletinBoard *board,
					const char *id, const gchar *title);
static OchushaAsyncBuffer *virtual_board_get_threadlist_source(
					OchushaBulletinBoard *board,
					OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					OchushaNetworkBrokerCacheMode mode);
static gboolean virtual_board_refresh_threadlist(OchushaBulletinBoard *board,
						 OchushaAsyncBuffer *buffer,
						 EachThreadCallback *cb,
						 gpointer callback_data);


GType
virtual_board_get_type(void)
{
  static GType virtual_board_type = 0;

  if (virtual_board_type == 0)
    {
      static const GTypeInfo virtual_board_info =
	{
	  sizeof(VirtualBoardClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)virtual_board_class_init,
	  NULL,	/* class_finalize */
	  NULL, /* class_data */
	  sizeof(VirtualBoard),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)virtual_board_init,
	};

      virtual_board_type = g_type_register_static(OCHUSHA_TYPE_BULLETIN_BOARD,
						  "VirtualBoard",
						  &virtual_board_info, 0);
    }

  return virtual_board_type;
}


static void
virtual_board_class_init(VirtualBoardClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);
  OchushaBulletinBoardClass *b_class = OCHUSHA_BULLETIN_BOARD_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = virtual_board_finalize;

  b_class->generate_base_path = virtual_board_generate_base_path;
  b_class->generate_board_id = virtual_board_generate_board_id;
#if 0
  b_class->lookup_thread_by_url = virtual_board_lookup_thread_by_url;
  b_class->lookup_kako_thread_by_url
    = virtual_board_lookup_kako_thread_by_url;
#endif

  b_class->thread_new = virtual_board_bbs_thread_new;
  b_class->get_threadlist_source = virtual_board_get_threadlist_source;
  b_class->refresh_threadlist = virtual_board_refresh_threadlist;

#if 0
  b_class->get_response_character_encoding
    = virtual_board_get_response_character_encoding;
  b_class->get_response_iconv_helper
    = virtual_board_get_response_iconv_helper;
#endif

  broker_id = g_quark_from_static_string("VirtualBoard::Broker");
  cache_mode_id = g_quark_from_static_string("VirtualBoard::CacheMode");
}


static void
virtual_board_init(VirtualBoard *board)
{
  OCHUSHA_BULLETIN_BOARD(board)->bbs_type = OCHUSHA_BBS_TYPE_VIRTUAL_BOARD;
}


static void
virtual_board_finalize(GObject *object)
{
  /* ΤȤoverrideƤ̣ʤ */
  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


OchushaBulletinBoard *
virtual_board_new(const gchar *name, const char *url,
		  OchushaApplication *application)
{
  VirtualBoard *virtual_board = VIRTUAL_BOARD(g_object_new(VIRTUAL_BOARD_TYPE,
							   "name", name,
							   "base_url", url,
							   NULL));
  OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(virtual_board);
  virtual_board->application = application;

  ensure_bulletin_board_info(board, application);

  return board;
}


static void
update_board_list(gpointer list_data, gpointer user_data)
{
  OchushaBBSThread *thread = (OchushaBBSThread *)list_data;
  VirtualBoard *virtual_board = (VirtualBoard *)user_data;
  OchushaBulletinBoard *board = ochusha_bbs_thread_get_board(thread);

  if (board != NULL)
    {
      if (g_slist_find(virtual_board->board_list, board) == NULL)
	virtual_board->board_list
	  = g_slist_append(virtual_board->board_list, board);
    }
}


void
virtual_board_remove_thread(VirtualBoard *virtual_board,
			    OchushaBBSThread *thread)
{
  OchushaBulletinBoard *board = (OchushaBulletinBoard *)virtual_board;
  g_return_if_fail(IS_VIRTUAL_BOARD(virtual_board) && IS_THREAD_PROXY(thread));

  if (g_hash_table_remove(board->thread_table, thread->id))
    {
      board->thread_list = g_slist_remove(board->thread_list, thread);
      virtual_board->recent_thread_list
	= g_slist_remove(virtual_board->recent_thread_list, thread);
      g_slist_free(virtual_board->board_list);
      virtual_board->board_list = NULL;
      g_slist_foreach(board->thread_list, update_board_list, virtual_board);
    }
}


static int
compare_by_id(OchushaBBSThread *a, OchushaBBSThread *b)
{
  OchushaBBSThread *real_a;
  OchushaBBSThread *real_b;
  char tmp_a[PATH_MAX];
  char tmp_b[PATH_MAX];
  int id_a;
  int id_b;
  int result;

  g_return_val_if_fail(IS_THREAD_PROXY(a) && IS_THREAD_PROXY(b), 0);
  real_a = THREAD_PROXY(a)->real_thread;
  real_b = THREAD_PROXY(b)->real_thread;

  if (real_a == NULL || real_b == NULL)
    return strcmp(ochusha_bbs_thread_get_url(a),
		  ochusha_bbs_thread_get_url(b));

  g_strlcpy(tmp_a, ochusha_bulletin_board_get_server(real_a->board), PATH_MAX);
  g_strlcpy(tmp_b, ochusha_bulletin_board_get_server(real_b->board), PATH_MAX);
  g_strreverse(tmp_a);
  g_strreverse(tmp_b);
  result = strcmp(tmp_a, tmp_b);
  if (result != 0)
    return result;

  snprintf(tmp_a, PATH_MAX, "%s%s",
	   real_a->board->base_path, real_a->board->id);
  snprintf(tmp_b, PATH_MAX, "%s%s",
	   real_b->board->base_path, real_b->board->id);
  result = strcmp(tmp_a, tmp_b);
  if (result != 0)
    return result;

  if (real_a->id != NULL && sscanf(real_a->id, "%d", &id_a) == 1
      && real_b->id != NULL && sscanf(real_b->id, "%d", &id_b) == 1)
    return id_a - id_b;

  return strcmp(ochusha_bbs_thread_get_url(a),
		ochusha_bbs_thread_get_url(b));
}


void
collect_unread_thread(OchushaBBSThread *thread, GSList **removed_list_p)
{
  if (ochusha_bbs_thread_get_number_of_responses_read(thread) == 0)
    *removed_list_p = g_slist_append(*removed_list_p, thread);
}


void
remove_thread(OchushaBBSThread *thread, VirtualBoard *virtual_board)
{
  virtual_board_remove_thread(virtual_board, thread);
}


void
virtual_board_clean_up_threadlist(VirtualBoard *virtual_board,
				  gboolean remove_unread_thread)
{
  OchushaBulletinBoard *board;
  BulletinBoardGUIInfo *info;
  g_return_if_fail(IS_VIRTUAL_BOARD(virtual_board));

  board = OCHUSHA_BULLETIN_BOARD(virtual_board);

  if (remove_unread_thread)
    {
      GSList *removed_list = NULL;
      g_slist_foreach(board->thread_list, (GFunc)collect_unread_thread,
		      &removed_list);
      g_slist_foreach(removed_list, (GFunc)remove_thread, virtual_board);
      g_slist_free(removed_list);
    }

  board->thread_list = g_slist_sort(board->thread_list,
				    (GCompareFunc)compare_by_id);
  info = ensure_bulletin_board_info(board, virtual_board->application);
  if (info->use_static_threadlist)
    {
      g_slist_free(virtual_board->recent_thread_list);
      virtual_board->recent_thread_list = g_slist_copy(board->thread_list);
    }
  else
    {
      g_slist_free(virtual_board->recent_thread_list);
      virtual_board->recent_thread_list = NULL;
    }
}


static char *
virtual_board_generate_base_path(OchushaBulletinBoard *board, const char *url)
{
  return G_STRDUP("");	/*  */
}


static char *
virtual_board_generate_board_id(OchushaBulletinBoard *board, const char *url)
{
  char *abs_path = ochusha_utils_url_extract_http_absolute_path(url);
  char *id;
  if (abs_path != NULL && abs_path[0] != '/')
    {
      G_FREE(abs_path);
      abs_path = NULL;
    }
  g_return_val_if_fail(abs_path != NULL, NULL);

  id = G_STRDUP(abs_path + 1);
  G_FREE(abs_path);

  return id;
}


static OchushaBBSThread *
virtual_board_bbs_thread_new(OchushaBulletinBoard *board, const char *id,
			     const gchar *title)
{
  VirtualBoard *virtual_board;
  OchushaBBSThread *thread;
  OchushaBulletinBoard *real_board = NULL;
  BulletinBoardGUIInfo *info;
  g_return_val_if_fail(IS_VIRTUAL_BOARD(board), NULL);
  g_return_val_if_fail(id != NULL && title != NULL, NULL);

  thread = thread_proxy_new(id, title);
  ochusha_bbs_thread_set_board(thread, board);

  virtual_board = VIRTUAL_BOARD(board);

#if 0
  fprintf(stderr, "ThreadProxy: %s\n", id);
#endif

  if (ochusha_bbs_table_check_url(virtual_board->application->table, id,
				  &real_board, NULL, NULL, NULL, NULL, NULL,
				  NULL))
    {
      if (real_board != NULL)
	{
	  OchushaBBSThread *real_thread;
	  if (g_slist_find(virtual_board->board_list, real_board) == NULL)
	    {
	      virtual_board->board_list
		= g_slist_append(virtual_board->board_list, real_board);
#if 0
	      fprintf(stderr, "board added: %s\n", real_board->base_url);
#endif
	      if (real_board->thread_list == NULL)
		ochusha_bulletin_board_read_threadlist_xml(real_board,
							   &virtual_board->application->config,
							   virtual_board->application->session_subdir,
							   TRUE);
	    }
	  real_thread = ochusha_bulletin_board_lookup_bbs_thread_by_url(real_board, id);
	  if (real_thread != NULL)
	    thread_proxy_set_real_thread(THREAD_PROXY(thread), real_thread);
	  else
	    fprintf(stderr, "Couldn't find real_thread by URL: %s\n", id);
	}
    }

  info = ensure_bulletin_board_info(board, virtual_board->application);
  if (info->use_static_threadlist)
    {
      virtual_board->recent_thread_list
	= g_slist_append(virtual_board->recent_thread_list, thread);
    }
  else if (virtual_board->recent_thread_list != NULL)
    {
      g_slist_free(virtual_board->recent_thread_list);
      virtual_board->recent_thread_list = NULL;
    }

  return thread;
}


static OchushaAsyncBuffer *
virtual_board_get_threadlist_source(OchushaBulletinBoard *board,
				    OchushaNetworkBroker *broker,
				    OchushaAsyncBuffer *buffer,
				    OchushaNetworkBrokerCacheMode mode)
{
  g_return_val_if_fail(IS_VIRTUAL_BOARD(board), NULL);

  /* ΤȤܾۤ */
  if (buffer == NULL || ochusha_async_buffer_reset(buffer))
    buffer = ochusha_async_buffer_new(NULL, 0, NULL);
  else
    ochusha_async_buffer_update_length(buffer, 0);
  g_object_set_qdata(G_OBJECT(buffer), broker_id, broker);
  g_object_set_qdata(G_OBJECT(buffer), cache_mode_id, (gpointer)mode);

  return buffer;
}


typedef struct _RefreshThreadlistArgs
{
  OchushaApplication *application;
  OchushaBulletinBoard *board;
  OchushaNetworkBroker *broker;
  OchushaAsyncBuffer *buffer;
  OchushaNetworkBrokerCacheMode mode;
  EachThreadCallback *each_thread_cb;
  gpointer callback_data;
  GSList *recent_list;
} RefreshThreadlistArgs;


static gboolean
internal_each_thread_cb(OchushaBBSThread *thread, gpointer user_data)
{
  RefreshThreadlistArgs *args = (RefreshThreadlistArgs *)user_data;
  ThreadProxy *proxy;
  OchushaBBSThread *proxy_thread
    = ochusha_bulletin_board_lookup_bbs_thread_by_id(args->board,
						     ochusha_bbs_thread_get_url(thread));
  if (proxy_thread == NULL)
    return TRUE;	/* ̵طʥ */

  g_return_val_if_fail(IS_THREAD_PROXY(proxy_thread), TRUE);
  proxy = THREAD_PROXY(proxy_thread);

  if (proxy->real_thread == NULL)
    {
      /* bookmark줿ĤξΤߤ */
      g_object_ref(G_OBJECT(thread));
      proxy->real_thread = thread;
    }

  if (args->each_thread_cb != NULL)
    (*args->each_thread_cb)(proxy_thread, args->callback_data);

  args->recent_list = g_slist_append(args->recent_list, proxy_thread);

  return TRUE;
}


static void
update_by_board(gpointer list_data, gpointer user_data)
{
  RefreshThreadlistArgs *args = (RefreshThreadlistArgs *)user_data;
  OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(list_data);
  OchushaAsyncBuffer *buffer
    = ochusha_bulletin_board_get_threadlist_source(board, args->broker,
						   args->buffer, args->mode);
  OchushaApplication *application = args->application;

  if (board->thread_list == NULL)
    ochusha_bulletin_board_read_threadlist_xml(board,
					       &application->config,
					       application->session_subdir,
					       TRUE);

  ochusha_bulletin_board_refresh_threadlist(OCHUSHA_BULLETIN_BOARD(list_data),
					    buffer,
					    internal_each_thread_cb,
					    user_data);
}


static gboolean
virtual_board_refresh_threadlist(OchushaBulletinBoard *board,
				 OchushaAsyncBuffer *buffer,
				 EachThreadCallback *each_thread_cb,
				 gpointer callback_data)
{
  VirtualBoard *virtual_board;
  RefreshThreadlistArgs args =
    {
      NULL,
      board,
      (OchushaNetworkBroker *)g_object_get_qdata(G_OBJECT(buffer),
						 broker_id),
      buffer,
      (OchushaNetworkBrokerCacheMode)g_object_get_qdata(G_OBJECT(buffer),
							cache_mode_id),
      each_thread_cb, callback_data,
      NULL,
    };
  BulletinBoardGUIInfo *info;

  g_return_val_if_fail(IS_VIRTUAL_BOARD(board), FALSE);

  virtual_board = VIRTUAL_BOARD(board);

  args.application = virtual_board->application;

  info = ensure_bulletin_board_info(board, virtual_board->application);

  if ((args.mode == OCHUSHA_NETWORK_BROKER_CACHE_ONLY
       || info->use_static_threadlist)
      && virtual_board->recent_thread_list != NULL)
    g_slist_foreach(virtual_board->recent_thread_list,
		    (GFunc)each_thread_cb, callback_data);
  else
    {
      g_slist_free(virtual_board->recent_thread_list);
      virtual_board->recent_thread_list = NULL;
      g_slist_foreach(virtual_board->board_list, update_by_board, &args);
      virtual_board->recent_thread_list = args.recent_list;
      /* ȤäĤget_threadlist_sourceޤǺѲǽ */
      g_object_set_qdata(G_OBJECT(buffer), cache_mode_id,
			 (gpointer)OCHUSHA_NETWORK_BROKER_CACHE_ONLY);
    }

  return TRUE;
}
