/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_async_buffer.c,v 1.22 2004/01/13 22:59:47 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"

#include "ochusha_async_buffer.h"
#include "marshal.h"

#include "worker.h"

#include <glib.h>

#include <errno.h>
#include <fcntl.h>

#include <pthread.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/mman.h>

#ifdef HAVE_NANOSLEEP
#include <time.h>
#else
#include <unistd.h>
#endif

#include <zlib.h>


#ifndef MAP_NOCORE
# define MAP_NOCORE	0
#endif


#define OCHUSHA_INITIAL_ASYNC_BUFFER_SIZE	4096


static void ochusha_async_buffer_class_init(OchushaAsyncBufferClass *klass);
static void ochusha_async_buffer_init(OchushaAsyncBuffer *buffer);
static void ochusha_async_buffer_finalize(GObject *object);


GType
ochusha_async_buffer_get_type(void)
{
  static GType async_buffer_type = 0;

  if (async_buffer_type == 0)
    {
      static const GTypeInfo async_buffer_info =
	{
	  sizeof(OchushaAsyncBufferClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_async_buffer_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaAsyncBuffer),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_async_buffer_init,
	};

      async_buffer_type = g_type_register_static(G_TYPE_OBJECT,
						 "OchushaAsyncBuffer",
						 &async_buffer_info, 0);
    }

  return async_buffer_type;
}


struct _BufferSyncObject
{
  pthread_mutex_t mutex;
  pthread_cond_t cond;
};


static BufferSyncObject *get_sync_object(void);

static GObjectClass *parent_class = NULL;
static pthread_mutex_t ochusha_async_buffer_global_lock;
static pthread_cond_t ochusha_async_buffer_global_condition;


/*
 * pthreadϢΥ֥ȤλȤΤƤƧߤʤΤǡѤ
 * ݤ롣
 */
static GSList *sync_object_list = NULL;

/*
 * 饤֤(newƤfinalizeޤǤδ֤)Ƥ
 * OchushaAsyncBufferݻ
 */
static GSList *ochusha_async_buffer_list = NULL;


#define LOCK_GLOBAL_LOCK						\
  if (pthread_mutex_lock(&ochusha_async_buffer_global_lock) != 0)	\
    {									\
      fprintf(stderr, "Couldn't lock a mutex.\n");			\
      abort();								\
    }

#define UNLOCK_GLOBAL_LOCK						\
  if (pthread_mutex_unlock(&ochusha_async_buffer_global_lock) != 0)	\
    {									\
      fprintf(stderr, "Couldn't unlock a mutex.\n");			\
      abort();								\
    }

#define WAIT_GLOBAL_CONDITION						\
  if (pthread_cond_wait(&ochusha_async_buffer_global_condition,		\
			&ochusha_async_buffer_global_lock) != 0)	\
    {									\
      fprintf(stderr, "Couldn't wait a condition.\n");			\
      abort();								\
    }

#define SIGNAL_GLOBAL_CONDITION						\
  if (pthread_cond_signal(&ochusha_async_buffer_global_condition) != 0)	\
    {									\
      fprintf(stderr, "Couldn't unlock a mutex.\n");			\
      abort();								\
    }

#define LOCK_BUFFER_LOCK(buffer)					\
  do									\
    {									\
      int result = pthread_mutex_trylock(&buffer->sync_object->mutex);	\
      if (result != 0)							\
	{								\
	  if (result == EBUSY)						\
	    {								\
	      LOCK_GLOBAL_LOCK;						\
	      buffer->number_of_lock_waiters++;				\
	      UNLOCK_GLOBAL_LOCK;					\
	      if (pthread_mutex_lock(&buffer->sync_object->mutex) != 0)	\
		{							\
		  fprintf(stderr, "Couldn't lock a mutex.\n");		\
		  abort();						\
		}							\
	      LOCK_GLOBAL_LOCK;						\
	      buffer->number_of_lock_waiters--;				\
	      UNLOCK_GLOBAL_LOCK;					\
	    }								\
	  else								\
	    {								\
	      fprintf(stderr, "Couldn't trylock a mutex.\n");		\
	      abort();							\
	    }								\
	}								\
    } while (0)
#if 0
#define LOCK_BUFFER_LOCK(buffer)				\
  if (pthread_mutex_lock(&buffer->sync_object->mutex) != 0)	\
    {								\
      fprintf(stderr, "Couldn't lock a mutex.\n");		\
      abort();							\
    }
#endif

#define UNLOCK_BUFFER_LOCK(buffer)				\
  if (pthread_mutex_unlock(&buffer->sync_object->mutex) != 0)	\
    {								\
      fprintf(stderr, "Couldn't unlock a mutex.\n");		\
      abort();							\
    }

#define WAIT_BUFFER_CONDITION(buffer)				\
  if (pthread_cond_wait(&buffer->sync_object->cond,		\
			&buffer->sync_object->mutex) != 0)	\
    {								\
      fprintf(stderr, "Couldn't wait a condition.\n");		\
      abort();							\
    }

#define SIGNAL_BUFFER_CONDITION(buffer)				\
  if (pthread_cond_signal(&buffer->sync_object->cond) != 0)	\
    {								\
      fprintf(stderr, "Couldn't siganl a condition.\n");	\
      abort();							\
    }

#define BROADCAST_BUFFER_CONDITION(buffer)			\
  if (pthread_cond_broadcast(&buffer->sync_object->cond) != 0)	\
    {								\
      fprintf(stderr, "Couldn't siganl a condition.\n");	\
      abort();							\
    }


enum {
  WAKEUP_NOW_SIGNAL,
  LAST_SIGNAL
};


static int async_buffer_signals[LAST_SIGNAL] = { 0, };


void
blind_awake(OchushaAsyncBuffer *buffer)
{
  LOCK_BUFFER_LOCK(buffer);

  if (buffer->state == OCHUSHA_ASYNC_BUFFER_OK
      && buffer->number_of_active_users > 0)
    {
#if 0
      fprintf(stderr, "blind awake on %p\n", buffer);
#endif
      BROADCAST_BUFFER_CONDITION(buffer);
    }

  UNLOCK_BUFFER_LOCK(buffer);

  g_object_unref(G_OBJECT(buffer));	/* ڥˤʤäƤϤ */
}


static GSList *to_be_signaled_buffer_list = NULL;


static void
iterate_buffers_to_be_signaled(OchushaAsyncBuffer *buffer)
{
  g_object_ref(G_OBJECT(buffer));	/* ڥˤʤäƤϤ */
  to_be_signaled_buffer_list = g_slist_append(to_be_signaled_buffer_list,
					      buffer);
}


static void
force_awake_job(WorkerThread *employee, void *unused)
{
#ifdef HAVE_NANOSLEEP
  struct timespec interval = { 0, 100000000 };	/* == 100ms */
#endif

  while (TRUE)
    {
      LOCK_GLOBAL_LOCK;
      to_be_signaled_buffer_list = NULL;
      g_slist_foreach(ochusha_async_buffer_list,
		      (GFunc)iterate_buffers_to_be_signaled, NULL);
      UNLOCK_GLOBAL_LOCK;

      g_slist_foreach(to_be_signaled_buffer_list, (GFunc)blind_awake, NULL);

      if (employee->command == DIE_NOW)
	break;
#ifdef HAVE_NANOSLEEP
      nanosleep(&interval, NULL);
#else
      sleep(1);		/* Ĺʤ뤬ʤ */
#endif
    }
}


static void
ochusha_async_buffer_class_init(OchushaAsyncBufferClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;
  WorkerJob *job = G_NEW0(WorkerJob, 1);

  if (pthread_mutex_init(&ochusha_async_buffer_global_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't initialize a mutex.\n");
      abort();
    }

  if (pthread_cond_init(&ochusha_async_buffer_global_condition, NULL) != 0)
    {
      fprintf(stderr, "Couldn't initialize a condvar.\n");
      abort();
    }

  parent_class = g_type_class_peek_parent(klass);

  async_buffer_signals[WAKEUP_NOW_SIGNAL] =
    g_signal_new("wakeup_now",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaAsyncBufferClass, wakeup_now),
		 NULL, NULL,
		 libochusha_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);

  o_class->finalize = ochusha_async_buffer_finalize;

  job->canceled = FALSE;
  job->job = force_awake_job;
  job->args = NULL;

  commit_job(job);
}


static void
ochusha_async_buffer_init(OchushaAsyncBuffer *buffer)
{
  buffer->fixed = FALSE;
  buffer->number_of_lock_waiters = 0;
}


static void
ochusha_async_buffer_finalize(GObject *object)
{
  OchushaAsyncBuffer *buffer = OCHUSHA_ASYNC_BUFFER(object);
#if DEBUG_OCHUSHA_MEMORY_USAGE
  fprintf(stderr, "ochusha_async_buffer_finalize: OchushaAsyncBuffer(%p)\n",
	  object);
#endif
  if (buffer->destructor != NULL)
    (*buffer->destructor)(buffer);

  LOCK_GLOBAL_LOCK;
  if (buffer->sync_object != NULL)
    {
      sync_object_list = g_slist_append(sync_object_list, buffer->sync_object);
      buffer->sync_object = NULL;
    }
  ochusha_async_buffer_list = g_slist_remove(ochusha_async_buffer_list, buffer);
  UNLOCK_GLOBAL_LOCK;

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static BufferSyncObject *
get_sync_object(void)
{
  GSList *recycled_sync_object;
  BufferSyncObject *sync_object;

  recycled_sync_object = sync_object_list;
  sync_object_list = g_slist_remove_link(sync_object_list,
					 recycled_sync_object);
  if (recycled_sync_object != NULL)
    {
      sync_object = (BufferSyncObject *)recycled_sync_object->data;
      g_slist_free_1(recycled_sync_object);
      return sync_object;
    }

  sync_object = (BufferSyncObject *)G_MALLOC(sizeof(BufferSyncObject));
  if (pthread_mutex_init(&sync_object->mutex, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }
  if (pthread_cond_init(&sync_object->cond, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a condition variable.\n");
      abort();
    }
  return sync_object;
}


OchushaAsyncBuffer *
ochusha_async_buffer_new(char *buffer, int length, DestructFunc *destructor)
{
  BufferSyncObject *sync_object;
  OchushaAsyncBuffer *buf
    = OCHUSHA_ASYNC_BUFFER(g_object_new(OCHUSHA_TYPE_ASYNC_BUFFER, NULL));

  LOCK_GLOBAL_LOCK
    {
      sync_object = get_sync_object();
      ochusha_async_buffer_list = g_slist_append(ochusha_async_buffer_list,
						 buf);
    }
  UNLOCK_GLOBAL_LOCK

  if (buffer == NULL && destructor == NULL)
    {
      if (length != 0)
	buffer = (char *)G_MALLOC(length);
      destructor = ochusha_async_buffer_free_when_finished;
      buf->length = 0;
    }
  else
    buf->length = length;
  
  buf->fixed = FALSE;
  buf->buffer = buffer;

  buf->buffer_length = length;	/* XXX: Ǥ餦٤ġġ*/

  buf->sync_object = sync_object;
  buf->destructor = destructor;

#if DEBUG_MEMORY_USAGE
  fprintf(stderr, "ochusha_async_buffer_new(): OchushaAsyncBuffer(%p)\n", buf);
#endif

  return buf;
}


OchushaAsyncBuffer *
ochusha_async_buffer_new_with_file(int fd)
{
  BufferSyncObject *sync_object;
  OchushaAsyncBuffer *buf
    = OCHUSHA_ASYNC_BUFFER(g_object_new(OCHUSHA_TYPE_ASYNC_BUFFER, NULL));
  gzFile gzfile;
  int chunk_size;

  LOCK_GLOBAL_LOCK
    {
      sync_object = get_sync_object();
      ochusha_async_buffer_list = g_slist_append(ochusha_async_buffer_list,
						 buf);
    }
  UNLOCK_GLOBAL_LOCK

  buf->fixed = FALSE;

  buf->sync_object = sync_object;
  buf->destructor = ochusha_async_buffer_free_when_finished;

  buf->length = 0;
  buf->buffer = 0;
  buf->buffer_length = 0;

  gzfile = gzdopen(fd, "rb");
  if (gzfile == NULL)
    return buf;

  chunk_size = OCHUSHA_INITIAL_ASYNC_BUFFER_SIZE;

  while (!gzeof(gzfile))
    {
      char *buffer;
      int result;
      if (!ochusha_async_buffer_ensure_free_space(buf, chunk_size))
	abort();	/* why? */

      buffer = (char *)buf->buffer + buf->length;
      chunk_size = buf->buffer_length - buf->length;
      result = gzread(gzfile, buffer, chunk_size);
      if (result == -1)
	{
	  fprintf(stderr, "gzread cause some error\n");
	  break;
	}

#if 0
      fprintf(stderr, "0x%x bytes read\n", result);

      if (result == 0)
	fprintf(stderr, "EOF\n");
      else if (result < chunk_size)
	fprintf(stderr, "Close to EOF\n");
#endif

      buf->length += result;
    }

  gzclose(gzfile);

#if DEBUG_MEMORY_USAGE
  fprintf(stderr, "ochusha_async_buffer_new(): OchushaAsyncBuffer(%p)\n", buf);
#endif

  return buf;
}


static void
munmap_when_finished(OchushaAsyncBuffer *buffer)
{
  munmap((void *)buffer->buffer, buffer->length);
#if DEBUG_MMAP
  fprintf(stderr, "mmapped buffer(at %p) is unmapped.\n",
	  buffer->buffer);
#endif
}


OchushaAsyncBuffer *
ochusha_async_buffer_new_with_file_mmap(int fd)
{
  BufferSyncObject *sync_object;
  char *buffer;
  OchushaAsyncBuffer *buf
    = OCHUSHA_ASYNC_BUFFER(g_object_new(OCHUSHA_TYPE_ASYNC_BUFFER, NULL));
  off_t len;

  LOCK_GLOBAL_LOCK
  {
    sync_object = get_sync_object();
    ochusha_async_buffer_list = g_slist_append(ochusha_async_buffer_list,
					       buf);
  }
  UNLOCK_GLOBAL_LOCK

  len = lseek(fd, 0, SEEK_END);
  buffer = mmap(NULL, len, PROT_READ, MAP_NOCORE | MAP_PRIVATE, fd, 0);
  if (buffer == MAP_FAILED)
    {
      fprintf(stderr, "mmap failed due to: %s (%d)\n", strerror(errno), errno);
      buf->buffer = NULL;
      buf->length = 0;
      buf->buffer_length = 0;
      buf->destructor = NULL;
    }
  else
    {
      buf->buffer = buffer;
      buf->length = len;
      buf->buffer_length = len;
      buf->destructor = munmap_when_finished;
    }
  close(fd);
  
  buf->fixed = FALSE;
  buf->sync_object = sync_object;

#if DEBUG_MEMORY_USAGE
  fprintf(stderr, "ochusha_async_buffer_new_with_file_mmap(): OchushaAsyncBuffer(%p)\n", buf);
#endif

  return buf;
}


gboolean
ochusha_async_buffer_reset(OchushaAsyncBuffer *buffer)
{
  g_return_val_if_fail(OCHUSHA_IS_ASYNC_BUFFER(buffer), FALSE);
  /* g_return_val_if_fail(buffer->number_of_active_users == 0, FALSE); */
  buffer->fixed = FALSE;
  buffer->state = OCHUSHA_ASYNC_BUFFER_OK;
  return TRUE;
}


/* ƱߴϢ */
/*
 * MEMO: OchushaAsyncBufferȤäƤƤΥåɤ⤷Υå
 *       Фᥤ󥹥åɤƱŪ˳ߤ򤫤ȻפäΤǡ
 *       ʲAPIѰդۤɰΤΤǤʤġġ
 *       pthreadUNIXʥ뤬ꤷƻȤФȤ⤢Τ
 *       pthreadUNIXʥȹ碌ϵʤΤ򤱤롣
 *
 * active user: OchushaAsyncBufferproducer⤷consumeråɡ
 *              ХåեݻƤΥƥȤϴޤޤʤ
 *              active userϥХåե˿ˡ
 *		ochusha_async_buffer_active_ref()ƤӽФ⤦Хåե
 *		ʤȤochusha_async_buffer_active_unref()
 *		ӽФȤˤ롣
 *
 *              δؿg_object_{ref,unref}Ƥǡ
 *              number_of_active_usersĴ롣
 *
 *              active userϻsuspend/resume/terminate׵᤬Ƥ
 *              ʤɤǧ뤳Ȥ롣
 */
gboolean
ochusha_async_buffer_active_ref(OchushaAsyncBuffer *buffer)
{
  gboolean result;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "OchushaAsyncBuffer(%p) is actively refered\n", buffer);
#endif
  g_object_ref(G_OBJECT(buffer));	/* Ĵ٤Ƥʤ */

  LOCK_BUFFER_LOCK(buffer);

  buffer->number_of_active_users++;
  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);
    }

  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
  if (!result)
    {
      buffer->number_of_active_users--;
      g_object_unref(G_OBJECT(buffer));	/* Ĵ٤Ƥʤ */
    }

  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


void
ochusha_async_buffer_active_unref(OchushaAsyncBuffer *buffer)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "OchushaAsyncBuffer(%p) actively refered is unrefered\n",
	  buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);

  buffer->number_of_active_users--;

  if (buffer->state != OCHUSHA_ASYNC_BUFFER_OK)
    {
      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;
    }

  UNLOCK_BUFFER_LOCK(buffer);

  g_object_unref(G_OBJECT(buffer));	/* Ĵ٤Ƥʤ */
}


void
ochusha_async_buffer_suspend_all(void)
{
  g_slist_foreach(ochusha_async_buffer_list,
		  (GFunc)ochusha_async_buffer_suspend, NULL);
}


void
ochusha_async_buffer_resume_all(void)
{
  g_slist_foreach(ochusha_async_buffer_list,
		  (GFunc)ochusha_async_buffer_resume, NULL);
}


void
ochusha_async_buffer_terminate_all(void)
{
  g_slist_foreach(ochusha_async_buffer_list,
		  (GFunc)ochusha_async_buffer_terminate, NULL);
}


void
ochusha_async_buffer_suspend(OchushaAsyncBuffer *buffer)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "suspend threads refering OchushaAsyncBuffer(%p)\n",
	  buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);
  buffer->number_of_suspended_users = 0;
  buffer->state = OCHUSHA_ASYNC_BUFFER_SUSPENDED;
  BROADCAST_BUFFER_CONDITION(buffer);
  LOCK_GLOBAL_LOCK;

  g_signal_emit(G_OBJECT(buffer),
		async_buffer_signals[WAKEUP_NOW_SIGNAL],
		0);

  UNLOCK_BUFFER_LOCK(buffer);		/* νǤʤȤʤʤ
					 * UNLOCK_BUFFER_LOCKִ֤
					 * ΥåԤäƤåɤ
					 * ưϤ롣åɤξ
					 * åSUSPENDED֤Ǥ
					 * ȤΤΤǡSUSPENDED֤
					 * ֲ򤱤ΤԤġwait롣
					 * BROADCASTϿSUSPENDED֤
					 * ʤäȤΤʤåɤ
					 * 뤿ΤΤǤ
					 * ա
					 */

  while (buffer->number_of_suspended_users < buffer->number_of_active_users)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "ochusha_async_buffer_suspend: suspended=%d, active_users=%d\n",
	      buffer->number_of_suspended_users,
	      buffer->number_of_active_users);
#endif
      WAIT_GLOBAL_CONDITION;
    }
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_suspend: suspended=%d, active_users=%d\n",
	  buffer->number_of_suspended_users,
	  buffer->number_of_active_users);
#endif

  UNLOCK_GLOBAL_LOCK;
}


void
ochusha_async_buffer_resume(OchushaAsyncBuffer *buffer)
{
  LOCK_BUFFER_LOCK(buffer);
  buffer->state = OCHUSHA_ASYNC_BUFFER_OK;
  BROADCAST_BUFFER_CONDITION(buffer);
  UNLOCK_BUFFER_LOCK(buffer);
}


void
ochusha_async_buffer_terminate(OchushaAsyncBuffer *buffer)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_terminate: buffer=%p\n", buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);
  buffer->state = OCHUSHA_ASYNC_BUFFER_TERMINATED;
  BROADCAST_BUFFER_CONDITION(buffer);

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_terminate: buffer condition broadcast done\n");
#endif

  LOCK_GLOBAL_LOCK;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_terminate: global lock locked\n");
#endif

  g_signal_emit(G_OBJECT(buffer),
		async_buffer_signals[WAKEUP_NOW_SIGNAL],
		0);

  UNLOCK_BUFFER_LOCK(buffer);

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_terminate: buffer lock unlocked\n");
#endif

  while (buffer->number_of_active_users > 0)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "ochusha_async_buffer_terminate: active_users=%d\n",
	      buffer->number_of_active_users);
#endif
      WAIT_GLOBAL_CONDITION;
    }
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_terminate: active_users=%d\n",
	  buffer->number_of_active_users);
#endif

  UNLOCK_GLOBAL_LOCK;
}


/*
 * ХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_update_length(OchushaAsyncBuffer *buffer, int length)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_update_length called\n");
#endif
  LOCK_BUFFER_LOCK(buffer);
  buffer->length = length;
  UNLOCK_BUFFER_LOCK(buffer);

  return ochusha_async_buffer_broadcast(buffer);
}


/*
 * ХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_fix(OchushaAsyncBuffer *buffer)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_fix called\n");
#endif

  LOCK_BUFFER_LOCK(buffer);
  buffer->fixed = TRUE;
  UNLOCK_BUFFER_LOCK(buffer);

  return ochusha_async_buffer_broadcast(buffer);
}


/*
 * Out of memory⤷ϥХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_resize(OchushaAsyncBuffer *buffer, int length)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_resize called\n");
#endif

  LOCK_BUFFER_LOCK(buffer);

  if (buffer->fixed)
    {
      fprintf(stderr, "Invalid use of ochusha_async_buffer: Fixed buffer isn't resizable.\n");
      abort();
    }

  buffer->buffer = (char *)G_REALLOC((void *)buffer->buffer, length);
  buffer->buffer_length = length;

  UNLOCK_BUFFER_LOCK(buffer);

  return ochusha_async_buffer_broadcast(buffer);
}


/*
 * ochusha_async_buffer_ensure_free_space:
 *
 * ХåեlengthʾĹĤȤݾڤ롣
 *
 * ХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_ensure_free_space(OchushaAsyncBuffer *buffer, int length)
{
  gboolean result = TRUE;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_ensure_free_space(%p, %d) called.\n",
	  buffer, length);
#endif

  LOCK_BUFFER_LOCK(buffer);
  if (buffer->fixed)
    {
      fprintf(stderr,
	      "ochusha_async_buffer_ensure_free_space(): buffer is fixed.\n");
      abort();
    }

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "buffer(%p) is suspended at ochusha_async_buffer_free_space.\n",
	      buffer);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);
      /* λbuffer->state != OCHUSHA_ASYNC_BUFFER_SUSPENDEDʤ
       * ʥsuspend򤯤˥ᥤ󥹥åɤäƤΡ
       *
       * buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDEDʤ顢waitƤ֤
       * ⤦ٿSUSPENDED֤ˤʤäȤȡ
       */
    }

  while (buffer->buffer_length - buffer->length < length)
    {
      size_t new_buf_len
	= (buffer->buffer_length > 0)
	? (buffer->buffer_length * 2) : OCHUSHA_INITIAL_ASYNC_BUFFER_SIZE;
      buffer->buffer = (char *)G_REALLOC((void *)buffer->buffer, new_buf_len);
      buffer->buffer_length = new_buf_len;
    }
  result = (buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED);
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_append_data(OchushaAsyncBuffer *buffer, const char *data,
				 int length)
{
  gboolean result = TRUE;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_append_data called.\n");
#endif

  if (ochusha_async_buffer_ensure_free_space(buffer, length))
    {
      memcpy((char *)buffer->buffer + buffer->length, data, length);
      buffer->length += length;
    }
  else
    result = FALSE;

  return ochusha_async_buffer_broadcast(buffer) && result;
}


#define MAXIMUM_CHUNK_SIZE	65536


gboolean
ochusha_async_buffer_read_file(OchushaAsyncBuffer *buffer, int fd)
{
  gzFile gzfile;
  int chunk_size;
  int result;

  g_return_val_if_fail(fd >= 0, FALSE);

  gzfile = gzdopen(fd, "rb");
  g_return_val_if_fail(gzfile != NULL, FALSE);

  chunk_size = OCHUSHA_INITIAL_ASYNC_BUFFER_SIZE;
  result = 0;

  while (!gzeof(gzfile))
    {
      char *buf;
      if (!ochusha_async_buffer_ensure_free_space(buffer, chunk_size))
	{
	  result = -1;
	  break;
	}

      buf = (char *)buffer->buffer + buffer->length;
      chunk_size = MIN(buffer->buffer_length - buffer->length,
		       MAXIMUM_CHUNK_SIZE);
      result = gzread(gzfile, buf, chunk_size);
      if (result == -1)
	{
	  fprintf(stderr, "gzread cause some error\n");
	  break;
	}

      if (!ochusha_async_buffer_update_length(buffer, buffer->length + result))
	{
	  result = -1;
	  break;
	}
    }

  gzclose(gzfile);

  return result >= 0;
}


void
ochusha_async_buffer_lock(OchushaAsyncBuffer *buffer)
{
  LOCK_BUFFER_LOCK(buffer);
}


void
ochusha_async_buffer_unlock(OchushaAsyncBuffer *buffer)
{
  UNLOCK_BUFFER_LOCK(buffer);
}


gboolean
ochusha_async_buffer_check_active(OchushaAsyncBuffer *buffer)
{
  gboolean result;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_check_active(%p)\n",
	  buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "buffer(%p) is suspended at ochusha_async_buffer_check_signal.\n",
	      buffer);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer(%p) resumed.\n", buffer);
#endif
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   * ޤλǤOCHUSHA_ASYNC_BUFFER_OKʤΤǡbuffercondvarԤäƤ
   * ΤconsumerΤߤǤ뤳ȤݾڤƤ롣
   */
  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;

  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_signal(OchushaAsyncBuffer *buffer)
{
  gboolean result;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_signal(%p)\n", buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "buffer(%p) is suspended at ochusha_async_buffer_signal.\n",
	      buffer);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer(%p) resumed.\n", buffer);
#endif
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   * ޤλǤOCHUSHA_ASYNC_BUFFER_OKʤΤǡbuffercondvarԤäƤ
   * ΤconsumerΤߤǤ뤳ȤݾڤƤ롣
   */
  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
  if (result)
    SIGNAL_BUFFER_CONDITION(buffer);	/* λǡbufferwaitƤ
					 * åɤԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΤǤϤʤ
					 */
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_broadcast(OchushaAsyncBuffer *buffer)
{
  gboolean result;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_broadcast(%p)\n", buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "buffer(%p) is suspended at ochusha_async_buffer_broadcast.\n",
	      buffer);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer(%p) resumed.\n", buffer);
#endif
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   */
  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
  if (result)
      BROADCAST_BUFFER_CONDITION(buffer);/* λǡbufferwaitƤ
					  * åɤԤäƤ륷ʥ
					  * ᥤ󥹥åɤΤΤǤϤʤ
					  */
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_wait(OchushaAsyncBuffer *buffer)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_wait(%p)\n", buffer);
#endif

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "buffer(%p) is suspended at ochusha_async_buffer_wait.\n",
	      buffer);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);
      /* λbuffer->state != OCHUSHA_ASYNC_BUFFER_SUSPENDEDʤ
       * ʥsuspend򤯤˥ᥤ󥹥åɤäƤΡ
       */
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer(%p) resumed.\n", buffer);
#endif
    }

  if (buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED)
    WAIT_BUFFER_CONDITION(buffer);

  /* λOCHUSHA_ASYNC_BUFFER_SUSPENDEDʾ⤢뤬
   * ξsuspendϼȤˤ롣
   */

  return buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
}


/*
 * ochusha_async_buffer_is_busy:
 *
 * bufferΥåcontentionäƤȤˤTRUE֤
 */
gboolean
ochusha_async_buffer_is_busy(OchushaAsyncBuffer *buffer)
{
  g_return_val_if_fail(OCHUSHA_IS_ASYNC_BUFFER(buffer), FALSE);

  return buffer->state != OCHUSHA_ASYNC_BUFFER_OK
    || buffer->number_of_lock_waiters > 0;
}


void
ochusha_async_buffer_free_when_finished(OchushaAsyncBuffer *buffer)
{
  if (buffer->buffer != NULL)
    G_FREE((void *)buffer->buffer);
}
