/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ts_engine.c,v 1.5 2003/12/31 16:53:19 fuyu Exp $
 */

#include "config.h"

#include "ts_core.h"
#include "ts_engine.h"

#include <glib.h>

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


static void ts_engine_class_init(TSEngineClass *klass);
static void ts_engine_init(TSEngine *ts_engine);
static void ts_engine_finalize(GObject *object);

typedef struct _TSCellHandleClass TSCellHandleClass;

static void ts_cell_handle_class_init(TSCellHandleClass *klass);
static void ts_cell_handle_init(TSCellHandle *ts_cell_handle);
static void ts_cell_handle_finalize(GObject *object);
static TSCellHandle *ts_cell_handle_new(TSEngine *engine, TSCell *cell);


GType
ts_engine_get_type(void)
{
  static GType ts_engine_type = 0;

  if (ts_engine_type == 0)
    {
      static const GTypeInfo ts_engine_info =
	{
	  sizeof(TSEngineClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)ts_engine_class_init,
	  NULL,	/* class_finalize */
	  NULL,	/* class_data */
	  sizeof(TSEngine),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ts_engine_init,
	};

      ts_engine_type = g_type_register_static(G_TYPE_OBJECT,
					      "TSEngine",
					      &ts_engine_info, 0);
    }

  return ts_engine_type;
}


static GObjectClass *parent_class = NULL;


static void
ts_engine_class_init(TSEngineClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = ts_engine_finalize;
}


static void
ts_engine_init(TSEngine *engine)
{
  engine->ts_core = ts_core_init_new();
  engine->is_busy
    = (engine->ts_core == NULL || engine->ts_core->no_memory);

  if (engine->is_busy)
    {
      if (engine->ts_core != NULL)
	{
	  free(engine->ts_core);
	  engine->ts_core = NULL;
	}
    }
  else
    {
      engine->nil = ts_cell_handle_new(NULL, engine->ts_core->nil);
      engine->t = ts_cell_handle_new(NULL, engine->ts_core->t);
      engine->f = ts_cell_handle_new(NULL, engine->ts_core->f);
    }
}


static void
ts_engine_finalize(GObject *object)
{
  TSEngine *engine = TS_ENGINE(object);

  if (engine->ts_core != NULL)
    {
#if 0
      if (engine->is_busy)
	fprintf(stderr, "Cannot finalize TSCore because it is busy.\n");
#endif
      g_return_if_fail(!engine->is_busy);

      ts_core_deinit(engine->ts_core);
      free(engine->ts_core);
      engine->ts_core = NULL;

      g_object_unref(G_OBJECT(engine->nil));
      engine->nil = NULL;
      g_object_unref(G_OBJECT(engine->t));
      engine->t = NULL;
      g_object_unref(G_OBJECT(engine->f));
      engine->f = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


#define TS_TYPE_CELL_HANDLE		(ts_cell_handle_get_type())
#define TS_CELL_HANDLE(obj)		(G_TYPE_CHECK_INSTANCE_CAST((obj), TS_TYPE_CELL_HANDLE, TSCellHandle))
#define TS_CELL_HANDLE_CLASS(klass)	(G_TYPE_CHECK_CLASS_CAST((klass), TS_TYPE_CELL_HANDLE, TSCellHandleClass))
#define TS_IS_CELL_HANDLE(obj)		(G_TYPE_CHECK_INSTANCE_TYPE((obj), TS_TYPE_CELL_HANDLE))
#define TS_IS_CELL_HANDLE_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE((klass), TS_TYPE_CELL_HANDLE))
#define TS_CELL_HANDLE_GET_CLASS(obj) 	(G_TYPE_INSTANCE_GET_CLASS((obj), TS_TYPE_CELL_HANDLE, TSCellHandleClass))


struct _TSCellHandle
{
  GObject parent_object;

  TSEngine *engine;
  TSCell *cell;
};


struct _TSCellHandleClass
{
  GObjectClass parent_class;
};


static GType
ts_cell_handle_get_type(void)
{
  static GType ts_cell_handle_type = 0;

  if (ts_cell_handle_type == 0)
    {
      static const GTypeInfo ts_cell_handle_info =
	{
	  sizeof(TSCellHandleClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)ts_cell_handle_class_init,
	  NULL,	/* class_finalize */
	  NULL,	/* class_data */
	  sizeof(TSCellHandle),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ts_cell_handle_init,
	};

      ts_cell_handle_type = g_type_register_static(G_TYPE_OBJECT,
						   "TSCellHandle",
						   &ts_cell_handle_info, 0);
    }

  return ts_cell_handle_type;
}


GObjectClass *cell_handle_parent_class = NULL;


static void
ts_cell_handle_class_init(TSCellHandleClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);

  cell_handle_parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = ts_cell_handle_finalize;
}


static void
ts_cell_handle_init(TSCellHandle *handle)
{
  /* nothing to do here */
}


static void
ts_cell_handle_finalize(GObject *object)
{
  TSCellHandle *handle = TS_CELL_HANDLE(object);

  if (handle->engine != NULL)
    {
      if (handle->cell != NULL)
	{
	  ts_core_unregister_external_root(handle->engine->ts_core,
					   handle->cell);
	  handle->cell = NULL;
	}

      g_object_unref(handle->engine);
      handle->engine = NULL;
    }

  if (G_OBJECT_CLASS(cell_handle_parent_class)->finalize)
    (*G_OBJECT_CLASS(cell_handle_parent_class)->finalize)(object);
}


static TSCellHandle *
ts_cell_handle_new(TSEngine *engine, TSCell *cell)
{
  TSCellHandle *handle = TS_CELL_HANDLE(g_object_new(TS_TYPE_CELL_HANDLE,
						     NULL));
  handle->engine = engine;
  handle->cell = cell;

  if (engine != NULL)
    {
      if (cell != engine->ts_core->nil)
	{
	  ts_core_register_external_root(engine->ts_core, cell);
	  g_object_ref(G_OBJECT(engine));
	}
      else
	handle->engine = NULL;
    }

  return handle;
}


static const char *default_init_scm =
  "(define (caar x) (car (car x)))"
  "(define (cadr x) (car (cdr x)))"
  "(define (cdar x) (cdr (car x)))"
  "(define (cddr x) (cdr (cdr x)))"
  "(define (caaar x) (car (car (car x))))"
  "(define (caadr x) (car (car (cdr x))))"
  "(define (cadar x) (car (cdr (car x))))"
  "(define (caddr x) (car (cdr (cdr x))))"
  "(define (cdaar x) (cdr (car (car x))))"
  "(define (cdadr x) (cdr (car (cdr x))))"
  "(define (cddar x) (cdr (cdr (car x))))"
  "(define (cdddr x) (cdr (cdr (cdr x))))"
  "(define (caaaar x) (car (car (car (car x)))))"
  "(define (caaadr x) (car (car (car (cdr x)))))"
  "(define (caadar x) (car (car (cdr (car x)))))"
  "(define (caaddr x) (car (car (cdr (cdr x)))))"
  "(define (cadaar x) (car (cdr (car (car x)))))"
  "(define (cadadr x) (car (cdr (car (cdr x)))))"
  "(define (caddar x) (car (cdr (cdr (car x)))))"
  "(define (cadddr x) (car (cdr (cdr (cdr x)))))"
  "(define (cdaaar x) (cdr (car (car (car x)))))"
  "(define (cdaadr x) (cdr (car (car (cdr x)))))"
  "(define (cdadar x) (cdr (car (cdr (car x)))))"
  "(define (cdaddr x) (cdr (car (cdr (cdr x)))))"
  "(define (cddaar x) (cdr (cdr (car (car x)))))"
  "(define (cddadr x) (cdr (cdr (car (cdr x)))))"
  "(define (cdddar x) (cdr (cdr (cdr (car x)))))"
  "(define (cddddr x) (cdr (cdr (cdr (cdr x)))))"
  "(macro (unless form) `(if (not ,(cadr form)) (begin ,@(cddr form))))"
  "(macro (when form) `(if ,(cadr form) (begin ,@(cddr form))))";


TSEngine *
ts_engine_new(gboolean use_internal_init_scm)
{
  TSEngine *engine = TS_ENGINE(g_object_new(TS_TYPE_ENGINE, NULL));

  if (use_internal_init_scm)
    ts_engine_load_string(engine, (char *)default_init_scm);

  return engine;
}


void
ts_engine_set_gc_verbose(TSEngine *engine, gboolean verbose)
{
  g_return_if_fail(TS_IS_ENGINE(engine) && !engine->is_busy);

  engine->ts_core->gc_verbose = verbose;
  return;
}


gboolean
ts_engine_get_gc_verbose(TSEngine *engine)
{
  g_return_val_if_fail(TS_IS_ENGINE(engine) && !engine->is_busy, FALSE);

  return engine->ts_core->gc_verbose;
}


void
ts_engine_set_input_file(TSEngine *engine, FILE *file)
{
  g_return_if_fail(TS_IS_ENGINE(engine) && file != NULL);
  g_return_if_fail(!engine->is_busy);

  ts_core_set_input_port_file(engine->ts_core, file);
}


void
ts_engine_set_input_buffer(TSEngine *engine, char *buf, size_t len)
{
  g_return_if_fail(TS_IS_ENGINE(engine) && buf != NULL);
  g_return_if_fail(!engine->is_busy);

  ts_core_set_input_port_string(engine->ts_core, buf, buf + len - 1);
}


void
ts_engine_set_output_file(TSEngine *engine, FILE *file)
{
  g_return_if_fail(TS_IS_ENGINE(engine) && file != NULL);
  g_return_if_fail(!engine->is_busy);

  ts_core_set_output_port_file(engine->ts_core, file);
}


void
ts_engine_set_output_buffer(TSEngine *engine, char *buf, size_t len)
{
  g_return_if_fail(TS_IS_ENGINE(engine) && buf != NULL);
  g_return_if_fail(!engine->is_busy);

  ts_core_set_output_port_string(engine->ts_core, buf, buf + len - 1);
}


int
ts_engine_load_file(TSEngine *engine, FILE *file)
{
  int retcode;
  g_return_val_if_fail(TS_IS_ENGINE(engine) && file != NULL, -1);
  g_return_val_if_fail(!engine->is_busy, -1);

  engine->is_busy = TRUE;
  ts_core_load_file(engine->ts_core, file);
  retcode = engine->ts_core->retcode;
  engine->is_busy = FALSE;

  return retcode;
}


int
ts_engine_load_string(TSEngine *engine, char *text)
{
  int retcode;
  g_return_val_if_fail(TS_IS_ENGINE(engine) && text != NULL, -1);
  g_return_val_if_fail(!engine->is_busy, -1);

  engine->is_busy = TRUE;
  ts_core_load_string(engine->ts_core, text);
  retcode = engine->ts_core->retcode;
  engine->is_busy = FALSE;

  return retcode;
}


TSCellHandle *
ts_engine_get_global_env(TSEngine *engine)
{
  g_return_val_if_fail(TS_IS_ENGINE(engine) && engine->ts_core != NULL, NULL);

  return ts_cell_handle_new(engine, engine->ts_core->global_env);
}


void
ts_engine_define(TSEngine *engine, const TSCellHandle *env,
		 const TSCellHandle *symbol, const TSCellHandle *value)
{
  g_return_if_fail(TS_IS_ENGINE(engine));
  g_return_if_fail(TS_IS_CELL_HANDLE(env));
  g_return_if_fail(TS_IS_CELL_HANDLE(symbol) && TS_IS_CELL_HANDLE(value));
  g_return_if_fail(!engine->is_busy);

  ts_core_define(engine->ts_core, env->cell, symbol->cell, value->cell);
}


void
ts_engine_define_global(TSEngine *engine,
			const char *name, const TSCellHandle *value)
{
  TSCell *symbol;
  g_return_if_fail(TS_IS_ENGINE(engine) && TS_IS_CELL_HANDLE(value));
  g_return_if_fail(name != NULL);
  g_return_if_fail(!engine->is_busy);

  symbol = ts_core_mk_cell_symbol(engine->ts_core, name);
  ts_core_define(engine->ts_core, engine->ts_core->global_env,
		 symbol, value->cell);
}


void
ts_engine_define_global_long(TSEngine *engine,
			     const char *name, long value)
{
  TSCell *symbol;
  g_return_if_fail(TS_IS_ENGINE(engine));
  g_return_if_fail(name != NULL);
  g_return_if_fail(!engine->is_busy);

  symbol = ts_core_mk_cell_symbol(engine->ts_core, name);
  ts_core_define(engine->ts_core, engine->ts_core->global_env,
		 symbol, ts_core_mk_cell_integer(engine->ts_core, value));
}


void
ts_engine_define_global_double(TSEngine *engine,
			       const char *name, double value)
{
  TSCell *symbol;
  g_return_if_fail(TS_IS_ENGINE(engine));
  g_return_if_fail(name != NULL);
  g_return_if_fail(!engine->is_busy);

  symbol = ts_core_mk_cell_symbol(engine->ts_core, name);
  ts_core_define(engine->ts_core, engine->ts_core->global_env,
		 symbol, ts_core_mk_cell_real(engine->ts_core, value));
}


void
ts_engine_define_global_string(TSEngine *engine,
			       const char *name, const char *value)
{
  TSCell *symbol;
  g_return_if_fail(TS_IS_ENGINE(engine));
  g_return_if_fail(name != NULL);
  g_return_if_fail(!engine->is_busy);

  symbol = ts_core_mk_cell_symbol(engine->ts_core, name);
  ts_core_define(engine->ts_core, engine->ts_core->global_env,
		 symbol, ts_core_mk_cell_string(engine->ts_core, value));
}


TSCellHandle *
ts_engine_eval(TSEngine *engine,
	       const TSCellHandle *symbol, const TSCellHandle *args)
{
  g_return_val_if_fail(TS_IS_ENGINE(engine), NULL);
  g_return_val_if_fail(!engine->is_busy, NULL);
  g_return_val_if_fail(TS_IS_CELL_HANDLE(symbol), NULL);

  return ts_cell_handle_new(engine,
			    ts_core_eval(engine->ts_core, symbol->cell,
					 args != NULL
					 ? args->cell : engine->ts_core->nil));
}


TSCellHandle *
ts_engine_evalf0(TSEngine *engine, const TSCellHandle *symbol,
		 const char *fmt, ...)
{
  TSCellHandle *ret;
  va_list ap;
  va_start(ap, fmt);
  ret = ts_engine_vevalf0(engine, symbol, fmt, ap);
  va_end(ap);
  return ret;
}


TSCellHandle *
ts_engine_evalf(TSEngine *engine, const char *name, const char *fmt, ...)
{
  TSCellHandle *ret;
  va_list ap;
  va_start(ap, fmt);
  ret = ts_engine_vevalf(engine, name, fmt, ap);
  va_end(ap);
  return ret;
}


TSCellHandle *
ts_engine_vevalf0(TSEngine *engine,
		 const TSCellHandle *symbol, const char *fmt, va_list ap)
{
  TSCell *args;
  TSCore *ts_core;

  g_return_val_if_fail(TS_IS_ENGINE(engine), NULL);
  g_return_val_if_fail(TS_IS_CELL_HANDLE(symbol), NULL);
  g_return_val_if_fail(IS_SYMBOL(symbol->cell), NULL);
  g_return_val_if_fail(!engine->is_busy, NULL);

  ts_core = engine->ts_core;
  ts_core->args = ts_core->nil;

  if (fmt != NULL)
    {
      gboolean post_percent_state = FALSE;
      gboolean long_argument = FALSE;
      TSCellHandle *handle;
      TSCell *value;

      while (*fmt != '\0')
	{
	  switch (*fmt)
	    {
	    case '%':
	      post_percent_state = TRUE;
	      break;

	    case 'l':
	      if (!post_percent_state)
		break;
	      long_argument = TRUE;
	      break;

	    case 'd':
	      if (!post_percent_state)
		break;

	      if (long_argument)
		value = ts_core_mk_cell_integer(ts_core, va_arg(ap, long));
	      else
		value = ts_core_mk_cell_integer(ts_core,
						(long)va_arg(ap, int));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case 'f':
	      if (!post_percent_state)
		break;

	      value = ts_core_mk_cell_real(ts_core, va_arg(ap, double));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case 'c':
	      if (!post_percent_state)
		break;

	      value = ts_core_mk_cell_character(ts_core, va_arg(ap, int));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case 'C':
	      if (!post_percent_state)
		break;

	      handle = va_arg(ap, TSCellHandle *);
	      value = TS_IS_CELL_HANDLE(handle) ? handle->cell : ts_core->nil;
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case 's':
	      value = ts_core_mk_cell_string(ts_core, va_arg(ap, char *));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case 'S':
	      value = ts_core_mk_cell_symbol(ts_core, va_arg(ap, char *));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case '#':
	      value = ts_core_mk_cell_sharp_constant(ts_core,
						     va_arg(ap, char *));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    default:
	      if (post_percent_state)
		fprintf(stderr, "unknown format '%%%c'\n", *fmt);
	      else
		fprintf(stderr, "non format character '%c' ignored.\n", *fmt);
	      break;	/* simply ignore invalid part of the specification */
	    }
	  fmt++;
	}
    }

  args = ts_core->args;
  ts_core->args = ts_core->nil;
  args = ts_core_list_reverse_in_place(engine->ts_core, ts_core->nil, args);

  return ts_cell_handle_new(engine,
			    ts_core_eval(engine->ts_core, symbol->cell, args));
}


TSCellHandle *
ts_engine_vevalf(TSEngine *engine,
		 const char *name, const char *fmt, va_list ap)
{
  TSCell *args;
  TSCore *ts_core;
  TSCell *symbol;

  g_return_val_if_fail(TS_IS_ENGINE(engine), NULL);
  g_return_val_if_fail(!engine->is_busy, NULL);

  ts_core = engine->ts_core;
  ts_core->args = ts_core->nil;

  if (fmt != NULL)
    {
      gboolean post_percent_state = FALSE;
      gboolean long_argument = FALSE;
      TSCellHandle *handle;
      TSCell *value;

      while (*fmt != '\0')
	{
	  switch (*fmt)
	    {
	    case '%':
	      post_percent_state = TRUE;
	      break;

	    case 'l':
	      if (!post_percent_state)
		break;
	      long_argument = TRUE;
	      break;

	    case 'd':
	      if (!post_percent_state)
		break;

	      if (long_argument)
		value = ts_core_mk_cell_integer(ts_core, va_arg(ap, long));
	      else
		value = ts_core_mk_cell_integer(ts_core,
						(long)va_arg(ap, int));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case 'f':
	      if (!post_percent_state)
		break;

	      value = ts_core_mk_cell_real(ts_core, va_arg(ap, double));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case 'c':
	      if (!post_percent_state)
		break;

	      value = ts_core_mk_cell_character(ts_core, va_arg(ap, int));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case 'C':
	      if (!post_percent_state)
		break;

	      handle = va_arg(ap, TSCellHandle *);
	      value = TS_IS_CELL_HANDLE(handle) ? handle->cell : ts_core->nil;
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case 's':
	      value = ts_core_mk_cell_string(ts_core, va_arg(ap, char *));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case 'S':
	      value = ts_core_mk_cell_symbol(ts_core, va_arg(ap, char *));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    case '#':
	      value = ts_core_mk_cell_sharp_constant(ts_core,
						     va_arg(ap, char *));
	      ts_core->args = ts_core_mk_cell_cons(ts_core, value,
						   ts_core->args, 0);
	      post_percent_state = FALSE;
	      long_argument = FALSE;
	      break;

	    default:
	      if (post_percent_state)
		fprintf(stderr, "unknown format '%%%c'\n", *fmt);
	      else
		fprintf(stderr, "non format character '%c' ignored.\n", *fmt);
	      break;	/* simply ignore invalid part of the specification */
	    }
	  fmt++;
	}
    }

  symbol = ts_core_mk_cell_symbol(engine->ts_core, name);

  args = ts_core->args;
  ts_core->args = ts_core->nil;
  args = ts_core_list_reverse_in_place(engine->ts_core, ts_core->nil, args);

  return ts_cell_handle_new(engine,
			    ts_core_eval(engine->ts_core, symbol, args));
}


TSCellHandle *
ts_engine_mk_cell_cons(TSEngine *engine,
		       const TSCellHandle *a, const TSCellHandle *d,
		       int immutable)
{
  g_return_val_if_fail(TS_IS_ENGINE(engine), NULL);
  g_return_val_if_fail(TS_IS_CELL_HANDLE(a) && TS_IS_CELL_HANDLE(d), NULL);
  g_return_val_if_fail(!engine->is_busy, NULL);

  return ts_cell_handle_new(engine,
			    ts_core_mk_cell_cons(engine->ts_core,
						 a->cell, d->cell, immutable));
}


TSCellHandle *
ts_engine_mk_cell_long(TSEngine *engine, long value)
{
  g_return_val_if_fail(TS_IS_ENGINE(engine), NULL);
  g_return_val_if_fail(!engine->is_busy, NULL);

  return ts_cell_handle_new(engine,
			    ts_core_mk_cell_integer(engine->ts_core, value));
}


TSCellHandle *
ts_engine_mk_cell_double(TSEngine *engine, double value)
{
  g_return_val_if_fail(TS_IS_ENGINE(engine), NULL);
  g_return_val_if_fail(!engine->is_busy, NULL);

  return ts_cell_handle_new(engine,
			    ts_core_mk_cell_real(engine->ts_core, value));
}


TSCellHandle *
ts_engine_mk_cell_symbol(TSEngine *engine, const char *name)
{
  g_return_val_if_fail(TS_IS_ENGINE(engine), NULL);
  g_return_val_if_fail(!engine->is_busy, NULL);

  return ts_cell_handle_new(engine,
			    ts_core_mk_cell_symbol(engine->ts_core, name));
}


TSCellHandle *
ts_engine_mk_cell_string(TSEngine *engine, const char *str)
{
  g_return_val_if_fail(TS_IS_ENGINE(engine), NULL);
  g_return_val_if_fail(!engine->is_busy, NULL);

  return ts_cell_handle_new(engine,
			    ts_core_mk_cell_string(engine->ts_core, str));
}


TSCellHandle *
ts_engine_list_reverse(TSEngine *engine, const TSCellHandle *list)
{
  g_return_val_if_fail(TS_IS_ENGINE(engine), NULL);
  g_return_val_if_fail(TS_IS_CELL_HANDLE(list), NULL);
  g_return_val_if_fail(!engine->is_busy, NULL);

  return ts_cell_handle_new(engine,
			    ts_core_list_reverse(engine->ts_core, list->cell));
}


TSCellHandle *
ts_engine_list_reverse_in_place(TSEngine *engine,
				const TSCellHandle *term,
				const TSCellHandle *list)
{
  g_return_val_if_fail(TS_IS_ENGINE(engine), NULL);
  g_return_val_if_fail(!engine->is_busy, NULL);
  g_return_val_if_fail(TS_IS_CELL_HANDLE(term)
		       && TS_IS_CELL_HANDLE(list), NULL);

  return ts_cell_handle_new(engine,
			    ts_core_list_reverse_in_place(engine->ts_core,
							  term->cell,
							  list->cell));
}


TSCellHandle *
ts_engine_list_append(TSEngine *engine,
		      const TSCellHandle *a, const TSCellHandle *b)
{
  g_return_val_if_fail(TS_IS_ENGINE(engine), NULL);
  g_return_val_if_fail(!engine->is_busy, NULL);
  g_return_val_if_fail(TS_IS_CELL_HANDLE(a)
		       && TS_IS_CELL_HANDLE(b), NULL);

  return ts_cell_handle_new(engine,
			    ts_core_list_append(engine->ts_core,
						a->cell, b->cell));
}


gboolean
ts_cell_handle_is_long(const TSCellHandle *handle)
{
  TSCell *cell;
  g_return_val_if_fail(TS_IS_CELL_HANDLE(handle), FALSE);

  cell = handle->cell;
  return IS_NUMBER(cell) && IS_INTEGER(cell);
}


long
ts_cell_handle_get_long_value(const TSCellHandle *handle)
{
  TSCell *cell;
  g_return_val_if_fail(TS_IS_CELL_HANDLE(handle), 0);

  cell = handle->cell;
  g_return_val_if_fail(IS_NUMBER(cell), 0);

  return IVALUE(cell);
}


gboolean
ts_cell_handle_is_double(const TSCellHandle *handle)
{
  TSCell *cell;
  g_return_val_if_fail(TS_IS_CELL_HANDLE(handle), FALSE);

  cell = handle->cell;
  return IS_NUMBER(cell) && IS_REAL(cell);
}


double
ts_cell_handle_get_double_value(const TSCellHandle *handle)
{
  TSCell *cell;
  g_return_val_if_fail(TS_IS_CELL_HANDLE(handle), 0.0);

  cell = handle->cell;
  g_return_val_if_fail(IS_NUMBER(cell), 0.0);

  return RVALUE(cell);
}


gboolean
ts_cell_handle_is_string(const TSCellHandle *handle)
{
  g_return_val_if_fail(TS_IS_CELL_HANDLE(handle), FALSE);

  return IS_STRING(handle->cell);
}


const char *
ts_cell_handle_get_string(const TSCellHandle *handle)
{
  TSCell *cell;
  g_return_val_if_fail(TS_IS_CELL_HANDLE(handle), NULL);

  cell = handle->cell;
  g_return_val_if_fail(IS_STRING(cell), NULL);

  return STRVALUE(cell);
}


char *
ts_cell_handle_mighty_get_string(const TSCellHandle *handle)
{
  TSCell *cell;
  g_return_val_if_fail(TS_IS_CELL_HANDLE(handle), NULL);

  cell = handle->cell;
  if (IS_STRING(cell))
    return strdup(STRVALUE(cell));

  if (IS_NUMBER(cell))
    {
      char buffer[256];
      if (IS_INTEGER(cell))
	sprintf(buffer, "%ld", IVALUE_UNCHECKED(cell));
      else
	sprintf(buffer, "%.10g", RVALUE_UNCHECKED(cell));
      return strdup(buffer);
    }

  return strdup("<not yet implemented>");
}


gboolean
ts_cell_handle_is_symbol(const TSCellHandle *handle)
{
  g_return_val_if_fail(TS_IS_CELL_HANDLE(handle), FALSE);

  return IS_SYMBOL(handle->cell);
}


gboolean
ts_cell_handle_is_pair(const TSCellHandle *handle)
{
  g_return_val_if_fail(TS_IS_CELL_HANDLE(handle), FALSE);

  return IS_PAIR(handle->cell);
}


TSCellHandle *
ts_cell_handle_get_car(const TSCellHandle *handle)
{
  TSCell *cell;
  g_return_val_if_fail(TS_IS_CELL_HANDLE(handle), NULL);

  cell = handle->cell;
  g_return_val_if_fail(IS_PAIR(cell), NULL);

  return ts_cell_handle_new(handle->engine, CAR(cell));
}


TSCellHandle *
ts_cell_handle_get_cdr(const TSCellHandle *handle)
{
  TSCell *cell;;
  g_return_val_if_fail(TS_IS_CELL_HANDLE(handle), NULL);

  cell = handle->cell;
  g_return_val_if_fail(IS_PAIR(cell), NULL);

  return ts_cell_handle_new(handle->engine, CDR(cell));
}
