/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: thread_proxy.c,v 1.8 2004/01/14 10:03:18 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_async_buffer.h"
#include "ochusha_bbs_thread.h"

#include "bbs_thread_ui.h"
#include "thread_proxy.h"

#include <glib-object.h>
#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>


static OchushaBBSThreadClass *parent_class = NULL;


static void thread_proxy_class_init(ThreadProxyClass *klass);
static void thread_proxy_init(ThreadProxy *thread);
static void thread_proxy_finalize(GObject *object);
static OchushaBulletinBoard *thread_proxy_get_board(OchushaBBSThread *thread);
static int thread_proxy_get_number_of_responses_on_server(OchushaBBSThread *thread);
static int thread_proxy_get_number_of_responses_read(OchushaBBSThread *thread);
static void thread_proxy_set_number_of_responses_read(OchushaBBSThread *thread,
						      int number);
#if 0
static int thread_proxy_get_flags(OchushaBBSThread *thread);
static void thread_proxy_set_flags(OchushaBBSThread *thread, int flags);
#endif

static OchushaAsyncBuffer *thread_proxy_get_responses_source(
					OchushaBBSThread *thread,
					OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					OchushaNetworkBrokerCacheMode mode);
static gboolean thread_proxy_parse_responses(OchushaBBSThread *thread,
					     OchushaAsyncBuffer *buffer,
					     int start, int number,
					     gboolean no_wait,
					     StartThreadCallback *st_cb,
					     EachResponseCallback *r_cb,
					     BrokenResponseCallback *bcb,
					     EndThreadCallback *end_cb,
					     gpointer callback_data);
static time_t thread_proxy_get_last_modified_utc(OchushaBBSThread *thread);
static const char *thread_proxy_get_url(OchushaBBSThread *thread);
static char *thread_proxy_get_url_for_response(OchushaBBSThread *thread,
					       int from, int to);
static const char *thread_proxy_get_url_to_post(OchushaBBSThread *thread);
static gboolean thread_proxy_check_url(OchushaBBSThread *thread,
				       const char *url,
				       unsigned int *from_p,
				       unsigned int *to_p);
static void thread_proxy_remove_cache(OchushaBBSThread *thread,
				      OchushaConfig *config);
static gboolean thread_proxy_preview_response(OchushaBBSThread *thread,
					const OchushaBBSResponse *response,
					StartThreadCallback *start_cb,
					EachResponseCallback *response_cb,
					EndThreadCallback *end_cb,
					gpointer callback_data);
static gboolean thread_proxy_post_supported(OchushaBBSThread *thread);
static gboolean thread_proxy_post_response(OchushaBBSThread *thread,
					   OchushaNetworkBroker *broker,
					   const OchushaBBSResponse *response);


GType
thread_proxy_get_type(void)
{
  static GType thread_proxy_type = 0;

  if (thread_proxy_type == 0)
    {
      static const GTypeInfo thread_proxy_info =
	{
	  sizeof(ThreadProxyClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)thread_proxy_class_init,
	  NULL,	/* class_finalize */
	  NULL,	/* class_data */
	  sizeof(ThreadProxy),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)thread_proxy_init,
	};

      thread_proxy_type = g_type_register_static(OCHUSHA_TYPE_BBS_THREAD,
						 "ThreadProxy",
						 &thread_proxy_info, 0);
    }

  return thread_proxy_type;
}


static void
thread_proxy_class_init(ThreadProxyClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);
  OchushaBBSThreadClass *b_class = OCHUSHA_BBS_THREAD_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = thread_proxy_finalize;

  b_class->get_board = thread_proxy_get_board;
  b_class->get_number_of_responses_on_server
    = thread_proxy_get_number_of_responses_on_server;
  b_class->get_number_of_responses_read
    = thread_proxy_get_number_of_responses_read;
  b_class->set_number_of_responses_read
    = thread_proxy_set_number_of_responses_read;
#if 0
  b_class->get_flags = thread_proxy_get_flags;
  b_class->set_flags = thread_proxy_set_flags;
#endif

  b_class->get_responses_source = thread_proxy_get_responses_source;
  b_class->parse_responses = thread_proxy_parse_responses;
  b_class->get_last_modified_utc = thread_proxy_get_last_modified_utc;
  b_class->get_url = thread_proxy_get_url;
  b_class->get_url_for_response = thread_proxy_get_url_for_response;
  b_class->get_url_to_post = thread_proxy_get_url_to_post;
  b_class->check_url = thread_proxy_check_url;
  b_class->remove_cache = thread_proxy_remove_cache;

  b_class->preview_response = thread_proxy_preview_response;
  b_class->post_supported = thread_proxy_post_supported;
  b_class->post_response = thread_proxy_post_response;
}


static void
thread_proxy_init(ThreadProxy *thread)
{
}


static void
thread_proxy_finalize(GObject *object)
{
  ThreadProxy *proxy = THREAD_PROXY(object);

#if 0
  fprintf(stderr, "thread_proxy_finalize: %p\n", object);
#endif

  if (proxy->real_thread != NULL)
    {
      OCHU_OBJECT_UNREF(proxy->real_thread);
      proxy->real_thread = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static OchushaBulletinBoard *
thread_proxy_get_board(OchushaBBSThread *thread)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), NULL);
  proxy = THREAD_PROXY(thread);

  if (!OCHUSHA_IS_BBS_THREAD(proxy->real_thread))
    {
#if 0
      fprintf(stderr, "real_thread unknown for thread->id: %s\n", thread->id);
#endif
      return NULL;
    }

  return ochusha_bbs_thread_get_board(proxy->real_thread);
}


static int
thread_proxy_get_number_of_responses_on_server(OchushaBBSThread *thread)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), 0);
  proxy = THREAD_PROXY(thread);

  if (proxy->real_thread == NULL)
    return 0;

  return ochusha_bbs_thread_get_number_of_responses_on_server(proxy->real_thread);
}


static int
thread_proxy_get_number_of_responses_read(OchushaBBSThread *thread)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), 0);
  proxy = THREAD_PROXY(thread);

  if (proxy->real_thread == NULL)
    return 0;

  return ochusha_bbs_thread_get_number_of_responses_read(proxy->real_thread);
}


static void
thread_proxy_set_number_of_responses_read(OchushaBBSThread *thread,
					  int number)
{
  ThreadProxy *proxy;
  g_return_if_fail(IS_THREAD_PROXY(thread));
  proxy = THREAD_PROXY(thread);
  thread->number_of_responses_read = number;

  ochusha_bbs_thread_set_number_of_responses_read(proxy->real_thread, number);
}


#if 0
static int
thread_proxy_get_flags(OchushaBBSThread *thread)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), 0);
  proxy = THREAD_PROXY(thread);

  if (proxy->real_thread == NULL)
    return 0;

  return ochusha_bbs_thread_get_flags(proxy->real_thread);
}


static void
thread_proxy_set_flags(OchushaBBSThread *thread, int flags)
{
  ThreadProxy *proxy;
  g_return_if_fail(IS_THREAD_PROXY(thread));
  proxy = THREAD_PROXY(thread);

  ochusha_bbs_thread_set_flags(proxy->real_thread, flags);
}
#endif


static OchushaAsyncBuffer *
thread_proxy_get_responses_source(OchushaBBSThread *thread,
				  OchushaNetworkBroker *broker,
				  OchushaAsyncBuffer *buffer,
				  OchushaNetworkBrokerCacheMode mode)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), NULL);
  proxy = THREAD_PROXY(thread);

  return ochusha_bbs_thread_get_responses_source(proxy->real_thread,
						 broker, buffer, mode);
}


static gboolean
thread_proxy_parse_responses(OchushaBBSThread *thread,
			     OchushaAsyncBuffer *buffer,
			     int start, int number,
			     gboolean no_wait,
			     StartThreadCallback *st_cb,
			     EachResponseCallback *r_cb,
			     BrokenResponseCallback *b_cb,
			     EndThreadCallback *end_cb,
			     gpointer callback_data)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), FALSE);
  proxy = THREAD_PROXY(thread);

  return ochusha_bbs_thread_parse_responses(proxy->real_thread,
					    buffer, start, number, no_wait,
					    st_cb, r_cb, b_cb, end_cb,
					    callback_data);
}


static time_t
thread_proxy_get_last_modified_utc(OchushaBBSThread *thread)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), 0);
  proxy = THREAD_PROXY(thread);

  return ochusha_bbs_thread_get_last_modified_utc(proxy->real_thread);
}


static const char *
thread_proxy_get_url(OchushaBBSThread *thread)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), NULL);
  proxy = THREAD_PROXY(thread);

#if 0
  return ochusha_bbs_thread_get_url(proxy->real_thread);
#else
  return thread->id;
#endif
}


static char *
thread_proxy_get_url_for_response(OchushaBBSThread *thread,
				  int from, int to)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), NULL);
  proxy = THREAD_PROXY(thread);

  return ochusha_bbs_thread_get_url_for_response(proxy->real_thread, from, to);
}


static const char *
thread_proxy_get_url_to_post(OchushaBBSThread *thread)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), NULL);
  proxy = THREAD_PROXY(thread);

  return ochusha_bbs_thread_get_url_to_post_response(proxy->real_thread);
}


static gboolean
thread_proxy_check_url(OchushaBBSThread *thread, const char *url,
		       unsigned int *from_p, unsigned int *to_p)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), FALSE);
  proxy = THREAD_PROXY(thread);

  return ochusha_bbs_thread_check_url(proxy->real_thread, url, from_p, to_p);
}


static void
thread_proxy_remove_cache(OchushaBBSThread *thread, OchushaConfig *config)
{
  ThreadProxy *proxy;
  g_return_if_fail(IS_THREAD_PROXY(thread));
  proxy = THREAD_PROXY(thread);

  ochusha_bbs_thread_remove_cache(proxy->real_thread, config);
}


static gboolean
thread_proxy_preview_response(OchushaBBSThread *thread,
			      const OchushaBBSResponse *response,
			      StartThreadCallback *start_cb,
			      EachResponseCallback *response_cb,
			      EndThreadCallback *end_cb,
			      gpointer callback_data)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), FALSE);
  proxy = THREAD_PROXY(thread);

  return ochusha_bbs_thread_preview_response(proxy->real_thread, response,
					     start_cb, response_cb, end_cb,
					     callback_data);
}


static gboolean
thread_proxy_post_supported(OchushaBBSThread *thread)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), FALSE);
  proxy = THREAD_PROXY(thread);

  return ochusha_bbs_thread_is_post_supported(proxy->real_thread);
}


static gboolean
thread_proxy_post_response(OchushaBBSThread *thread,
			   OchushaNetworkBroker *broker,
			   const OchushaBBSResponse *response)
{
  ThreadProxy *proxy;
  g_return_val_if_fail(IS_THREAD_PROXY(thread), FALSE);
  proxy = THREAD_PROXY(thread);

  return ochusha_bbs_thread_post_response(proxy->real_thread,
					  broker, response);
}


OchushaBBSThread *
thread_proxy_new(const char *url, const gchar *title)
{
  return OCHUSHA_BBS_THREAD(g_object_new(THREAD_PROXY_TYPE,
					 "id", url,
					 "title", title,
					 NULL));
}


static void
copy_info_from_real_thread(ThreadProxy *proxy)
{
  BBSThreadGUIInfo *info;
  BBSThreadGUIInfo *proxy_info;

  g_return_if_fail(IS_THREAD_PROXY(proxy));
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(proxy->real_thread));

  info = ensure_bbs_thread_info(proxy->real_thread);
  proxy_info = ensure_bbs_thread_info(OCHUSHA_BBS_THREAD(proxy));

  proxy_info->view_rank = info->view_rank;
  proxy_info->show_mailto_mode = info->show_mailto_mode;

  if (proxy_info->last_name == NULL && info->last_name != NULL)
    proxy_info->last_name = G_STRDUP(info->last_name);
  if (proxy_info->last_mail == NULL && info->last_mail != NULL)
    proxy_info->last_mail = G_STRDUP(info->last_mail);

  proxy_info->thread_local_a_bone = info->thread_local_a_bone;

  proxy_info->a_bone_by_name = info->a_bone_by_name;
  if (proxy_info->a_bone_by_name_pattern == NULL
      && info->a_bone_by_name_pattern != NULL)
    proxy_info->a_bone_by_name_pattern
      = G_STRDUP(info->a_bone_by_name_pattern);

  proxy_info->a_bone_by_id = info->a_bone_by_id;
  if (proxy_info->a_bone_by_id_pattern == NULL
      && info->a_bone_by_id_pattern != NULL)
    proxy_info->a_bone_by_id_pattern = G_STRDUP(info->a_bone_by_id_pattern);

  proxy_info->a_bone_by_content = info->a_bone_by_content;
  if (proxy_info->a_bone_by_content_pattern == NULL
      && info->a_bone_by_content_pattern != NULL)
    proxy_info->a_bone_by_content_pattern
      = G_STRDUP(info->a_bone_by_content_pattern);
  proxy_info->bookmark_response_number = info->bookmark_response_number;
}


OchushaBBSThread *
thread_proxy_new_with_thread(OchushaBBSThread *thread)
{
  OchushaBBSThread *proxy
    = OCHUSHA_BBS_THREAD(g_object_new(THREAD_PROXY_TYPE,
				      "id", ochusha_bbs_thread_get_url(thread),
				      "title", thread->title,
				      NULL));
  OCHU_OBJECT_REF(thread);
  THREAD_PROXY(proxy)->real_thread = thread;
  copy_info_from_real_thread(THREAD_PROXY(proxy));

  return proxy;
}


OchushaBBSThread *
thread_proxy_get_real_thread(ThreadProxy *proxy)
{
  g_return_val_if_fail(IS_THREAD_PROXY(proxy), NULL);

  return proxy->real_thread;
}


void
thread_proxy_set_real_thread(ThreadProxy *proxy, OchushaBBSThread *thread)
{
  g_return_if_fail(IS_THREAD_PROXY(proxy));
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  if (proxy->real_thread != NULL)
    OCHU_OBJECT_UNREF(proxy->real_thread);
  OCHU_OBJECT_REF(thread);
  proxy->real_thread = thread;
  copy_info_from_real_thread(proxy);
}
