/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_thread_jbbs.c,v 1.26 2004/01/14 09:59:48 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_board_jbbs.h"
#include "ochusha_thread_jbbs.h"

#include "worker.h"

#include <glib-object.h>
#include <glib.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>


#if HAVE_ONIG_ONIGURUMA_H
# include <onig/oniguruma.h>
#else
# include <oniguruma.h>
#endif
#define ENABLE_HTML_TO_DAT_CONVERTER	1


#define DEBUG_DAT	0
#define DEBUG_DAT_MOST	0

#define ANALYSIS_INTERVAL	20


static void ochusha_thread_jbbs_class_init(OchushaThreadJBBSClass *klass);
static void ochusha_thread_jbbs_init(OchushaThreadJBBS *thread);

static char *ochusha_thread_jbbs_get_base_path(OchushaThread2ch *thread_2ch);

static OchushaAsyncBuffer *ochusha_thread_jbbs_get_responses_source(
					OchushaBBSThread *thread,
					OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					OchushaNetworkBrokerCacheMode mode);
static gboolean ochusha_thread_jbbs_parse_responses(OchushaBBSThread *thread,
						    OchushaAsyncBuffer *buffer,
						    int start, int number,
						    gboolean no_wait,
						    StartThreadCallback *st_cb,
						    EachResponseCallback *r_cb,
						    BrokenResponseCallback *bcb,
						    EndThreadCallback *end_cb,
						    gpointer callback_data);
static const char *ochusha_thread_jbbs_get_base_url(OchushaBBSThread *thread);

static gboolean ochusha_thread_jbbs_preview_response(OchushaBBSThread *thread,
					const OchushaBBSResponse *response,
					StartThreadCallback *start_cb,
					EachResponseCallback *response_cb,
					EndThreadCallback *end_cb,
					gpointer callback_data);

static gboolean ochusha_thread_jbbs_post_supported(OchushaBBSThread *thread);

static char *ochusha_thread_jbbs_make_post_response_message(
						OchushaThread2ch *thread_2ch,
						const char *from,
						const char *mail,
						const char *message,
						const char *bbs,
						const char *key,
						time_t time);

static char *ochusha_thread_jbbs_get_url_for_response(OchushaBBSThread *thread,
						      int from, int to);
static const char *ochusha_thread_jbbs_get_url_to_post(OchushaBBSThread *thread);

static OchushaAsyncBuffer *ochusha_thread_jbbs_fake_responses_source(
					OchushaThread2ch *thread_2ch,
					OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					OchushaNetworkBrokerCacheMode mode);


GType
ochusha_thread_jbbs_get_type(void)
{
  static GType thread_jbbs_type = 0;

  if (thread_jbbs_type == 0)
    {
      static const GTypeInfo thread_jbbs_info =
	{
	  sizeof(OchushaThreadJBBSClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_thread_jbbs_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaThreadJBBS),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_thread_jbbs_init,
	};

      thread_jbbs_type = g_type_register_static(OCHUSHA_TYPE_THREAD_2CH,
						"OchushaThreadJBBS",
						&thread_jbbs_info, 0);
    }

  return thread_jbbs_type;
}


static OchushaThread2chClass *parent_class = NULL;

static void
ochusha_thread_jbbs_class_init(OchushaThreadJBBSClass *klass)
{
  OchushaBBSThreadClass *b_class = OCHUSHA_BBS_THREAD_CLASS(klass);
  OchushaThread2chClass *class_2ch = OCHUSHA_THREAD_2CH_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  b_class->get_board = NULL;
  b_class->get_number_of_responses_on_server = NULL;
  b_class->get_number_of_responses_read = NULL;
  b_class->set_number_of_responses_read = NULL;
  b_class->get_flags = NULL;
  b_class->set_flags = NULL;

  b_class->get_responses_source = ochusha_thread_jbbs_get_responses_source;
  b_class->parse_responses = ochusha_thread_jbbs_parse_responses;
  b_class->get_url
    = (const char *(*)(OchushaBBSThread *))ochusha_thread_jbbs_get_base_url;
  b_class->get_url_for_response = ochusha_thread_jbbs_get_url_for_response;
  b_class->get_url_to_post = ochusha_thread_jbbs_get_url_to_post;

  b_class->preview_response = ochusha_thread_jbbs_preview_response;
  b_class->post_supported = ochusha_thread_jbbs_post_supported;

  class_2ch->get_base_path = ochusha_thread_jbbs_get_base_path;
  class_2ch->make_post_response_message
    = ochusha_thread_jbbs_make_post_response_message;
}


static void
ochusha_thread_jbbs_init(OchushaThreadJBBS *thread)
{
  /* äɬפʤ */
}


OchushaBBSThread *
ochusha_thread_jbbs_new(OchushaBoardJBBS *board,
			const char *thread_id,
			const gchar *title)
{
  return OCHUSHA_BBS_THREAD(g_object_new(OCHUSHA_TYPE_THREAD_JBBS,
					 "board", board,
					 "id", thread_id,
					 "title", title,
					 NULL));
}


static char *
ochusha_thread_jbbs_get_base_path(OchushaThread2ch *thread_2ch)
{
  char base_path[PATH_MAX];
  OchushaBBSThread *thread;

  thread = OCHUSHA_BBS_THREAD(thread_2ch);

  g_return_val_if_fail(thread->board != NULL && thread->id != NULL, NULL);

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MACHIBBS:
      if (snprintf(base_path, PATH_MAX, "/bbs/read.pl?BBS=%s&KEY=%s",
		   ochusha_bulletin_board_get_id(thread->board), thread->id)
	  >= PATH_MAX)
	return NULL;
      break;

    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      if (snprintf(base_path, PATH_MAX, "/bbs/read.cgi%s%s/%s/",
		   ochusha_bulletin_board_get_base_path(thread->board),
		   ochusha_bulletin_board_get_id(thread->board),
		   thread->id) >= PATH_MAX)
	return NULL;
      break;

    case OCHUSHA_BBS_TYPE_MITINOKU:
      if (snprintf(base_path, PATH_MAX, "/read.cgi?bbs=%s&key=%s",
		   ochusha_bulletin_board_get_id(thread->board), thread->id)
	  >= PATH_MAX)
	return NULL;
      break;

    case OCHUSHA_BBS_TYPE_JBBS:
    default:	/* ̤ݡ */
      return NULL;
    }

  return G_STRDUP(base_path);
}


static OchushaAsyncBuffer *
ochusha_thread_jbbs_get_responses_source(OchushaBBSThread *thread,
					 OchushaNetworkBroker *broker,
					 OchushaAsyncBuffer *buffer,
					 OchushaNetworkBrokerCacheMode mode)
{
  OchushaThread2ch *thread_2ch;
  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread)
		       && OCHUSHA_IS_NETWORK_BROKER(broker), NULL);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  if (thread->flags & OCHUSHA_BBS_THREAD_DAT_DROPPED)
    mode = OCHUSHA_NETWORK_BROKER_CACHE_ONLY;

  return ochusha_thread_jbbs_fake_responses_source(thread_2ch, broker, buffer,
						   mode);
}


/*
 * extract_jbb_response
 *
 * cur_posʸJBBSDATǡ1쥹ʬȸʤƲϤ
 * 1쥹ɽOchushaBBSResponse¤Τ˳Ǽ롣
 * δؿϡߤΰ(cur_pos)龯ʤȤ1ʬDATǡ
 * ¸ߤꤹΤǸƤӽФ¦դɬסʾ֤ǸƤӽФȡ
 * Хåեۤǡ˥뤳Ȥˤʤ롣
 *
 * 1쥹ʬΥǡ줿硢쥹ľΥǥߥ"<>"
 * ľʸؤݥ󥿤֤ԻˤNULL֤
 */
static char *
extract_jbbs_response(char *start_pos, char *eol_pos,
		      OchushaBBSResponse *response)
{
  /*
   * XXX: 2chDATե
   * DATեγƹԤ
   * ̾<>ᥤ<><>쥹<>[åɥȥ]
   * ȤʤäƤ롣
   *
   * ǶDATɤ߹褦ˤʤäJBBSФʷǼĤξ硢
   * ֹ<>ᥤ<><>쥹<>[åɥȥ]<>[ۥorID]
   * ȤˤʤäƤ롣
   *
   * ᤤˤη򥵥ݡȤޤBBSʤɤǹƤDATե
   * ηѹͽ2chΤޤޤˤʤäƤ롣
   */
  char *clone_buffer = NULL;
  char *cur_pos = start_pos;
  char *tmp_pos;
  gboolean comma_delimiter = FALSE;
  int i = 0;

  /* DATǡƤ뤳Ȥ⤢Τǥå */
  tmp_pos = memchr(start_pos, '\0', eol_pos - start_pos);
  if (tmp_pos != NULL)
    {
      int rest;
      int clone_len = eol_pos - start_pos;
      /* 줿DATǡ */
      clone_buffer = G_MALLOC(clone_len + 1);
      memcpy(clone_buffer, start_pos, clone_len);
      rest = clone_len;

      cur_pos = clone_buffer + (tmp_pos - start_pos);
      eol_pos = clone_buffer + clone_len;
      /*
       * DATե1ʬ˴ޤޤƤ롢
       * ٤Ǥʤ'\0''*'֤롣
       */
      while (*cur_pos == '\0' && *cur_pos != '\n')
	{
	  *cur_pos = '*';
	  cur_pos++;
	}

      rest = clone_len;
#if DEBUG_DAT
      clone_buffer[clone_len] = '\0';
      fprintf(stderr, "broken line: %s\n", clone_buffer);
#endif
    }


  tmp_pos = g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  while (tmp_pos != NULL)
    {
      i++;
      tmp_pos = g_strstr_len(tmp_pos + 2, eol_pos - tmp_pos - 2, "<>");
    }

  if (i == 0)
    goto error_exit;

  if (i == 1)
    {
      i = 0;
      tmp_pos = memchr(start_pos, ',', eol_pos - start_pos);
      while (tmp_pos != NULL)
	{
	  i++;
	  tmp_pos = memchr(tmp_pos + 1, ',', eol_pos - tmp_pos - 1);
	}
      if (i != 3)
	goto error_exit;	/* ȤΤʤ */
      comma_delimiter = TRUE;
    }

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  response->name = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos - cur_pos > 0)
    response->mailto = G_STRNDUP(cur_pos, tmp_pos - cur_pos);
  else
    response->mailto = NULL;

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  response->date_id = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);

  tmp_pos = g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  response->content = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

  if (clone_buffer != NULL)
    {
      G_FREE(clone_buffer);
      return start_pos + (tmp_pos - clone_buffer) + 2;
    }
  else
    return tmp_pos + 2;

 error_exit:
#if DEBUG_DAT_MOST
  tmp_pos = G_STRNDUP(start_pos, eol_pos - start_pos);
  fprintf(stderr, "DAT Parsing failed: \"%s\"", tmp_pos);
  G_FREE(tmp_pos);
#endif
  if (response->name != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "name=%s\n", response->name);
#endif
      G_FREE((gpointer)response->name);
      response->name = NULL;
    }

  if (response->mailto != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "mailto=%s\n", response->mailto);
#endif
      G_FREE((gpointer)response->mailto);
      response->mailto = NULL;
    }

  if (response->date_id != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "date_id=%s\n", response->date_id);
#endif
      G_FREE((gpointer)response->date_id);
      response->date_id = NULL;
    }

  if (response->content != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "content=%s\n", response->content);
#endif
      G_FREE((gpointer)response->content);
      response->content = NULL;
    }

  if (clone_buffer != NULL)
    G_FREE(clone_buffer);

  return NULL;
}


static gboolean
advance_parsing(gpointer data)
{
  ochusha_async_buffer_broadcast((OchushaAsyncBuffer *)data);
  OCHU_OBJECT_UNREF(data);
  return FALSE;
}


/*
 * ochusha_thread_jbbs_parse_responses
 *
 * bufferƤåɤDATǡǤȸʤƲϤåɤ
 * ɽOchushaThread2ch¤Τˤ륹åɤʬ(responses)ۤ롣
 *
 * åɤγϻåɥȥˤstart_thread_cb
 * ġΥ쥹Фeach_response_cbåΤβϽλˤ
 * end_thread_cb򤽤줾ƤӽФΥХåFALSE
 * 顢λǲϤŪˤ롣
 *
 * start+1ܤnumberʬΥ쥹ϤġĤȤؿˤͽꡣ
 * åã硢number˴ؤ餺Ϥλ롣
 *
 * number == -1ʤ顢start饹åޤǡ
 * start == -1ʤ顢ånumberʬΥ쥹
 *
 * Ĥޤꡢstart=0,number=-1ǡ쥹ϡ
 *         start=-1,number=100Ǻǿ100쥹Ȥä
 * 褺start=-1ξ̤Τޤ֢ʤͽ
 */
static gboolean
ochusha_thread_jbbs_parse_responses(OchushaBBSThread *thread,
				    OchushaAsyncBuffer *buffer,
				    int start, int number, gboolean no_wait,
				    StartThreadCallback *start_thread_cb,
				    EachResponseCallback *each_response_cb,
				    BrokenResponseCallback *broken_response_cb,
				    EndThreadCallback *end_thread_cb,
				    gpointer callback_data)
{
  OchushaThread2ch *thread_2ch;
  unsigned int *responses;
  gboolean result = TRUE;
  OchushaNetworkBrokerBufferStatus *status;
  OchushaNetworkBrokerBufferStatus fake_status;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread)
		       && OCHUSHA_IS_ASYNC_BUFFER(buffer), FALSE);

  status = g_object_get_data(G_OBJECT(buffer),
			     "OchushaNetworkBroker::BufferStatus");

  if (status == NULL)
    {
      fake_status.state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;
      fake_status.status_code = 304;
      fake_status.last_modified = NULL;
      fake_status.date = NULL;
      status = &fake_status;
    }

  if (!ochusha_async_buffer_active_ref(buffer))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      return FALSE;
    }

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  responses = thread_2ch->responses;

  if (responses == NULL)
    {
      /*
       * 1000쥹ʬǤ⡹4000ХȤʤΤǡ
       * ǽ餫MAX_RESPONSE쥹ʬ϶Ȥ롣
       */
      responses = G_NEW0(unsigned int, MAX_RESPONSE);
      thread_2ch->responses = responses;
      thread_2ch->responses_length = MAX_RESPONSE;
    }

  if (number < -1)
    number = thread_2ch->responses_length;

  ochusha_async_buffer_lock(buffer);
  {
    int rest_of_responses;
    unsigned int offset;
    int i;
    gboolean buffer_fixed = FALSE;

  retry_analysis:
    rest_of_responses = number;
    offset = 0;
    i = 0;
    if (status->state != OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
      {
	if (start > 1 && start < thread_2ch->responses_length)
	  {
	    offset = responses[start];
	    if (offset != 0)
	      i = start;
	    else
	      {
		/*
		 * nʬΥ쥹ɤ߹n+1ܰʹߤɤ
		 * ġĤȤΤ̰ƤХʥꥵ
		 * ȤơǤᤤϺʬõȤǤ뤬
		 * ޤǤɬפʤ
		 */
		offset = responses[start - 1];
		if (offset != 0)
		  i = start - 1;
		else
		  offset = 0;
	      }
	  }
      }

    while (result)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	int rest_of_data = buffer->length - offset;
	char *eol_pos = NULL;
	int interval = ANALYSIS_INTERVAL;

#if DEBUG_DAT_MOST
	fprintf(stderr, "1: ochusha_thread_jbbs.c: length=%d\n",
		buffer->length);
#endif
	while (rest_of_data > 0
	       && (rest_of_responses > 0 || rest_of_responses == -1)
	       && interval-- > 0
	       && (eol_pos = memchr(cur_pos, '\n', rest_of_data)) != NULL)
	  {
	    OchushaBBSResponse response = { NULL, NULL, NULL, NULL };
	    char *title_pos;

	    if (i >= thread_2ch->responses_length)
	      {
		guint new_len = thread_2ch->responses_length * 2;
		thread_2ch->responses = G_REALLOC(thread_2ch->responses,
						  new_len);
		memset(thread_2ch->responses + thread_2ch->responses_length, 0,
		       (new_len - thread_2ch->responses_length)
		       * sizeof(guint));
		thread_2ch->responses_length = new_len;
		responses = thread_2ch->responses;
	      }
#if DEBUG_DAT_MOST
	    fprintf(stderr, "i=%d, offset=%d, rest_of_data=%d, cur_pos=%p\n",
		    i, offset, rest_of_data, cur_pos);
#endif

	    title_pos = extract_jbbs_response(cur_pos, eol_pos, &response);

#if DEBUG_DAT_MOST
	    fprintf(stderr, "title_pos=%p\n", title_pos);
#endif

	    if (i == 0 && start == 0 && title_pos != NULL)
	      {
		char *title = G_STRNDUP(title_pos, eol_pos - title_pos);
		if (start_thread_cb != NULL)
		  result = (*start_thread_cb)(thread, title, callback_data);
		G_FREE(title);
		if (!result)
		  goto terminated;
	      }
	    responses[i] = offset;

	    i++;
	    if (i > start)
	      {
		result = TRUE;
		if (title_pos != NULL)
		  {
		    if (each_response_cb != NULL)
		      result = (*each_response_cb)(thread, i, &response,
						   callback_data);
		  }
		else
		  {
		    if (broken_response_cb != NULL)
		      result = (*broken_response_cb)(thread, i, callback_data);
		  }
		if (rest_of_responses != -1)
		  rest_of_responses--;
		if (!result)
		  goto terminated;
	      }

	    offset = (eol_pos + 1) - buffer_top;
	    buffer_top = (char *)buffer->buffer;
	    cur_pos = buffer_top + offset;
	    rest_of_data = buffer->length - offset;

#if DEBUG_DAT
	    if (title_pos == NULL)
	      {
		char *line;
		fprintf(stderr, "response #%d: Couldn't analyze DAT file...ignoring parts.\n", i);
		line = G_STRNDUP(buffer_top + responses[i - 1],
				 offset - responses[i - 1]);
		fprintf(stderr, "failed: \"%s\"\n", line);
		G_FREE(line);
		fprintf(stderr, "offset=%d\n", responses[i - 1]);
	      }
#endif

	    if (response.name != NULL)
	      G_FREE((gpointer)response.name);
	    if (response.mailto != NULL)
	      G_FREE((gpointer)response.mailto);
	    if (response.date_id != NULL)
	      G_FREE((gpointer)response.date_id);
	    if (response.content != NULL)
	      G_FREE((gpointer)response.content);

	    eol_pos = NULL;
	  }

	if (!buffer_fixed && buffer->fixed)
	  {
	    buffer_fixed = TRUE;
	    eol_pos = NULL;
	    continue;
	  }

	if (buffer_fixed && eol_pos == NULL && interval > 0)
	  goto terminated;

	if (interval > 0)
	  {
	    /*
	     * ɤ٤ǡʤ硣
	     */
	    unsigned int old_len = buffer->length;
	    if (no_wait)
	      goto terminated;	/* ξ */

	    if (ochusha_async_buffer_is_busy(buffer))
	      do
		{
		  if (!ochusha_async_buffer_wait(buffer))
		    {
#if DEBUG_ASYNC_BUFFER_MOST
		      fprintf(stderr, "ochusha_thread_jbbs_parse_responses(): buffer has been terminated.\n");
#endif
		      goto terminated;
		    }
		} while (buffer->length == old_len && !buffer->fixed);
	  }
	else
	  {
	    if (ochusha_async_buffer_is_busy(buffer))
	      {
		/*
		 * buffer˥륹åɤsuspend/terminate
		 * ȿ뤿˥å
		 */
		if (!ochusha_async_buffer_wait(buffer))
		  goto terminated;
	      }
	    else
	      {
		/*
		 * GUIΥ쥹ݥ󥹤βܻؤ¸
		 */
		OCHU_OBJECT_REF(buffer);
		g_idle_add_full(G_PRIORITY_HIGH_IDLE + 15,
				advance_parsing, buffer, NULL);
		if (!ochusha_async_buffer_wait(buffer))
		  goto terminated;
	      }
	  }
	interval = ANALYSIS_INTERVAL;

	if (status->state
	    == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
	  {
	    if (i > start && end_thread_cb != NULL)
	      result = (*end_thread_cb)(thread, FALSE, callback_data);
	    /* å夬äƤ뤳ȤϤ⤦狼ä */
	    status->state
	      = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;
	    goto retry_analysis;
	  }
      }

  terminated:
    if (number == -1 || thread->number_of_responses_read < i)
      thread->number_of_responses_read = i;
  }
  ochusha_async_buffer_unlock(buffer);

  if (status->last_modified != NULL)
    {
      if (thread_2ch->last_modified != NULL)
	G_FREE(thread_2ch->last_modified);
      thread_2ch->last_modified = G_STRDUP(status->last_modified);
    }

  if (status->date != NULL)
    {
      if (thread_2ch->date != NULL)
	G_FREE(thread_2ch->date);
      thread_2ch->date = G_STRDUP(status->date);
    }

  ochusha_async_buffer_active_unref(buffer);

  if (end_thread_cb != NULL)
    result = (*end_thread_cb)(thread, TRUE, callback_data);

  return TRUE;
}


static const char *
ochusha_thread_jbbs_get_base_url(OchushaBBSThread *thread)
{
  /* XXX: ľɬ */
  char url[PATH_MAX];
  const char *base_path;
  const char *server;
  OchushaThread2ch *thread_2ch;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), NULL);
  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  if (thread_2ch->base_url != NULL)
    return thread_2ch->base_url;

  base_path = ochusha_thread_2ch_get_base_path(thread_2ch);

  thread = OCHUSHA_BBS_THREAD(thread_2ch);
  server = ochusha_bulletin_board_get_server(thread->board);

  g_return_val_if_fail(base_path != NULL && server != NULL, NULL);
  if (snprintf(url, PATH_MAX, "http://%s%s", server, base_path) >= PATH_MAX)
    return NULL;

  thread_2ch->base_url = G_STRDUP(url);
  return thread_2ch->base_url;
}


static char *
ochusha_thread_jbbs_get_url_for_response(OchushaBBSThread *thread, int from,
					 int to)
{
  OchushaThread2ch *thread_2ch;
  char url[PATH_MAX];

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), NULL);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MACHIBBS:
      {
	if (from >= to)
	  to = from;
	snprintf(url, PATH_MAX, "http://%s/bbs/read.pl?BBS=%s&KEY=%s&START=%d&END=%d&NOFIRST=TRUE",
		 ochusha_bulletin_board_get_server(thread->board),
		 ochusha_bulletin_board_get_id(thread->board),
		 thread->id, from, to);
	break;
      }

    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      {
	if (from >= to)
	  to = from;
	snprintf(url, PATH_MAX, "http://%s%sbbs/read.cgi?BBS=%s&KEY=%s&START=%d&END=%d&NOFIRST=TRUE",
		 ochusha_bulletin_board_get_server(thread->board),
		 ochusha_bulletin_board_get_base_path(thread->board),
		 ochusha_bulletin_board_get_id(thread->board),
		 thread->id, from, to);
	break;
      }

    case OCHUSHA_BBS_TYPE_MITINOKU:
      {
	if (from >= to)
	  to = from;
	snprintf(url, PATH_MAX, "./read.cgi?bbs=%s&key=%s&st=%d&to=%d&nofirst=true",
		 ochusha_bulletin_board_get_id(thread->board),
		 thread->id, from, to);
	break;
      }

    default:	/* ̤ݡ */
      return NULL;
    }

  return G_STRDUP(url);
}


static const char *
ochusha_thread_jbbs_get_url_to_post(OchushaBBSThread *thread)
{
  OchushaThread2ch *thread_2ch;
  char url[PATH_MAX];

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), NULL);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);
  if (thread_2ch->post_url != NULL)
    return thread_2ch->post_url;

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      snprintf(url, PATH_MAX, "%s?BBS=%s&KEY=%s&LAST=10",
	       ochusha_board_2ch_get_read_cgi_url(OCHUSHA_BOARD_2CH(thread->board)),
	       ochusha_bulletin_board_get_id(thread->board),
	       thread->id);
      break;

    case OCHUSHA_BBS_TYPE_MITINOKU:
      snprintf(url, PATH_MAX, "%s?bbs=%s&key=%s&ls=10",
	       ochusha_board_2ch_get_read_cgi_url(OCHUSHA_BOARD_2CH(thread->board)),
	       ochusha_bulletin_board_get_id(thread->board),
	       thread->id);
      break;

    default:	/* ̤ݡ */
      break;
    }

  thread_2ch->post_url = G_STRDUP(url);
  return thread_2ch->post_url;
}


static gboolean
ochusha_thread_jbbs_preview_response(OchushaBBSThread *thread,
				     const OchushaBBSResponse *response,
				     StartThreadCallback *start_cb,
				     EachResponseCallback *response_cb,
				     EndThreadCallback *end_cb,
				     gpointer callback_data)
{
  return FALSE;
}


static gboolean
ochusha_thread_jbbs_post_supported(OchushaBBSThread *thread)
{
  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), FALSE);
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(thread->board), FALSE);

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MACHIBBS:
      return TRUE;

    default:
      return FALSE;
    }
}


static char *
ochusha_thread_jbbs_make_post_response_message(OchushaThread2ch *thread_2ch,
					       const char *from,
					       const char *mail,
					       const char *message,
					       const char *bbs,
					       const char *key,
					       time_t time)
{
  /* time_tΤޤޤˤ٤10ʿʸؤѴ
   * ԤˡʤΤǡ褺longѴƤޤ
   * long32bitʥƥˤ2038ǯ()¸ߤ롣
   */
  long long_time = time;
  OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(thread_2ch);

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_MACHIBBS:
      return g_strdup_printf("submit=%%8F%%91%%82%%AB%%8D%%9E%%82%%DE&NAME=%s&MAIL=%s&MESSAGE=%s&BBS=%s&KEY=%s&TIME=%ld", from, mail, message, bbs, key, long_time);

    default:
      return NULL;	/* ̤ݡ */
    }
}


#define DEFAULT_LINE_BUFFER_SIZE		8192
#define DEBUG_HTML_TO_DAT_CONVERTER		0
#define DEBUG_HTML_TO_DAT_CONVERTER_MOST	0


typedef struct _HTML2DATConverterJobArgs
{
  OchushaAsyncBuffer *html_buffer;
  OchushaAsyncBuffer *dat_buffer;
  OchushaNetworkBroker *broker;
  OchushaThread2ch *thread_2ch;
  int incoming_offset;
  int number_of_cached_responses;
} HTML2DATConverterJobArgs;


static const char *title_pattern_string = NULL;
static const char *title_pattern_end = NULL;
static const char *jbbs_shitaraba_response_pattern_string = NULL;
static const char *jbbs_shitaraba_response_pattern_end = NULL;
static const char *machibbs_response_pattern_string = NULL;
static const char *machibbs_response_pattern_end = NULL;
static const char *mitinoku_response_pattern_string = NULL;
static const char *mitinoku_response_pattern_end = NULL;


static void
initialize_default_regexp_patterns(void)
{
  title_pattern_string = "<title>(.*)</title>";
  title_pattern_end
    = title_pattern_string + strlen(title_pattern_string);

  /* "\314\276\301\260\241\247" == "̾" in EUC-JP */
  /* "\305\352\271\306\306\374\241\247" == "" in EUC-JP */
#if 0
  jbbs_shitaraba_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) \314\276\301\260\241\247[^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\305\352\271\306\306\374\241\247 ([^<]*)<[bB][rR]><[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#else
  /* ʤ"̾"ѥˤȥޥåʤġġ*/
  jbbs_shitaraba_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) [^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\305\352\271\306\306\374\241\247 ([^<]*)<[bB][rR]><[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#endif
  jbbs_shitaraba_response_pattern_end
    = (jbbs_shitaraba_response_pattern_string
       + strlen(jbbs_shitaraba_response_pattern_string));

  /* "\226\274\221\117\201\106" == "̾" in Shift_JIS */
  /* "\223\212\215\145\223\372\201\106" == "" in Shift_JIS */
#if 0
  machibbs_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) \226\274\221\117\201\106[^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\223\212\215\145\223\372\201\106 (.*)<[bB][rR]><[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#else
  machibbs_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) [^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\223\212\215\145\223\372\201\106 (.*)<[bB][rR]><[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#endif
  machibbs_response_pattern_end
    = (machibbs_response_pattern_string
       + strlen(machibbs_response_pattern_string));

  /* "\314\276\301\260\241\247" == "̾" in EUC-JP */
  /* "\305\352\271\306\306\374\241\247" == "" in EUC-JP */
#if 0
  mitinoku_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) \314\276\301\260\241\247[^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\305\352\271\306\306\374\241\247(.*)<[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#else
  mitinoku_response_pattern_string = "<[dD][tT]>([1-9][0-9]*) [^<]*(<[fF][oO][nN][tT][^>]*|<[aA][^>]*)>(.*)\305\352\271\306\306\374\241\247(.*)<[dD][dD]>(.*)<[bB][rR]><[bB][rR]>$";
#endif
  mitinoku_response_pattern_end
    = (mitinoku_response_pattern_string
       + strlen(mitinoku_response_pattern_string));
}


static void
convert_html_to_dat(WorkerThread *employee, HTML2DATConverterJobArgs *args)
{
  OchushaAsyncBuffer *html_buffer = args->html_buffer;
  OchushaAsyncBuffer *dat_buffer = args->dat_buffer;
  OchushaThread2ch *thread_2ch = args->thread_2ch;
  OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(thread_2ch);
  int incoming_offset = args->incoming_offset;
  int number_of_cached_responses = args->number_of_cached_responses;
  gboolean signal_result;
  unsigned int html_offset;
  unsigned int response_number;
  char *title = NULL;
  gboolean dump_html = FALSE;
  regex_t *title_pattern = NULL;
  regex_t *response_pattern = NULL;
  RegRegion *match_region = NULL;

  if (title_pattern_string == NULL)
    initialize_default_regexp_patterns();

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      if (regex_new(&title_pattern,
		    (char *)title_pattern_string, (char *)title_pattern_end,
#ifdef REG_ENCODING_EUC_JP
		    REG_OPTION_EXTEND, REG_ENCODING_EUC_JP,
#else /* old style */
		    REG_OPTION_EXTEND, REGCODE_EUCJP,
#endif
		    REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
      if (regex_new(&response_pattern,
		    (char *)jbbs_shitaraba_response_pattern_string,
		    (char *)jbbs_shitaraba_response_pattern_end,
#ifdef REG_ENCODING_EUC_JP
		    REG_OPTION_EXTEND, REG_ENCODING_EUC_JP,
#else /* old style */
		    REG_OPTION_EXTEND, REGCODE_EUCJP,
#endif
		    REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
      fprintf(stderr, "title_pattern: \"%s\"\n", title_pattern_string);
      fprintf(stderr, "response_pattern: \"%s\"\n",
	      jbbs_shitaraba_response_pattern_string);
#endif
      break;

    case OCHUSHA_BBS_TYPE_MACHIBBS:
      if (regex_new(&title_pattern,
		    (char *)title_pattern_string, (char *)title_pattern_end,
#ifdef REG_ENCODING_SJIS
		    REG_OPTION_EXTEND, REG_ENCODING_SJIS,
#else /* old style */
		    REG_OPTION_EXTEND, REGCODE_SJIS,
#endif
		    REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
      if (regex_new(&response_pattern,
		    (char *)machibbs_response_pattern_string,
		    (char *)machibbs_response_pattern_end,
#ifdef REG_ENCODING_SJIS
		    REG_OPTION_EXTEND, REG_ENCODING_SJIS,
#else /* old style */
		    REG_OPTION_EXTEND, REGCODE_SJIS,
#endif
		    REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
      fprintf(stderr, "title_pattern: \"%s\"\n", title_pattern_string);
      fprintf(stderr, "response_pattern: \"%s\"\n",
	      machibbs_response_pattern_string);
#endif
      break;

    case OCHUSHA_BBS_TYPE_MITINOKU:
      if (regex_new(&title_pattern,
		    (char *)title_pattern_string, (char *)title_pattern_end,
#ifdef REG_ENCODING_EUC_JP
		    REG_OPTION_EXTEND, REG_ENCODING_EUC_JP,
#else /* old style */
		    REG_OPTION_EXTEND, REGCODE_EUCJP,
#endif
		    REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
      if (regex_new(&response_pattern,
		    (char *)mitinoku_response_pattern_string,
		    (char *)mitinoku_response_pattern_end,
#ifdef REG_ENCODING_EUC_JP
		    REG_OPTION_EXTEND, REG_ENCODING_EUC_JP,
#else /* old style */
		    REG_OPTION_EXTEND, REGCODE_EUCJP,
#endif
		    REG_SYNTAX_POSIX_EXTENDED, NULL) != REG_NORMAL)
	{
	  fprintf(stderr, "invalid regular expression\n");
	  goto terminated;
	}
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
      fprintf(stderr, "title_pattern: \"%s\"\n", title_pattern_string);
      fprintf(stderr, "response_pattern: \"%s\"\n",
	      mitinoku_response_pattern_string);
#endif
      break;

    case OCHUSHA_BBS_TYPE_JBBS:
      dump_html = TRUE;
      break;

    default:
      goto terminated;
    }

#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
  fprintf(stderr, "convert_html_to_dat called\n");
#endif

  if (!ochusha_async_buffer_active_ref(html_buffer))
    {
      g_signal_emit_by_name(G_OBJECT(args->broker),
			    "access_terminated",
			    dat_buffer,
			    &signal_result);
      goto terminated;
    }

  if (!ochusha_async_buffer_active_ref(dat_buffer))
    {
      g_signal_emit_by_name(G_OBJECT(args->broker),
			    "access_terminated",
			    dat_buffer,
			    &signal_result);
      ochusha_async_buffer_active_unref(html_buffer);
      goto terminated;
    }

  html_offset = 0;
  response_number = number_of_cached_responses + 1;

  g_assert(title_pattern != NULL);
  g_assert(response_pattern != NULL);
  match_region = regex_region_new();

  ochusha_async_buffer_lock(html_buffer);
  while (TRUE)
    {
      char *html_buffer_top = (char *)html_buffer->buffer;
      char *html_buffer_pos = html_buffer_top + html_offset;
      int rest_of_data = html_buffer->length - html_offset;
      char *eol_pos = NULL;
      char *previous_line = NULL;

#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
      fprintf(stderr,
	      "html_buffer_top=%p, html_buffer_pos=%p, rest_of_data=%d\n",
	      html_buffer_top, html_buffer_pos, rest_of_data);
#endif

      while (rest_of_data > 0 && (eol_pos = memchr(html_buffer_pos, '\n',
						   rest_of_data)) != NULL)
	{
	  int line_size = eol_pos - html_buffer_pos;
	  char *html_line_buffer = G_STRNDUP(html_buffer_pos, line_size);
	  char *html_line_buffer_tail = html_line_buffer + line_size;
	  line_size++;

	  if (previous_line != NULL)
	    {
	      char *tmp_line = g_strconcat(previous_line, html_line_buffer,
					   NULL);
	      g_free(previous_line);
	      previous_line = html_line_buffer;
	      html_line_buffer_tail = tmp_line + strlen(tmp_line);
	      html_line_buffer = tmp_line;
	    }
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
	  fprintf(stderr, "html_buffer_pos=%p, eol_pos=%p, line_size=%d\n",
		  html_buffer_pos, eol_pos, line_size);
	  fprintf(stderr, "html_line: \"%s\"\n", html_line_buffer);
#endif

	  regex_region_clear(match_region);
	  if (response_number == 1 && title == NULL
	      && regex_search(title_pattern,
			      html_line_buffer, html_line_buffer_tail,
			      html_line_buffer, html_line_buffer_tail,
			      match_region, 0) >= 0)
	    {
	      char *title_head = html_line_buffer + match_region->beg[1];
	      int title_len = match_region->end[1] - match_region->beg[1];
	      title = G_STRNDUP(title_head, title_len);
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
	      fprintf(stderr, "title: %s\n", title);
#endif
	    }
	  else if (dump_html)
	    {
	      fprintf(stderr, "html_line: \"%s\"\n", html_line_buffer);
	    }
	  else if (regex_search(response_pattern,
				html_line_buffer, html_line_buffer_tail,
				html_line_buffer, html_line_buffer_tail,
				match_region, 0) >= 0)
	    {
	      int res_num;
	      char *tag_head = html_line_buffer + match_region->beg[2];
	      int tag_len = match_region->end[2] - match_region->beg[2];

	      char *name_head = html_line_buffer + match_region->beg[3];
	      int name_len = match_region->end[3] - match_region->beg[3];

	      char *date_head = html_line_buffer + match_region->beg[4];
	      int date_len = match_region->end[4] - match_region->beg[4];

	      char *body_head = html_line_buffer + match_region->beg[5];
	      int body_len = match_region->end[5] - match_region->beg[5];

	      char *mailto_head = NULL;
	      int mailto_len = 0;

	      if (tag_len > 2 && tag_head[1] == 'a')
		{
		  int tag_rest = tag_len - 2;
		  mailto_head = g_strstr_len(tag_head + 2, tag_rest,
					     "href=\"mailto:");
		  if (mailto_head != NULL)
		    {
		      mailto_head += 13;	/* "href=\"mailto:"Ф */
		      tag_rest -= 13;
		      if (tag_rest > 0)
			{
			  char *mailto_tail
			    = memchr(mailto_head, '"', tag_rest);
			  if (mailto_tail != NULL)
			    mailto_len = mailto_tail - mailto_head;
			}
		    }
		}

	      sscanf(html_line_buffer + match_region->beg[1], "%d", &res_num);

#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
	      fprintf(stderr, "res_num=%d : response_number=%d\n",
		      res_num, response_number);
#endif

	      while (res_num > response_number)
		{
		  /* ŪDATƤ */
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
		  fprintf(stderr, "breaking DAT for: \"%s\"\n",
			  html_line_buffer);
#endif
		  ochusha_async_buffer_append_data(dat_buffer, "\n", 1);
		  response_number++;
		}

	      if (res_num == response_number)
		{
		  char dat_line_buffer[DEFAULT_LINE_BUFFER_SIZE];
		  char *dat_line_buffer_pos = dat_line_buffer;
		  int dat_line_buffer_rest = DEFAULT_LINE_BUFFER_SIZE;

#define APPEND_TEXT_TO_DAT_LINE_BUFFER(text, text_len)			\
  do									\
    {									\
      if ((text_len) > 0)						\
	{								\
	  if (dat_line_buffer_rest > (text_len))			\
	    {								\
	      memcpy(dat_line_buffer_pos, (text), (text_len));		\
	      dat_line_buffer_rest -= (text_len);			\
	      dat_line_buffer_pos += (text_len);			\
	    }								\
	  else								\
	    dat_line_buffer_rest = -1;					\
	}								\
      if (dat_line_buffer_rest > 2)					\
	{								\
	  dat_line_buffer_pos[0] = '<';					\
	  dat_line_buffer_pos[1] = '>';					\
	  dat_line_buffer_rest -= 2;					\
	  dat_line_buffer_pos += 2;					\
	}								\
      else								\
	dat_line_buffer_rest = -1;					\
    } while (0)

#if 1	/* ɥۥå᤮ */
		  /*
		   * ̾礤Ȥ줤ˤ
		   */
		  /* 1: ̾ζ */
		  while (name_len > 0 && name_head[name_len - 1] == ' ')
		    name_len--;

		  /* 2: ᥤ󤬶Ǥʤϡ</a>
		   *    ξϡ</font>ĤƤΤ
		   *    
		   */
		  if (mailto_len > 0)
		    {
		      /* </a>ĤƤ */
		      if (name_len > 4
			  && g_ascii_strncasecmp(name_head + name_len - 4,
						 "</a>", 4) == 0)
			name_len -= 4;
		    }
		  else
		    {
		      /* </font>ĤƤ */
		      if (name_len > 7
			  && g_ascii_strncasecmp(name_head + name_len - 7,
						 "</font>", 7) == 0)
			name_len -= 7;
		    }

		  /* 3: ᥤ<b></b>ǰϤޤƤɤ */
		  if (name_len > 4
		      && g_ascii_strncasecmp(name_head + name_len - 4,
					      "</b>", 4) == 0)
		    {
		      name_len -= 4;
		      if (name_len > 3
			  && g_ascii_strncasecmp(name_head, "<b>", 3) == 0)
			{
			  name_len -= 3;
			  name_head += 3;
			}
		    }

		  /* 4: ̾Ƭζ */
		  while (name_len > 0 && *name_head == ' ')
		    {
		      name_len--;
		      name_head++;
		    }

		  /* 5: ̾ζ */
		  while (name_len > 0 && name_head[name_len - 1] == ' ')
		    name_len--;
#endif

		  APPEND_TEXT_TO_DAT_LINE_BUFFER(name_head, name_len);
		  APPEND_TEXT_TO_DAT_LINE_BUFFER(mailto_head, mailto_len);
		  APPEND_TEXT_TO_DAT_LINE_BUFFER(date_head, date_len);
		  APPEND_TEXT_TO_DAT_LINE_BUFFER(body_head, body_len);
		  if (response_number == 1 && title != NULL)
		    {
		      int title_len = strlen(title);
		      if (dat_line_buffer_rest > title_len)
			{
			  memcpy(dat_line_buffer_pos, title, title_len);
			  dat_line_buffer_rest -= title_len;
			  dat_line_buffer_pos += title_len;
			}
		      else
			dat_line_buffer_rest = -1;
		    }
		  if (dat_line_buffer_rest > 1)
		    {
		      dat_line_buffer_pos[0] = '\n';
		      dat_line_buffer_rest -= 1;
		      dat_line_buffer_pos += 1;
		    }
		  else
		    dat_line_buffer_rest = -1;
#undef APPEND_TEXT_TO_DAT_LINE_BUFFER
		  if (dat_line_buffer_rest > 0)
		    {
		      ochusha_async_buffer_append_data(dat_buffer,
					dat_line_buffer,
					dat_line_buffer_pos - dat_line_buffer);
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
		      {
			dat_line_buffer_pos[0] = '\0';
			fprintf(stderr, "DAT: %s\n", dat_line_buffer);
		      }
#endif
		    }
		  else
		    {
		      /* Ĺ */
		      fprintf(stderr, "Implement this!\n");
		    }

		  response_number++;
		}

	      if (title != NULL)
		{
		  G_FREE(title);
		  title = NULL;
		}
	    }
	  else
	    {
	      if (previous_line != NULL)
		{
#if DEBUG_HTML_TO_DAT_CONVERTER
		  fprintf(stderr, "IRREGULAR html_line: \"%s\"\n",
			  html_line_buffer);
#endif
		  G_FREE(html_line_buffer);
		  html_line_buffer = previous_line;
		  previous_line = NULL;
		}

	      if (g_ascii_strncasecmp("<dt>", html_line_buffer, 4) == 0)
		{
		  int len = strlen(html_line_buffer) - 1;
		  while (len > 0 &&
			 (html_line_buffer[len] == '\n'
			  || html_line_buffer[len] == '\r'))
		    {
		      html_line_buffer[len--] = '\0';
		    }
		  previous_line = html_line_buffer;
#if DEBUG_HTML_TO_DAT_CONVERTER
		  fprintf(stderr, "IRREGULAR html_line: \"%s\"\n",
			  html_line_buffer);
#endif
		  html_line_buffer = NULL;
		}
	    }

	  if (html_line_buffer != NULL)
	    G_FREE(html_line_buffer);

	  html_buffer_pos = eol_pos + 1;
	  rest_of_data -= line_size;
	  html_offset += line_size;
	}

      if (html_buffer->fixed)
	{
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
	  fprintf(stderr, "conversion done.\n");
#endif
	  g_signal_emit_by_name(G_OBJECT(args->broker),
				"access_completed",
				dat_buffer,
				&signal_result);

	  if (previous_line != NULL)
	    {
	      G_FREE(previous_line);
	      previous_line = NULL;
	    }
	  break;
	}

      if (!ochusha_async_buffer_wait(html_buffer))
	{
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
	  fprintf(stderr, "wait terminated.\n");
#endif
	  g_signal_emit_by_name(G_OBJECT(args->broker),
				"access_terminated",
				dat_buffer,
				&signal_result);
	  if (previous_line != NULL)
	    {
	      G_FREE(previous_line);
	      previous_line = NULL;
	    }
	  break;
	}
#if DEBUG_HTML_TO_DAT_CONVERTER_MOST
      else
	fprintf(stderr, "wait signaled.\n");
#endif
    }
  ochusha_async_buffer_unlock(html_buffer);

  ochusha_async_buffer_active_unref(html_buffer);
  ochusha_async_buffer_active_unref(dat_buffer);

  /* ɬפʤХåեƤ򥭥å˽񤭹 */
  if (dat_buffer->length > incoming_offset)
    {
      const char *fake_dat_url = ochusha_thread_2ch_get_dat_url(thread_2ch);
      int fd = ochusha_config_cache_open_file(args->broker->config,
					      fake_dat_url,
					      O_WRONLY | O_TRUNC | O_CREAT);
      if (fd >= 0)
	{
	  ssize_t len = write(fd, (void *)dat_buffer->buffer,
			      dat_buffer->length);
	  close(fd);
	  if (len != dat_buffer->length)
	    {
	      ochusha_config_cache_unlink_file(args->broker->config,
					       fake_dat_url);
	      fd = -1;
	    }
	}
#if DEBUG_NETWORK
      if (fd < 0)
	fprintf(stderr, "Couldn't update cache file for %s: %s (%d)\n",
		fake_dat_url, strerror(errno), errno);
#endif	      
    }

  ochusha_async_buffer_fix(dat_buffer);

 terminated:
  if (title_pattern != NULL)
    regex_free(title_pattern);
  if (response_pattern != NULL)
    regex_free(response_pattern);
  if (match_region != NULL)
    regex_region_free(match_region, 1);
  OCHU_OBJECT_UNREF(dat_buffer);
  OCHU_OBJECT_UNREF(args->broker);
}


static void
unref_html_buffer(OchushaAsyncBuffer *html_buffer)
{
  /* dat_buffer̤Ȥ˸ƤФ롣*/
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "unref_html_buffer: %p\n", html_buffer);
#endif
  OCHU_OBJECT_UNREF(html_buffer);
}


/*
 * ochusha_utils_jbbs_fake_responses_source
 *
 * cgiͳhtmlʥɤ߹ߡ2chDATѴХåե֤
 * cgiͳǤɹϥͥåȥ򤹤륹åɤǹԤ2chDAT
 * ѴΤѤΥåɤǹԤ
 *
 * δؿ֤Τϡ̥åɤˤäDATѴ줿̤Ǽ
 * ХåեǤꡢͥåȥľܻȤХåե̤˺롣
 */
static OchushaAsyncBuffer *
ochusha_thread_jbbs_fake_responses_source(OchushaThread2ch *thread_2ch,
					  OchushaNetworkBroker *broker,
					  OchushaAsyncBuffer *buffer,
					  OchushaNetworkBrokerCacheMode mode)
{
  OchushaAsyncBuffer *dat_buffer = NULL;
  OchushaAsyncBuffer *html_buffer;
  OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(thread_2ch);
  OchushaBoard2ch *board_2ch = OCHUSHA_BOARD_2CH(thread->board);
  const char *fake_dat_url = ochusha_thread_2ch_get_dat_url(thread_2ch);
  WorkerJob *job;
  HTML2DATConverterJobArgs *args;
  int incoming_offset = 0;
  int number_of_cached_responses = 0;
  char cgi_url[PATH_MAX];

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
    case OCHUSHA_BBS_TYPE_MITINOKU:
      break;

    default:
      return NULL;	/* ̤ */
    }

  if (mode != OCHUSHA_NETWORK_BROKER_CACHE_IGNORE)
    {
      if (buffer == NULL || !ochusha_async_buffer_reset(buffer))
	{
	  int fd = ochusha_config_cache_open_file(broker->config, fake_dat_url,
						  O_RDONLY);
	  if (fd >= 0)
	    {
	      /* å夢ꡣ*/
	      dat_buffer = ochusha_async_buffer_new(NULL, 0, NULL);
	      incoming_offset = lseek(fd, 0, SEEK_END);
	      lseek(fd, 0, SEEK_SET);
	      if (ochusha_async_buffer_resize(dat_buffer, incoming_offset))
		{
		  incoming_offset = read(fd, (void *)dat_buffer->buffer,
					 incoming_offset);
		  if (!ochusha_async_buffer_update_length(dat_buffer,
							  incoming_offset))
		    {
#if DEBUG_ASYNC_BUFFER
		      fprintf(stderr, "ochusha_utils_jbbs_fake_responses_source(): buffer may have been terminated.\n");
#endif
		      /* ξϤɤ褦ʤ */
		    }
		}
	      close(fd);
	    }
	}
      else
	{
	  dat_buffer = buffer;
	  incoming_offset = buffer->length;
	}

      /* ¸Υ쥹 */
      if (incoming_offset > 0)
	{
	  char *tmp_pos = (char *)dat_buffer->buffer;
	  char *tail_pos = tmp_pos + incoming_offset;
	  while ((tmp_pos = memchr(tmp_pos, '\n', tail_pos - tmp_pos))
		 != NULL)
	    {
	      number_of_cached_responses++;
	      tmp_pos++;
	    }
	}
    }

  /* λǼεDATե뤬ѲǽʾΤdat_bufferNULL */

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_JBBS:
    case OCHUSHA_BBS_TYPE_MACHIBBS:
    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      if (snprintf(cgi_url, PATH_MAX, "%s?BBS=%s&KEY=%s&START=%d%s",
		   ochusha_board_2ch_get_read_cgi_url(board_2ch),
		   thread->board->id, thread->id,
		   number_of_cached_responses + 1,
		   number_of_cached_responses == 0 ? "" : "&NOFIRST=TRUE")
	  >= PATH_MAX)
	cgi_url[0] = '\0';
      break;

    case OCHUSHA_BBS_TYPE_MITINOKU:
      if (thread->number_of_responses_on_server <= number_of_cached_responses)
	return dat_buffer;

      if (snprintf(cgi_url, PATH_MAX, "%s?bbs=%s&key=%s&st=%d&to=%d%s",
		   ochusha_board_2ch_get_read_cgi_url(board_2ch),
		   thread->board->id, thread->id,
		   number_of_cached_responses + 1,
		   thread->number_of_responses_on_server,
		   number_of_cached_responses == 0 ? "" : "&nofirst=true")
	  >= PATH_MAX)
	cgi_url[0] = '\0';
      break;

    default:	/* ̤ݡ */
      cgi_url[0] = '\0';
      break;
    }

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "cgi_url=\"%s\"\n", cgi_url);
#endif

  if (cgi_url[0] != '\0')
#if 1	/* ʤǤΤasync⡼ɤǥȼԤ롣*/
    html_buffer = ochusha_network_broker_read_from_cgi(broker, NULL, cgi_url);
#else
    html_buffer = ochusha_network_broker_read_from_url(broker, NULL, cgi_url,
						       NULL, mode);
#endif
  else
    html_buffer = NULL;

  if (html_buffer == NULL)
    {
      if (dat_buffer != NULL)
	{
	  gboolean signal_result;
	  g_signal_emit_by_name(G_OBJECT(broker),
				"access_failed",
				dat_buffer,
				OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
				_("Access failed: unknown reason"),
				&signal_result);
	  ochusha_async_buffer_fix(dat_buffer);
	}
      return dat_buffer;
    }

  if (dat_buffer == NULL)
    {
      if (buffer == NULL || !ochusha_async_buffer_reset(buffer))
	dat_buffer = ochusha_async_buffer_new(NULL, 0, NULL);
      else
	{
	  dat_buffer = buffer;
	  ochusha_async_buffer_update_length(buffer, 0);
	}
    }

  args = G_NEW0(HTML2DATConverterJobArgs, 1);
  args->html_buffer = html_buffer;
  args->dat_buffer = dat_buffer;
  args->broker = broker;
  args->thread_2ch = thread_2ch;
  args->incoming_offset = incoming_offset;
  args->number_of_cached_responses = number_of_cached_responses;

  g_object_set_data(G_OBJECT(dat_buffer), "OchushaNetworkBroker::BufferStatus",
		    g_object_get_data(G_OBJECT(html_buffer),
				      "OchushaNetworkBroker::BufferStatus"));
  g_object_set_data_full(G_OBJECT(dat_buffer), "OchushaUtilsJBBS::HTMLBuffer",
			 html_buffer, (GDestroyNotify)unref_html_buffer);

  job = G_NEW0(WorkerJob, 1);
  job->canceled = FALSE;
  job->job = (JobFunc *)convert_html_to_dat;
  job->args = args;

  /* åɤΤ˳ */
  OCHU_OBJECT_REF(dat_buffer);
  OCHU_OBJECT_REF(broker);

  commit_job(job);

  return dat_buffer;
}

