/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_utils_2ch.c,v 1.35 2004/01/14 09:59:48 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_bbs_table.h"
#include "ochusha_board_2ch.h"
#include "ochusha_thread_2ch.h"
#include "ochusha_utils_2ch.h"

#include "utils.h"

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define GUESS_2CH_BOARD_URL	1


static gboolean
is_thread_id(const char *string)
{
  int len;
  const char *pos;
  g_return_val_if_fail(string != NULL, FALSE);

  len = strlen(string);
  if (len < 9)
    return FALSE;

  pos = string;
  if (strchr("123456789", *pos) == NULL)
    return FALSE;

  pos++;
  while (*pos != '\0')
    {
      if (strchr("0123456789", *pos) == NULL)
	return FALSE;
      pos++;
    }

  return TRUE;
}


static gboolean
parse_thread_path(char *path, char **thread_id_p,
		  unsigned int *from_p, unsigned int *to_p)
{
  char *thread_id = NULL;
  unsigned int from = 0;
  unsigned int to = 0;
  char *slash_pos = strchr(path + 1, '/');

#if 0
  fprintf(stderr, "parse_thread_path()\n");
  fprintf(stderr, "  path: %s\n", path);
#endif

  if (slash_pos != NULL)
    {
      if (thread_id_p != NULL)
	thread_id = G_STRNDUP(path, slash_pos - path);
      else
	thread_id = path;

      if (is_thread_id(thread_id))
	{
	  path = slash_pos + 1;

	  if (*path != '\0')
	    {
	      char tmp_char;
	      char *hyphen_pos = strchr(path, '-');
	      if (hyphen_pos != NULL)
		{
		  tmp_char = *hyphen_pos;
		  *hyphen_pos = '\0';
		}
	      else
		tmp_char = '\0';

	      if (hyphen_pos != path)
		{
		  if (sscanf(path, "%d", &from) != 1)
		    from = 0;
		}

	      if (hyphen_pos != NULL && hyphen_pos[1] != '\0')
		{
		  if (sscanf(hyphen_pos + 1, "%d", &to) != 1)
		    to = from;
		}
	    }
	}
      else
	{
	  if (thread_id != path)
	    G_FREE(thread_id);
	  thread_id = NULL;
	}
    }
  else
    {
      if (thread_id_p != NULL && *path != '\0')
	thread_id = G_STRDUP(path);
      else
	thread_id = NULL;
    }

  if (thread_id_p != NULL)
    *thread_id_p = thread_id;

  if (from_p != NULL)
    *from_p = from;

  if (to_p != NULL)
    *to_p = to;

#if 0
  fprintf(stderr, "  thread_id: %s\n", thread_id);
  fprintf(stderr, "  from=%d, to=%d\n", from, to);
#endif

  return thread_id != NULL;
}


static gboolean
parse_cgi_query(const char *query, char **board_id_p, char **thread_id_p,
		unsigned int *from_p, unsigned int *to_p)
{
  char board_id[PATH_MAX];
  char thread_id[PATH_MAX];
  char tmp_buf[PATH_MAX];
  unsigned int from = 0;
  unsigned int to = 0;
  char *tmp_pos;
  CGIQueryKeyTuple tuple;

#if 0
  fprintf(stderr, "parse_cgi_query()\n");
  fprintf(stderr, "  query: %s\n", query);
#endif

  board_id[0] = '\0';
  thread_id[0] = '\0';

  tmp_pos = strchr(query, '?');
  if (tmp_pos != NULL)
    query = tmp_pos + 1;	/* ǽ'?'ޤǤФ */

  while ((tmp_pos = ochusha_utils_find_cgi_query_key(query, &tuple)) != NULL)
    {
      if (g_ascii_strncasecmp("BBS", tuple.name, tuple.name_len) == 0)
	{
	  if (tuple.value_len < (PATH_MAX - 1))
	    {
	      memcpy(board_id, tuple.value, tuple.value_len);
	      board_id[tuple.value_len] = '\0';
	    }
	}
      else if (g_ascii_strncasecmp("KEY", tuple.name, tuple.name_len) == 0)
	{
	  if (tuple.value_len < (PATH_MAX - 1))
	    {
	      memcpy(thread_id, tuple.value, tuple.value_len);
	      thread_id[tuple.value_len] = '\0';
	    }
	}
      else if (g_ascii_strncasecmp("START", tuple.name, tuple.name_len) == 0
	       || g_ascii_strncasecmp("ST", tuple.name, tuple.name_len) == 0)
	{
	  if (tuple.value_len > 0 && tuple.value_len < (PATH_MAX - 1))
	    {
	      memcpy(tmp_buf, tuple.value, tuple.value_len);
	      tmp_buf[tuple.value_len] = '\0';
	      if (sscanf(tmp_buf, "%d", &from) != 1)
		from = 0;
	    }
	}
      else if (g_ascii_strncasecmp("END", tuple.name, tuple.name_len) == 0
	       || g_ascii_strncasecmp("TO", tuple.name, tuple.name_len) == 0)
	{
	  if (tuple.value_len > 0 && tuple.value_len < (PATH_MAX - 1))
	    {
	      memcpy(tmp_buf, tuple.value, tuple.value_len);
	      tmp_buf[tuple.value_len] = '\0';
	      if (sscanf(tmp_buf, "%d", &to) != 1)
		to = 0;
	    }
	}

      query = tmp_pos;
    }

  if (from == 0 && to != 0)
    from = to;
  else if (to == 0 && from != 0)
    to = from;

  if (from_p != NULL)
    *from_p = from;

  if (to_p != NULL)
    *to_p = to;

  if (board_id[0] == '\0')
    thread_id[0] = '\0';

  if (board_id_p != NULL)
    {
      if (board_id[0] != '\0')
	*board_id_p = G_STRDUP(board_id);
      else
	*board_id_p = NULL;
    }

  if (thread_id_p != NULL)
    {
      if (thread_id[0] != '\0')
	*thread_id_p = G_STRDUP(thread_id);
      else
	*thread_id_p = NULL;
    }

#if 0
  fprintf(stderr, "  board_id: %s\n", board_id);
  fprintf(stderr, "  thread_id: %s\n", thread_id);
  fprintf(stderr, "  from=%d, to=%d\n", from, to);
#endif

  return board_id[0] != '\0';
}


/*
 * Ϳ줿URLϤбƤʷǼĤΤΤɤ
 * Ƚꤹ롣
 *
 * Ƚ˲äʲΤ褦Ѥ򵯤
 *
 * ȤͿ줿board_url_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ
 *   tableˤ̤Ͽʾ硢ĤURLȤ餷ʸΥԡ
 *   board_url_p˳Ǽ롣
 * ȤͿ줿bbs_type_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ
 *   tableˤ̤Ͽʾ硢ļ̤Ȥ餷ͤ
 *   bbs_type_p˳Ǽ롣
 * ȤͿ줿thread_id_pfrom_pto_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ硢
 *   ID(thread_id)䡢쥹ϰϻƬ(from)(to)򤽤줾
 *   thread_id_pfrom_pto_p˳Ǽ롣thread_id_p˳ǼΤ
 *   IDʸΥԡ
 */
gboolean
ochusha_utils_2ch_check_url(const char *url, char **board_url_p,
			    int *bbs_type_p,
			    char **board_id_p, char **thread_id_p,
			    unsigned int *from_p, unsigned int *to_p,
			    gboolean *is_kako_html_p)
{
  char *server;
  char *abs_path;
  char *path_pos;
  int bbs_type = OCHUSHA_BBS_TYPE_UNKNOWN;
  char *thread_id = NULL;
  char *board_id = NULL;
  int from = 0;
  int to = 0;
  char board_url[PATH_MAX];
  char tmp_buf[PATH_MAX];

  g_return_val_if_fail(url != NULL, FALSE);

  board_url[0] = '\0';

  server = ochusha_utils_url_extract_http_server(url);
  if (server != NULL)
    abs_path = ochusha_utils_url_extract_http_absolute_path(url);
  else if (board_url_p == NULL)
    {
      const char *tmp_pos = url;
      server = G_STRDUP("localhost");	/*  */
      while (TRUE)
	{
	  if (strncmp(tmp_pos, "../", 3) == 0)
	    {
	      tmp_pos += 2;
	      continue;
	    }
	  if (strncmp(tmp_pos, "/../", 4) == 0)
	    {
	      tmp_pos += 3;
	      continue;
	    }
	  if (strncmp(tmp_pos, "./", 2) == 0)
	    {
	      tmp_pos ++;
	      continue;
	    }
	  if (strncmp(tmp_pos, "/./", 3) == 0)
	    {
	      tmp_pos += 2;
	      continue;
	    }
	  break;
	}
      abs_path = G_STRDUP(tmp_pos);
    }
  else
    abs_path = NULL;

#if 0
  fprintf(stderr, "ochusha_utils_2ch_check_url()\n");
  fprintf(stderr, "  url: %s\n", url);
  fprintf(stderr, "  server: %s\n", server);
  fprintf(stderr, "  abs_path: %s\n", abs_path);
#endif

  if (server == NULL || abs_path == NULL || *abs_path != '/')
    goto done;

  path_pos = abs_path + 1;
  if (strncmp(path_pos, "test/read.cgi", 13) == 0)
    {
      bbs_type = OCHUSHA_BBS_TYPE_2CH;
      if (path_pos[13] == '/')
	{
	  char *slash_pos = strchr(path_pos + 14, '/');
	  if (slash_pos != NULL)
	    {
	      char tmp_char = slash_pos[1];
	      slash_pos[1] = '\0';
	      if (snprintf(board_url, PATH_MAX, "http://%s%s", server,
			   path_pos + 13) < PATH_MAX)
		{
#if 0
		  fprintf(stderr, "  board_url: %s\n", board_url);
#endif

		  slash_pos[1] = tmp_char;
		  tmp_char = slash_pos[0];
		  slash_pos[0] = '\0';
		  board_id = G_STRDUP(path_pos + 14);
		  slash_pos[0] = tmp_char;

		  parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
		}
	      else
		board_url[0] = '\0';
	    }
	}
      else if (path_pos[13] == '?')
	{
	  if (parse_cgi_query(path_pos + 14, &board_id, &thread_id,
			      &from, &to))
	    {
	      if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server,
			   board_id) >= PATH_MAX)
		board_url[0] = '\0';
#if 0
	      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	    }
	}
    }
  else if (g_ascii_strcasecmp(server, "jbbs.shitaraba.com") == 0)
    {
      /* JBBSФ̯ʴ*/
      char *tmp_pos = strstr(path_pos, "bbs/read.cgi");
      bbs_type = OCHUSHA_BBS_TYPE_JBBS_SHITARABA;
      if (tmp_pos == NULL)
	{
	  /* URLȤƤ
	   * http://jbbs.shitaraba.com/computer/351/
	   * ߤʤΤ⤢롣
	   */
	  tmp_pos = strchr(path_pos, '/');
	  if (tmp_pos != NULL)
	    {
	      char *board_id_head = tmp_pos + 1;
	      tmp_pos = strchr(board_id_head, '/');
	      if (tmp_pos != NULL)
		{
		  char tmp_char = tmp_pos[1];
		  tmp_pos[1] = '\0';
		  if (snprintf(board_url, PATH_MAX, "http://%s/%s", server,
			       path_pos) < PATH_MAX)
		    board_id = G_STRDUP(board_id_head);
		  else
		    board_url[0] = '\0';
		  tmp_pos[1] = tmp_char;
		}
	    }
	}
      else
	{
	  /*
	   * URLȤƤϰʲ2ĤηΤʤ¾ˤ⤢뤫⡣
	   */
	  if (tmp_pos[12] == '/')
	    {
	      /*
	       * http://jbbs.shitaraba.com/bbs/read.cgi/computer/351/1040452877/109
	       * URL
	       */
	      char *slash_pos = strchr(tmp_pos + 13, '/');
	      if (slash_pos != NULL)
		{
		  char *board_id_head = slash_pos + 1;
		  slash_pos = strchr(board_id_head, '/'); /* 2ܤ'/'õ*/
		  if (slash_pos != NULL)
		    {
		      /* (tmp_pos + 12)slash_posޤǤʸ
		       * "/computer/351/"ʬˤʤ
		       */
		      *slash_pos = '\0';
		      if (snprintf(board_url, PATH_MAX, "http://%s%s/", server,
				   tmp_pos + 12) < PATH_MAX)
			{
			  board_id = G_STRDUP(board_id_head);
			  *slash_pos = '/';
			  parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
			}
		      else
			board_url[0] = '\0';
		    }
		}
	    }
	  else if (tmp_pos[12] == '?')
	    {
	      /*
	       * http://jbbs.shitaraba.com/computer/bbs/read.cgi?BBS=351&KEY=1040452877&START=109&END=109&NOFIRST=TRUE
	       * URL
	       */
	      if (parse_cgi_query(tmp_pos + 13, &board_id, &thread_id,
				  &from, &to))
		{
		  *tmp_pos = '\0';
		  if (snprintf(board_url, PATH_MAX, "http://%s%s%s/",
			       server, abs_path, board_id) >= PATH_MAX)
		    board_url[0] = '\0';
		}
	    }
	}
    }
  else if (strncmp(path_pos, "bbs/read.pl?", 12) == 0)
    {
      bbs_type = OCHUSHA_BBS_TYPE_MACHIBBS;
      if (parse_cgi_query(path_pos + 12, &board_id, &thread_id,
			  &from, &to))
	{
	  if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server, board_id)
	      >= PATH_MAX)
	    board_url[0] = '\0';
#if 0
	  fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	}
    }
  else if (strncmp(path_pos, "read.cgi?", 9) == 0)
    {
      bbs_type = OCHUSHA_BBS_TYPE_MITINOKU;
      if (parse_cgi_query(path_pos + 9, &board_id, &thread_id,
			  &from, &to))
	{
	  if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server, board_id)
	      >= PATH_MAX)
	    board_url[0] = '\0';
#if 0
	  fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	}
    }
  else if (strncmp(path_pos, "bbs/read.cgi", 12) == 0)
    {
      if (path_pos[12] == '/')
	{
	  /* JBBSФ../../bbs/read.cgi/ϤURL⤳ˤ뤫*/
	  char *slash_pos = strchr(path_pos + 13, '/');
#if 0
	  fprintf(stderr, "slash_pos=%p\n", slash_pos);
#endif
	  if (slash_pos != NULL)
	    {
	      char *tmp_pos = strchr(slash_pos + 1, '/');
	      int len;
#if 0
	      fprintf(stderr, "tmp_pos=%p\n", tmp_pos);
#endif
	      if (tmp_pos != NULL && tmp_pos < (slash_pos + 10))
		{
		  /* ɥۥåʵ§ġġ*/
		  bbs_type = OCHUSHA_BBS_TYPE_JBBS_SHITARABA;
		  *tmp_pos = '\0';
		  board_id = G_STRDUP(slash_pos + 1);
		  *tmp_pos = '/';
		  slash_pos = tmp_pos;
		}
	      else
		{
		  /* ФǤʤJBBSäƸߤΤ*/
		  bbs_type = OCHUSHA_BBS_TYPE_JBBS;
		}
	      len = slash_pos - (path_pos + 13);
	      memcpy(tmp_buf, path_pos + 13, len);
	      tmp_buf[len] = '\0';
	      if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server,
			   tmp_buf) < PATH_MAX)
		{
		  if (board_id == NULL)
		    board_id = G_STRDUP(tmp_buf);
#if 0
		  fprintf(stderr, "board_id=\"%s\"\n", board_id);
		  fprintf(stderr, "tmp_buf=\"%s\"\n", tmp_buf);
		  fprintf(stderr, "board_url=\"%s\"\n", board_url);
#endif
		  parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
		}
	      else
		{
		  bbs_type = OCHUSHA_BBS_TYPE_UNKNOWN;
		  board_url[0] = '\0';
		}
#if 0
	      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	    }
	}
      else if (path_pos[12] == '?')
	{
	  if (parse_cgi_query(path_pos + 13, &board_id, &thread_id,
			      &from, &to))
	    {
	      if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server,
			   board_id) < PATH_MAX)
		bbs_type = OCHUSHA_BBS_TYPE_JBBS;
	      else
		board_url[0] = '\0';
#if 0
	      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	    }
	}
    }
  else
    {
      /* 2chäݤġġ*/
      char *slash_pos = strchr(path_pos, '/');
      if (slash_pos != NULL)
	{
	  char *html_pos;
	  slash_pos[0] = '\0';
	  snprintf(board_url, PATH_MAX, "http://%s%s/", server, abs_path);
	  board_id = G_STRDUP(abs_path + 1);
	  slash_pos[0] = '/';
	  parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
	  if (thread_id == NULL && is_kako_html_p != NULL
	      && strncmp(slash_pos + 1, "kako/", 5) == 0
	      && (html_pos = strstr(slash_pos + 6, ".html")) != NULL)
	    {
	      slash_pos = strrchr(slash_pos + 6, '/');
	      if (slash_pos != NULL && slash_pos < html_pos)
		{
		  thread_id = G_STRNDUP(slash_pos + 1,
					html_pos - slash_pos - 1);
		  if (is_thread_id(thread_id))
		    *is_kako_html_p = TRUE;
		  else
		    {
		      G_FREE(thread_id);
		      thread_id = NULL;
		    }
		}
	    }
	  bbs_type = OCHUSHA_BBS_TYPE_2CH;
	}
#if 0
      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
    }

 done:
  if (server != NULL)
    G_FREE(server);

  if (abs_path != NULL)
    G_FREE(abs_path);

  if (board_url[0] == '\0')
    {
      /*  */
      if (board_id != NULL)
	{
	  G_FREE(board_id);
	  board_id = NULL;
	}
      if (thread_id != NULL)
	{
	  G_FREE(thread_id);
	  thread_id = NULL;
	}
      bbs_type = OCHUSHA_BBS_TYPE_UNKNOWN;
    }

  if (board_url_p != NULL)
    {
      if (board_url[0] != '\0')
	*board_url_p = G_STRDUP(board_url);
      else
	*board_url_p = NULL;
    }

  if (bbs_type_p != NULL)
    *bbs_type_p = bbs_type;

  if (board_id_p != NULL)
    *board_id_p = board_id;
  else if (board_id != NULL)
    G_FREE(board_id);

  if (thread_id_p != NULL)
    *thread_id_p = thread_id;
  else if (thread_id != NULL)
    G_FREE(thread_id);

  if (from_p != NULL)
    *from_p = from;

  if (to_p != NULL)
    *to_p = to;

  return board_url[0] != '\0';
}


OchushaAsyncBuffer *
ochusha_utils_2ch_get_bbsmenu(OchushaNetworkBroker *broker)
{
  return ochusha_network_broker_read_from_url(broker, NULL,
				broker->config->bbsmenu_url, NULL,
				OCHUSHA_NETWORK_BROKER_CACHE_IGNORE);
}


static gboolean
is_2ch_board_url(char *url)
{
  if (strstr(url, ".html") != NULL)
    return FALSE;
  if (strstr(url, "del_2ch") != NULL)
    return FALSE;
  return TRUE;
}


static gboolean
extract_boards(OchushaBBSTable *table, OchushaBoardCategory *category,
	       iconv_t converter,
	       char *head, char *tail,
	       EachBoardCallback *each_board_callback,
	       BoardMovedCallback *board_moved_callback,
	       gpointer callback_data)
{
  /* ǽ(HTMLA)õ*/
  char *cur_pos = g_strstr_len(head, tail - head, "<A HREF=http");
  if (cur_pos == NULL)
    return FALSE;
  cur_pos += 8;	/* skip "<A HREF=" */

  while (cur_pos != NULL && cur_pos < tail)
    {
      char *tag_tail = memchr(cur_pos, '>', tail - cur_pos);
      char *url_tail;
      gchar *name = NULL;
      char *url = NULL;
      char *close_tag;
      OchushaBulletinBoard *board;
      OchushaBulletinBoard *board_by_name;
      gboolean is_machibbs = FALSE;

      if (tag_tail == NULL)
	return FALSE;	/* Ƥ롩 */

      url_tail = strpbrk(cur_pos, " \t\r\n>");
      /* tag_tail != NULLʤΤurl_tail != NULL*/

      close_tag = g_strstr_len(url_tail, tail - url_tail, "</A>");
      if (close_tag == NULL)
	return FALSE;	/* Ƥ롩 */

#if GUESS_2CH_BOARD_URL
      {
	/* 2chİʳؤΥ󥫡ˤTARGE°դƤ̣*/
	char *target = g_strstr_len(cur_pos, close_tag - cur_pos, "TARGET=");
	if (target != NULL)
	  {
	    /* ޤBBS⥵ݡȤͽ */
	    if (g_strstr_len(cur_pos, close_tag - cur_pos, "machi.to") != NULL)
	      is_machibbs = TRUE;
	    else
	      goto search_next;
	  }
      }
#endif
      name = convert_string(converter, cp932_to_utf8_helper,
			    tag_tail + 1, close_tag - tag_tail - 1);
      url = G_STRNDUP(cur_pos, url_tail - cur_pos);
#if GUESS_2CH_BOARD_URL
      if (!is_2ch_board_url(url))
	goto search_next;
#endif
      board = ochusha_bbs_table_lookup_board_by_url(table, url);
      board_by_name = ochusha_bbs_table_lookup_board_by_name(table, name);

      if (board == NULL)
	{
#if GUESS_2CH_BOARD_URL
	  const char *id;
#endif
	  if (name == NULL)
	    {
	      G_FREE(url);
	      return FALSE;	/* Out of memory */
	    }

	  board = ochusha_board_2ch_new(name, url);
	  if (is_machibbs)
	    board->bbs_type = OCHUSHA_BBS_TYPE_MACHIBBS;

#if GUESS_2CH_BOARD_URL
	  id = ochusha_bulletin_board_get_id(board);
	  /* ФΥ롼ľηǼĤϸʤġġ*/
	  if (id == NULL || id[0] == '\0')
	    {
	      OCHU_OBJECT_UNREF(board);
	      board = NULL;
	      goto search_next;
	    }
#endif

	  if (board_by_name != NULL && board_moved_callback != NULL)
	    {
	      OchushaBulletinBoard *winner
		= (*board_moved_callback)(board_by_name, board, callback_data);
	      /* XXX:
	       *   winnerNULLʤ()boardɲá
	       *   winnerboard_by_nameʤ()boardΤƤƼء
	       *   winnerboardʤ(Ť)board_by_nameΤƤơ
	       *   ()boardɲäפˡƥ꤫board_by_name
	       *   
	       */
	      if (winner == board_by_name)
		{
		  /* boardϥơ֥ɲäʤ */
		  OCHU_OBJECT_UNREF(board);
		  board = NULL;
		  goto search_next;
		}

	      if (winner != NULL)
		{
		  /* (winner == board)ȸʤ */
		  ochusha_board_category_remove_board(category, board_by_name);
		  ochusha_bbs_table_remove_board(table, board_by_name);
		}
	    }

	  if (board != NULL)
	    {
	      if (each_board_callback != NULL)
		{
		  gboolean result = (*each_board_callback)(board,
							   callback_data);
		  if (!result)
		    {
		      OCHU_OBJECT_UNREF(board);
		      return FALSE;
		    }
		}
	      ochusha_bbs_table_add_board(table, board);
	      OCHU_OBJECT_UNREF(board);
	    }
	}

      board->killed = FALSE;
      ochusha_bulletin_board_set_name(board, name);
      ochusha_board_category_add_board(category, board);

    search_next:
      if (url != NULL)
	G_FREE(url);
      if (name != NULL)
	G_FREE(name);
      /* (HTMLA)õ*/
      cur_pos = g_strstr_len(close_tag + 4, tail - close_tag, "<A HREF=http");
      if (cur_pos == NULL)
	return TRUE;
      cur_pos += 8;	/* skip "<A HREF=" */
    }
  return TRUE;
}


/*
 * ochusha_utils_2ch_analyze_bbsmenu
 *
 * Ϳ줿bufferbbsmenu.htmlǤȸʤƲϤ˴ޤޤ
 * ƥƥбOchushaBoardCategoryȳĤб
 * OchushaBulletinBoard(Υ֥饹Υ֥OchushaBoard2ch)ۤ
 * Ϳ줿OchushaBBSTable򹹿롣
 *
 * ޤ˥ХåؿͿ줿硢ƥƥꡢĤˤĤ
 * б륳ХåؿƤ֡
 * ХåؿFALSE֤硢ǲϤλ롣
 *
 * OchushaBBSTableURLOchushaBulletinBoardʤäˤؤ餺
 * ̾Ǥϰ褦ʾ硢İžεǻʤΤǤ˴ؤƤ⥳
 * Хåؿ(ͿƤ)Ƥ֡ХåؿϰȤƥݥ
 * Ϳ뿷ĤOchushaBulletinBoard⡢OchushaBBSTable˻Ĥ
 * ؤΥݥ󥿤֤ޤNULL֤ˤϿξĤ롣
 *
 * Ϥ˽λTRUE֤
 */
gboolean
ochusha_utils_2ch_analyze_bbsmenu(OchushaBBSTable *table,
				  OchushaAsyncBuffer *buffer,
				  EachCategoryCallback *category_callback,
				  EachBoardCallback *board_callback,
				  BoardMovedCallback *moved_callback,
				  gpointer callback_data)
{
  gboolean result = TRUE;
  iconv_t converter;
  OchushaNetworkBrokerBufferStatus *status;

  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table)
		       && OCHUSHA_IS_ASYNC_BUFFER(buffer), FALSE);

  status = g_object_get_data(G_OBJECT(buffer),
			     "OchushaNetworkBroker::BufferStatus");

  g_return_val_if_fail(status != NULL, FALSE);

  converter = iconv_open("UTF-8//IGNORE", "CP932");  /* ϡɥǥ */
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  if (!ochusha_async_buffer_active_ref(buffer))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      iconv_close(converter);
      return FALSE;
    }

  ochusha_async_buffer_lock(buffer);
  {
    unsigned int offset = 0;

    while (result)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	unsigned int length = buffer->length;
	unsigned int rest_of_data = length - offset;

	while (rest_of_data > 0)
	  {
	    char *end_name_pos;
	    gchar *category_name;
	    char *end_category_pos;
	    OchushaBoardCategory *category;

	    cur_pos = g_strstr_len(cur_pos, rest_of_data, "<B>");

	    if (cur_pos == NULL)
	      break;	/* ǡ­ʤ⤦ƥ꤬ʤ */
	    cur_pos += 3;	/* skip "<B>" */
	    rest_of_data -= 3;

	    end_name_pos = g_strstr_len(cur_pos, rest_of_data, "</B>");

	    if (end_name_pos == NULL)
	      break;	/* ǡ­ʤ */

	    /* ߤΥƥϼΥƥľޤ */
	    end_category_pos = g_strstr_len(end_name_pos + 4,
					    rest_of_data - (end_name_pos
							    - cur_pos),
					    "<B>");
	    /* ǸΥƥ</BODY>ľޤ */
	    if (end_category_pos == NULL)
	      end_category_pos = g_strstr_len(end_name_pos + 4,
					      rest_of_data - (end_name_pos
							      - cur_pos),
					      "</BODY>");
	    if (end_category_pos == NULL)
	      break;

	    category_name = convert_string(converter, cp932_to_utf8_helper,
					   cur_pos, end_name_pos - cur_pos);

	    category = ochusha_bbs_table_lookup_category(table, category_name);
	    if (category == NULL)
	      category = ochusha_board_category_new(category_name);
	    G_FREE(category_name);

	    cur_pos = end_name_pos + 4;	/* skip "</B>" */
	    rest_of_data -= 4;
	      
	    result = extract_boards(table, category, converter,
				    cur_pos, end_category_pos,
				    board_callback, moved_callback,
				    callback_data);

	    if (!result || category->board_list == NULL)
	      OCHU_OBJECT_UNREF(category);
	    else
	      {
		category->killed = FALSE;
		ochusha_bbs_table_add_category(table, category);
		if (category_callback != NULL
		    && !(*category_callback)(category, callback_data))
		  {
		    result = FALSE;
		    break;
		  }
	      }

	    cur_pos = end_category_pos;
	    offset = cur_pos - buffer_top;
	    rest_of_data = (length - offset);
	  }

	if (buffer->fixed)
	  break;

	if (!ochusha_async_buffer_wait(buffer))
	  {
	    /* bufferϤ޲٤callerˤޤ */
#if DEBUG_ASYNC_BUFFER_MOST
	    fprintf(stderr, "ochusha_utils_2ch_analyze_bbsmenu(): buffer has been terminated.\n");
#endif
	    result = FALSE;
	    break;
	  }

	if (status->state
	    == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
	  {
	    result = FALSE;
	    break;
	  }
      }
  }
  ochusha_async_buffer_unlock(buffer);

  ochusha_async_buffer_active_unref(buffer);

  iconv_close(converter);

  return result;
}


OchushaUtils2chPostResult
ochusha_utils_2ch_try_post(OchushaNetworkBroker *broker,
			   OchushaBulletinBoard *board, const char *message)
{
  char url[PATH_MAX];
  const char *cookie;
  const char *new_cookie = NULL;
  OchushaBoard2ch *board_2ch;
  gboolean post_result = FALSE;
  int result;
  OchushaNetworkBrokerPostStatus post_status;

  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(board),
		       OCHUSHA_UTILS_2CH_POST_FAILURE);

  switch (board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
      result = snprintf(url, PATH_MAX, "http://%s/test/bbs.cgi",
			ochusha_bulletin_board_get_server(board));
      break;

    case OCHUSHA_BBS_TYPE_MACHIBBS:
      result = snprintf(url, PATH_MAX, "http://%s/bbs/write.cgi",
			ochusha_bulletin_board_get_server(board));
      break;

    case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
      result = snprintf(url, PATH_MAX, "http://%s/bbs/write.cgi",
			ochusha_bulletin_board_get_server(board));
      break;

    default:
      return OCHUSHA_UTILS_2CH_POST_FAILURE;
    }
  g_return_val_if_fail(result < PATH_MAX, OCHUSHA_UTILS_2CH_POST_FAILURE);

  board_2ch = OCHUSHA_BOARD_2CH(board);
  cookie = ochusha_board_2ch_get_cookie(board_2ch);
  post_result = ochusha_network_broker_try_post(broker, url,
				ochusha_bulletin_board_get_server(board),
				ochusha_bulletin_board_get_base_url(board),
				cookie, message, &post_status);

  if (post_result)
    {
      if (post_status.status_code == 200)
	{
	  const char *x_2ch = strstr(post_status.body, "2ch_X:");
#if DEBUG_POST_MOST
	  if (post_status.body != NULL)
	    {
	      const char *encoding
		= ochusha_bulletin_board_get_response_character_encoding(board);
	      iconv_helper *helper
		= ochusha_bulletin_board_get_response_iconv_helper(board);
	      iconv_t converter = iconv_open("UTF-8", encoding);
	      if (converter != (iconv_t)-1)
		{
		  gchar *body_utf8 = convert_string(converter, helper,
						    post_status.body, -1);
		  if (body_utf8 != NULL)
		    {
		      ochusha_network_broker_output_log(broker,
							"Result-Body: ");
		      ochusha_network_broker_output_log(broker, body_utf8);
		      ochusha_network_broker_output_log(broker, "\n");
		      G_FREE(body_utf8);
		    }
		  else
		    {
		      ochusha_network_broker_output_log(broker,
							"iconv_failed.\n");
		    }
		  iconv_close(converter);
		}
	      else
		{
		  ochusha_network_broker_output_log(broker,
						    "iconv_open failed.\n");
		}
	    }
#endif
	  if (x_2ch == NULL)
	    { /* "åǧ" in Shift_JISå */
	      if (strstr(post_status.body, "\203\116\203\142\203\114\201\133\212\155\224\106\201\111") != NULL)
		x_2ch = "2ch_X:cookie";	/*  */
	    }
	  if (x_2ch != NULL)
	    {
	      post_result = FALSE;
	      if (strstr(x_2ch, "2ch_X:cookie") != NULL)
		{
		  new_cookie = post_status.set_cookie;
		  if (new_cookie != NULL && *new_cookie != '\0')
		    {
		      char *pos = strchr(new_cookie, ';');
		      if (pos != NULL)
			{
			  *pos = '\0';
			  ochusha_board_2ch_set_cookie(board_2ch, new_cookie);
			}
		    }
		}

#if DEBUG_POST && !DEBUG_POST_MOST
	      if (post_status.body != NULL)
		{
		  const char *encoding
		    = ochusha_bulletin_board_get_response_character_encoding(board);
		  iconv_helper *helper
		    = ochusha_bulletin_board_get_response_iconv_helper(board);
		  iconv_t converter = iconv_open("UTF-8", encoding);
		  if (converter != (iconv_t)-1)
		    {
		      gchar *body_utf8 = convert_string(converter, helper,
							post_status.body, -1);
		      if (body_utf8 != NULL)
			{
			  ochusha_network_broker_output_log(broker,
							    "Result-Body: ");
			  ochusha_network_broker_output_log(broker, body_utf8);
			  ochusha_network_broker_output_log(broker, "\n");
			  G_FREE(body_utf8);
			}
		      else
			{
			  ochusha_network_broker_output_log(broker,
							    "iconv_failed.\n");
			}
		      iconv_close(converter);
		    }
		  else
		    {
		      ochusha_network_broker_output_log(broker,
							"iconv_open failed.\n");
		    }
		}
#endif
	    }
	}
#if DEBUG_POST
      else
	{
	  if (post_status.body != NULL)
	    {
	      const char *encoding
		= ochusha_bulletin_board_get_response_character_encoding(board);
	      iconv_helper *helper
		= ochusha_bulletin_board_get_response_iconv_helper(board);
	      iconv_t converter = iconv_open("UTF-8", encoding);
	      if (converter != (iconv_t)-1)
		{
		  gchar *body_utf8 = convert_string(converter, helper,
						    post_status.body, -1);
		  if (body_utf8 != NULL)
		    {
		      ochusha_network_broker_output_log(broker,
							"Result-Body: ");
		      ochusha_network_broker_output_log(broker, body_utf8);
		      ochusha_network_broker_output_log(broker, "\n");
		      G_FREE(body_utf8);
		    }
		  else
		    {
		      ochusha_network_broker_output_log(broker,
							"iconv_failed.\n");
		    }
		  iconv_close(converter);
		}
	      else
		{
		  ochusha_network_broker_output_log(broker,
						    "iconv_open failed.\n");
		}
	    }
	}
#endif
    }

  if (post_status.body != NULL)
    G_FREE(post_status.body);

  if (post_status.set_cookie != NULL)
    G_FREE(post_status.set_cookie);

#if 0
  if (new_cookie != NULL
      && (cookie == NULL || strcmp(cookie, new_cookie) != 0))
    return OCHUSHA_UTILS_2CH_POST_NO_COOKIE;
#else
  if (new_cookie != NULL)
    return OCHUSHA_UTILS_2CH_POST_NO_COOKIE;
#endif

  return post_result
    ? OCHUSHA_UTILS_2CH_POST_SUCCESS : OCHUSHA_UTILS_2CH_POST_FAILURE;
}
