/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * This is completely derived from TinyScheme 1.33 even though
 * the source code literally seems not like that ;-).
 *
 * $Id: ts_core.c,v 1.6 2004/01/05 13:16:32 fuyu Exp $
 */

/* T I N Y S C H E M E    1 . 3 3
 *   Dimitrios Souflis (dsouflis@acm.org)
 *   Based on MiniScheme (original credits follow)
 * (MINISCM)               coded by Atsushi Moriwaki (11/5/1989)
 * (MINISCM)           E-MAIL :  moriwaki@kurims.kurims.kyoto-u.ac.jp
 * (MINISCM) This version has been modified by R.C. Secrist.
 * (MINISCM)
 * (MINISCM) Mini-Scheme is now maintained by Akira KIDA.
 * (MINISCM)
 * (MINISCM) This is a revised and modified version by Akira KIDA.
 * (MINISCM)	current version is 0.85k4 (15 May 1994)
 *
 */

/*
 * LICENSE TERMS
 *
 * Copyright (c) 2000, Dimitrios Souflis
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * Neither the name of Dimitrios Souflis nor the names of the
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <ctype.h>
#include <float.h>
#include <limits.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef HAVE_STRCASECMP
# include <glib.h>
# define strcasecmp	g_ascii_strcasecmp
#endif

#define PROFILE_GC	1
#if PROFILE_GC
#include <sys/time.h>
#endif

#include <unistd.h>

#include "ts_core.h"


/* Used for documentation purposes, to signal functions in 'interface' */

#define TOK_EOF     (-1)
#define TOK_LPAREN  0
#define TOK_RPAREN  1
#define TOK_DOT     2
#define TOK_ATOM    3
#define TOK_QUOTE   4
#define TOK_COMMENT 5
#define TOK_DQUOTE  6
#define TOK_BQUOTE  7
#define TOK_COMMA   8
#define TOK_ATMARK  9
#define TOK_SHARP   10
#define TOK_SHARP_CONST 11
#define TOK_VEC     12

#ifndef PROMPT
# define PROMPT "> "
#endif

/*
 *  Basic memory allocation units
 */

#ifndef FIRST_CELLSEGS
# define FIRST_CELLSEGS 3
#endif


static char *
strdown_inplace(char *s)
{
  char *p = s;
  while (*s)
    {
      *s = tolower(*s);
      s++;
    }
  return p;
}


/* ADJ is enough slack to align cells in a TYPE_BITS-bit boundary */
#define ADJ		32
#define CLRATOM      49151    /* 1011111111111111 */   /* only for gc */
#define MARK         32768    /* 1000000000000000 */
#define UNMARK       32767    /* 0111111111111111 */
#define T_ATOM       16384    /* 0100000000000000 */   /* only for gc */


/* operator code */
enum scheme_opcodes { 
#define _OP_DEF(A, B, C, D, E, OP) OP, 
#include "ts_opdefines.h" 
  OP_MAXDEFINED 
}; 


#define IS_ATOM(p)		(TYPEFLAG(p) & T_ATOM)
#define SET_ATOM(p)		TYPEFLAG(p) |= T_ATOM
#define CLR_ATOM(p)		TYPEFLAG(p) &= CLRATOM

#define IS_MARK(p)		(TYPEFLAG(p) & MARK)
#define SET_MARK(p)		TYPEFLAG(p) |= MARK
#define CLR_MARK(p)		TYPEFLAG(p) &= UNMARK

#define IS_ZERO_DOUBLE(x)	((x) < DBL_MIN && (x) > -DBL_MIN)

#define CISALPHA(c) 	(isascii(c) && isalpha(c))
#define CISDIGIT(c)	(isascii(c) && isdigit(c))
#define CISSPACE(c)	(isascii(c) && isspace(c))
#define CISUPPER(c)	(isascii(c) && isupper(c))
#define CISLOWER(c)	(isascii(c) && islower(c))

#define PROCNUM(p)		IVALUE(p)
#define CONT_DUMP(p)		CDR(p)
#define SYMNAME(p)		STRVALUE(CAR(p))
#define CLOSURE_CODE(p)		CAR(p)
#define CLOSURE_ENV(p)		CDR(p)


static num num_zero;
static num num_one;


static num num_add(num a, num b);
static num num_mul(num a, num b);
static num num_div(num a, num b);
static num num_intdiv(num a, num b);
static num num_sub(num a, num b);
static num num_rem(num a, num b);
static num num_mod(num a, num b);
static int num_eq(num a, num b);
static int num_gt(num a, num b);
static int num_ge(num a, num b);
static int num_lt(num a, num b);
static int num_le(num a, num b);

static double round_per_R5RS(double x);

static void putstr(TSCore *sc, const char *s);
static void fill_vector(TSCell *vec, TSCell *obj);
static TSCell *vector_elem(TSCell *vec, int ielem);
static TSCell *set_vector_elem(TSCell *vec, int ielem, TSCell *a);
static TSCell *gensym(TSCore *sc);

static TSCell *mk_proc(TSCore *sc, enum scheme_opcodes op);
static const char *procname(TSCell *x);

static int alloc_cellseg(TSCore *sc, int n);
static long binary_decode(const char *s);
static TSCell *get_consecutive_cells(TSCore *sc, int n);
static TSCell *find_consecutive_cells(TSCore *sc, int n);
static void finalize_cell(TSCore *sc, TSCell *a);
static int count_consecutive_cells(TSCell *x, int needed);
static TSCell *find_slot_in_env(TSCore *sc, TSCell *env, TSCell *sym, int all);

static char *store_string(TSCore *sc, size_t len_str, const char *str,
			  char fill);
static int file_push(TSCore *sc, const char *fname);
static void file_pop(TSCore *sc);
static int file_interactive(TSCore *sc);

static TSCell *port_from_filename(TSCore *sc, const char *fn, int prop);
static TSCell *port_from_file(TSCore *sc, FILE *, int prop);
static TSCell *port_from_string(TSCore *sc, char *start, char *past_the_end,
				int prop);
static port *port_rep_from_filename(TSCore *sc, const char *fn, int prop);
static port *port_rep_from_file(TSCore *sc, FILE *, int prop);
static port *port_rep_from_string(TSCore *sc, char *start, char *past_the_end,
				  int prop);
static void port_close(TSCore *sc, TSCell *p, int flag);
static void mark(TSCell *a);
static void gc(TSCore *sc, TSCell *a, TSCell *b);
static int basic_inchar(port *pt);
static int inchar(TSCore *sc);
static void backchar(TSCore *sc, int c);
static char *readstr_upto(TSCore *sc, const char *delim);
static TSCell *readstrexp(TSCore *sc);
static int token(TSCore *sc);
static void printslashstring(TSCore *sc, char *s, size_t len);
static void atom2str(TSCore *sc, TSCell *l, int f, const char **pp,
		     size_t *plen);
static void printatom(TSCore *sc, TSCell *l, int f);

static int list_length(TSCore *sc, TSCell *a);
static int eqv(TSCell *a, TSCell *b);
static TSCell *opexe_0(TSCore *sc, enum scheme_opcodes op);
static TSCell *opexe_1(TSCore *sc, enum scheme_opcodes op);
static TSCell *opexe_2(TSCore *sc, enum scheme_opcodes op);
static TSCell *opexe_3(TSCore *sc, enum scheme_opcodes op);
static TSCell *opexe_4(TSCore *sc, enum scheme_opcodes op);
static TSCell *opexe_5(TSCore *sc, enum scheme_opcodes op);
static TSCell *opexe_6(TSCore *sc, enum scheme_opcodes op);
static void Eval_Cycle(TSCore *sc, enum scheme_opcodes op);
static void assign_syntax(TSCore *sc, const char *name);
static int syntaxnum(TSCell *p);
static void assign_proc(TSCore *sc, enum scheme_opcodes, const char *name);


static inline num
num_add(num a, num b)
{
  num ret;
  ret.is_fixnum = a.is_fixnum && b.is_fixnum;
  if (ret.is_fixnum)
    ret.value.ivalue = a.value.ivalue + b.value.ivalue;
  else
    ret.value.rvalue = NUM_RVALUE(a) + NUM_RVALUE(b);
  return ret;
}


static inline num
num_mul(num a, num b)
{
  num ret;
  ret.is_fixnum = a.is_fixnum && b.is_fixnum;
  if (ret.is_fixnum)
    ret.value.ivalue = a.value.ivalue * b.value.ivalue;
  else
    ret.value.rvalue = NUM_RVALUE(a) * NUM_RVALUE(b);
  return ret;
}


static inline num
num_div(num a, num b)
{
  num ret;
  ret.is_fixnum
    = a.is_fixnum && b.is_fixnum && a.value.ivalue && b.value.ivalue != 0;
  if (ret.is_fixnum)
    ret.value.ivalue = a.value.ivalue / b.value.ivalue;
  else
    ret.value.rvalue = NUM_RVALUE(a) / NUM_RVALUE(b);
  return ret;
}


static inline num
num_intdiv(num a, num b)
{
  num ret;
  ret.is_fixnum = a.is_fixnum && b.is_fixnum;
  if (ret.is_fixnum)
    ret.value.ivalue = a.value.ivalue / b.value.ivalue;
  else
    ret.value.rvalue = NUM_RVALUE(a) / NUM_RVALUE(b);
  return ret;
}


static inline num
num_sub(num a, num b)
{
  num ret;
  ret.is_fixnum = a.is_fixnum && b.is_fixnum;
  if (ret.is_fixnum)
    ret.value.ivalue = a.value.ivalue - b.value.ivalue;
  else
    ret.value.rvalue = NUM_RVALUE(a) - NUM_RVALUE(b);
  return ret;
}


static inline num
num_rem(num a, num b)
{
  num ret;
  long e1, e2, res;
  ret.is_fixnum = a.is_fixnum && b.is_fixnum;
  e1 = NUM_IVALUE(a);
  e2 = NUM_IVALUE(b);
  res = e1 % e2;
  if (res * e1 < 0)
    {    /* remainder should have same sign as first operand */
      e2 = labs(e2);
      if (res > 0)
	res -= e2;
      else
	res += e2;
    }
  ret.value.ivalue = res;
  return ret;
}


static inline num
num_mod(num a, num b)
{
  num ret;
  long e1, e2, res;
  ret.is_fixnum = a.is_fixnum && b.is_fixnum;
  e1 = NUM_IVALUE(a);
  e2 = NUM_IVALUE(b);
  res = e1 % e2;
  if (res * e2 < 0)
    {    /* modulo should have same sign as second operand */
      e2 = labs(e2);
      if (res > 0)
	res -= e2;
      else
	res += e2;
    }
  ret.value.ivalue = res;
  return ret;
}


static inline int
num_eq(num a, num b)
{
  if (a.is_fixnum && b.is_fixnum)
    return a.value.ivalue == b.value.ivalue;
  return NUM_RVALUE(a) == NUM_RVALUE(b);
}


static inline int
num_gt(num a, num b)
{
  if(a.is_fixnum && b.is_fixnum)
    return a.value.ivalue > b.value.ivalue;
  return NUM_RVALUE(a) > NUM_RVALUE(b);
}


static inline int
num_lt(num a, num b)
{
  if(a.is_fixnum && b.is_fixnum)
    return a.value.ivalue < b.value.ivalue;
  return NUM_RVALUE(a) < NUM_RVALUE(b);
}


static inline int
num_ge(num a, num b)
{
  return !num_lt(a, b);
}


static inline int
num_le(num a, num b)
{
  return !num_gt(a, b);
}


/* Round to nearest. Round to even if midway */
static double
round_per_R5RS(double x)
{
  double fl = floor(x);
  double ce = ceil(x);
  double dfl = x - fl;
  double dce = ce - x;

  if (dfl > dce)
    return ce;

  if (dfl < dce)
    return fl;

  if (fmod(fl, 2.0) == 0.0)
    return fl;	/* I imagine this holds */

  return ce;
}


static long
binary_decode(const char *s)
{
  long x = 0;

  while (*s != 0 && (*s == '1' || *s == '0'))
    {
      x <<= 1;
      x += *s - '0';
      s++;
    }

 return x;
}

#define CELL_SEGSIZE    5000  /* # of cells in one segment */


/* allocate new cell segment */
static int
alloc_cellseg(TSCore *sc, int n)
{
  TSCell *newp;
  TSCell *last;
  TSCell *p;
  char *cp;
  long i;
  int k;
  size_t adj = ADJ;

  if (adj < sizeof(TSCell))
    adj = sizeof(TSCell);

  for (k = 0; k < n; k++)
    {
      if (sc->last_cell_seg >= CELL_NSEGMENT - 1)
	return k;

      cp = (char*)sc->malloc(CELL_SEGSIZE * sizeof(TSCell) + adj);
      if (cp == NULL)
	return k;

      i = ++sc->last_cell_seg;
      sc->alloc_seg[i] = cp;
      /* adjust in TYPE_BITS-bit boundary */
      if (((int)cp) % adj != 0)
	{
	  cp = (char*)(adj * ((long)cp / adj + 1));
	}
      /* insert new segment in address order */
      newp = (TSCell *)cp;
      sc->cell_seg[i] = newp;
      while (i > 0 && sc->cell_seg[i - 1] > sc->cell_seg[i])
	{
	  p = sc->cell_seg[i];
	  sc->cell_seg[i] = sc->cell_seg[i - 1];
	  sc->cell_seg[--i] = p;
        }
      sc->fcells += CELL_SEGSIZE;
      last = newp + CELL_SEGSIZE - 1;
      for (p = newp; p <= last; p++)
	{
	  TYPEFLAG(p) = 0;
	  CDR(p) = p + 1;
	  CAR(p) = sc->nil;
	}
      /* insert new cells in address order on free list */
      if (sc->free_cell == sc->nil || p < sc->free_cell)
	{
	  CDR(last) = sc->free_cell;
	  sc->free_cell = newp;
	}
      else
	{
	  p = sc->free_cell;
	  while (CDR(p) != sc->nil && newp > CDR(p))
	    p = CDR(p);
	  CDR(last) = CDR(p);
	  CDR(p) = newp;
	}
    }
  return n;
}


/* get new cell.  parameter a, b is marked by gc. */
static TSCell *
_get_cell(TSCore *sc, TSCell *a, TSCell *b)
{
  TSCell *x;

  if (sc->no_memory)
    return sc->sink;
  
  if (sc->free_cell == sc->nil)
    {
      gc(sc, a, b);
      if (sc->fcells < sc->last_cell_seg * 8 || sc->free_cell == sc->nil)
	{
	  /* if only a few recovered, get more to avoid fruitless gc's */
	  if (!alloc_cellseg(sc, 1) && sc->free_cell == sc->nil)
	    {
	      sc->no_memory = 1;
	      return sc->sink;
	    }
	}
    }

  x = sc->free_cell;
  sc->free_cell = CDR(x);
  --sc->fcells;

  return x;
}

static inline TSCell *
get_cell(TSCore *sc, TSCell *a, TSCell *b)
{
  if (sc->free_cell != sc->nil)
    {
      TSCell *x = sc->free_cell;
      sc->free_cell = CDR(x);
      --sc->fcells;
      return x;
    } 
  return _get_cell(sc, a, b);
}


static TSCell *
get_consecutive_cells(TSCore *sc, int n)
{
  TSCell *x;

  if (sc->no_memory)
    return sc->sink;
  
  /* Are there any cells available? */
  x = find_consecutive_cells(sc, n);
  if (x == sc->nil)
    {
      /* If not, try gc'ing some */
      gc(sc, sc->nil, sc->nil);
      x = find_consecutive_cells(sc, n);
      if (x == sc->nil)
	{
	  /* If there still aren't, try getting more heap */
	  if (!alloc_cellseg(sc, 1))
	    {
	      sc->no_memory = 1;
	      return sc->sink;
	    }
	}
      x = find_consecutive_cells(sc, n);
      if (x == sc->nil)
	{
	  /* If all fail, report failure */
	  sc->no_memory = 1;
	  return sc->sink;
	}
    }
  return x;
}


static int
count_consecutive_cells(TSCell *x, int needed)
{
  int n = 1;
  while (CDR(x) == x + 1)
    {
      x = CDR(x);
      n++;
      if (n > needed)
	return n;
    }
  return n;
}


static TSCell *
find_consecutive_cells(TSCore *sc, int n)
{
  TSCell **pp = &sc->free_cell;
  
  while (*pp != sc->nil)
    {
      int cnt = count_consecutive_cells(*pp, n);
      if (cnt >= n)
	{
	  TSCell *x = *pp;
	  *pp = CDR(*pp + n - 1);
	  sc->fcells -= n;
	  return x;
	}
      pp = &CDR(*pp + cnt - 1);
    }
  return sc->nil;
}


/* get new cons cell */
TSCell *
ts_core_mk_cell_cons(TSCore *sc, TSCell *a, TSCell *d, int immutable)
{
  TSCell *x = get_cell(sc, a, d);
  TYPEFLAG(x) = T_PAIR;
  if (immutable)
    SET_IMMUTABLE(x);
  CAR(x) = a;
  CDR(x) = d;
  return x;
}

/* ========== oblist implementation  ========== */ 

static int hash_fn(const char *key, int table_size); 


static TSCell *
oblist_initial_value(TSCore *sc)
{
  /* probably should be bigger */
  return ts_core_mk_cell_vector(sc, 461);
}


/* returns the new symbol */ 
static TSCell *
oblist_add_by_name(TSCore *sc, const char *name)
{
  int location;
  TSCell *x = IMMUTABLE_CONS(sc, ts_core_mk_cell_string(sc, name), sc->nil);

  TYPEFLAG(x) = T_SYMBOL;
  SET_IMMUTABLE(CAR(x));
  location = hash_fn(name, IVALUE_UNCHECKED(sc->oblist));
  set_vector_elem(sc->oblist, location,
                  IMMUTABLE_CONS(sc, x, vector_elem(sc->oblist, location)));
  return x;
}


static inline TSCell *
oblist_find_by_name(TSCore *sc, const char *name)
{
  int location = hash_fn(name, IVALUE_UNCHECKED(sc->oblist));
  TSCell *x;

  for (x = vector_elem(sc->oblist, location); x != sc->nil; x = CDR(x))
    {
      char *s = SYMNAME(CAR(x));
      /* case-insensitive, per R5RS section 2. */
      if(strcasecmp(name, s) == 0)
	return CAR(x);
    }

  return sc->nil;
}


static TSCell *
oblist_all_symbols(TSCore *sc)
{
  int i;
  TSCell *ob_list = sc->nil;

  for (i = 0; i < IVALUE_UNCHECKED(sc->oblist); i++)
    {
      TSCell *x;
      for (x = vector_elem(sc->oblist, i); x != sc->nil; x = CDR(x))
	ob_list = CONS(sc, x, ob_list);
    }

  return ob_list;
}


TSCell *
ts_core_mk_cell_port(TSCore *sc, port *p)
{
  TSCell *x = get_cell(sc, sc->nil, sc->nil);
  TYPEFLAG(x) = T_PORT | T_ATOM;
  x->object.port = p;
  return x;
}


TSCell *
ts_core_mk_cell_foreign_func(TSCore *sc, TSForeignFunc f)
{
  TSCell *x = get_cell(sc, sc->nil, sc->nil);
  TYPEFLAG(x) = (T_FOREIGN | T_ATOM);
  x->object.ff = f;
  return x;
}


TSCell *
ts_core_mk_cell_character(TSCore *sc, int c)
{
  TSCell *x = get_cell(sc, sc->nil, sc->nil);
  TYPEFLAG(x) = (T_CHARACTER | T_ATOM);
  IVALUE_UNCHECKED(x) = c;
  SET_INTEGER(x);
  return x;
}


/* get number atom (integer) */
TSCell *
ts_core_mk_cell_integer(TSCore *sc, long value)
{
  TSCell *x = get_cell(sc, sc->nil, sc->nil);
  TYPEFLAG(x) = (T_NUMBER | T_ATOM);
  IVALUE_UNCHECKED(x) = value;
  SET_INTEGER(x);
  return x;
}


TSCell *
ts_core_mk_cell_real(TSCore *sc, double n)
{
  TSCell *x = get_cell(sc, sc->nil, sc->nil);
  TYPEFLAG(x) = (T_NUMBER | T_ATOM);
  RVALUE_UNCHECKED(x) = n;
  SET_REAL(x);
  return x;
}


static inline TSCell *
mk_number(TSCore *sc, num n)
{
  if (n.is_fixnum)
    return ts_core_mk_cell_integer(sc, n.value.ivalue);
  return ts_core_mk_cell_real(sc, n.value.rvalue);
}


TSCell *
ts_core_mk_cell_number(TSCore *sc, num n)
{
  return mk_number(sc, n);
}


/* allocate name to string area */
static char *
store_string(TSCore *sc, size_t len_str, const char *str, char fill)
{
  char *q = (char*)sc->malloc(len_str + 1);
  if (q == NULL)
    {
      sc->no_memory = 1;
      return sc->strbuff;
    }
  if (str != NULL)
    strcpy(q, str);
  else
    {
      memset(q, fill, len_str);
      q[len_str] = 0;
    }
  return q;
}


/* get new string */
TSCell *
ts_core_mk_cell_string(TSCore *sc, const char *str)
{
  return ts_core_mk_cell_counted_string(sc, str, strlen(str));
}


TSCell *
ts_core_mk_cell_counted_string(TSCore *sc, const char *str, size_t len)
{
  TSCell *x = get_cell(sc, sc->nil, sc->nil);
  STRVALUE(x) = store_string(sc, len, str, '\0');
  TYPEFLAG(x) = (T_STRING | T_ATOM);
  STRLENGTH(x) = len;
  return x;
}


TSCell *
ts_core_mk_cell_empty_string(TSCore *sc, size_t len, char fill)
{
  TSCell *x = get_cell(sc, sc->nil, sc->nil);
  STRVALUE(x) = store_string(sc, len, 0, fill);
  TYPEFLAG(x) = (T_STRING | T_ATOM);
  STRLENGTH(x) = len;
  return x;
}


TSCell *
ts_core_mk_cell_vector(TSCore *sc, int len)
{
  TSCell *x = get_consecutive_cells(sc, len / 2 + len % 2 + 1);
  TYPEFLAG(x) = (T_VECTOR | T_ATOM);
  IVALUE_UNCHECKED(x) = len;
  SET_INTEGER(x);
  fill_vector(x, sc->nil);
  return x;
}


static void
fill_vector(TSCell *vec, TSCell *obj)
{
  int i;
  int n = IVALUE(vec) / 2 + IVALUE(vec) % 2;
  for (i = 0; i < n; i++)
    {
      TYPEFLAG(vec + 1 + i) = T_PAIR;
      SET_IMMUTABLE(vec + 1 + i);
      CAR(vec + 1 + i) = obj;
      CDR(vec + 1 + i) = obj;
    }
}


static TSCell *
vector_elem(TSCell *vec, int ielem)
{
  int n = ielem / 2;
  if (ielem % 2 == 0)
    return CAR(vec + 1 + n);
  return CDR(vec + 1 + n);
}


static TSCell *
set_vector_elem(TSCell *vec, int ielem, TSCell *a)
{
  int n = ielem / 2;
  if (ielem % 2 == 0)
    return CAR(vec + 1 + n) = a;
  return CDR(vec + 1 + n) = a;
}


/* get new symbol */
TSCell *
ts_core_mk_cell_symbol(TSCore *sc, const char *name)
{
  /* first check oblist */
  TSCell *x = oblist_find_by_name(sc, name);
  if (x != sc->nil)
    return x;
  return oblist_add_by_name(sc, name);
}


static TSCell *
gensym(TSCore *sc)
{
  char name[40];

  for (; sc->gensym_cnt < LONG_MAX; sc->gensym_cnt++)
    {
      sprintf(name, "gensym-%ld", sc->gensym_cnt);

      /* first check oblist */
      if (oblist_find_by_name(sc, name) != sc->nil)
	continue;

      return oblist_add_by_name(sc, name);
    }

  return sc->nil;
}


/* make symbol or number atom from string */
TSCell *
ts_core_mk_cell_atom(TSCore *sc, char *q)
{
  char c, *p;
  int has_dec_point = 0;
  int has_fp_exp = 0;

#if USE_COLON_HOOK
  if ((p = strstr(q, "::")) != NULL)
    {
      *p = '\0';
      return CONS(sc,
		  sc->colon_hook,
		  CONS(sc,
		       CONS(sc,
			    sc->quote,
			    CONS(sc,
				 ts_core_mk_cell_atom(sc, p + 2),
				 sc->nil)),
		       CONS(sc,
			    ts_core_mk_cell_symbol(sc, strdown_inplace(q)),
			    sc->nil)));
    }
#endif
  p = q;
  c = *p++;
  if (c == '+' || c == '-')
    {
      c = *p++;
      if (c == '.')
	{
	  has_dec_point = 1;
	  c = *p++;
	}
      if (!isdigit(c))
	return ts_core_mk_cell_symbol(sc, strdown_inplace(q)); 
    }
  else if (c == '.')
    { 
      has_dec_point = 1;
      c = *p++;
      if (!isdigit(c))
	return ts_core_mk_cell_symbol(sc, strdown_inplace(q));
    }
  else if (!isdigit(c))
    return ts_core_mk_cell_symbol(sc, strdown_inplace(q));

  for (; (c = *p) != '\0'; p++)
    {
      if (!isdigit(c))
	{
	  if (c == '.')
	    {
	      if (!has_dec_point)
		{
		  has_dec_point = 1;
		  continue;
		}
	    }
	  else if (c == 'e' || c == 'E')
	    {
	      if (!has_fp_exp)
		{
		  has_dec_point = 1; /* decimal point illegal
					from now on */
		  p++;
		  if (*p == '-' || *p == '+' || isdigit(*p))
		    continue;
		}
	    }
	  return ts_core_mk_cell_symbol(sc, strdown_inplace(q));
	}
    }

  if (has_dec_point)
    return ts_core_mk_cell_real(sc, atof(q));
  return ts_core_mk_cell_integer(sc, atol(q));
}


/* make constant */
TSCell *
ts_core_mk_cell_sharp_constant(TSCore *sc, char *name)
{
  long x;
  char tmp[256];

  if (strcmp(name, "t") == 0)
    return sc->t;
  if (strcmp(name, "f") == 0)
    return sc->f;

  if (*name == 'o')
    { /* #o (octal) */
      sprintf(tmp, "0%s", name + 1);
      sscanf(tmp, "%lo", &x);
      return ts_core_mk_cell_integer(sc, x);
    }

  if (*name == 'd')
    { /* #d (decimal) */
      sscanf(name + 1, "%ld", &x);
      return ts_core_mk_cell_integer(sc, x);
    }

  if (*name == 'x')
    { /* #x (hex) */
      sprintf(tmp, "0x%s", name + 1);
      sscanf(tmp, "%lx", &x);
      return ts_core_mk_cell_integer(sc, x);
    }

  if (*name == 'b')
    { /* #b (binary) */
      x = binary_decode(name + 1);
      return ts_core_mk_cell_integer(sc, x);
    }

  if (*name == '\\')
    { /* #\w (character) */
      int c = 0;
      if (strcasecmp(name + 1, "space") == 0)
	c = ' ';
      else if (strcasecmp(name + 1, "newline") == 0)
	c = '\n';
      else if (strcasecmp(name + 1, "return") == 0)
	c = '\r';
      else if (strcasecmp(name + 1, "tab") == 0)
	c = '\t';
      else if (name[1] == 'x' && name[2] != '\0')
	{
          int c1 = 0;
          if (sscanf(name + 2, "%x", &c1) == 1 && c1 < 256)
	    c = c1;
	  else
	    return sc->nil;
	}
      else if (name[2] == '\0')
	c = name[1];
      else
	return sc->nil;

      return ts_core_mk_cell_character(sc, c);
    }
  else
    return sc->nil;
}


/* ========== garbage collector ========== */

/*--
 *  We use algorithm E (Knuth, The Art of Computer Programming Vol.1,
 *  sec. 2.3.5), the Schorr-Deutsch-Waite link-inversion algorithm, 
 *  for marking. 
 */
static void
mark(TSCell *a)
{
  TSCell *p = a;
  TSCell *q;
  TSCell *t = NULL;

 E2:
  SET_MARK(p);
  if (IS_VECTOR(p))
    {
      int i;
      int n = IVALUE_UNCHECKED(p) / 2 + IVALUE_UNCHECKED(p) % 2;
      for (i = 0; i < n; i++)
	{
	  /* Vector cells will be treated like ordinary cells */
	  mark(p + 1 + i);
	}
    }
  if (IS_ATOM(p))
    goto E6;

  /* E4: down car */
  q = CAR(p);
  if (q && !IS_MARK(q))
    {
      SET_ATOM(p);  /* a note that we have moved car */ 
      CAR(p) = t;
      t = p;
      p = q;
      goto E2;
    }

 E5:	/* down cdr */
  q = CDR(p);
  if (q && !IS_MARK(q))
    {
      CDR(p) = t;
      t = p;
      p = q;
      goto E2;
    }

 E6:	/* up.  Undo the link switching from steps E4 and E5. */ 
  if (t == NULL)
    return;

  q = t;
  if (IS_ATOM(q))
    {
      CLR_ATOM(q);
      t = CAR(q);
      CAR(q) = p;
      p = q;
      goto E5;
    }
  else
    {
      t = CDR(q);
      CDR(q) = p;
      p = q;
      goto E6;
    }
}


/* this structure holds all the interpreter's registers */ 
struct dump_stack_frame
{
  enum scheme_opcodes op; 
  TSCell *args; 
  TSCell *envir; 
  TSCell *code; 
}; 


static inline void
dump_stack_mark(TSCore *sc)
{
  int nframes = (int)sc->dump;
  int i;
  for (i = 0; i < nframes; i++)
    {
      struct dump_stack_frame *frame;
      frame = (struct dump_stack_frame *)sc->dump_base + i;
      mark(frame->args);
      mark(frame->envir);
      mark(frame->code);
    }
}


/* garbage collection. parameter a, b is marked. */
static void
gc(TSCore *sc, TSCell *a, TSCell *b)
{
  TSCell *p;
  int i;
#if PROFILE_GC
  struct timeval gc_start;
  struct timeval gc_end;
#endif
  
  if (sc->gc_verbose)
    {
#if PROFILE_GC
      gettimeofday(&gc_start, NULL);
#endif
      putstr(sc, "gc...");
    }

  /* mark system globals */
  mark(sc->oblist);
  mark(sc->global_env);

  /* mark external roots */
  mark(sc->ext_roots);

  /* mark current registers */
  mark(sc->args);
  mark(sc->envir);
  mark(sc->code);
  dump_stack_mark(sc); 
  mark(sc->value);
  mark(sc->inport);
  mark(sc->save_inport);
  mark(sc->outport);
  mark(sc->loadport);

  /* mark variables a, b */
  mark(a);
  mark(b);

  /* garbage collect */
  CLR_MARK(sc->nil);
  sc->fcells = 0;
  sc->free_cell = sc->nil;

  /* free-list is kept sorted by address so as to maintain consecutive
   * ranges, if possible, for use with vectors. Here we scan the cells
   * (which are also kept sorted by address) downwards to build the
   * free-list in sorted order.
   */

  for (i = sc->last_cell_seg; i >= 0; i--)
    {
      p = sc->cell_seg[i] + CELL_SEGSIZE;
      while (--p >= sc->cell_seg[i])
	{
	  if (IS_MARK(p))
	    {
	      CLR_MARK(p);
	    }
	  else
	    {
	      /* reclaim cell */
	      if (TYPEFLAG(p) != 0)
		{
		  finalize_cell(sc, p);
		  TYPEFLAG(p) = 0;
		  ++sc->fcells;
		  CAR(p) = sc->nil;
		}
	      CDR(p) = sc->free_cell;
	      sc->free_cell = p;
	    }
	}
    }

  if (sc->gc_verbose)
    {
      char msg[80];
#if PROFILE_GC
      gettimeofday(&gc_end, NULL);
#endif
      sprintf(msg, "done: %ld cells were recovered.\n", sc->fcells);
      putstr(sc, msg);
#if PROFILE_GC
      sprintf(msg, "GC consumes %d sec and %d usec.\n",
	      gc_end.tv_sec - gc_start.tv_sec,
	      gc_end.tv_usec - gc_start.tv_usec);
      putstr(sc, msg);
#endif
    }
}


static void
finalize_cell(TSCore *sc, TSCell *a)
{
  if (IS_STRING(a) && STRVALUE(a) != sc->strbuff)
    {
      sc->free(STRVALUE(a));
    }
  else if (IS_PORT(a))
    {
      if (a->object.port->kind & port_file 
	  && a->object.port->rep.stdio.closeit)
	{
	  port_close(sc, a, port_input|port_output);
	}
      sc->free(a->object.port);
    }
  else if (IS_FOREIGN_OBJECT(a) && a->object.fo.destructor != NULL)
    {
      (*a->object.fo.destructor)(a->object.fo.pointer);
    }
}


/* ========== Routines for Reading ========== */

static int
file_push(TSCore *sc, const char *fname)
{
  FILE *fin = fopen(fname, "r");
  if (fin != NULL)
    {
      sc->file_i++;
      sc->load_stack[sc->file_i].kind = port_file | port_input;
      sc->load_stack[sc->file_i].rep.stdio.file = fin;
      sc->load_stack[sc->file_i].rep.stdio.closeit = 1;
      sc->nesting_stack[sc->file_i] = 0;
      sc->loadport->object.port = sc->load_stack + sc->file_i;
    }
  return fin != NULL;
}


static void
file_pop(TSCore *sc)
{
  sc->nesting = sc->nesting_stack[sc->file_i];
  if (sc->file_i != 0)
    {
      port_close(sc, sc->loadport, port_input);
      sc->file_i--;
      sc->loadport->object.port = sc->load_stack + sc->file_i;
      if (file_interactive(sc))
	putstr(sc, PROMPT);
    }
}


static int
file_interactive(TSCore *sc)
{
  return sc->file_i == 0 && sc->load_stack[0].rep.stdio.file == stdin
    && sc->inport->object.port->kind & port_file;
}


static port *
port_rep_from_filename(TSCore *sc, const char *fn, int prop)
{
  FILE *f;
  const char *rw;
  port *pt;

  if (prop == (port_input|port_output))
    rw = "a+";
  else if (prop == port_output)
    rw = "w";
  else
    rw = "r";

  f = fopen(fn, rw);
  if (f == NULL)
    return NULL;

  pt = port_rep_from_file(sc, f, prop);
  pt->rep.stdio.closeit = 1;

  return pt;
}


static TSCell *
port_from_filename(TSCore *sc, const char *fn, int prop)
{
  port *pt = port_rep_from_filename(sc, fn, prop);

  if (pt == NULL)
    return sc->nil;

  return ts_core_mk_cell_port(sc, pt);
}


static port *
port_rep_from_file(TSCore *sc, FILE *f, int prop)
{
  const char *rw;
  port *pt = sc->malloc(sizeof(port));

  if (pt == NULL)
    return NULL;

  if (prop == (port_input | port_output))
    rw = "a+";
  else if (prop == port_output)
    rw = "w";
  else
    rw = "r";

  pt->kind = port_file | prop;
  pt->rep.stdio.file = f;
  pt->rep.stdio.closeit = 0;
  return pt;
}


static TSCell *
port_from_file(TSCore *sc, FILE *f, int prop)
{
  port *pt = port_rep_from_file(sc, f, prop);
  if (pt == NULL)
    return sc->nil;

  return ts_core_mk_cell_port(sc, pt);
}


static port *
port_rep_from_string(TSCore *sc, char *start, char *past_the_end, int prop)
{
  port *pt = sc->malloc(sizeof(port));
  if (pt == NULL)
    return NULL;

  pt->kind = port_string | prop;
  pt->rep.string.start = start;
  pt->rep.string.curr = start;
  pt->rep.string.past_the_end = past_the_end;
  return pt;
}


static TSCell *
port_from_string(TSCore *sc, char *start, char *past_the_end, int prop)
{
  port *pt = port_rep_from_string(sc, start, past_the_end, prop);
  if (pt == NULL)
    return sc->nil;
  return ts_core_mk_cell_port(sc, pt);
}


static void
port_close(TSCore *sc, TSCell *p, int flag)
{
  port *pt = p->object.port;
  pt->kind &= ~flag;
  if ((pt->kind & (port_input | port_output)) == 0)
    {
      if (pt->kind & port_file)
	fclose(pt->rep.stdio.file);
      pt->kind = port_free;
    }
}


/* get new character from input file */
static int
inchar(TSCore *sc)
{
  int c;
  port *pt;

 again:
  pt = sc->inport->object.port;
  c = basic_inchar(pt);
  if (c == EOF && sc->inport == sc->loadport && sc->file_i != 0)
    {
      file_pop(sc);
      if (sc->nesting != 0)
	return EOF;
      goto again;
    }
  return c;
}


static int
basic_inchar(port *pt)
{
  if (pt->kind & port_file)
      return fgetc(pt->rep.stdio.file);

  if (*pt->rep.string.curr == 0
      || pt->rep.string.curr == pt->rep.string.past_the_end)
    return EOF;
  return *pt->rep.string.curr++;
}


/* back character to input buffer */
static void
backchar(TSCore *sc, int c)
{
  port *pt;
  if (c == EOF)
    return;

  pt = sc->inport->object.port;
  if (pt->kind & port_file)
    ungetc(c, pt->rep.stdio.file);
  else
    {
      if(pt->rep.string.curr!=pt->rep.string.start)
	--pt->rep.string.curr;
    }
}


static void
putstr(TSCore *sc, const char *s)
{
  port *pt = sc->outport->object.port;

  if (pt->kind == port_free)
    return;

  if (pt->kind & port_file)
    fputs(s, pt->rep.stdio.file);
  else
    {
      for (; *s != '\0'; s++)
	{
	  if (pt->rep.string.curr != pt->rep.string.past_the_end)
	    *pt->rep.string.curr++ = *s;
	}
    }
}


static void
putchars(TSCore *sc, const char *s, size_t len)
{
  port *pt = sc->outport->object.port;

  if (pt->kind == port_free)
    return;

  if (pt->kind & port_file)
    fwrite(s, 1, len, pt->rep.stdio.file);
  else
    {
      for (; len > 0; len--)
	{
	  if (pt->rep.string.curr != pt->rep.string.past_the_end)
	    *pt->rep.string.curr++ = *s++;
	}
    }
}


static void
putcharacter(TSCore *sc, int c)
{
  port *pt = sc->outport->object.port;

  if (pt->kind == port_free)
    return;

  if (pt->kind & port_file)
    {
      fputc(c, pt->rep.stdio.file);
    }
  else
    {
      if (pt->rep.string.curr != pt->rep.string.past_the_end)
	*pt->rep.string.curr++ = c;
    }
}


/* check c is in chars */
static inline int
is_one_of(const char *s, int c)
{
  if (c == EOF)
    return 1;
  while (*s)
    if (*s++ == c)
      return 1;
  return 0;
}


/* read characters up to delimiter, but cater to character constants */
static char *
readstr_upto(TSCore *sc, const char *delim)
{
  char *p = sc->strbuff;

  while (!is_one_of(delim, (*p++ = inchar(sc))));

  if (p == sc->strbuff + 2 && p[-2] == '\\')
    *p = '\0';
  else
    {
      backchar(sc, p[-1]);
      *--p = '\0';
    }
  return sc->strbuff;
}


/* read string expression "xxx...xxx" */
static TSCell *
readstrexp(TSCore *sc)
{
  char *p = sc->strbuff;
  int c;
  int c1 = 0;
  enum { st_ok, st_bsl, st_x1, st_x2 } state = st_ok;
  
  for (;;)
    {
      c = inchar(sc);
      if (c == EOF || (size_t)(p - sc->strbuff) > sizeof(sc->strbuff) - 1)
	return sc->f;

      switch (state)
	{
	case st_ok:
	  switch (c)
	    {
	    case '\\':
	      state = st_bsl;
	      break;
	    case '"':
	      *p = '\0';
	      return ts_core_mk_cell_counted_string(sc,
						    sc->strbuff,
						    (size_t)(p - sc->strbuff));
	    default:
	      *p++ = c;
	      break;
	    }
	  break;

	case st_bsl:
	  switch (c)
	    {
	    case 'x':
	    case 'X':
	      state = st_x1;
	      c1 = 0;
	      break;
	    case 'n':
	      *p++ = '\n';
	      state = st_ok;
	      break;
	    case 't':
	      *p++ = '\t';
	      state = st_ok;
	      break;
	    case 'r':
	      *p++ = '\r';
	      state = st_ok;
	      break;
	    case '"':
	      *p++ = '"';
	      state = st_ok;
	      break;
	    default:
	      *p++ = c;
	      state = st_ok;
	      break;
	    }
	  break;

	case st_x1:
	case st_x2:
	  c = toupper(c);
	  if (c >= '0' && c <= 'F')
	    {
	      if (c <= '9')
		c1 = (c1 << 4) + c - '0';
	      else
		c1 = (c1 << 4) + c - 'A' + 10;

	      if (state == st_x1)
		state = st_x2;
	      else
		{
		  *p++ = c1;
		  state = st_ok;
		}
	    }
	  else
	    return sc->f;
	  break;
	}
    }
}


/* skip white characters */
static inline void
skipspace(TSCore *sc)
{
  int c;
  while (isspace(c = inchar(sc)));

  if (c != EOF)
    backchar(sc, c);
}


/* get token */
static int
token(TSCore *sc)
{
  int c;
  skipspace(sc);
  switch (c = inchar(sc))
    {
    case EOF:
      return TOK_EOF;
    case '(':
      return TOK_LPAREN;
    case ')':
      return TOK_RPAREN;
    case '.':
      c = inchar(sc);
      if (is_one_of(" \n\t", c))
	return TOK_DOT;
      backchar(sc, c);
      backchar(sc, '.');
      return TOK_ATOM;

    case '\'':
      return TOK_QUOTE;
    case ';':
      return TOK_COMMENT;
    case '"':
      return TOK_DQUOTE;
    case '`':
      return TOK_BQUOTE;
    case ',':
      if ((c = inchar(sc)) == '@')
	return TOK_ATMARK;
      backchar(sc, c);
      return TOK_COMMA;

    case '#':
      c = inchar(sc);
      if (c == '(')
	return TOK_VEC;
      if(c == '!')
	return TOK_COMMENT;
      backchar(sc, c);
      if (is_one_of(" tfodxb\\", c))
	return TOK_SHARP_CONST;
      return TOK_SHARP;

    default:
      backchar(sc, c);
      return TOK_ATOM;
    }
}


/* ========== Routines for Printing ========== */
#define ok_abbrev(x)	(IS_PAIR(x) && CDR(x) == sc->nil)


static void
printslashstring(TSCore *sc, char *p, size_t len)
{
  size_t i;
  unsigned char *s = (unsigned char*)p;
  putcharacter(sc, '"');
  for (i = 0; i < len; i++)
    {
      if(*s == 0xff || *s == '"' || *s < ' ' || *s == '\\')
	{
	  putcharacter(sc, '\\');
	  switch (*s)
	    {
	    case '"':
	      putcharacter(sc, '"');
	      break;
	    case '\n':
	      putcharacter(sc, 'n');
	      break;
	    case '\t':
	      putcharacter(sc, 't');
	      break;
	    case '\r':
	      putcharacter(sc, 'r');
	      break;
	    case '\\':
	      putcharacter(sc, '\\');
	      break;
	    default:
	      {
		int d = *s / 16;
		putcharacter(sc, 'x');
		if (d < 10)
		  putcharacter(sc, d + '0');
		else
		  putcharacter(sc, d - 10 + 'A');
		d = *s % 16;
		if (d < 10)
		  putcharacter(sc, d + '0');
		else
		  putcharacter(sc, d - 10 + 'A');
	      }
	    }
	}
      else
	putcharacter(sc, *s);
      s++;
    }
  putcharacter(sc, '"');
}


/* print atoms */
static void
printatom(TSCore *sc, TSCell *l, int f)
{
  const char *p;
  size_t len;
  atom2str(sc, l, f, &p, &len);
  putchars(sc, p, len);
}


/* Uses internal buffer unless string pointer is already available */
static void
atom2str(TSCore *sc, TSCell *l, int f, const char **pp, size_t *plen)
{
  const char *p;

  if (l == sc->nil)
    p = "()";
  else if (l == sc->t)
    p = "#t";
  else if (l == sc->f)
    p = "#f";
  else if (l == sc->eof_obj)
    p = "#<EOF>";
  else if (IS_PORT(l))
    {
      strcpy(sc->strbuff, "#<PORT>");
      p = sc->strbuff;
    }
  else if (IS_NUMBER(l))
    {
      if(IS_INTEGER(l))
	sprintf(sc->strbuff, "%ld", IVALUE_UNCHECKED(l));
      else
	sprintf(sc->strbuff, "%.10g", RVALUE_UNCHECKED(l));
      p = sc->strbuff;
    }
  else if (IS_STRING(l))
    {
      if (!f)
	p = STRVALUE(l);
      else
	{ /* Hack, uses the fact that printing is needed */
	  *pp = sc->strbuff;
	  *plen = 0;
	  printslashstring(sc, STRVALUE(l), STRLENGTH(l));
	  return;
	}
    }
  else if (IS_CHARACTER(l))
    {
      int c = CHARVALUE(l);
      if (!f)
	{
	  sc->strbuff[0] = c;
	  sc->strbuff[1] = 0;
	}
      else
	{
	  switch(c)
	    {
	    case ' ':
	      sprintf(sc->strbuff, "#\\space");
	      break;
	    case '\n':
	      sprintf(sc->strbuff, "#\\newline");
	      break;
	    case '\r':
	      sprintf(sc->strbuff, "#\\return");
	      break;
	    case '\t':
	      sprintf(sc->strbuff, "#\\tab");
	      break;
	    default:
	      if (c < 32)
		{
		  sprintf(sc->strbuff, "#\\x%x", c);
		  break;
		}
	      sprintf(sc->strbuff, "#\\%c", c);
	      break;
	    }
	}
      p = sc->strbuff;
    }
  else if (IS_SYMBOL(l))
    p = SYMNAME(l);
  else if (IS_PROC(l))
    {
      sprintf(sc->strbuff, "#<%s PROCEDURE %ld>", procname(l), PROCNUM(l));
      p = sc->strbuff;
    }
  else if (IS_MACRO(l))
    p = "#<MACRO>";
  else if (IS_CLOSURE(l))
    p = "#<CLOSURE>";
  else if (IS_PROMISE(l))
    p = "#<PROMISE>";
  else if (IS_FOREIGN(l))
    {
      sprintf(sc->strbuff, "#<FOREIGN PROCEDURE %ld>", PROCNUM(l));
      p = sc->strbuff;
    }
  else if (IS_CONTINUATION(l))
    p = "#<CONTINUATION>";
  else if (IS_FOREIGN_OBJECT(l))
    {
      sprintf(sc->strbuff, "#<FOREIGN OBJECT %p>", l->object.fo.pointer);
      p = sc->strbuff;
    }
  else
    p = "#<ERROR>";

  *pp = p;
  *plen = strlen(p);
}

/* ========== Routines for Evaluation Cycle ========== */

/* make closure. c is code. e is environment */
TSCell *
ts_core_mk_cell_closure(TSCore *sc, TSCell *c, TSCell *e)
{
  TSCell *x = get_cell(sc, c, e);
  TYPEFLAG(x) = T_CLOSURE;
  CAR(x) = c;
  CDR(x) = e;
  return x;
}


/* make continuation. */
TSCell *
ts_core_mk_cell_continuation(TSCore *sc, TSCell *d)
{
  TSCell *x = get_cell(sc, sc->nil, d);
  TYPEFLAG(x) = T_CONTINUATION;
  CONT_DUMP(x) = d;
  return x;
}


TSCell *
ts_core_mk_cell_foreign_object(TSCore *sc, void *pointer,
			       void (*destructor)(void *))
{
  TSCell *x = get_cell(sc, sc->nil, sc->nil);
  TYPEFLAG(x) = (T_FOREIGN_OBJECT | T_ATOM);
  x->object.fo.pointer = pointer;
  x->object.fo.destructor = destructor;
  return x;
}


static TSCell *
list_star(TSCore *sc, TSCell *d)
{
  TSCell *p, *q;
  if (CDR(d) == sc->nil)
    return CAR(d);
  p = CONS(sc, CAR(d), CDR(d));
  q = p;
  while (CDDR(p) != sc->nil)
    {
      d = CONS(sc, CAR(p), CDR(p));
      if (CDDR(p) != sc->nil)
	p = CDR(d);
    }
  CDR(p) = CADR(p);
  return q;
}


/* reverse list -- produce new list */
TSCell *
ts_core_list_reverse(TSCore *sc, TSCell *list)
{
  /* a must be checked by gc */
  TSCell *p = sc->nil;
  TSCell *q;
  for (q = list; IS_PAIR(q); q = CDR(q))
    p = CONS(sc, CAR(q), p);

  return p;
}


/* reverse list --- in-place */
TSCell *
ts_core_list_reverse_in_place(TSCore *sc, TSCell *term, TSCell *list)
{
  TSCell *p = list;
  TSCell *result = term;
  TSCell *q;

  while (p != sc->nil)
    {
      q = CDR(p);
      CDR(p) = result;
      result = p;
      p = q;
    }
  return result;
}


/* append list -- produce new list */
TSCell *
ts_core_list_append(TSCore *sc, TSCell *a, TSCell *b)
{
  TSCell *p = b;
  TSCell *q;

  if (a != sc->nil)
    {
      a = ts_core_list_reverse(sc, a);
      while (a != sc->nil)
	{
	  q = CDR(a);
	  CDR(a) = p;
	  p = a;
	  a = q;
	}
    }
  return p;
}


void
ts_core_register_external_root(TSCore *sc, TSCell *cell)
{
  sc->ext_roots = ts_core_mk_cell_cons(sc, cell, sc->ext_roots, 0);
}


void
ts_core_unregister_external_root(TSCore *sc, TSCell *cell)
{
  TSCell *p = sc->nil;
  TSCell *q = sc->ext_roots;
  while (q != sc->nil)
    {
      if (CAR(q) == cell)
	break;
      p = q;
      q = CDR(q);
    }

  if (q == sc->nil)
    return;	/* cell not found */

  if (p == sc->nil)
    { /* cell found as the first element of sc->ext_roots list */
      sc->ext_roots = CDR(q);
      return;
    }

  CDR(p) = CDR(q);
}


/* equivalence of atoms */
static int
eqv(TSCell *a, TSCell *b)
{
  if (IS_STRING(a))
    {
      if (IS_STRING(b))
	return STRVALUE(a) == STRVALUE(b);
      return 0;
    }
  if (IS_NUMBER(a))
    {
      if (IS_NUMBER(b))
	return num_eq(NVALUE(a), NVALUE(b));
      return 0;
    }
  if (IS_CHARACTER(a))
    {
      if (IS_CHARACTER(b))
	return CHARVALUE(a) == CHARVALUE(b);
      return 0;
    }
#if 0	/* Redundant! */
  if (IS_PORT(a))
    {
      if (IS_PORT(b))
	return a == b;
      return 0;
    }
#endif
  if (IS_PROC(a))
    {
      if (IS_PROC(b))
	return PROCNUM(a) == PROCNUM(b);
      return 0;
    }

  return (a == b);
}


/* true or false value macro */
/* () is #t in R5RS */
#define IS_TRUE(p)	((p) != sc->f)
#define IS_FALSE(p)	((p) == sc->f)


/* ========== Environment implementation  ========== */ 

static int
hash_fn(const char *key, int table_size)
{
  const char *c;
  unsigned int hashed = 0;
  int bits_per_int = sizeof(unsigned int) * 8;

  for (c = key; *c != '\0'; c++)
    {
      /* letters have about 5 bits in them */
      hashed = (hashed << 5) | (hashed >> (bits_per_int - 5));
      hashed ^= *c;
    }
  return hashed % table_size;
}


/* 
 * In this implementation, each frame of the environment may be 
 * a hash table: a vector of alists hashed by variable name. 
 * In practice, we use a vector only for the initial frame; 
 * subsequent frames are too small and transient for the lookup 
 * speed to out-weigh the cost of making a new vector. 
 */ 

static void
new_frame_in_env(TSCore *sc, TSCell *old_env)
{
  TSCell *new_frame;

  /* The interaction-environment has about 300 variables in it. */ 
  if (old_env == sc->nil)
    new_frame = ts_core_mk_cell_vector(sc, 461);
  else
    new_frame = sc->nil;

  sc->envir = IMMUTABLE_CONS(sc, new_frame, old_env);
  SET_ENVIRONMENT(sc->envir);
}


static inline void
new_slot_spec_in_env(TSCore *sc, TSCell *env,
		     TSCell *variable, TSCell *value)
{
  TSCell *slot = IMMUTABLE_CONS(sc, variable, value);
  if (IS_VECTOR(CAR(env)))
    {
      int location = hash_fn(SYMNAME(variable), IVALUE_UNCHECKED(CAR(env)));
      set_vector_elem(CAR(env), location,
		      IMMUTABLE_CONS(sc,
				     slot, vector_elem(CAR(env), location)));
    }
  else
    CAR(env) = IMMUTABLE_CONS(sc, slot, CAR(env));
} 


static TSCell *
find_slot_in_env(TSCore *sc, TSCell *env, TSCell *hdl, int all)
{
  TSCell *x, *y; 
  int location; 

  for (x = env; x != sc->nil; x = CDR(x))
    {
      if (IS_VECTOR(CAR(x)))
	{
	  location = hash_fn(SYMNAME(hdl), IVALUE_UNCHECKED(CAR(x)));
	  y = vector_elem(CAR(x), location);
	}
      else
	y = CAR(x);

      for (; y != sc->nil; y = CDR(y))
	{
	  if (CAAR(y) == hdl)
	    break; 
	} 
      if (y != sc->nil)
	break; 
      if (!all)
	return sc->nil; 
    }
  if (x != sc->nil)
    return CAR(y); 

  return sc->nil; 
} 


static inline void
new_slot_in_env(TSCore *sc, TSCell *variable, TSCell *value)
{
  new_slot_spec_in_env(sc, sc->envir, variable, value);
}


static inline void
set_slot_in_env(TSCore *sc, TSCell *slot, TSCell *value)
{
  CDR(slot) = value;
}


static inline TSCell *
slot_value_in_env(TSCell *slot)
{
  return CDR(slot);
}


/* ========== Evaluation Cycle ========== */


static TSCell *
_Error_1(TSCore *sc, const char *s, TSCell *a)
{
#if USE_ERROR_HOOK
  TSCell *hdl = sc->error_hook;
  TSCell *x = find_slot_in_env(sc, sc->envir, hdl, 1);
  if (x != sc->nil)
    {
      if (a != NULL)
	sc->code = CONS(sc,
			CONS(sc, sc->quote, CONS(sc, a, sc->nil)),
			sc->nil);
      else
	sc->code = sc->nil;
      sc->code = CONS(sc, ts_core_mk_cell_string(sc, s), sc->code);
      SET_IMMUTABLE(CAR(sc->code));
      sc->code = CONS(sc, slot_value_in_env(x), sc->code); 
      sc->op = (int)OP_EVAL;
      return sc->t;
    }
#endif

    if (a != NULL)
      sc->args = CONS(sc, a, sc->nil);
    else
      sc->args = sc->nil;
    sc->args = CONS(sc, ts_core_mk_cell_string(sc, s), sc->args);
    SET_IMMUTABLE(CAR(sc->args));
    sc->op = (int)OP_ERR0;
    return sc->t;
}


#define Error_1(sc, s, a)	return _Error_1(sc, s, a)
#define Error_0(sc, s)		return _Error_1(sc, s, NULL)


/* Too small to turn into function */
#define s_goto(sc, a)					    \
  do							    \
    {							    \
      sc->op = (int)(a);				    \
      return sc->t;					    \
    } while (0)

#define s_return(sc, a)		return _s_return(sc, a)


#define STACK_GROWTH 3 

static void
s_save(TSCore *sc, enum scheme_opcodes op, TSCell *args, TSCell *code)
{
  int nframes = (int)sc->dump;
  struct dump_stack_frame *next_frame;

  /* enough room for the next frame? */
  if (nframes >= sc->dump_size)
    {
      sc->dump_size += STACK_GROWTH;
      /* alas there is no sc->realloc */
      sc->dump_base = realloc(sc->dump_base,
			      sizeof(struct dump_stack_frame) * sc->dump_size);
    }
  next_frame = (struct dump_stack_frame *)sc->dump_base + nframes;
  next_frame->op = op;
  next_frame->args = args;
  next_frame->envir = sc->envir;
  next_frame->code = code;
  sc->dump = (TSCell *)(nframes + 1);
}


static TSCell *
_s_return(TSCore *sc, TSCell *a)
{
  int nframes = (int)sc->dump;
  struct dump_stack_frame *frame;

  sc->value = a;
  if (nframes <= 0)
    return sc->nil; 
  nframes--;
  frame = (struct dump_stack_frame *)sc->dump_base + nframes;
  sc->op = frame->op;
  sc->args = frame->args;
  sc->envir = frame->envir;
  sc->code = frame->code;
  sc->dump = (TSCell *)nframes;
  return sc->t;
}


static inline void
dump_stack_reset(TSCore *sc)
{
  /* in this implementation, sc->dump is the number of frames on the stack */
  sc->dump = NULL;
}


static inline void
dump_stack_initialize(TSCore *sc)
{
  sc->dump_size = 0;
  sc->dump_base = NULL;
  dump_stack_reset(sc);
}


static void
dump_stack_free(TSCore *sc)
{
  free(sc->dump_base);
  sc->dump_base = NULL;
  sc->dump = NULL;
  sc->dump_size = 0;
}


#define s_retbool(tf)	s_return(sc, (tf) ? sc->t : sc->f)


static TSCell *
opexe_0(TSCore *sc, enum scheme_opcodes op)
{
  TSCell *x, *y;

  switch (op)
    {
    case OP_LOAD:		/* load */
      if (file_interactive(sc))
	{
	  fprintf(sc->outport->object.port->rep.stdio.file, 
		  "Loading %s\n", STRVALUE(CAR(sc->args)));
	}
      if (!file_push(sc, STRVALUE(CAR(sc->args))))
	{
	  Error_1(sc, "unable to open", CAR(sc->args));
	}
      s_goto(sc, OP_T0LVL);


    case OP_T0LVL:		/* top level */
      if (file_interactive(sc))
	{
	  putstr(sc, "\n");
	}
      sc->nesting = 0;
      dump_stack_reset(sc); 
      sc->envir = sc->global_env;
      sc->save_inport = sc->inport;
      sc->inport = sc->loadport;
      s_save(sc, OP_T0LVL, sc->nil, sc->nil);
      s_save(sc, OP_VALUEPRINT, sc->nil, sc->nil);
      s_save(sc, OP_T1LVL, sc->nil, sc->nil);
      if (file_interactive(sc))
	{
	  putstr(sc, PROMPT);
	}
      s_goto(sc, OP_READ_INTERNAL);


    case OP_T1LVL:		/* top level */
      sc->code = sc->value;
      sc->inport = sc->save_inport;
      s_goto(sc, OP_EVAL);


    case OP_READ_INTERNAL:	/* internal read */
      sc->tok = token(sc);
      if (sc->tok == TOK_EOF)
	{
	  if (sc->inport == sc->loadport)
	    {
	      sc->args = sc->nil;
	      s_goto(sc, OP_QUIT);
	    }
	  else
	    {
	      s_return(sc, sc->eof_obj);
	    }
	}
      s_goto(sc, OP_RDSEXPR);


    case OP_GENSYM:
      s_return(sc, gensym(sc));


    case OP_VALUEPRINT:		/* print evaluation result */
      /* OP_VALUEPRINT is always pushed, because when changing from
	 non-interactive to interactive mode, it needs to be
	 already on the stack */
      if (sc->tracing)
	{
	  putstr(sc, "\nGives: ");
	}
      if (file_interactive(sc))
	{
	  sc->print_flag = 1;
	  sc->args = sc->value;
	  s_goto(sc, OP_P0LIST);
	}
      else
	{
	  s_return(sc, sc->value);
	}


    case OP_EVAL:		/* main part of evaluation */
#if USE_TRACING
      if (sc->tracing)
	{
	  /* s_save(sc, OP_VALUEPRINT, sc->nil, sc->nil); */
	  s_save(sc, OP_REAL_EVAL, sc->args, sc->code);
	  sc->args = sc->code;
	  putstr(sc, "\nEval: ");
	  s_goto(sc, OP_P0LIST);
	}
      /* fall through */
    case OP_REAL_EVAL:
#endif
      if (IS_SYMBOL(sc->code))
	{ /* symbol */
	  x = find_slot_in_env(sc, sc->envir, sc->code, 1);
	  if (x != sc->nil)
	    {
	      s_return(sc, slot_value_in_env(x)); 
	    }
	  else
	    {
	      Error_1(sc, "eval: unbound variable:", sc->code);
	    }
	}
      else if (IS_PAIR(sc->code))
	{
	  if (IS_SYNTAX(x = CAR(sc->code)))
	    { /* SYNTAX */
	      sc->code = CDR(sc->code);
	      s_goto(sc, syntaxnum(x));
	    }
	  else
	    { /* first, eval top element and eval arguments */
	      s_save(sc, OP_E0ARGS, sc->nil, sc->code);
	      /* If no macros
	       * => s_save(sc, OP_E1ARGS, sc->nil, CDR(sc->code));
	       */
	      sc->code = CAR(sc->code);
	      s_goto(sc, OP_EVAL);
	    }
	}
      else
	{
	  s_return(sc, sc->code);
	}


    case OP_E0ARGS:		/* eval arguments */
      if (IS_MACRO(sc->value))
	{ /* macro expansion */
	  s_save(sc, OP_DOMACRO, sc->nil, sc->nil);
	  sc->args = CONS(sc, sc->code, sc->nil);
	  sc->code = sc->value;
	  s_goto(sc, OP_APPLY);
	}
      else
	{
	  sc->code = CDR(sc->code);
	  s_goto(sc, OP_E1ARGS);
	}


    case OP_E1ARGS:		/* eval arguments */
      sc->args = CONS(sc, sc->value, sc->args);
      if (IS_PAIR(sc->code))
	{ /* continue */
	  s_save(sc, OP_E1ARGS, sc->args, CDR(sc->code));
	  sc->code = CAR(sc->code);
	  sc->args = sc->nil;
	  s_goto(sc, OP_EVAL);
	}
      else
	{ /* end */
	  sc->args = ts_core_list_reverse_in_place(sc, sc->nil, sc->args); 
	  sc->code = CAR(sc->args);
	  sc->args = CDR(sc->args);
	  s_goto(sc, OP_APPLY);
	}


#if USE_TRACING
    case OP_TRACING:
      {
	int tr = sc->tracing;
	sc->tracing = IVALUE(CAR(sc->args));
	s_return(sc, ts_core_mk_cell_integer(sc, tr));
      }
#endif


    case OP_APPLY:		/* apply 'code' to 'args' */
#if USE_TRACING
      if (sc->tracing)
	{
	  s_save(sc, OP_REAL_APPLY, sc->args, sc->code);
	  sc->print_flag = 1;
	  /* sc->args = CONS(sc, sc->code, sc->args); */
	  putstr(sc, "\nApply to: ");
	  s_goto(sc, OP_P0LIST);
	}
      /* fall through */
    case OP_REAL_APPLY:
#endif
      if (IS_PROC(sc->code))
	{
	  s_goto(sc, PROCNUM(sc->code));	/* PROCEDURE */
	}
      else if (IS_FOREIGN(sc->code))
	{
	  x = sc->code->object.ff(sc, sc->args);
	  s_return(sc, x);
	}
      else if (IS_CLOSURE(sc->code) || IS_MACRO(sc->code)
	       || IS_PROMISE(sc->code))
	{ /* CLOSURE */
	  /* Should not accept promise */
	  /* make environment */
	  new_frame_in_env(sc, CLOSURE_ENV(sc->code)); 
	  for (x = CAR(CLOSURE_CODE(sc->code)), y = sc->args;
	       IS_PAIR(x);
	       x = CDR(x), y = CDR(y))
	    {
	      if (y == sc->nil)
		{
		  Error_0(sc, "not enough arguments");
		}
	      else
		{
		  new_slot_in_env(sc, CAR(x), CAR(y)); 
		}
	    }
	  if (x == sc->nil)
	    {
	      /*--
	       * if (y != sc->nil) {
	       *   Error_0(sc, "too many arguments");
	       * }
	       */
	    }
	  else if (IS_SYMBOL(x))
	    new_slot_in_env(sc, x, y); 
	  else
	    {
	      Error_1(sc, "syntax error in closure: not a symbol:", x); 
	    }
	  sc->code = CDR(CLOSURE_CODE(sc->code));
	  sc->args = sc->nil;
	  s_goto(sc, OP_BEGIN);
	}
      else if (IS_CONTINUATION(sc->code))
	{ /* CONTINUATION */
	  sc->dump = CONT_DUMP(sc->code);
	  s_return(sc, sc->args != sc->nil ? CAR(sc->args) : sc->nil);
	}
      else
	{
	  Error_0(sc, "illegal function");
	}


    case OP_DOMACRO:		/* do macro */
      sc->code = sc->value;
      s_goto(sc, OP_EVAL);


    case OP_LAMBDA:		/* lambda */
      s_return(sc, ts_core_mk_cell_closure(sc, sc->code, sc->envir));


    case OP_MKCLOSURE:		/* make-closure */
      x = CAR(sc->args);
      if (CAR(x) == sc->lambda)
	{
	  x = CDR(x);
	}
      if (CDR(sc->args) == sc->nil)
	{
	  y = sc->envir;
	}
      else
	{
	  y = CADR(sc->args);
	}
      s_return(sc, ts_core_mk_cell_closure(sc, x, y));


    case OP_QUOTE:		/* quote */
      x = CAR(sc->code);
      s_return(sc, CAR(sc->code));


    case OP_DEF0:		/* define */
      if (IS_PAIR(CAR(sc->code)))
	{
	  x = CAAR(sc->code);
	  sc->code = CONS(sc,
			  sc->lambda,
			  CONS(sc, CDAR(sc->code), CDR(sc->code)));
	}
      else
	{
	  x = CAR(sc->code);
	  sc->code = CADR(sc->code);
	}
      if (!IS_SYMBOL(x))
	{
	  Error_0(sc, "variable is not a symbol");
	}
      s_save(sc, OP_DEF1, sc->nil, x);
      s_goto(sc, OP_EVAL);


    case OP_DEF1:		/* define */
      x = find_slot_in_env(sc, sc->envir, sc->code, 0);
      if (x != sc->nil)
	set_slot_in_env(sc, x, sc->value); 
      else
	new_slot_in_env(sc, sc->code, sc->value); 
      s_return(sc, sc->code);


    case OP_DEFP:		/* defined? */
      x = sc->envir;
      if (CDR(sc->args) != sc->nil)
	x = CADR(sc->args);
      s_retbool(find_slot_in_env(sc, x, CAR(sc->args), 1) != sc->nil);


    case OP_SET0:		/* set! */
      s_save(sc, OP_SET1, sc->nil, CAR(sc->code));
      sc->code = CADR(sc->code);
      s_goto(sc, OP_EVAL);


    case OP_SET1:		/* set! */
      y = find_slot_in_env(sc, sc->envir, sc->code, 1);
      if (y != sc->nil)
	{
	  set_slot_in_env(sc, y, sc->value); 
	  s_return(sc, sc->value);
	}
      else
	{
	  Error_1(sc, "set!: unbound variable:", sc->code); 
	}


    case OP_BEGIN:		/* begin */
      if (!IS_PAIR(sc->code))
	{
	  s_return(sc, sc->code);
	}
      if (CDR(sc->code) != sc->nil)
	{
	  s_save(sc, OP_BEGIN, sc->nil, CDR(sc->code));
	}
      sc->code = CAR(sc->code);
      s_goto(sc, OP_EVAL);


    case OP_IF0:		/* if */
      s_save(sc, OP_IF1, sc->nil, CDR(sc->code));
      sc->code = CAR(sc->code);
      s_goto(sc, OP_EVAL);


    case OP_IF1:		/* if */
      if (IS_TRUE(sc->value))
	sc->code = CAR(sc->code);
      else
	sc->code = CADR(sc->code);	/* (if #f 1) ==> () because
					 * CAR(sc->nil) = sc->nil */
      s_goto(sc, OP_EVAL);


    case OP_LET0:		/* let */
      sc->args = sc->nil;
      sc->value = sc->code;
      sc->code = IS_SYMBOL(CAR(sc->code)) ? CADR(sc->code) : CAR(sc->code);
      s_goto(sc, OP_LET1);


    case OP_LET1:		/* let (calculate parameters) */
      sc->args = CONS(sc, sc->value, sc->args);
      if (IS_PAIR(sc->code))
	{ /* continue */
	  s_save(sc, OP_LET1, sc->args, CDR(sc->code));
	  sc->code = CADAR(sc->code);
	  sc->args = sc->nil;
	  s_goto(sc, OP_EVAL);
	}
      else
	{ /* end */
	  sc->args = ts_core_list_reverse_in_place(sc, sc->nil, sc->args);
	  sc->code = CAR(sc->args);
	  sc->args = CDR(sc->args);
	  s_goto(sc, OP_LET2);
	}


    case OP_LET2:		/* let */
      new_frame_in_env(sc, sc->envir); 
      for (x = IS_SYMBOL(CAR(sc->code))
	     ? CADR(sc->code) : CAR(sc->code), y = sc->args;
	   y != sc->nil;
	   x = CDR(x), y = CDR(y))
	{
	  new_slot_in_env(sc, CAAR(x), CAR(y)); 
	}
      if (IS_SYMBOL(CAR(sc->code)))
	{ /* named let */
	  for (x = CADR(sc->code), sc->args = sc->nil;
	       x != sc->nil;
	       x = CDR(x))
	    {
	      sc->args = CONS(sc, CAAR(x), sc->args);
	    }
	  x = ts_core_mk_cell_closure(sc,
				      CONS(sc,
					   ts_core_list_reverse_in_place(sc,
								  sc->nil,
								  sc->args),
					   CDDR(sc->code)), sc->envir);
	  new_slot_in_env(sc, CAR(sc->code), x);
	  sc->code = CDDR(sc->code);
	  sc->args = sc->nil;
	}
      else
	{
	  sc->code = CDR(sc->code);
	  sc->args = sc->nil;
	}
      s_goto(sc, OP_BEGIN);


    case OP_LET0AST:		/* let* */
      if (CAR(sc->code) == sc->nil)
	{
	  new_frame_in_env(sc, sc->envir); 
	  sc->code = CDR(sc->code);
	  s_goto(sc, OP_BEGIN);
	}
      s_save(sc, OP_LET1AST, CDR(sc->code), CAR(sc->code));
      sc->code = CADAAR(sc->code);
      s_goto(sc, OP_EVAL);


    case OP_LET1AST:		/* let* (make new frame) */
      new_frame_in_env(sc, sc->envir); 
      s_goto(sc, OP_LET2AST);


    case OP_LET2AST:		/* let* (calculate parameters) */
      new_slot_in_env(sc, CAAR(sc->code), sc->value); 
      sc->code = CDR(sc->code);
      if (IS_PAIR(sc->code))
	{ /* continue */
	  s_save(sc, OP_LET2AST, sc->args, sc->code);
	  sc->code = CADAR(sc->code);
	  sc->args = sc->nil;
	  s_goto(sc, OP_EVAL);
	}
      else
	{ /* end */
	  sc->code = sc->args;
	  sc->args = sc->nil;
	  s_goto(sc, OP_BEGIN);
	}


    default:
      sprintf(sc->strbuff, "%d: illegal operator", sc->op);
      Error_0(sc, sc->strbuff);
    }
  return sc->t;
}


static TSCell *
opexe_1(TSCore *sc, enum scheme_opcodes op)
{
  TSCell *x, *y;

  switch (op)
    {
    case OP_LET0REC:		/* letrec */
      new_frame_in_env(sc, sc->envir); 
      sc->args = sc->nil;
      sc->value = sc->code;
      sc->code = CAR(sc->code);
      s_goto(sc, OP_LET1REC);


    case OP_LET1REC:		/* letrec (calculate parameters) */
      sc->args = CONS(sc, sc->value, sc->args);
      if (IS_PAIR(sc->code))
	{ /* continue */
	  s_save(sc, OP_LET1REC, sc->args, CDR(sc->code));
	  sc->code = CADAR(sc->code);
	  sc->args = sc->nil;
	  s_goto(sc, OP_EVAL);
	}
      else
	{ /* end */
	  sc->args = ts_core_list_reverse_in_place(sc, sc->nil, sc->args); 
	  sc->code = CAR(sc->args);
	  sc->args = CDR(sc->args);
	  s_goto(sc, OP_LET2REC);
	}


    case OP_LET2REC:		/* letrec */
      for (x = CAR(sc->code), y = sc->args;
	   y != sc->nil;
	   x = CDR(x), y = CDR(y))
	{
	  new_slot_in_env(sc, CAAR(x), CAR(y)); 
	}
      sc->code = CDR(sc->code);
      sc->args = sc->nil;
      s_goto(sc, OP_BEGIN);


    case OP_COND0:		/* cond */
      if (!IS_PAIR(sc->code))
	{
	  Error_0(sc, "syntax error in cond");
	}
      s_save(sc, OP_COND1, sc->nil, sc->code);
      sc->code = CAAR(sc->code);
      s_goto(sc, OP_EVAL);


    case OP_COND1:		/* cond */
      if (IS_TRUE(sc->value))
	{
	  if ((sc->code = CDAR(sc->code)) == sc->nil)
	    {
	      s_return(sc, sc->value);
	    }
	  if (CAR(sc->code) == sc->feed_to)
	    {
	      if (!IS_PAIR(CDR(sc->code)))
		{
		  Error_0(sc, "syntax error in cond");
		}
	      x = CONS(sc, sc->quote, CONS(sc, sc->value, sc->nil));
	      sc->code = CONS(sc, CADR(sc->code), CONS(sc, x, sc->nil));
	      s_goto(sc, OP_EVAL);
	    }
	  s_goto(sc, OP_BEGIN);
	}
      else
	{
	  if ((sc->code = CDR(sc->code)) == sc->nil)
	    {
	      s_return(sc, sc->nil);
	    }
	  else
	    {
	      s_save(sc, OP_COND1, sc->nil, sc->code);
	      sc->code = CAAR(sc->code);
	      s_goto(sc, OP_EVAL);
	    }
	}


    case OP_DELAY:		/* delay */
      x = ts_core_mk_cell_closure(sc, CONS(sc, sc->nil, sc->code), sc->envir);
      TYPEFLAG(x) = T_PROMISE;
      s_return(sc, x);


    case OP_AND0:		/* and */
      if (sc->code == sc->nil)
	{
	  s_return(sc, sc->t);
	}
      s_save(sc, OP_AND1, sc->nil, CDR(sc->code));
      sc->code = CAR(sc->code);
      s_goto(sc, OP_EVAL);


    case OP_AND1:		/* and */
      if (IS_FALSE(sc->value))
	{
	  s_return(sc, sc->value);
	}
      else if (sc->code == sc->nil)
	{
	  s_return(sc, sc->value);
	}
      else
	{
	  s_save(sc, OP_AND1, sc->nil, CDR(sc->code));
	  sc->code = CAR(sc->code);
	  s_goto(sc, OP_EVAL);
	}


    case OP_OR0:		/* or */
      if (sc->code == sc->nil)
	{
	  s_return(sc, sc->f);
	}
      s_save(sc, OP_OR1, sc->nil, CDR(sc->code));
      sc->code = CAR(sc->code);
      s_goto(sc, OP_EVAL);


    case OP_OR1:		/* or */
      if (IS_TRUE(sc->value))
	{
	  s_return(sc, sc->value);
	}
      else if (sc->code == sc->nil)
	{
	  s_return(sc, sc->value);
	}
      else
	{
	  s_save(sc, OP_OR1, sc->nil, CDR(sc->code));
	  sc->code = CAR(sc->code);
	  s_goto(sc, OP_EVAL);
	}


    case OP_C0STREAM:		/* cons-stream */
      s_save(sc, OP_C1STREAM, sc->nil, CDR(sc->code));
      sc->code = CAR(sc->code);
      s_goto(sc, OP_EVAL);


    case OP_C1STREAM:		/* cons-stream */
      sc->args = sc->value;  /* save sc->value to register sc->args for gc */
      x = ts_core_mk_cell_closure(sc, CONS(sc, sc->nil, sc->code), sc->envir);
      TYPEFLAG(x) = T_PROMISE;
      s_return(sc, CONS(sc, sc->args, x));

    case OP_MACRO0:		/* macro */
      if (IS_PAIR(CAR(sc->code)))
	{
	  x = CAAR(sc->code);
	  sc->code = CONS(sc,
			  sc->lambda,
			  CONS(sc, CDAR(sc->code), CDR(sc->code)));
	}
      else
	{
	  x = CAR(sc->code);
	  sc->code = CADR(sc->code);
	}
      if (!IS_SYMBOL(x))
	{
	  Error_0(sc, "variable is not a symbol");
	}
      s_save(sc, OP_MACRO1, sc->nil, x);
      s_goto(sc, OP_EVAL);


    case OP_MACRO1:		/* macro */
      TYPEFLAG(sc->value) = T_MACRO;
      x = find_slot_in_env(sc, sc->envir, sc->code, 0); 
      if (x != sc->nil)
	set_slot_in_env(sc, x, sc->value); 
      else
	new_slot_in_env(sc, sc->code, sc->value); 
      s_return(sc, sc->code);


    case OP_CASE0:		/* case */
      s_save(sc, OP_CASE1, sc->nil, CDR(sc->code));
      sc->code = CAR(sc->code);
      s_goto(sc, OP_EVAL);


    case OP_CASE1:		/* case */
      for (x = sc->code; x != sc->nil; x = CDR(x))
	{
	  if (!IS_PAIR(y = CAAR(x)))
	    break;
	  for (; y != sc->nil; y = CDR(y))
	    {
	      if (eqv(CAR(y), sc->value))
		break;
	    }
	  if (y != sc->nil)
	    break;
	}
      if (x != sc->nil)
	{
	  if (IS_PAIR(CAAR(x)))
	    {
	      sc->code = CDAR(x);
	      s_goto(sc, OP_BEGIN);
	    }
	  else
	    { /* else */
	      s_save(sc, OP_CASE2, sc->nil, CDAR(x));
	      sc->code = CAAR(x);
	      s_goto(sc, OP_EVAL);
	    }
	}
      else
	{
	  s_return(sc, sc->nil);
	}


    case OP_CASE2:		/* case */
      if (IS_TRUE(sc->value))
	{
	  s_goto(sc, OP_BEGIN);
	}
      else
	{
	  s_return(sc, sc->nil);
	}


    case OP_PAPPLY:		/* apply */
      sc->code = CAR(sc->args);
      sc->args = list_star(sc, CDR(sc->args));
      /* sc->args = CADR(sc->args); */
      s_goto(sc, OP_APPLY);


    case OP_PEVAL: /* eval */
      if (CDR(sc->args) != sc->nil)
	{
	  sc->envir = CADR(sc->args);
	}
      sc->code = CAR(sc->args);
      s_goto(sc, OP_EVAL);


    case OP_CONTINUATION:	/* call-with-current-continuation */
      sc->code = CAR(sc->args);
      sc->args = CONS(sc, ts_core_mk_cell_continuation(sc, sc->dump), sc->nil);
      s_goto(sc, OP_APPLY);


    default:
      sprintf(sc->strbuff, "%d: illegal operator", sc->op);
      Error_0(sc, sc->strbuff);
    }
  return sc->t;
}


static int
list_length(TSCore *sc, TSCell *a)
{
  int v = 0;
  TSCell *x;
  for (x = a, v = 0; IS_PAIR(x); x = CDR(x))
    v++;

  if (x == sc->nil)
    return v;

  return -1;
}


static TSCell *
opexe_2(TSCore *sc, enum scheme_opcodes op)
{
  TSCell *x;
  num v;
  double dd;

  switch (op)
    {
    case OP_INEX2EX:		/* inexact->exact */
      x = CAR(sc->args);
      if(IS_INTEGER(x))
	{
	  s_return(sc, x);
	}
      else if(modf(RVALUE_UNCHECKED(x), &dd) == 0.0)
	{
	  s_return(sc, ts_core_mk_cell_integer(sc, IVALUE(x)));
	}
      else
	{
	  Error_1(sc, "inexact->exact: not integral:", x);
	}


    case OP_EXP:
      x = CAR(sc->args);
      s_return(sc, ts_core_mk_cell_real(sc, exp(RVALUE(x))));

    case OP_LOG:
      x = CAR(sc->args);
      s_return(sc, ts_core_mk_cell_real(sc, log(RVALUE(x))));

    case OP_SIN:
      x = CAR(sc->args);
      s_return(sc, ts_core_mk_cell_real(sc, sin(RVALUE(x))));

    case OP_COS:
      x = CAR(sc->args);
      s_return(sc, ts_core_mk_cell_real(sc, cos(RVALUE(x))));

    case OP_TAN:
      x = CAR(sc->args);
      s_return(sc, ts_core_mk_cell_real(sc, tan(RVALUE(x))));

    case OP_ASIN:
      x = CAR(sc->args);
      s_return(sc, ts_core_mk_cell_real(sc, asin(RVALUE(x))));

    case OP_ACOS:
      x = CAR(sc->args);
      s_return(sc, ts_core_mk_cell_real(sc, acos(RVALUE(x))));

    case OP_ATAN:
      x = CAR(sc->args);
      if (CDR(sc->args) == sc->nil)
	{
	  s_return(sc, ts_core_mk_cell_real(sc, atan(RVALUE(x))));
	}
      else
	{
	  TSCell *y = CADR(sc->args);
	  s_return(sc, ts_core_mk_cell_real(sc, atan2(RVALUE(x), RVALUE(y))));
	}

    case OP_SQRT:
      x = CAR(sc->args);
      s_return(sc, ts_core_mk_cell_real(sc, sqrt(RVALUE(x))));

    case OP_EXPT:
      x = CAR(sc->args);
      if (CDR(sc->args) == sc->nil)
	{
	  Error_0(sc, "expt: needs two arguments");
	}
      else
	{
	  TSCell *y = CADR(sc->args);
	  s_return(sc, ts_core_mk_cell_real(sc, pow(RVALUE(x), RVALUE(y))));
	}

    case OP_FLOOR:
      x = CAR(sc->args);
      s_return(sc, ts_core_mk_cell_real(sc, floor(RVALUE(x))));

    case OP_CEILING:
      x = CAR(sc->args);
      s_return(sc, ts_core_mk_cell_real(sc, ceil(RVALUE(x))));

    case OP_TRUNCATE:
      {
	double rvalue_of_x;
	x = CAR(sc->args);
	rvalue_of_x = RVALUE(x);
	if (rvalue_of_x > 0)
	  {
	    s_return(sc, ts_core_mk_cell_real(sc, floor(rvalue_of_x)));
	  }
	else
	  {
	    s_return(sc, ts_core_mk_cell_real(sc, ceil(rvalue_of_x)));
	  }
      }

    case OP_ROUND:
      x = CAR(sc->args);
      s_return(sc, ts_core_mk_cell_real(sc, round_per_R5RS(RVALUE(x))));

    case OP_ADD:	/* + */
      v = num_zero;
      for (x = sc->args; x != sc->nil; x = CDR(x))
	{
	  v = num_add(v, NVALUE(CAR(x)));
	}
      s_return(sc, mk_number(sc, v));

    case OP_MUL:	/* * */
      v = num_one;
      for (x = sc->args; x != sc->nil; x = CDR(x))
	{
	  v = num_mul(v, NVALUE(CAR(x)));
	}
      s_return(sc, mk_number(sc, v));

    case OP_SUB:	/* - */
      if (CDR(sc->args) == sc->nil)
	{
	  x = sc->args;
	  v = num_zero;
	}
      else
	{
	  x = CDR(sc->args);
	  v = NVALUE(CAR(sc->args));
	}
      for (; x != sc->nil; x = CDR(x))
	{
	  v = num_sub(v, NVALUE(CAR(x)));
	}
      s_return(sc, mk_number(sc, v));

    case OP_DIV:	/* / */
      if (CDR(sc->args) == sc->nil)
	{
	  x = sc->args;
	  v = num_one;
	}
      else
	{
	  x = CDR(sc->args);
	  v = NVALUE(CAR(sc->args));
	}
      for (; x != sc->nil; x = CDR(x))
	{
	  if (!IS_ZERO_DOUBLE(RVALUE(CAR(x))))
	    v = num_div(v, NVALUE(CAR(x)));
	  else
	    {
	      Error_0(sc, "/: division by zero");
	    }
	}
      s_return(sc, mk_number(sc, v));

    case OP_INTDIV:	/* quotient */
      if (CDR(sc->args) == sc->nil)
	{
	  x = sc->args;
	  v = num_one;
	}
      else
	{
	  x = CDR(sc->args);
	  v = NVALUE(CAR(sc->args));
	}
      for (; x != sc->nil; x = CDR(x))
	{
	  if (IVALUE(CAR(x)) != 0)
	    v = num_intdiv(v, NVALUE(CAR(x)));
	  else
	    {
	      Error_0(sc, "quotient: division by zero");
	    }
	}
      s_return(sc, mk_number(sc, v));

    case OP_REM:	/* remainder */
      v = NVALUE(CAR(sc->args));
      if (IVALUE(CADR(sc->args)) != 0)
	v = num_rem(v, NVALUE(CADR(sc->args)));
      else
	{
	  Error_0(sc, "remainder: division by zero");
	}
      s_return(sc, mk_number(sc, v));

    case OP_MOD:	/* modulo */
      v = NVALUE(CAR(sc->args));
      if (IVALUE(CADR(sc->args)) != 0)
	v = num_mod(v, NVALUE(CADR(sc->args)));
      else
	{
	  Error_0(sc, "modulo: division by zero");
	}
      s_return(sc, mk_number(sc, v));

    case OP_CAR:	/* car */
      s_return(sc, CAAR(sc->args));

    case OP_CDR:	/* cdr */
      s_return(sc, CDAR(sc->args));

    case OP_CONS:	/* cons */
      CDR(sc->args) = CADR(sc->args);
      s_return(sc, sc->args);

    case OP_SETCAR:	/* set-car! */
      if (!IS_IMMUTABLE(CAR(sc->args)))
	{
	  CAAR(sc->args) = CADR(sc->args);
	  s_return(sc, CAR(sc->args));
	}
      else
	{
	  Error_0(sc, "set-car!: unable to alter immutable pair");
	}

    case OP_SETCDR:	/* set-cdr! */
      if (!IS_IMMUTABLE(CAR(sc->args)))
	{
	  CDAR(sc->args) = CADR(sc->args);
	  s_return(sc, CAR(sc->args));
	}
      else
	{
	  Error_0(sc, "set-cdr!: unable to alter immutable pair");
	}

    case OP_CHAR2INT:	/* char->integer */
      {
	char c;
	c = (char)IVALUE(CAR(sc->args));
	s_return(sc, ts_core_mk_cell_integer(sc, (unsigned char)c));
      }

    case OP_INT2CHAR:	/* integer->char */
      {
	unsigned char c;
	c = (unsigned char)IVALUE(CAR(sc->args));
	s_return(sc, ts_core_mk_cell_character(sc, (char)c));
      }

    case OP_CHARUPCASE:
      {
	unsigned char c;
	c = (unsigned char)IVALUE(CAR(sc->args));
	c = toupper(c);
	s_return(sc, ts_core_mk_cell_character(sc, (char)c));
      }

    case OP_CHARDNCASE:
      {
	unsigned char c;
	c = (unsigned char)IVALUE(CAR(sc->args));
	c = tolower(c);
	s_return(sc, ts_core_mk_cell_character(sc, (char)c));
      }

    case OP_STR2SYM:	/* string->symbol */
      s_return(sc, ts_core_mk_cell_symbol(sc, STRVALUE(CAR(sc->args))));

    case OP_STR2ATOM:	/* string->atom */
      {
	char *s = STRVALUE(CAR(sc->args));
	if (*s == '#')
	  {
	    s_return(sc, ts_core_mk_cell_sharp_constant(sc, s + 1));
	  }
	else
	  {
	    s_return(sc, ts_core_mk_cell_atom(sc, s));
	  }
      }

    case OP_SYM2STR:	/* symbol->string */
      x = ts_core_mk_cell_string(sc, SYMNAME(CAR(sc->args)));
      SET_IMMUTABLE(x);
      s_return(sc, x);

    case OP_ATOM2STR:	/* atom->string */
      x = CAR(sc->args);
      if (IS_NUMBER(x) || IS_CHARACTER(x) || IS_STRING(x) || IS_SYMBOL(x))
	{
	  const char *p;
	  size_t len;
	  atom2str(sc, x, 0, &p, &len);
	  s_return(sc, ts_core_mk_cell_counted_string(sc, p, len));
	}
      else
	{
	  Error_1(sc, "atom->string: not an atom:", x);
	}

    case OP_MKSTRING:	/* make-string */
      {
	int fill = ' ';
	size_t len = IVALUE(CAR(sc->args));

	if (CDR(sc->args) != sc->nil)
	  fill = CHARVALUE(CADR(sc->args));
	s_return(sc, ts_core_mk_cell_empty_string(sc, len, (char)fill));
      }

    case OP_STRLEN:	/* string-length */
      s_return(sc, ts_core_mk_cell_integer(sc,
					   (long)STRLENGTH(CAR(sc->args))));

    case OP_STRREF:	/* string-ref */
      {
	char *str = STRVALUE(CAR(sc->args));
	size_t index = IVALUE(CADR(sc->args));

	if (index >= STRLENGTH(CAR(sc->args)))
	  {
	    Error_1(sc, "string-ref: out of bounds:", CADR(sc->args));
          }
	s_return(sc,
		 ts_core_mk_cell_character(sc, ((unsigned char*)str)[index]));
      }

    case OP_STRSET:	/* string-set! */
      {
	char *str;
	size_t index;
	int c;

	if (IS_IMMUTABLE(CAR(sc->args)))
	  {
	    Error_1(sc, "string-set!: unable to alter immutable string:",
		    CAR(sc->args));
          }
	str = STRVALUE(CAR(sc->args));
	index = IVALUE(CADR(sc->args));
	if (index >= STRLENGTH(CAR(sc->args)))
	  {
	    Error_1(sc, "string-set!: out of bounds:", CADR(sc->args));
          }
	c = CHARVALUE(CADDR(sc->args));
	str[index] = (char)c;
	s_return(sc, CAR(sc->args));
      }

    case OP_STRAPPEND:	/* string-append */
      {
	/* in 1.29 string-append was in Scheme in init.scm but was too slow */
	size_t len = 0;
	TSCell *newstr;
	char *pos;

	/* compute needed length for new string */
	for (x = sc->args; x != sc->nil; x = CDR(x))
	  len += STRLENGTH(CAR(x));

	newstr = ts_core_mk_cell_empty_string(sc, len, ' ');
	/* store the contents of the argument strings into the new string */
	for (pos = STRVALUE(newstr), x = sc->args;
	     x != sc->nil;
	     pos += STRLENGTH(CAR(x)), x = CDR(x))
	  {
	    memcpy(pos, STRVALUE(CAR(x)), STRLENGTH(CAR(x)));
	  }
	s_return(sc, newstr);
      }

    case OP_SUBSTR:	/* substring */
      {
	char *str = STRVALUE(CAR(sc->args));
	size_t index0 = IVALUE(CADR(sc->args));
	size_t index1;
	size_t len;

	if (index0 > STRLENGTH(CAR(sc->args)))
	  {
	    Error_1(sc, "substring: start out of bounds:", CADR(sc->args));
          }
	if (CDDR(sc->args) != sc->nil)
	  {
	    index1 = IVALUE(CADDR(sc->args));
	    if (index1 > STRLENGTH(CAR(sc->args)) || index1 < index0)
	      {
		Error_1(sc, "substring: end out of bounds:", CADDR(sc->args));
	      }
          }
	else
	  {
	    index1 = STRLENGTH(CAR(sc->args));
          }

	len = index1 - index0;
	x = ts_core_mk_cell_empty_string(sc, len, ' ');
	memcpy(STRVALUE(x), str + index0, len);
	STRVALUE(x)[len] = '\0';

	s_return(sc, x);
     }

    case OP_VECTOR:	/* vector */
      {
	int i;
	TSCell *vec;
	int len = list_length(sc, sc->args);
	if (len < 0)
	  {
	    Error_1(sc, "vector: not a proper list:", sc->args);
          }
	vec = ts_core_mk_cell_vector(sc, len);
	for (x = sc->args, i = 0; IS_PAIR(x); x = CDR(x), i++)
	  set_vector_elem(vec, i, CAR(x));
	s_return(sc, vec);
      }

    case OP_MKVECTOR:	/* make-vector */
      {
	TSCell *fill = sc->nil;
	int len =IVALUE(CAR(sc->args));
	TSCell *vec;

	if (CDR(sc->args) != sc->nil)
	  {
	    fill = CADR(sc->args);
          }
	vec = ts_core_mk_cell_vector(sc, len);
	if (fill != sc->nil)
	  {
	    fill_vector(vec, fill);
          }
	s_return(sc, vec);
      }

    case OP_VECLEN:	/* vector-length */
      s_return(sc, ts_core_mk_cell_integer(sc, IVALUE(CAR(sc->args))));

    case OP_VECREF:	/* vector-ref */
      {
	int index = IVALUE(CADR(sc->args));
	if (index >= IVALUE(CAR(sc->args)))
	  {
	    Error_1(sc, "vector-ref: out of bounds:", CADR(sc->args));
          }
	s_return(sc, vector_elem(CAR(sc->args), index));
      }

    case OP_VECSET:	/* vector-set! */
      {
	int index;

	if (IS_IMMUTABLE(CAR(sc->args)))
	  {
	    Error_1(sc, "vector-set!: unable to alter immutable vector:",
		    CAR(sc->args));
          }

	index = IVALUE(CADR(sc->args));
	if (index >= IVALUE(CAR(sc->args)))
	  {
	    Error_1(sc, "vector-set!: out of bounds:", CADR(sc->args));
          }
	set_vector_elem(CAR(sc->args), index, CADDR(sc->args));
	s_return(sc, CAR(sc->args));
      }

    default:
      sprintf(sc->strbuff, "%d: illegal operator", sc->op);
      Error_0(sc, sc->strbuff);
    }
  return sc->t;
}


static TSCell *
opexe_3(TSCore *sc, enum scheme_opcodes op)
{
  TSCell *x;
  num v;
  int (*comp_func)(num, num) = 0;

  switch (op)
    {
    case OP_NOT:	/* not */
      s_retbool(IS_FALSE(CAR(sc->args)));
    case OP_BOOLP:	/* boolean? */
      s_retbool(CAR(sc->args) == sc->f || CAR(sc->args) == sc->t);
    case OP_EOFOBJP:	/* eof-object? */
      s_retbool(CAR(sc->args) == sc->eof_obj);
    case OP_NULLP:	/* null? */
      s_retbool(CAR(sc->args) == sc->nil);
    case OP_NUMEQ:	/* = */
    case OP_LESS:	/* < */
    case OP_GRE:	/* > */
    case OP_LEQ:	/* <= */
    case OP_GEQ:	/* >= */
      switch(op)
	{
	case OP_NUMEQ: comp_func = num_eq; break;
	case OP_LESS:  comp_func = num_lt; break;
	case OP_GRE:   comp_func = num_gt; break;
	case OP_LEQ:   comp_func = num_le; break;
	case OP_GEQ:   comp_func = num_ge; break;
	default: abort();	/* never reached */
	}
      x = sc->args;
      v = NVALUE(CAR(x));
      x = CDR(x);
      for (; x != sc->nil; x = CDR(x))
	{
	  if (!comp_func(v, NVALUE(CAR(x))))
	    {
	      s_retbool(0);
	    }
	  v = NVALUE(CAR(x));
	}
      s_retbool(1);

    case OP_SYMBOLP:	/* symbol? */
      s_retbool(IS_SYMBOL(CAR(sc->args)));
    case OP_NUMBERP:	/* number? */
      s_retbool(IS_NUMBER(CAR(sc->args)));
    case OP_STRINGP:	/* string? */
      s_retbool(IS_STRING(CAR(sc->args)));
    case OP_INTEGERP:	/* integer? */
      s_retbool(IS_INTEGER(CAR(sc->args)));
    case OP_REALP:	/* real? */
      s_retbool(IS_NUMBER(CAR(sc->args)));	/* All numbers are real */
    case OP_CHARP:	/* char? */
      s_retbool(IS_CHARACTER(CAR(sc->args)));
    case OP_CHARAP:	/* char-alphabetic? */
      s_retbool(CISALPHA(IVALUE(CAR(sc->args))));
    case OP_CHARNP:	/* char-numeric? */
      s_retbool(CISDIGIT(IVALUE(CAR(sc->args))));
    case OP_CHARWP:	/* char-whitespace? */
      s_retbool(CISSPACE(IVALUE(CAR(sc->args))));
    case OP_CHARUP:	/* char-upper-case? */
      s_retbool(CISUPPER(IVALUE(CAR(sc->args))));
    case OP_CHARLP:	/* char-lower-case? */
      s_retbool(CISLOWER(IVALUE(CAR(sc->args))));
    case OP_PORTP:	/* port? */
      s_retbool(IS_PORT(CAR(sc->args)));
    case OP_INPORTP:	/* input-port? */
      s_retbool(IS_INPORT(CAR(sc->args)));
    case OP_OUTPORTP:	/* output-port? */
      s_retbool(IS_OUTPORT(CAR(sc->args)));
    case OP_PROCP:	/* procedure? */
      /*--
       * continuation should be procedure by the example
       * (call-with-current-continuation procedure?) ==> #t
       * in R^3 report sec. 6.9
       */
      s_retbool(IS_PROC(CAR(sc->args))
		|| IS_CLOSURE(CAR(sc->args))
		|| IS_CONTINUATION(CAR(sc->args))
		|| IS_FOREIGN(CAR(sc->args)));

    case OP_PAIRP:	/* pair? */
      s_retbool(IS_PAIR(CAR(sc->args)));
    case OP_LISTP:	/* list? */
      {
	TSCell *slow, *fast;
	slow = fast = CAR(sc->args);
	while (1)
	  {
	    if (!IS_PAIR(fast))
	      s_retbool(fast == sc->nil);

	    fast = CDR(fast);
	    if (!IS_PAIR(fast))
	      s_retbool(fast == sc->nil);

	    fast = CDR(fast);
	    slow = CDR(slow);
	    if (fast == slow)
	      {
		/* the fast (TSCell *) has looped back around and caught up
		   with the slow (TSCell *), hence the structure is circular,
		   not of finite length, and therefore not a list */
		s_retbool(0);
	      }
          }
      }

    case OP_ENVP:	/* environment? */
      s_retbool(IS_ENVIRONMENT(CAR(sc->args)));
    case OP_VECTORP:	/* vector? */
      s_retbool(IS_VECTOR(CAR(sc->args)));
    case OP_EQ:		/* eq? */
      s_retbool(CAR(sc->args) == CADR(sc->args));
    case OP_EQV:	/* eqv? */
      s_retbool(eqv(CAR(sc->args), CADR(sc->args)));

    default:
      sprintf(sc->strbuff, "%d: illegal operator", sc->op);
      Error_0(sc, sc->strbuff);
    }
  return sc->t;
}


static TSCell *
opexe_4(TSCore *sc, enum scheme_opcodes op)
{
  TSCell *x, *y;

  switch (op)
    {
    case OP_FORCE:	/* force */
      sc->code = CAR(sc->args);
      if (IS_PROMISE(sc->code))
	{
	  /* Should change type to closure here */
	  s_save(sc, OP_SAVE_FORCED, sc->nil, sc->code);
	  sc->args = sc->nil;
	  s_goto(sc, OP_APPLY);
	}
      else
	{
	  s_return(sc, sc->code);
	}

    case OP_SAVE_FORCED:	/* Save forced value replacing promise */
      memcpy(sc->code, sc->value, sizeof(TSCell));
      s_return(sc, sc->value);

    case OP_WRITE:	/* write */
    case OP_DISPLAY:	/* display */
    case OP_WRITE_CHAR:	/* write-char */
      if (IS_PAIR(CDR(sc->args)))
	{
	  if (CADR(sc->args) != sc->outport)
	    {
	      x = CONS(sc, sc->outport, sc->nil);
	      s_save(sc, OP_SET_OUTPORT, x, sc->nil);
	      sc->outport = CADR(sc->args);
	    }
	}
      sc->args = CAR(sc->args);
      if (op == OP_WRITE)
	{
	  sc->print_flag = 1;
	}
      else
	{
	  sc->print_flag = 0;
	}
      s_goto(sc, OP_P0LIST);

    case OP_NEWLINE:	/* newline */
      if (IS_PAIR(sc->args))
	{
	  if (CAR(sc->args) != sc->outport)
	    {
	      x = CONS(sc, sc->outport, sc->nil);
	      s_save(sc, OP_SET_OUTPORT, x, sc->nil);
	      sc->outport = CAR(sc->args);
	    }
	}
      putstr(sc, "\n");
      s_return(sc, sc->t);

    case OP_ERR0:	/* error */
      sc->retcode = -1;
      if (!IS_STRING(CAR(sc->args)))
	{
	  sc->args = CONS(sc, ts_core_mk_cell_string(sc, " -- "), sc->args);
	  SET_IMMUTABLE(CAR(sc->args));
	}
      putstr(sc, "Error: ");
      putstr(sc, STRVALUE(CAR(sc->args)));
      sc->args = CDR(sc->args);
      s_goto(sc, OP_ERR1);

    case OP_ERR1:	/* error */
      putstr(sc, " ");
      if (sc->args != sc->nil)
	{
	  s_save(sc, OP_ERR1, CDR(sc->args), sc->nil);
	  sc->args = CAR(sc->args);
	  sc->print_flag = 1;
	  s_goto(sc, OP_P0LIST);
	}
      else
	{
	  putstr(sc, "\n");
	  if (sc->interactive_repl)
	    {
	      s_goto(sc, OP_T0LVL);
	    }
	  else
	    {
	      return sc->nil;
	    }
	}

    case OP_REVERSE:	/* reverse */
      s_return(sc, ts_core_list_reverse(sc, CAR(sc->args)));

    case OP_LIST_STAR:	/* list* */
      s_return(sc, list_star(sc, sc->args));

    case OP_APPEND:	/* append */
      if (sc->args == sc->nil)
	{
	  s_return(sc, sc->nil);
	}
      x = CAR(sc->args);
      if (CDR(sc->args) == sc->nil)
	{
	  s_return(sc, sc->args);
	}
      for (y = CDR(sc->args); y != sc->nil; y = CDR(y))
	{
	  x = ts_core_list_append(sc, x, CAR(y));
	}
      s_return(sc, x);

    case OP_QUIT:	/* quit */
      if (IS_PAIR(sc->args))
	{
	  sc->retcode = IVALUE(CAR(sc->args));
	}
      return sc->nil;

    case OP_GC:		/* gc */
      gc(sc, sc->nil, sc->nil);
      s_return(sc, sc->t);

    case OP_GCVERB:	/* gc-verbose */
      {
	int was = sc->gc_verbose;
	sc->gc_verbose = (CAR(sc->args) != sc->f);
	s_retbool(was);
      }

    case OP_NEWSEGMENT:	/* new-segment */
      if (!IS_PAIR(sc->args) || !IS_NUMBER(CAR(sc->args)))
	{
	  Error_0(sc, "new-segment: argument must be a number");
	}
      alloc_cellseg(sc, (int)IVALUE(CAR(sc->args)));
      s_return(sc, sc->t);

    case OP_OBLIST:	/* oblist */
      s_return(sc, oblist_all_symbols(sc)); 

    case OP_CURR_INPORT:	/* current-input-port */
      s_return(sc, sc->inport);

    case OP_CURR_OUTPORT:	/* current-output-port */
      s_return(sc, sc->outport);

    case OP_OPEN_INFILE:	/* open-input-file */
    case OP_OPEN_OUTFILE:	/* open-output-file */
    case OP_OPEN_INOUTFILE:	/* open-input-output-file */
      {
	int prop = 0;
	TSCell *p;
	switch(op)
	  {
	  case OP_OPEN_INFILE:	prop = port_input; break;
	  case OP_OPEN_OUTFILE:	prop = port_output; break;
	  case OP_OPEN_INOUTFILE:	prop = port_input | port_output; break;
	  default: abort();	/* never reached */
	  }
	p = port_from_filename(sc, STRVALUE(CAR(sc->args)), prop);
	if (p == sc->nil)
	  {
	    s_return(sc, sc->f);
	  }
	s_return(sc, p);
      }
     
    case OP_OPEN_INSTRING:	/* open-input-string */
     case OP_OPEN_OUTSTRING:	/* open-output-string */
     case OP_OPEN_INOUTSTRING:	/* open-input-output-string */
       {
	 int prop = 0;
	 TSCell *p;
	 switch(op)
	   {
	   case OP_OPEN_INSTRING:	prop = port_input; break;
	   case OP_OPEN_OUTSTRING:	prop = port_output; break;
	   case OP_OPEN_INOUTSTRING:	prop = port_input | port_output; break;
	   default: abort();	/* never reached */
	   }
	 p = port_from_string(sc, STRVALUE(CAR(sc->args)),
			      (STRVALUE(CAR(sc->args))
			       + STRLENGTH(CAR(sc->args))),
			      prop);
	 if (p == sc->nil)
	   {
	     s_return(sc, sc->f);
	   }
	 s_return(sc, p);
       }

    case OP_CLOSE_INPORT:	/* close-input-port */
      port_close(sc, CAR(sc->args), port_input);
      s_return(sc, sc->t);

    case OP_CLOSE_OUTPORT:	/* close-output-port */
      port_close(sc, CAR(sc->args), port_output);
      s_return(sc, sc->t);

    case OP_INT_ENV:		/* interaction-environment */
      s_return(sc, sc->global_env);

    case OP_CURR_ENV:		/* current-environment */
      s_return(sc, sc->envir);

    default:
      sprintf(sc->strbuff, "%d: illegal operator", sc->op);
      Error_0(sc, sc->strbuff);
    }
  return sc->t;
}


static TSCell *
opexe_5(TSCore *sc, enum scheme_opcodes op)
{
  TSCell *x;

  if (sc->nesting != 0)
    {
      int n = sc->nesting;
      sc->nesting = 0;
      sc->retcode = -1;
      Error_1(sc, "unmatched parentheses:", ts_core_mk_cell_integer(sc, n));
    }

  switch (op)
    {
      /* ========== reading part ========== */
    case OP_READ:
      if (!IS_PAIR(sc->args))
	{
	  s_goto(sc, OP_READ_INTERNAL);
	}
      if (!IS_INPORT(CAR(sc->args)))
	{
	  Error_1(sc, "read: not an input port:", CAR(sc->args));
	}
      if (CAR(sc->args) == sc->inport)
	{
	  s_goto(sc, OP_READ_INTERNAL);
	}
      x = sc->inport;
      sc->inport = CAR(sc->args);
      x = CONS(sc, x, sc->nil);
      s_save(sc, OP_SET_INPORT, x, sc->nil);
      s_goto(sc, OP_READ_INTERNAL);

    case OP_READ_CHAR:	/* read-char */
    case OP_PEEK_CHAR:	/* peek-char */ {
      int c;
      if (IS_PAIR(sc->args))
	{
	  if (CAR(sc->args) != sc->inport)
	    {
	      x = sc->inport;
	      x = CONS(sc, x, sc->nil);
	      s_save(sc, OP_SET_INPORT, x, sc->nil);
	      sc->inport = CAR(sc->args);
	    }
	}
      c = inchar(sc);
      if (c == EOF)
	{
	  s_return(sc, sc->eof_obj);
	}
      if (sc->op == OP_PEEK_CHAR)
	{
	  backchar(sc, c);
	}
      s_return(sc, ts_core_mk_cell_character(sc, c));
    }

    case OP_CHAR_READY:	/* char-ready? */
      {
	TSCell *p = sc->inport;
	if (IS_PAIR(sc->args))
	  p = CAR(sc->args);
	s_retbool(p->object.port->kind & port_string);
      }

    case OP_SET_INPORT:		/* set-input-port */
      sc->inport = CAR(sc->args);
      s_return(sc, sc->value);

    case OP_SET_OUTPORT:	/* set-output-port */
      sc->outport = CAR(sc->args);
      s_return(sc, sc->value);

    case OP_RDSEXPR:
      switch (sc->tok)
	{
	case TOK_EOF:
	  if (sc->inport == sc->loadport)
	    {
	      sc->args = sc->nil;
	      s_goto(sc, OP_QUIT);
	    }
	  else
	    {
	      s_return(sc, sc->eof_obj);
	    }

	case TOK_COMMENT:
	  {
	    int c;
	    while ((c = inchar(sc)) != '\n' && c != EOF);

	    sc->tok = token(sc);
	    s_goto(sc, OP_RDSEXPR);
          }

	case TOK_VEC:
	  s_save(sc, OP_RDVEC, sc->nil, sc->nil);
	  /* fall through */
	case TOK_LPAREN:
	  sc->tok = token(sc);
	  if (sc->tok == TOK_RPAREN)
	    {
	      s_return(sc, sc->nil);
	    }
	  else if (sc->tok == TOK_DOT)
	    {
	      Error_0(sc, "syntax error: illegal dot expression");
	    }
	  else
	    {
	      sc->nesting_stack[sc->file_i]++;
	      s_save(sc, OP_RDLIST, sc->nil, sc->nil);
	      s_goto(sc, OP_RDSEXPR);
	    }

	case TOK_QUOTE:
	  s_save(sc, OP_RDQUOTE, sc->nil, sc->nil);
	  sc->tok = token(sc);
	  s_goto(sc, OP_RDSEXPR);

	case TOK_BQUOTE:
	  sc->tok = token(sc);
	  if (sc->tok == TOK_VEC)
	    {
	      s_save(sc, OP_RDQQUOTEVEC, sc->nil, sc->nil);
	      sc->tok = TOK_LPAREN;
	      s_goto(sc, OP_RDSEXPR);
	    }
	  else
	    {
	      s_save(sc, OP_RDQQUOTE, sc->nil, sc->nil);
	    }
	  s_goto(sc, OP_RDSEXPR);

	case TOK_COMMA:
	  s_save(sc, OP_RDUNQUOTE, sc->nil, sc->nil);
	  sc->tok = token(sc);
	  s_goto(sc, OP_RDSEXPR);

	case TOK_ATMARK:
	  s_save(sc, OP_RDUQTSP, sc->nil, sc->nil);
	  sc->tok = token(sc);
	  s_goto(sc, OP_RDSEXPR);

	case TOK_ATOM:
	  s_return(sc,
		   ts_core_mk_cell_atom(sc, readstr_upto(sc, "();\t\n\r ")));

	case TOK_DQUOTE:
	  x = readstrexp(sc);
	  if (x == sc->f)
	    {
	      Error_0(sc, "Error reading string");
	    }
	  SET_IMMUTABLE(x);
	  s_return(sc, x);

	case TOK_SHARP:
	  {
	    TSCell *f = find_slot_in_env(sc, sc->envir, sc->sharp_hook, 1);
	    if (f == sc->nil)
	      {
		Error_0(sc, "undefined sharp expression");
	      }
	    else
	      {
		sc->code = CONS(sc, slot_value_in_env(f), sc->nil); 
		s_goto(sc, OP_EVAL);
	      }
          }

	case TOK_SHARP_CONST:
	  if ((x = ts_core_mk_cell_sharp_constant(sc,
						  readstr_upto(sc,
							       "();\t\n\r ")))
	      == sc->nil)
	    {
	      Error_0(sc, "undefined sharp expression");
	    }
	  else
	    {
	      s_return(sc, x);
	    }

	default:
	  Error_0(sc, "syntax error: illegal token");
	}
      break;

    case OP_RDLIST:
      {
	sc->args = CONS(sc, sc->value, sc->args);
	sc->tok = token(sc);
	if (sc->tok == TOK_COMMENT)
	  {
	    int c;
	    while ((c = inchar(sc)) != '\n' && c != EOF);

	    sc->tok = token(sc);
          }
	if (sc->tok == TOK_RPAREN)
	  {
	    int c = inchar(sc);
	    if (c != '\n')
	      backchar(sc, c);
	    sc->nesting_stack[sc->file_i]--;
	    s_return(sc, ts_core_list_reverse_in_place(sc, sc->nil, sc->args));
          }
	else if (sc->tok == TOK_DOT)
	  {
	    s_save(sc, OP_RDDOT, sc->args, sc->nil);
	    sc->tok = token(sc);
	    s_goto(sc, OP_RDSEXPR);
          }
	else
	  {
	    s_save(sc, OP_RDLIST, sc->args, sc->nil);;
	    s_goto(sc, OP_RDSEXPR);
          }
      }

    case OP_RDDOT:
      if (token(sc) != TOK_RPAREN)
	{
	  Error_0(sc, "syntax error: illegal dot expression");
	}
      else
	{
	  sc->nesting_stack[sc->file_i]--;
	  s_return(sc, ts_core_list_reverse_in_place(sc, sc->value, sc->args));
	}

    case OP_RDQUOTE:
      s_return(sc, CONS(sc, sc->quote, CONS(sc, sc->value, sc->nil)));

    case OP_RDQQUOTE:
      s_return(sc, CONS(sc, sc->qquote, CONS(sc, sc->value, sc->nil)));

    case OP_RDQQUOTEVEC:
      s_return(sc, CONS(sc, ts_core_mk_cell_symbol(sc, "apply"),
			CONS(sc, ts_core_mk_cell_symbol(sc, "vector"), 
			     CONS(sc, CONS(sc, sc->qquote, 
					   CONS(sc, sc->value, sc->nil)),
				  sc->nil))));

    case OP_RDUNQUOTE:
      s_return(sc, CONS(sc, sc->unquote, CONS(sc, sc->value, sc->nil)));

    case OP_RDUQTSP:
      s_return(sc, CONS(sc, sc->unquotesp, CONS(sc, sc->value, sc->nil)));

    case OP_RDVEC:
      /* sc->code = CONS(sc, mk_proc(sc, OP_VECTOR), sc->value);
	 s_goto(sc, OP_EVAL); Cannot be quoted*/
      /* x = CONS(sc, mk_proc(sc, OP_VECTOR), sc->value);
	 s_return(sc, x); Cannot be part of pairs*/
      /* sc->code = mk_proc(sc, OP_VECTOR);
	 sc->args = sc->value;
	 s_goto(sc, OP_APPLY); */
      sc->args = sc->value;
      s_goto(sc, OP_VECTOR);


      /* ========== printing part ========== */
    case OP_P0LIST:
      if (IS_VECTOR(sc->args))
	{
	  putstr(sc, "#(");
	  sc->args = CONS(sc, sc->args, ts_core_mk_cell_integer(sc, 0));
	  s_goto(sc, OP_PVECFROM);
	}
      else if(IS_ENVIRONMENT(sc->args))
	{
	  putstr(sc, "#<ENVIRONMENT>");
	  s_return(sc, sc->t);
	}
      else if (!IS_PAIR(sc->args))
	{
	  printatom(sc, sc->args, sc->print_flag);
	  s_return(sc, sc->t);
	}
      else if (CAR(sc->args) == sc->quote && ok_abbrev(CDR(sc->args)))
	{
	  putstr(sc, "'");
	  sc->args = CADR(sc->args);
	  s_goto(sc, OP_P0LIST);
	}
      else if (CAR(sc->args) == sc->qquote && ok_abbrev(CDR(sc->args)))
	{
	  putstr(sc, "`");
	  sc->args = CADR(sc->args);
	  s_goto(sc, OP_P0LIST);
	}
      else if (CAR(sc->args) == sc->unquote && ok_abbrev(CDR(sc->args)))
	{
	  putstr(sc, ",");
	  sc->args = CADR(sc->args);
	  s_goto(sc, OP_P0LIST);
	}
      else if (CAR(sc->args) == sc->unquotesp && ok_abbrev(CDR(sc->args)))
	{
	  putstr(sc, ",@");
	  sc->args = CADR(sc->args);
	  s_goto(sc, OP_P0LIST);
	}
      else
	{
	  putstr(sc, "(");
	  s_save(sc, OP_P1LIST, CDR(sc->args), sc->nil);
	  sc->args = CAR(sc->args);
	  s_goto(sc, OP_P0LIST);
	}

    case OP_P1LIST:
      if (IS_PAIR(sc->args))
	{
	  s_save(sc, OP_P1LIST, CDR(sc->args), sc->nil);
	  putstr(sc, " ");
	  sc->args = CAR(sc->args);
	  s_goto(sc, OP_P0LIST);
	}
      else if(IS_VECTOR(sc->args))
	{
	  s_save(sc, OP_P1LIST, sc->nil, sc->nil);
	  putstr(sc, " . ");
	  s_goto(sc, OP_P0LIST);
	}
      else
	{
	  if (sc->args != sc->nil)
	    {
	      putstr(sc, " . ");
	      printatom(sc, sc->args, sc->print_flag);
	    }
	  putstr(sc, ")");
	  s_return(sc, sc->t);
	}

     case OP_PVECFROM:
       {
	 int i = IVALUE_UNCHECKED(CDR(sc->args));
	 TSCell *vec = CAR(sc->args);
	 int len = IVALUE_UNCHECKED(vec);
	 if (i == len)
	   {
	     putstr(sc, ")");
	     s_return(sc, sc->t);
	   }
	 else
	   {
	     TSCell *elem = vector_elem(vec, i);
	     IVALUE_UNCHECKED(CDR(sc->args)) = i + 1;
	     s_save(sc, OP_PVECFROM, sc->args, sc->nil);
	     sc->args = elem;
	     putstr(sc, " ");
	     s_goto(sc, OP_P0LIST);
	   }
       }

     default:
       sprintf(sc->strbuff, "%d: illegal operator", sc->op);
       Error_0(sc, sc->strbuff);
    }
  return sc->t;
}


static TSCell *
opexe_6(TSCore *sc, enum scheme_opcodes op)
{
  TSCell *x, *y;
  long v;

  switch (op)
    {
    case OP_LIST_LENGTH:	/* length */   /* a.k */
      v = list_length(sc, CAR(sc->args));
      if (v < 0)
	{
	  Error_1(sc, "length: not a list:", CAR(sc->args));
	}
      s_return(sc, ts_core_mk_cell_integer(sc, v));

    case OP_ASSQ:		/* assq */     /* a.k */
      x = CAR(sc->args);
      for (y = CADR(sc->args); IS_PAIR(y); y = CDR(y))
	{
	  if (!IS_PAIR(CAR(y)))
	    {
	      Error_0(sc, "unable to handle non pair element");
	    }
	  if (x == CAAR(y))
	    break;
	}
      if (IS_PAIR(y))
	{
	  s_return(sc, CAR(y));
	}
      else
	{
	  s_return(sc, sc->f);
	}
          
          
    case OP_GET_CLOSURE:	/* get-closure-code */   /* a.k */
      sc->args = CAR(sc->args);
      if (sc->args == sc->nil)
	{
	  s_return(sc, sc->f);
	}
      else if (IS_CLOSURE(sc->args))
	{
	  s_return(sc, CONS(sc, sc->lambda, CLOSURE_CODE(sc->value)));
	}
      else if (IS_MACRO(sc->args))
	{
	  s_return(sc, CONS(sc, sc->lambda, CLOSURE_CODE(sc->value)));
	}
      else
	{
	  s_return(sc, sc->f);
	}

    case OP_CLOSUREP:		/* closure? */
      /*
       * Note, macro object is also a closure.
       * Therefore, (closure? <#MACRO>) ==> #t
       */
      s_retbool(IS_CLOSURE(CAR(sc->args)));

    case OP_MACROP:		/* macro? */
      s_retbool(IS_MACRO(CAR(sc->args)));

    default:
      sprintf(sc->strbuff, "%d: illegal operator", sc->op);
      Error_0(sc, sc->strbuff);
    }
  return sc->t; /* NOTREACHED */
}


/* For arguments test */
typedef int (*test_predicate)(TSCell *);

static int
arg_is_any(TSCell *p)
{
  return 1;
}


static int
arg_is_string(TSCell *p)
{
  return IS_STRING(p);
}


static int
arg_is_symbol(TSCell *p)
{
  return IS_SYMBOL(p);
}


static int
arg_is_port(TSCell *p)
{
  return IS_PORT(p);
}


static int
arg_is_environment(TSCell *p)
{
  return IS_ENVIRONMENT(p);
}


static int
arg_is_pair(TSCell *p)
{
  return IS_PAIR(p);
}


static int
arg_is_character(TSCell *p)
{
  return IS_CHARACTER(p);
}


static int
arg_is_vector(TSCell *p)
{
  return IS_VECTOR(p);
}


static int
arg_is_number(TSCell *p)
{
  return IS_NUMBER(p);
}


static int
arg_is_num_integer(TSCell *p)
{
  return IS_NUMBER(p) && ((p)->object.number.is_fixnum);
}


static int
arg_is_nonneg(TSCell *p)
{
  return IS_NUMBER(p) && ((p)->object.number.is_fixnum) && IVALUE(p) >= 0;
}


/* Correspond carefully with following defines! */
static struct {
  test_predicate fct;
  const char *kind;
} tests[] = {
  { NULL,		NULL }, /* unused */
  { arg_is_any, 	NULL },
  { arg_is_string,	"string" },
  { arg_is_symbol,	"symbol" },
  { arg_is_port,	"port" },
  { NULL,		"input port" },
  { NULL,		"output_port" },
  { arg_is_environment, "environment" },
  { arg_is_pair, 	"pair" },
  { NULL, 		"pair or '()" },
  { arg_is_character, 	"character" },
  { arg_is_vector, 	"vector" },
  { arg_is_number, 	"number" },
  { arg_is_num_integer, "integer" },
  { arg_is_nonneg, 	"non-negative integer" }
};


#define TST_NONE		0
#define TST_ANY			"\001"
#define TST_STRING		"\002"
#define TST_SYMBOL		"\003"
#define TST_PORT		"\004"
#define TST_INPORT		"\005"
#define TST_OUTPORT		"\006"
#define TST_ENVIRONMENT		"\007"
#define TST_PAIR		"\010"
#define TST_LIST		"\011"
#define TST_CHAR		"\012"
#define TST_VECTOR		"\013"
#define TST_NUMBER		"\014"
#define TST_INTEGER		"\015"
#define TST_NATURAL		"\016"


typedef TSCell *(*dispatch_func)(TSCore *, enum scheme_opcodes);

typedef struct
{
  dispatch_func func;
  const char *name;
  int min_arity;
  int max_arity;
  const char *arg_tests_encoding;
} op_code_info;

#define INF_ARG 0xffff

static op_code_info dispatch_table[] = { 
#define _OP_DEF(A, B, C, D, E, OP) { A, B, C, D, E }, 
#include "ts_opdefines.h" 
  { 0 } 
}; 


static const char *
procname(TSCell *x)
{
  int n = PROCNUM(x);
  const char *name = dispatch_table[n].name;
  if (name == NULL)
    name = "ILLEGAL!";
  return name;
}


/* kernel of this interpreter */
static void
Eval_Cycle(TSCore *sc, enum scheme_opcodes op)
{
  int count = 0;
  int old_op;
  
  sc->op = op;
  for (;;)
    {
      op_code_info *pcd = dispatch_table + sc->op;
      if (pcd->name != NULL)
	{ /* if built-in function, check arguments */
	  char msg[512];
	  int ok = 1;
	  int n = list_length(sc, sc->args);
      
	  /* Check number of arguments */
	  if (n < pcd->min_arity)
	    {
	      ok = 0;
	      sprintf(msg, "%s: needs%s %d argument(s)",
		      pcd->name,
		      pcd->min_arity == pcd->max_arity ? "" : " at least",
		      pcd->min_arity);
	    }
	  if (ok && n > pcd->max_arity)
	    {
	      ok = 0;
	      sprintf(msg, "%s: needs%s %d argument(s)",
		      pcd->name,
		      pcd->min_arity == pcd->max_arity ? "" : " at most",
		      pcd->max_arity);
	    }
	  if (ok)
	    {
	      if (pcd->arg_tests_encoding != 0)
		{
		  int i = 0;
		  int j;
		  const char *t = pcd->arg_tests_encoding;
		  TSCell *arglist = sc->args;
		  do
		    {
		      TSCell *arg = CAR(arglist);
		      j = (int)t[0];
		      if (j == TST_INPORT[0])
			{
			  if (!IS_INPORT(arg)) break;
			}
		      else if (j == TST_OUTPORT[0])
			{
			  if (!IS_OUTPORT(arg)) break;
			}
		      else if (j == TST_LIST[0])
			{
			  if (arg != sc->nil && !IS_PAIR(arg)) break; 	      
			}
		      else
			{
			  if (!tests[j].fct(arg)) break;
			}

		      if (t[1] != 0)
			{ /* last test is replicated as necessary */
			  t++;
			}
		      arglist = CDR(arglist);
		      i++;
		    } while (i < n);
		  if (i < n)
		    {
		      ok = 0;
		      sprintf(msg, "%s: argument %d must be: %s",
			      pcd->name,
			      i + 1,
			      tests[j].kind);
		    }
		}
	    }
	  if (!ok)
	    {
	      if (_Error_1(sc, msg, 0) == sc->nil)
		{
		  return;
		}
	      pcd = dispatch_table + sc->op;
	    }
	}
      old_op = sc->op;
      if (pcd->func(sc, sc->op) == sc->nil)
	{
	  return;
	}
      if (sc->no_memory)
	{
	  fprintf(stderr, "No memory!\n");
	  return;
	}
      count++;
    }
}


/* ========== Initialization of internal keywords ========== */

static void
assign_syntax(TSCore *sc, const char *name)
{
  TSCell *x = oblist_add_by_name(sc, name);
  TYPEFLAG(x) |= T_SYNTAX;
}


static void
assign_proc(TSCore *sc, enum scheme_opcodes op, const char *name)
{
  TSCell *x = ts_core_mk_cell_symbol(sc, name);
  TSCell *y = mk_proc(sc, op);
  new_slot_in_env(sc, x, y);
}


static TSCell *
mk_proc(TSCore *sc, enum scheme_opcodes op)
{
  TSCell *y = get_cell(sc, sc->nil, sc->nil);
  TYPEFLAG(y) = (T_PROC | T_ATOM);
  IVALUE_UNCHECKED(y) = (long) op;
  SET_INTEGER(y);
  return y;
}


/* Hard-coded for the given keywords. Remember to rewrite if more are added! */
static int
syntaxnum(TSCell *p)
{
  const char *s = STRVALUE(CAR(p));
  switch (STRLENGTH(CAR(p)))
    {
    case 2:
      if (s[0] == 'i')
	return OP_IF0;		/* if */
      else
	return OP_OR0;		/* or */
    case 3:
      if (s[0] == 'a')
	return OP_AND0;		/* and */
      else
	return OP_LET0;		/* let */
    case 4:
      switch (s[3])
	{
	case 'e':
	  return OP_CASE0;	/* case */
	case 'd':
	  return OP_COND0;	/* cond */
	case '*':
	  return OP_LET0AST;	/* let* */
	default:
	  return OP_SET0;	/* set! */
	}
    case 5:
      switch (s[2])
	{
	case 'g':
	  return OP_BEGIN;	/* begin */
	case 'l':
	  return OP_DELAY;	/* delay */
	case 'c':
	  return OP_MACRO0;	/* macro */
	default:
	  return OP_QUOTE;	/* quote */
	}
    case 6:
      switch (s[2])
	{
	case 'm':
	  return OP_LAMBDA;	/* lambda */
	case 'f':
	  return OP_DEF0;	/* define */
	default:
	  return OP_LET0REC;	/* letrec */
	}
    default:
      return OP_C0STREAM;	/* cons-stream */
    }
}


/* initialization of TinyScheme */
TSCore *
ts_core_init_new(void)
{
  TSCore *sc = (TSCore *)malloc(sizeof(TSCore));
  if (!ts_core_init(sc))
    {
      free(sc);
      return NULL;
    }
  return sc;
}


TSCore *
ts_core_init_new_custom_alloc(func_alloc malloc_func, func_dealloc free_func)
{
  TSCore *sc = (TSCore *)malloc(sizeof(TSCore));
  if (!ts_core_init_custom_alloc(sc, malloc_func, free_func))
    {
      free(sc);
      return NULL;
    }

  return sc;
}


int
ts_core_init(TSCore *sc)
{
  return ts_core_init_custom_alloc(sc, malloc, free);
}


int
ts_core_init_custom_alloc(TSCore *sc,
			  func_alloc malloc_func, func_dealloc free_func)
{
  int i, n = sizeof(dispatch_table) / sizeof(dispatch_table[0]);
  TSCell *x;

  num_zero.is_fixnum = 1;
  num_zero.value.ivalue = 0;
  num_one.is_fixnum = 1;
  num_one.value.ivalue = 1;

  sc->gensym_cnt = 0;
  sc->malloc = malloc_func;
  sc->free = free_func;
  sc->last_cell_seg = -1;
  sc->sink = &sc->_sink;
  sc->nil = &sc->_nil;
  sc->t = &sc->_t;
  sc->f = &sc->_f;
  sc->eof_obj = &sc->_eof_obj;
  sc->free_cell = &sc->_nil;
  sc->fcells = 0;
  sc->no_memory = 0;
  sc->inport = sc->nil;
  sc->outport = sc->nil;
  sc->save_inport = sc->nil;
  sc->loadport = sc->nil;
  sc->nesting = 0;
  sc->interactive_repl = 0;
  
  if (alloc_cellseg(sc, FIRST_CELLSEGS) != FIRST_CELLSEGS)
    {
      sc->no_memory = 1;
      return 0;
    }
  sc->gc_verbose = 0;
  dump_stack_initialize(sc); 
  sc->code = sc->nil;
  sc->tracing = 0;
  
  /* init sc->nil */
  TYPEFLAG(sc->nil) = (T_ATOM | MARK);
  CAR(sc->nil) = CDR(sc->nil) = sc->nil;

  /* init T */
  TYPEFLAG(sc->t) = (T_ATOM | MARK);
  CAR(sc->t) = CDR(sc->t) = sc->t;

  /* init F */
  TYPEFLAG(sc->f) = (T_ATOM | MARK);
  CAR(sc->f) = CDR(sc->f) = sc->f;

  sc->oblist = oblist_initial_value(sc); 

  /* init global_env */
  new_frame_in_env(sc, sc->nil); 
  sc->global_env = sc->envir; 

  /* init else */
  sc->ext_roots = sc->nil;
  x = ts_core_mk_cell_symbol(sc, "else");
  new_slot_in_env(sc, x, sc->t); 

  assign_syntax(sc, "lambda");
  assign_syntax(sc, "quote");
  assign_syntax(sc, "define");
  assign_syntax(sc, "if");
  assign_syntax(sc, "begin");
  assign_syntax(sc, "set!");
  assign_syntax(sc, "let");
  assign_syntax(sc, "let*");
  assign_syntax(sc, "letrec");
  assign_syntax(sc, "cond");
  assign_syntax(sc, "delay");
  assign_syntax(sc, "and");
  assign_syntax(sc, "or");
  assign_syntax(sc, "cons-stream");
  assign_syntax(sc, "macro");
  assign_syntax(sc, "case");
  
  for (i = 0; i < n; i++)
    {
      if (dispatch_table[i].name != NULL)
	{
	  assign_proc(sc, i, dispatch_table[i].name);
	}
    }

  /* initialization of global pointers to TSCell of special symbols */
  sc->lambda = ts_core_mk_cell_symbol(sc, "lambda");
  sc->quote = ts_core_mk_cell_symbol(sc, "quote");
  sc->qquote = ts_core_mk_cell_symbol(sc, "quasiquote");
  sc->unquote = ts_core_mk_cell_symbol(sc, "unquote");
  sc->unquotesp = ts_core_mk_cell_symbol(sc, "unquote-splicing");
  sc->feed_to = ts_core_mk_cell_symbol(sc, "=>");
  sc->colon_hook = ts_core_mk_cell_symbol(sc, "*colon-hook*");
  sc->error_hook = ts_core_mk_cell_symbol(sc, "*error-hook*");
  sc->sharp_hook = ts_core_mk_cell_symbol(sc, "*sharp-hook*");

  return !sc->no_memory;
}


void
ts_core_set_input_port_file(TSCore *sc, FILE *fin)
{
  sc->inport = port_from_file(sc, fin, port_input);
}


void
ts_core_set_input_port_string(TSCore *sc, char *start, char *past_the_end)
{
  sc->inport = port_from_string(sc, start, past_the_end, port_input);
}


void
ts_core_set_output_port_file(TSCore *sc, FILE *fout)
{
  sc->outport = port_from_file(sc, fout, port_output);
}


void
ts_core_set_output_port_string(TSCore *sc, char *start, char *past_the_end)
{
  sc->outport = port_from_string(sc, start, past_the_end, port_output);
}


void
ts_core_set_external_data(TSCore *sc, void *p)
{
  sc->ext_data = p;
}


void
ts_core_deinit(TSCore *sc)
{
  int i;

  sc->oblist = sc->nil;
  sc->global_env = sc->nil;
  dump_stack_free(sc); 
  sc->envir = sc->nil;
  sc->code = sc->nil;
  sc->args = sc->nil;
  sc->value = sc->nil;
  if (IS_PORT(sc->inport))
    {
      TYPEFLAG(sc->inport) = T_ATOM;
    }
  sc->inport = sc->nil;
  sc->outport = sc->nil;
  if (IS_PORT(sc->save_inport))
    {
      TYPEFLAG(sc->save_inport) = T_ATOM;
    }
  sc->save_inport = sc->nil;
  if (IS_PORT(sc->loadport))
    {
      TYPEFLAG(sc->loadport) = T_ATOM;
    }
  sc->loadport = sc->nil;
  sc->gc_verbose = 0;
  gc(sc, sc->nil, sc->nil);

  for (i = 0; i <= sc->last_cell_seg; i++)
    sc->free(sc->alloc_seg[i]);
}


void
ts_core_load_file(TSCore *sc, FILE *fin)
{
  dump_stack_reset(sc); 
  sc->envir = sc->global_env;
  sc->file_i = 0;
  sc->load_stack[0].kind = port_input | port_file;
  sc->load_stack[0].rep.stdio.file = fin;
  sc->loadport = ts_core_mk_cell_port(sc, sc->load_stack);
  sc->retcode = 0;
  if (fin == stdin)
    sc->interactive_repl = 1;

  sc->inport = sc->loadport;
  Eval_Cycle(sc, OP_T0LVL);
  TYPEFLAG(sc->loadport) = T_ATOM;

  if (sc->retcode == 0)
    sc->retcode = sc->nesting != 0;
}


void
ts_core_load_string(TSCore *sc, const char *cmd)
{
  dump_stack_reset(sc); 
  sc->envir = sc->global_env;
  sc->file_i = 0;

  /* This func respects const */
  sc->load_stack[0].kind = port_input | port_string;
  sc->load_stack[0].rep.string.start = (char*)cmd;
  sc->load_stack[0].rep.string.past_the_end = (char*)cmd + strlen(cmd);
  sc->load_stack[0].rep.string.curr = (char*)cmd;
  sc->loadport = ts_core_mk_cell_port(sc, sc->load_stack);

  sc->retcode = 0;
  sc->interactive_repl = 0;
  sc->inport = sc->loadport;
  Eval_Cycle(sc, OP_T0LVL);

  TYPEFLAG(sc->loadport) = T_ATOM;
  if (sc->retcode == 0)
    sc->retcode = sc->nesting != 0;
}


void
ts_core_define(TSCore *sc, TSCell *envir, TSCell *symbol, TSCell *value)
{
  TSCell *x = find_slot_in_env(sc, envir, symbol, 0);
  if (x != sc->nil)
    set_slot_in_env(sc, x, value);
  else
    new_slot_spec_in_env(sc, envir, symbol, value);
}


void
ts_core_apply0(TSCore *sc, const char *name)
{
  TSCell *carx = ts_core_mk_cell_symbol(sc, name);
  TSCell *cdrx = sc->nil;
  dump_stack_reset(sc);
  sc->envir = sc->global_env;
  sc->code = CONS(sc, carx, cdrx);
  sc->interactive_repl = 0;
  sc->retcode = 0;
  Eval_Cycle(sc, OP_EVAL);
}


TSCell *
ts_core_eval(TSCore *sc, TSCell *symbol, TSCell *args)
{
  dump_stack_reset(sc);
  sc->envir = sc->global_env;
  sc->code = CONS(sc, symbol, args);
  sc->interactive_repl = 0;
  sc->retcode = 0;
  Eval_Cycle(sc, OP_EVAL);

  return sc->value;
}


TSCell *
ts_core_call(TSCore *sc, TSCell *func, TSCell *args)
{
  dump_stack_reset(sc);
  sc->envir = sc->global_env;
  sc->args = args;
  sc->code = func;
  sc->interactive_repl = 0;
  sc->retcode = 0;
  Eval_Cycle(sc, OP_APPLY);

  return sc->value;
}
