﻿using System;
using System.Linq;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.ComponentModel.Composition;
using System.Diagnostics;

namespace OGSNET.Plugins.GameOn
{
    using OGSNET.Plugin;

    /**
     * \brief GameOn の flashvars 汎用プラグインクラス
     * 
     * 起動にフラッシュを用い、flashvars で起動 URL を指定するゲームに適用できる
     */
    public class GameOnFlashvarsBase : GamePluginBase
    {
        /** \brief プラグインの開始処理 */
        protected override sealed void StartPlugin()
        {
            // ログインする
            this.CallPlugin("GameOn", this.LoadGameStartPage);
        }

        /** \brief ゲームを開始するページを読み込む */
        private void LoadGameStartPage()
        {
            this.Notice("ゲームを開始するページを読み込んでいます。");
            this.Navigate(this.GameStartUrl, this.GameStart);
        }

        /** \brief ゲームを開始する */
        private void GameStart()
        {
            // Flash の要素を取得
            var flash = this.Document.GetElementById(this.FlashId);

            if (flash == null)
            {
                this.Error("ゲーム開始パラメーターが見つかりませんでした。");
                return;
            }

            // Flash の中のすべての param 要素を取得
            var elements = flash.GetElementsByTagName("param").Cast<HtmlElement>();

            // name 属性が flashvars な param 要素を LINQ で取得 (複数ある場合あり)
            var query =
                from A
                in elements
                where A.Name == "flashvars"
                select A.GetAttribute("value");

            // 起動に成功したかのフラグ
            var success = false;

            foreach (var value in query)
            {
                // javascript:～ の形で入ってる
                var regex = new Regex(this.FlashvarsName + "=([^&]+)");

                // マッチ処理
                var match = regex.Match(value);

                // マッチした場合
                if (match.Success)
                {
                    // ゲームを開始する
                    this.Navigate(match.Groups[1].Value);

                    // 成功フラグを立てる
                    success = true;
                    break;
                }
            }

            // 成功した場合
            if (success)
            {
                this.Notice("ゲームの起動に成功しました。");
            }
            else
            {
                this.Notice("ゲーム開始に必要なパラメーターが見つかりません。");
            }
        }

        /**
         * \brief Flash の object 要素の id
         * 
         * ExternalInterface には必ず ID 属性が必要なため、開始する Flash の object 要素には必ず ID 属性がある
         */
        protected string FlashId
        {
            get;
            set;
        }

        /**
         * \brief flashvars で探す名前
         * 
         * FlashId の中の param 要素で name 属性が flashvars な要素の value 属性の中で検索する
         * value 属性が 名前=値&名前=値& ... となってるので、その名前を指定する
         */
        protected string FlashvarsName
        {
            get;
            set;
        }

        private string GameStartUrl_ = null;

        /** \brief ゲームを開始する URL */
        protected string GameStartUrl
        {
            get { return this.GameStartUrl_ ?? this.GameInfo.Url.AbsoluteUri; }
            set { this.GameStartUrl_ = value; }
        }
    }

    /** \brief AVAの起動プラグイン */
    [Export(typeof(IPlugin))]
    [Export(typeof(IGamePlugin))]
    [ExportMetadata("PluginId", "GameOn.AVA")]
    public class AVA : GameOnFlashvarsBase
    {
        /** \brief プラグインの情報を初期化する */
        public AVA()
        {
            this.PluginInfo =
                new PluginInfo("1.0.1.0", "Alliance of Valiant Arms (GameOn) プラグイン ", "pine",
                    "Alliance of Valiant Arms (GameOn) を起動するプラグインです。");

            this.GameInfo =
                new GameInfo("ゲームオン", "GameOn", "Alliance of Valiant Arms",
                    "http://ava.gamechu.jp/");

            this.FlashId = "flash_gamestart_loginSWF";
            this.FlashvarsName = "GAMESTART_LOGIN_URL";
        }
    }

    /** \brief Soul of the Ultimate Nation の起動プラグイン */
    [Export(typeof(IPlugin))]
    [Export(typeof(IGamePlugin))]
    [ExportMetadata("PluginId", "GameOn.SUN")]
    public class SUN : GameOnFlashvarsBase
    {
        /** \brief プラグインの情報を初期化する */
        public SUN()
        {
            this.PluginInfo =
                new PluginInfo("1", "Soul of the Ultimate Nation (GameOn) プラグイン ", "pine",
                    "Soul of the Ultimate Nation (GameOn) を起動するプラグインです。");

            this.GameInfo =
                new GameInfo("ゲームオン", "GameOn", "Soul of the Ultimate Nation",
                    "http://sunonline.gamechu.jp/member/");

            this.FlashId = "flash_start";
            this.FlashvarsName = "START_URL";
        }
    }

    /** \brief ALLODS ONLINE の起動プラグイン */
    [Export(typeof(IPlugin))]
    [Export(typeof(IGamePlugin))]
    [ExportMetadata("PluginId", "GameOn.Allods")]
    public class Allods : GameOnFlashvarsBase
    {
        /** \brief プラグインの情報を初期化する */
        public Allods()
        {
            this.PluginInfo =
                new PluginInfo("1", "ALLODS ONLINE (GameOn) プラグイン ", "pine",
                    "ALLODS ONLINE (GameOn) を起動するプラグインです。");

            this.GameInfo =
                new GameInfo("ゲームオン", "GameOn", "ALLODS ONLINE",
                    "http://allods.gamechu.jp/");

            this.FlashId = "flash_navi";
            this.FlashvarsName = "naviURL3";
        }
    }

    /** \brief RF ONLINE Z の起動プラグイン */
    [Export(typeof(IPlugin))]
    [Export(typeof(IGamePlugin))]
    [ExportMetadata("PluginId", "GameOn.RF")]
    public class RF : GameOnFlashvarsBase
    {
        /** \brief プラグインの情報を初期化する */
        public RF()
        {
            this.PluginInfo =
                new PluginInfo("1", "RF ONLINE Z (GameOn) プラグイン ", "pine",
                    "RF ONLINE Z (GameOn) を起動するプラグインです。");

            this.GameInfo =
                new GameInfo("ゲームオン", "GameOn", "RF ONLINE Z",
                    "http://rfonline.gamechu.jp/member/");

            this.FlashId = "header_menu";
            this.FlashvarsName = "myURL1";
        }
    }

    /** \brief PRIUS ONLINE の起動プラグイン */
    [Export(typeof(IPlugin))]
    [Export(typeof(IGamePlugin))]
    [ExportMetadata("PluginId", "GameOn.Prius")]
    public class Prius : GamePluginBase
    {
        /** \brief プラグインの基本情報の初期化 */
        public Prius()
        {
            this.PluginInfo =
                new PluginInfo("1", "PRIUS ONLINE (GameOn) プラグイン ", "pine",
                    "PRIUS ONLINE (GameOn) を起動するプラグインです。");

            this.GameInfo =
                new GameInfo("ゲームオン", "GameOn", "PRIUS ONLINE",
                    "https://secure.priusonline.jp/member/");
        }

        /** \brief プラグインの開始処理 */
        protected override void StartPlugin()
        {
            this.CallPlugin("GameOn", this.LoadGameStartPage);
        }

        /** \brief ゲーム開始ページへ移動 */
        private void LoadGameStartPage()
        {
            this.Notice("ゲーム開始ページへ接続しています。");

            this.Navigate(this.GameInfo.Url, this.GameStart);
        }

        /** \brief ゲームを開始する */
        private void GameStart()
        {
            this.EndPlugin("ゲームを開始しています。");

            this.Document.InvokeScript("ajaxReq");
        }
    }

    /** \brief 新・天上碑 の起動プラグイン */
    [Export(typeof(IPlugin))]
    [Export(typeof(IGamePlugin))]
    [ExportMetadata("PluginId", "GameOn.Tenjouhi")]
    public class Tenjouhi : GamePluginBase
    {
        /** \brief プラグインの基本情報の初期化 */
        public Tenjouhi()
        {
            this.PluginInfo =
                new PluginInfo("1", "新・天上碑 (GameOn) プラグイン ", "pine",
                    "新・天上碑 (GameOn) を起動するプラグインです。");

            this.GameInfo =
                new GameInfo("ゲームオン", "GameOn", "新・天上碑",
                    "http://tenjouhi.gamechu.jp/member/");
        }

        /** \brief プラグインの開始処理 */
        protected override void StartPlugin()
        {
            this.CallPlugin("GameOn", this.LoadGameStartPage);
        }

        /** \brief ゲーム開始ページを読み込む */
        private void LoadGameStartPage()
        {
            this.Notice("ゲーム開始ページに接続しています。");

            // ゲーム開始ページへ移動
            this.Navigate(this.GameInfo.Url, this.GameStart);
        }

        /** \brief ゲームを開始する */
        private void GameStart()
        {
            this.Notice("ゲームを開始しています。");

            // すべての div 要素を取得
            var divs = this.Document.GetElementsByTagName("div").Cast<HtmlElement>();

            // ゲーム開始ボタンの div 要素を取得
            var query =
                from A
                in divs
                where A.GetAttribute("className") == "gamestart"
                select A;

            if (!query.Any())
            {
                this.Error("ゲーム開始パラメーターが取得できませんでした。");
                return;
            }

            // div 要素内のすべての img 要素を取得
            var imgs = query.First().GetElementsByTagName("img");

            if (imgs.Count == 0)
            {
                this.Error("ゲーム開始パラメーターが取得できませんでした。");
                return;
            }

            // ゲームを開始する
            imgs[0].InvokeMember("click");
        }
    }

    /** \brief EA SPORTS TM FIFA Online 2 の起動プラグイン */
    [Export(typeof(IPlugin))]
    [Export(typeof(IGamePlugin))]
    [ExportMetadata("PluginId", "GameOn.Fifa")]
    public class Fifa : GameOnFlashvarsBase
    {
        /** \brief プラグインの情報を初期化する */
        public Fifa()
        {
            this.PluginInfo =
                new PluginInfo("1", "EA SPORTS TM FIFA Online 2 (GameOn) プラグイン ", "pine",
                    "EA SPORTS TM FIFA Online 2 (GameOn) を起動するプラグインです。");

            this.GameInfo =
                new GameInfo("ゲームオン", "GameOn", "EA SPORTS TM FIFA Online 2",
                    "https://secure.eafifaonline.jp/member/");

            this.FlashId = "flash_gamestart_loginSWF";
            this.FlashvarsName = "GAMESTART_URL";
        }
    }
}
