﻿using System;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text.RegularExpressions;

// Online Game Starter.NET プラグイン ベースクラス
// プラグインを簡単に実装するための、ベースとなるクラスです
// 仕様は固まっていないため、今後変更される可能性があります

namespace OGSNET.Plugins
{
    using OGSNET.Plugin;

    /** \brief IPlugin を実装したプラグインを作成するためのベースクラス */
    public abstract class PluginBase : IPlugin
    {
        /** \brief 外部プラグイン呼び出しのヘルパークラス */
        private class CallPluginHelper
        {
            private StartCallback startCallbackOriginal;
            private Action success;

            public CallPluginHelper(StartCallback callback, Action success)
            {
                this.startCallbackOriginal = callback;
                this.success               = success;

            }

            /** \brief StartCallback を取得する */
            public StartCallback GetStartCallback()
            {
                return new StartCallback(this.StartCallback);
            }

            /** \brief StartCallback のラッパー */
            private void StartCallback(CallbackStatus status, string message)
            {
                var statusOriginal = status;

                if (status == CallbackStatus.Finish)
                {
                    status = CallbackStatus.Notice;
                }

                this.startCallbackOriginal(status, message);

                if (statusOriginal == CallbackStatus.Finish)
                {
                    this.success();
                }
            }
        }

        /** \brief イベントを保存する連想配列 */
        private Dictionary<string, Action> urlAction;
        private Dictionary<Regex, Action> urlRegexAction;

        #region IPlugin の実装

        /**
         * \brief 開始する
         * StartPlugin 抽象メソッドを呼び出す
         *
         * \arg \c pluginStartInfo プラグインを開始するための引数
         */
        public void Start(PluginStartInfo pluginStartInfo)
        {
            this.Browser       = pluginStartInfo.Browser;
            this.Account       = pluginStartInfo.Account;
            this.GetPlugin     = pluginStartInfo.GetPlugin;
            this.StartCallback = pluginStartInfo.StartCallback;
            this.urlAction     = new Dictionary<string, Action>();
            this.urlRegexAction = new Dictionary<Regex, Action>();

            this.Browser.DocumentCompleted += new WebBrowserDocumentCompletedEventHandler(this.OnDocumentCompleted);

            this.StartPlugin(); // 抽象メソッドの呼び出し
        }

        /**
         * \brief プラグインの情報を返す
         * \return プラグインの情報
         */
        public PluginInfo GetPluginInfo()
        {
            return this.PluginInfo;
        }

        #endregion
        #region Abstracts

        /**
         * \brief 開始する
         * Start から呼び出される
         */
        protected abstract void StartPlugin();

        #endregion
        #region Utility

        /** \brief キャッシュを削除する */
        [DllImport("wininet", CharSet = CharSet.Unicode)]
        private extern static bool DeleteUrlCacheEntry(string lpszUrlName);

        protected bool DeleteCache(string url)
        {
            return PluginBase.DeleteUrlCacheEntry(url);
        }

        protected bool DeleteCache(Uri url)
        {
            return PluginBase.DeleteUrlCacheEntry(url.AbsoluteUri);
        }

        #endregion
        #region プラグインに関する各種関数

        /** \brief ページを移動する */
        protected void Navigate(string url)
        {
            this.DeleteCache(url);
            this.Browser.Navigate(url);
        }

        /** \brief ページを移動する */
        protected void Navigate(Uri url)
        {
            this.DeleteCache(url);
            this.Browser.Navigate(url);
        }

        /**
         * \brief 移動先URLにイベントデリゲートを追加し、ページを移動する
         * \arg \c url 移動先URL
         * \arg \c action 追加するデリゲート
         */
        protected void Navigate(string url, Action action)
        {
            this.AddAction(url, action);
            this.Navigate(url);
        }

        /**
         * \brief 移動先URLにイベントデリゲートを追加し、ページを移動する
         * \arg \c url 移動先 URL の System.Uri オブジェクト
         * \arg \c action 追加するデリゲート
         */
        protected void Navigate(Uri url, Action action)
        {
            this.AddAction(url, action);
            this.Navigate(url);
        }

        /**
         * \brief 指定されたURLにイベントデリゲートを追加し、ページを移動する
         * \arg \c url 移動先 URL
         * \arg \c action 追加するデリゲート
         * \arg \c observe イベントの監視対象の URL
         */
        protected void Navigate(string url, Action action, string observe)
        {
            this.Navigate(url);
            this.AddAction(observe, action);
        }

        /**
         * \brief 指定されたURLにイベントデリゲートを追加し、ページを移動する
         * \arg \c url 移動先 URL
         * \arg \c action 追加するデリゲート
         * \arg \c observe イベントの監視対象の URL の System.Uri オブジェクト
         */
        protected void Navigate(string url, Action action, Uri observe)
        {
            this.Navigate(url);
            this.AddAction(observe, action);
        }

        /**
         * \brief 指定されたURLにイベントデリゲートを追加し、ページを移動する
         * \arg \c url 移動先 URL
         * \arg \c action 追加するデリゲート
         * \arg \c regex イベントの監視対象の URL の正規表現
         */
        protected void Navigate(string url, Action action, Regex regex)
        {
            this.Navigate(url);
            this.AddAction(regex, action);
        }

        protected void Navigate(Uri url, Action action, Uri observe)
        {
            this.Navigate(url);
            this.AddAction(observe, action);
        }

        protected void Navigate(Uri url, Action action, string observe)
        {
            this.Navigate(url);
            this.AddAction(observe, action);
        }

        protected void Navigate(Uri url, Action action, Regex regex)
        {
            this.Navigate(url);
            this.AddAction(regex, action);
        }


        /** \brief URL移動に応じたイベントを追加 */
        protected void AddAction(string url, Action action)
        {
            this.urlAction[url] = action;
        }

        protected void AddAction(Regex regex, Action action)
        {
            this.urlRegexAction[regex] = action;
        }

        protected void AddAction(Uri url, Action action)
        {
            this.urlAction[url.AbsoluteUri] = action;
        }

        /** \brief URL移動に応じたイベントを削除 */
        protected bool RemoveAction(string url)
        {
            return this.urlAction.Remove(url);
        }

        protected bool RemoveAction(Uri url)
        {
            return this.urlAction.Remove(url.AbsoluteUri);
        }

        protected bool RemoveAction(Regex regex)
        {
            return this.urlRegexAction.Remove(regex);
        }

        /** \brief イベントをすべて削除 */
        protected void ClearAction()
        {
            this.urlAction.Clear();
            this.urlRegexAction.Clear();
        }

        /** \brief 外部プラグインの呼び出し */
        protected void CallPlugin(string pluginId, Action success)
        {
            PluginStartInfo pluginStartInfo;
            CallPluginHelper callPluginHelper
                = new CallPluginHelper(this.StartCallback, success);

            pluginStartInfo.Account       = this.Account;
            pluginStartInfo.Browser       = this.Browser;
            pluginStartInfo.GetPlugin     = this.GetPlugin;
            pluginStartInfo.StartCallback = callPluginHelper.GetStartCallback();

            this.GetPlugin(pluginId).Start(pluginStartInfo);
        }

        /** \brief 通知メッセージを発生させる */
        protected void Notice(string message)
        {
            this.StartCallback(CallbackStatus.Notice, message);
        }

        /** \brief 警告を発生させる */
        protected void Warning(string message)
        {
            this.StartCallback(CallbackStatus.Warning, message);
        }

        /** \brief エラーを発生させる */
        protected void Error(string message)
        {
            this.StartCallback(CallbackStatus.Error, message);
        }

        /** \brief プラグインの終了を知らせる */
        protected void EndPlugin(string message)
        {
            this.StartCallback(CallbackStatus.Finish, message);
        }

        #endregion
        #region プロパティ

        /** \brief WebBrowser オブジェクト */
        protected WebBrowser Browser
        {
            private set;
            get;
        }

        /**
         * \brief ブラウザの Document オブジェクト
         * 
         * リードオンリー
         */
        protected HtmlDocument Document
        {
            get { return this.Browser.Document; }
        }

        /** \brief アカウントの情報 */
        protected Account Account
        {
            private set;
            get;
        }

        /** \brief ユーザー名 */
        protected string UserName
        {
            get { return this.Account.UserName; }
        }

        /** \brief パスワード */
        protected string Password
        {
            get { return this.Account.Password; }
        }

        /** \brief プラグインを取得するデリゲート */
        protected GetPlugin GetPlugin
        {
            private set;
            get;
        }

        /** \brief コールバックデリゲート */
        protected StartCallback StartCallback
        {
            private set;
            get;
        }

        /** \brief プラグインの情報 */
        protected PluginInfo PluginInfo
        {
            set;
            get;
        }

        #endregion
        #region イベント

        /** \brief ページの読み込みが完了したときのイベント */
        private void OnDocumentCompleted(object sender, WebBrowserDocumentCompletedEventArgs arg)
        {
            // 実行するデリゲートを保存
            var delList = new List<Action>();

            // URLに一致したデリゲートがあれば実行
            if (this.urlAction.ContainsKey(arg.Url.AbsoluteUri))
            {
                delList.Add(this.urlAction[arg.Url.AbsoluteUri]);
            }

            // 正規表現にマッチしたデリゲートがあれば実行
            
            foreach (var regex in this.urlRegexAction)
            {
                if (regex.Key.IsMatch(arg.Url.AbsoluteUri))
                {
                    delList.Add(regex.Value);
                }
            }

            // デリゲートを実行
            foreach (var del in delList)
            {
                if (del != null)
                {
                    del();
                }
            }
        }

        #endregion
    }

    /** \brief IOperationPlugin を実装したプラグインを作成するためのベースクラス */
    public abstract class OperationPluginBase :
        PluginBase,
        IOperationPlugin
    {
        /** \brief 運営の情報を返す */
        public OperationInfo GetOperationInfo(){
            return this.OperationInfo;
        }

        /** \brief 運営の情報 */
        protected OperationInfo OperationInfo
        {
            set;
            get;
        }
    }

    /** \brief IGamePlugin を実装したプラグインを作成するためのベースクラス */
    public abstract class GamePluginBase :
        PluginBase,
        IGamePlugin
    {
        /**
         * \brief ゲームの情報を返す
         * \return ゲームの情報
         */
        public GameInfo GetGameInfo()
        {
            return this.GameInfo;
        }

        /**
         * \brief ゲームの情報
         * サブクラスから必ず呼び出す必要がある
         */
        protected GameInfo GameInfo
        {
            set;
            get;
        }
    }
}
