//
//  NSFileManager+OpenMetaKit.h
//  OpenMetaKit
//
//  Created by s-take on 09/02/08.
//  
//  The MIT License
//  
//  Copyright (c) 2009 s-take.
//  
//  Permission is hereby granted, free of charge, to any person obtaining a copy
//  of this software and associated documentation files (the "Software"), to deal
//  in the Software without restriction, including without limitation the rights
//  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//  copies of the Software, and to permit persons to whom the Software is
//  furnished to do so, subject to the following conditions:
//  
//  The above copyright notice and this permission notice shall be included in
//  all copies or substantial portions of the Software.
//  
//  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//  THE SOFTWARE.
//  

/*!
    @header
    @abstract   Header for NSFileManager (OpenMetaKit)
    @discussion Header for OpenMetaKit, a category for NSFileManager supporting OpenMeta attributes.
*/
#import <Cocoa/Cocoa.h>
#import "OMKConstants.h"


/*!
    @category
    @abstract    NSFileManager extention for OpenMeta
    @discussion  A category extension for supporting OpenMeta operations.
*/
@interface NSFileManager (OpenMetaKit)

/*!
    @method     
    @abstract   Get OpenMeta tags attached to the specificed file
    @discussion Get OpenMeta user tags ('com.apple.metadata:kOMUserTags') attached to the specificed file.
    @param      path File path, whose tags you want to get.
    @result     Array of tags. Return nil if no tags are attached.
*/
- (NSArray *)tagsOfFileAtPath:(NSString *)path;

/*!
    @method     
    @abstract   Set OpenMeta tags to the specificed file
    @discussion Set OpenMeta user tags ('com.apple.metadata:kOMUserTags') to the specificed file. It update user-tagged date.
    @param      tags Array of tags to attach.
    @param      path File path to attach tags.
    @param      flag whether updates tag history.
*/
- (void)setTags:(NSArray *)tags ofFileAtPath:(NSString *)path storesHistory:(BOOL)flag;

/*!
    @method     
    @abstract   Get OpenMeta rating of the specificed file
    @discussion Get OpenMeta user raing ('com.apple.metadata:kOMStarRating') of the specificed file.
    @param      path File path, whose rate you want to get.
    @result     Rating value, return 0 if not rated.
*/
- (float)ratingOfFileAtPath:(NSString *)path;

/*!
    @method     
    @abstract   Set OpenMeta rating to the specificed file
    @discussion Set OpenMeta user rating ('com.apple.metadata:kOMStarRating') to the specificed file.
    @param      rating Rating value to set.
    @param      path File path to rate.
*/
- (void)setRating:(float)rating ofFileAtPath:(NSString *)path;

/*!
    @method     
    @abstract   Get tag history
    @discussion Get recent user tags, stored in com.openmeta.shared.plist.
    @result     Array of tags.
*/
- (NSArray *)tagHistory;

@end
