/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.compiler.environment;

import java.util.*;
import java.util.Iterator;
import java.util.Set;

import org.onion_lang.onion.lang.kernel.type.TypeSymbol;


/**
 * @author Kota Mizushima
 * Date: 2005/06/28
 */
public class LocalFrame {
  private LocalFrame parent;
  private LocalScope scope;
  private List allScopes;
  private int maxIndex;
  private boolean closed;
  
  /**
   * Creates a new frame.
   * @param parent the parent frame
   */
  public LocalFrame(LocalFrame parent) {
    this.parent = parent;
    this.scope = new LocalScope(null);
    this.allScopes = new ArrayList();
    allScopes.add(scope);
  }
  
  /**
   * Creates a new child frame.
   * @return a LocalFrame object which parent is this instance.
   */
  public LocalFrame newChildFrame(){
    return new LocalFrame(this);
  }
  
  /**
   * Gets the parent frame.
   * @return
   */
  public LocalFrame getParent(){
    return parent;
  }
  
  /**
   * Opens a new scope.
   */
  public void openScope(){
    scope = scope.newChildScope();
    allScopes.add(scope);
  }
  
  LocalScope getScope(){
    return scope;
  }
  
  /**
   * Closes the current scope.
   */
  public void closeScope(){
    scope = scope.getParent();
  }
  
  public LocalBinding[] entries(){
    Set entries = entrySet();
    LocalBinding[] binds = new LocalBinding[entries.size()];
    Iterator iterator = entries.iterator();
    for(int i = 0; i < binds.length; i++){
      binds[i] = (LocalBinding) iterator.next();
    }
    Arrays.sort(binds, new Comparator(){
      public int compare(Object binding1, Object binding2) {
        return ((LocalBinding)binding1).getIndex() - ((LocalBinding)binding2).getIndex();
      }
    });
    return binds;
  }
  
  public int addEntry(String name, TypeSymbol type){
    LocalBinding bind = scope.get(name);    
    //if name is already registered, it returns -1 which means failure.
    if(bind != null) return -1;
    
    int index = maxIndex;
    maxIndex++;
    scope.put(name, new LocalBinding(index, type));
    return index;
  }
  
  public ClosureLocalBinding lookup(String name){
    LocalFrame frame = this;
    int frameIndex = 0;
    while(frame != null){
      LocalBinding binding = frame.scope.lookup(name);
      if(binding != null){
        return new ClosureLocalBinding(
          frameIndex, binding.getIndex(), binding.getType());
      }      
      frameIndex++;
      frame = frame.parent;
    }
    return null;
  }

  public ClosureLocalBinding lookupOnlyCurrentScope(String name) {
    LocalBinding binding = scope.get(name);
    if(binding != null){
      return new ClosureLocalBinding(
        0, binding.getIndex(), binding.getType());
    }
    return null;
  }
  
  public void setAllClosed(boolean closed){
    LocalFrame frame = this;
    while(frame != null){
      frame.setClosed(closed);
      frame = frame.getParent();
    }
  }
  
  public void setClosed(boolean closed){
    this.closed = closed;
  }
  
  public boolean isClosed(){
    return closed;
  }
  
  public int depth(){
    LocalFrame frame = this;
    int depth = -1;
    while(frame != null){
      depth++;
      frame = frame.getParent();
    }
    return depth;
  }
  
  private Set entrySet(){
    Set entries = new HashSet();
    for(Iterator scopes = allScopes.iterator(); scopes.hasNext();){
      entries.addAll(((LocalScope)scopes.next()).entries());
    }
    return entries;
  }
}