/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.compiler.phase;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.onion_lang.onion.compiler.CompilationConfiguration;
import org.onion_lang.onion.compiler.problem.CompilationFailureException;
import org.onion_lang.onion.compiler.problem.CompilationProblem;
import org.onion_lang.onion.compiler.utility.*;
import org.onion_lang.onion.compiler.utility.Inputs;
import org.onion_lang.onion.compiler.utility.Paths;
import org.onion_lang.onion.lang.syntax.CompilationUnit;
import org.onion_lang.onion.lang.syntax.SourcePosition;
import org.onion_lang.onion.lang.syntax.parser.*;
import org.onion_lang.onion.lang.syntax.parser.JavaccOnionParser;
import org.onion_lang.onion.lang.syntax.parser.ParseException;

/**
 * parsing phase
 * @author Kota Mizushima
 * Date: 2005/04/19
 */
public class ParsingPhase {
  private CompilationConfiguration config;

  public ParsingPhase(CompilationConfiguration config) {
    this.config = config;
  }
  
  /**
   * parses files indicated by paths.
   * @param paths an array of path which indicate file which will be parsed
   * @return
   */
  public CompilationUnit[] process(String[] paths){
    CompilationUnit[] units = new CompilationUnit[paths.length];
    List problems = new ArrayList();
    for(int i = 0; i < paths.length; i++){
      try{
        units[i] = parse(paths[i]);
      }catch(IOException e){
        problems.add(createFileCannotReadProblem(paths[i]));
      }catch(ParseException e){
        problems.add(createParseFailureProblem(paths[i], e));
      }
    }
    if(problems.size() > 0) throw new CompilationFailureException(problems);
    return units;
  }
  
  /**
   * creates a compilation problem for cannot reading file.
   * @param path a file path which cannot be read
   * @return compilation problem
   */
  private static CompilationProblem createFileCannotReadProblem(String path){
    return new CompilationProblem(
      null, null, Messages.get("error.fileRead", new String[]{path}));
  }
  
  /**
   * creates a compilation problem for cannot parsing file.
   * @param path a file path which cannot be parsed
   * @param exception
   * @return
   */
  private static CompilationProblem createParseFailureProblem(
    String path, ParseException exception){
    Token errorToken = exception.currentToken.next;
    int line = errorToken.beginLine;
    int column = errorToken.beginColumn;
    return new CompilationProblem(
      path, new SourcePosition(line, column),
      Messages.get(
        "error.parsing.syntax",
        new String[]{ exception.getLocalizedMessage() }));
  }
        
  private static String toString(int value){ return Integer.toString(value); }
  
  private CompilationUnit parse(String path) 
  	throws IOException, ParseException {
    JavaccOnionParser parser = new JavaccOnionParser(Inputs.newReader(path));
    CompilationUnit unit = parser.CompilationUnit();
    unit.setSourceFileName(path);
    return unit;
  }

}
