/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.compiler.phase.analysis;

import org.onion_lang.onion.compiler.environment.ClassTable;
import org.onion_lang.onion.compiler.environment.ImportItem;
import org.onion_lang.onion.compiler.environment.ImportList;
import org.onion_lang.onion.compiler.environment.NameResolution;
import org.onion_lang.onion.compiler.phase.CodeAnalysisPhase;
import org.onion_lang.onion.compiler.problem.SemanticErrorReporter;
import org.onion_lang.onion.compiler.utility.Paths;
import org.onion_lang.onion.lang.kernel.ClassNode;
import org.onion_lang.onion.lang.kernel.type.ClassSymbol;
import org.onion_lang.onion.lang.syntax.*;
import org.onion_lang.onion.lang.syntax.visitor.ASTVisitor;

/**
 * @author Kota Mizushima
 * Date: 2005/07/08
 */
public class ClassTableBuilder extends ASTVisitor
	implements SemanticErrorReporter.Constants{
  private CodeAnalysisPhase phase;

  public ClassTableBuilder(CodeAnalysisPhase phase) {
    this.phase = phase;
  }
  
  public void process(CompilationUnit unit){
    phase.setUnit(unit);
    ModuleDeclaration module = unit.getModuleDeclaration();
    String moduleName = module != null ? module.getName() : null;
    ImportList imports = createImport(moduleName, unit.getImportListDeclaration());
    phase.setImport(imports);
    TopLevelElement[] toplevels = unit.getTopLevels();
    int count = 0;
    for(int i = 0; i < toplevels.length; i++){
      TopLevelElement toplevel = toplevels[i];
      if(toplevel instanceof TypeDeclaration){
        accept(toplevel, moduleName);
      }else{
        count++;
      }
    }
    ClassTable table = phase.table();
    if(count > 0){
      ClassNode node = ClassNode.newClass(
        0, phase.topClass(), table.rootClass(), new ClassSymbol[0]);
      node.setSourceFile(Paths.getName(unit.getSourceFileName()));
      node.setResolutionComplete(true);
      table.addSourceClass(node);
      node.addDefaultConstructor();
      phase.put(unit, node);
      phase.putResolver(node.getName(), new NameResolution(imports, table));
    }
  }
  
  public Object visit(ClassDeclaration ast, Object context) {
    String module = (String)context;
    ClassNode node = ClassNode.newClass(
      ast.getModifier(), 
      createName(module, ast.getName()), null, null);
    node.setSourceFile(Paths.getName(phase.getUnit().getSourceFileName()));
    if(phase.table().lookup(node.getName()) != null){
      phase.report(DUPLICATE_CLASS,  ast, new Object[]{node.getName()});
      return null;
    }
    ClassTable table = phase.table();
    phase.table().addSourceClass(node);
    phase.put(ast, node);
    phase.putResolver(node.getName(), new NameResolution(phase.getImport(), table));
    return null;    
  }
  
  public Object visit(InterfaceDeclaration ast, Object context) {
    String module = (String)context;
    ClassNode node = ClassNode.newInterface(
      ast.getModifier(), createName(module, ast.getName()),
      null);
    node.setSourceFile(Paths.getName(phase.getUnit().getSourceFileName()));
    ClassTable table = phase.table();
    if(table.lookup(node.getName()) != null){
      phase.report(
        DUPLICATE_CLASS,  ast, new Object[]{node.getName()});
      return null;
    }
    table.addSourceClass(node);
    phase.put(ast, node);
    phase.putResolver(
      node.getName(), new NameResolution(phase.getImport(), phase.table()));
    return null;
  }  
  
  private ImportList createImport(String module, ImportListDeclaration imports){
    ImportList list = new ImportList();
    list.add(new ImportItem("*", "java.lang.*"));
    list.add(new ImportItem("*", "java.io.*"));
    list.add(new ImportItem("*", "java.util.*"));
    list.add(new ImportItem("*", "javax.swing.*"));
    list.add(new ImportItem("*", module != null ? module + ".*" : "*"));
    if(imports != null){
	    for(int i = 0; i < imports.size(); i++){
	      list.add(new ImportItem(imports.getName(i), imports.getFQCN(i)));
	    }
    }
    return list;
  }
  
  private String createName(String moduleName, String simpleName){
    return (moduleName != null ? moduleName + "." : "") + simpleName;
  }
}
