/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.compiler.phase.analysis;

import java.util.*;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import org.onion_lang.onion.compiler.environment.*;
import org.onion_lang.onion.compiler.environment.MethodSymbolComparator;
import org.onion_lang.onion.compiler.environment.NameResolution;
import org.onion_lang.onion.compiler.phase.CodeAnalysisPhase;
import org.onion_lang.onion.compiler.problem.SemanticErrorReporter;
import org.onion_lang.onion.compiler.utility.Methods;
import org.onion_lang.onion.lang.kernel.*;
import org.onion_lang.onion.lang.kernel.type.*;
import org.onion_lang.onion.lang.syntax.*;
import org.onion_lang.onion.lang.syntax.visitor.ASTVisitor;

/**
 * @author Kota Mizushima
 * Date: 2005/07/12
 */
public class DuplicationChecker extends ASTVisitor
	implements SemanticErrorReporter.Constants {
  private CodeAnalysisPhase state;
  private Set methods;
  private Set constructors;
  private Set fields;
  private Set globalVariables;
  private Set functions;
  
  public DuplicationChecker(CodeAnalysisPhase phase) {
    this.state = phase;
    this.methods = new TreeSet(new MethodSymbolComparator());
    this.fields = new TreeSet(new FieldSymbolComparator());
    this.constructors = new TreeSet(new ConstructorSymbolComparator());
    this.globalVariables = new TreeSet(new FieldSymbolComparator());
    this.functions = new TreeSet(new MethodSymbolComparator());
  }
  
  public void process(CompilationUnit unit){
    state.setUnit(unit);
    globalVariables.clear();
    functions.clear();
    TopLevelElement[] toplevels = unit.getTopLevels();    
    for(int i = 0; i < toplevels.length; i++){      
      state.setCurrentResolver(state.getResolver(state.topClass()));
      accept(toplevels[i]);
    }
  }
  
  public Object visit(ClassDeclaration ast, Object context) {
    ClassNode node = (ClassNode) state.lookupKernelNode(ast);
    if(node == null) return null;
    methods.clear();
    fields.clear();
    constructors.clear();
    state.setContextClass(node);
    state.setCurrentResolver(state.getResolver(node.getName()));
    if(ast.getDefaultSection() != null){
      accept(ast.getDefaultSection());
    }
    AccessSection[] sections = ast.getSections();
    for(int i = 0; i < sections.length; i++){
      accept(sections[i]);
    }
    generateMethods();
    return null;
  }
  
  private void generateMethods(){
    Set generated = new TreeSet(new MethodSymbolComparator());
    Set methodSet = new TreeSet(new MethodSymbolComparator());
    for(Iterator i = fields.iterator(); i.hasNext();){
      FieldNode node = (FieldNode)i.next();
      if(Modifier.isForwarded(node.getModifier())){
        generateDelegationMethods(node ,generated, methodSet);
      }
    }
  }
  
  private void generateDelegationMethods(FieldNode node, Set generated, Set methodSet){    
    ClassSymbol type = (ClassSymbol) node.getType();
    Set src = Methods.intefaceMethods(type);
    for (Iterator i = src.iterator(); i.hasNext();) {
      MethodSymbol method = (MethodSymbol) i.next();
      if(methodSet.contains(method)) continue;
      if(generated.contains(method)){
        report(
          DUPLICATE_GENERATED_METHOD, state.lookupAST(node),
          new Object[]{
            method.getClassType(), method.getName(), method.getArguments()
          });
        continue;
      }
      MethodNode generatedMethod = createEmptyMethod(node, method);
      generated.add(generatedMethod);
      state.getContextClass().addMethod(generatedMethod);
    }
  }
  
  private MethodNode createEmptyMethod(FieldSymbol field, MethodSymbol method){
    ExpressionNode target;
    target = new FieldRefNode(new SelfNode(state.getContextClass()), field);
    TypeSymbol[] args = method.getArguments();
    ExpressionNode[] params = new ExpressionNode[args.length];
    LocalFrame frame = new LocalFrame(null);
    for(int i = 0; i < params.length; i++){
      int index = frame.addEntry("arg" + i, args[i]);
      params[i] = new LocalRefNode(new ClosureLocalBinding(0, index, args[i]));
    }
    target = new MethodCallNode(target, method, params);
    BlockNode statement;
    if(method.getReturnType() != BasicSymbol.VOID){
      statement = new BlockNode(new StatementNode[]{new ReturnNode(target)});
    }else{
      statement = 
        new BlockNode(
          new StatementNode[]{
            new ExpressionStatementNode(target), new ReturnNode(null)
          });
    }
    MethodNode node = new MethodNode(
      Modifier.PUBLIC, state.getContextClass(), method.getName(),
      method.getArguments(), method.getReturnType(), statement);
    node.setFrame(frame);
    return node;
  }
    
  public Object visit(InterfaceDeclaration ast, Object context) {
    ClassNode node = (ClassNode) state.lookupKernelNode(ast);
    if(node == null) return null;
    methods.clear();
    fields.clear();
    constructors.clear();
    state.setContextClass(node);
    state.setCurrentResolver(state.getResolver(node.getName()));
    InterfaceMethodDeclaration[] members = ast.getDeclarations();
    for(int i = 0; i < members.length; i++){
      accept(members[i], context);
    }
    return null;
  }
  
  public Object visit(ConstructorDeclaration ast, Object context) {
    ConstructorNode node = (ConstructorNode) state.lookupKernelNode(ast);
    if(node == null) return null;
    if(constructors.contains(node)){
      ClassSymbol classType = node.getClassType();
      TypeSymbol[] args = node.getArguments();
      report(DUPLICATE_CONSTRUCTOR, ast, new Object[]{classType, args});
    }else{
      constructors.add(node);
    }
    return null;
  }
  
  public Object visit(DelegationDeclaration ast, Object context) {
    FieldNode node = (FieldNode) state.lookupKernelNode(ast);    
    if(node == null) return null;
    if(fields.contains(node)){
      ClassSymbol classType = node.getClassType();
      String name = node.getName();
      report(DUPLICATE_FIELD, ast, new Object[]{classType, name});
    }else{
      fields.add(node);
    }
    return null;
  }
  
  public Object visit(MethodDeclaration ast, Object context) {
    MethodNode node = (MethodNode) state.lookupKernelNode(ast);
    if(node == null) return null;
    if(methods.contains(node)){
      ClassSymbol classType = node.getClassType();
      String name = node.getName();
      TypeSymbol[] args = node.getArguments();
      report(DUPLICATE_METHOD, ast, new Object[]{classType, name, args});
    }else{
      methods.add(node);
    }
    return null;
  }
  
  public Object visit(FieldDeclaration ast, Object context) {
    FieldNode node = (FieldNode) state.lookupKernelNode(ast);
    if(node == null) return null;
    if(fields.contains(node)){
      ClassSymbol classType = node.getClassType();
      String name = node.getName();
      report(DUPLICATE_FIELD, ast, new Object[]{classType, name});
    }else{
      fields.add(node);
    }
    return null;
  }
  
  public Object visit(InterfaceMethodDeclaration ast, Object context) {
    MethodNode node = (MethodNode) state.lookupKernelNode(ast);
    if(node == null) return null;
    if(methods.contains(node)){
      ClassSymbol classType = node.getClassType();
      String name = node.getName();
      TypeSymbol[] args = node.getArguments();
      report(DUPLICATE_METHOD, ast, new Object[]{classType, name, args});
    }else{
      methods.add(node);
    }
    return null;
  }
  
  public Object visit(FunctionDeclaration ast, Object context) {
    MethodNode node = (MethodNode) state.lookupKernelNode(ast);
    if(node == null) return null;
    if(functions.contains(node)){
      ClassSymbol classType = node.getClassType();
      String name = node.getName();
      TypeSymbol[] args = node.getArguments();
      report(DUPLICATE_FUNCTION, ast, new Object[]{name, args});
    }else{
      functions.add(node);
    }
    return null;
  }
  
  public Object visit(GlobalVariableDeclaration ast, Object context) {
    FieldNode node = (FieldNode) state.lookupKernelNode(ast);
    if(node == null) return null;
    if(globalVariables.contains(node)){
      ClassSymbol classType = node.getClassType();
      String name = node.getName();
      report(DUPLICATE_GLOBAL_VARIABLE, ast, new Object[]{name});      
    }else{
      globalVariables.add(node);
    }
    return null;
  }
    
  public Object visit(AccessSection section, Object context){
    if(section == null) return null;    
    MemberDeclaration[] members = section.getMembers();
    for(int i = 0; i < members.length; i++){
      accept(members[i], context);
    }
    return null;
  }
  
  private void report(int error, AbstractSyntaxTreeNode ast, Object[] items){
    state.report(error, ast, items);
  }
}
