/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.compiler.phase.analysis;

import java.util.*;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import org.onion_lang.onion.compiler.environment.NameResolution;
import org.onion_lang.onion.compiler.phase.CodeAnalysisPhase;
import org.onion_lang.onion.compiler.problem.SemanticErrorReporter;
import org.onion_lang.onion.compiler.utility.Methods;
import org.onion_lang.onion.lang.kernel.*;
import org.onion_lang.onion.lang.kernel.type.*;
import org.onion_lang.onion.lang.syntax.*;
import org.onion_lang.onion.lang.syntax.visitor.ASTVisitor;

/**
 * @author Kota Mizushima
 * Date: 2005/07/08
 */
public class TypeHeaderAnalysis extends ASTVisitor
	implements SemanticErrorReporter.Constants {
  private CodeAnalysisPhase phase;
  private int countConstructor;
  
  public TypeHeaderAnalysis(CodeAnalysisPhase phase) {
    this.phase = phase;
  }
  
  public void process(CompilationUnit unit){
    phase.setUnit(unit);
    TopLevelElement[] toplevels = unit.getTopLevels();    
    for(int i = 0; i < toplevels.length; i++){      
      phase.setCurrentResolver(phase.getResolver(phase.topClass()));
      accept(toplevels[i]);
    }
  }
  
  public Object visit(ClassDeclaration ast, Object context) {
    countConstructor = 0;
    ClassNode node = (ClassNode) phase.lookupKernelNode(ast);
    phase.setContextClass(node);
    phase.setCurrentResolver(phase.getResolver(node.getName()));
    constructTypeHierarchy(node, new ArrayList());
    if(hasCyclicity(node)){
      report(CYCLIC_INHERITANCE, ast, new Object[]{node.getName()});
    }
    if(ast.getDefaultSection() != null){
      accept(ast.getDefaultSection());
    }
    AccessSection[] sections = ast.getSections();
    for(int i = 0; i < sections.length; i++){
      accept(sections[i]);
    }
    if(countConstructor == 0){
      node.addDefaultConstructor();
    }
    return null;
  }
    
  public Object visit(InterfaceDeclaration ast, Object context) {
    ClassNode node = (ClassNode) phase.lookupKernelNode(ast);
    phase.setContextClass(node);
    phase.setCurrentResolver(phase.getResolver(node.getName()));
    constructTypeHierarchy(node, new ArrayList());
    if(hasCyclicity(node)){
      report(CYCLIC_INHERITANCE, ast, new Object[]{node.getName()});
    }
    InterfaceMethodDeclaration[] members = ast.getDeclarations();
    for(int i = 0; i < members.length; i++){
      accept(members[i], context);
    }
    return null;
  }
  
  public Object visit(ConstructorDeclaration ast, Object context) {
    countConstructor++;
    TypeSymbol[] args = acceptTypes(ast.getArguments());
    ClassNode contextClass = phase.getContextClass();
    if(args == null) return null;
    int modifier = ast.getModifier() | phase.getAccess();
    ConstructorNode node = 
      new ConstructorNode(modifier, contextClass, args, null, null);
    phase.put(ast, node);
    contextClass.addConstructor(node);
    return null;
  }
  
  public Object visit(DelegationDeclaration ast, Object context) {
    TypeSymbol type = phase.resolve(ast.getType());
    if(type == null) return null;  
    if(!(type.isReferenceType() && ((ObjectSymbol)type).isInterface())){
      report(INTERFACE_REQUIRED, ast.getType(), new Object[]{type});
      return null;
    }
    ClassNode contextClass = phase.getContextClass();
    int modifier = ast.getModifier() | phase.getAccess() | Modifier.FORWARDED;
    String name = ast.getName();
    FieldNode node = new FieldNode(modifier, contextClass, name, type);
    phase.put(ast, node);
    contextClass.addField(node);
    return null;
  }
  
  public Object visit(MethodDeclaration ast, Object context) {
    TypeSymbol[] args = acceptTypes(ast.getArguments());
    TypeSymbol returnType;
    if(ast.getReturnType() != null){
      returnType = phase.resolve(ast.getReturnType());
    }else{
      returnType = BasicSymbol.VOID;
    }
    if(args == null || returnType == null) return null;
    
    ClassNode cotextClass = phase.getContextClass();
    int modifier = ast.getModifier() | phase.getAccess();
    String name = ast.getName();    
    MethodNode node = 
      new MethodNode(modifier, cotextClass, name, args, returnType, null);     
    phase.put(ast, node);
    cotextClass.addMethod(node);
    return null;
  }
  
  public Object visit(FieldDeclaration ast, Object context) {
    TypeSymbol type = phase.resolve(ast.getType());
    if(type == null) return null;
    ClassNode contextClass = phase.getContextClass();
    int modifier = ast.getModifier() | phase.getAccess();
    String name = ast.getName();    
    FieldNode node = new FieldNode(modifier, contextClass, name, type);
    phase.put(ast, node);
    contextClass.addField(node);
    return node;
  }
  
  private TypeSymbol[] acceptTypes(ArgumentDeclaration[] ast){
    TypeSymbol[] types = new TypeSymbol[ast.length];
    boolean success = true;
    for (int i = 0; i < ast.length; i++) {
      types[i] = (TypeSymbol) accept(ast[i]);
      if(types[i] == null) success = false;
    }
    if(success){
      return types;
    }else{
      return null;
    }
  }
  
  public Object visit(ArgumentDeclaration ast, Object context){
    TypeSymbol type = phase.resolve(ast.getType());
    return type;
  }
  
  private FieldNode createFieldNode(FieldDeclaration ast){
    TypeSymbol type = phase.resolve(ast.getType());
    if(type == null) return null;
    FieldNode node = new FieldNode(
      ast.getModifier() | phase.getAccess(), phase.getContextClass(), 
      ast.getName(), type);
    return node;
  }
    
  public Object visit(InterfaceMethodDeclaration ast, Object context) {
    TypeSymbol[] args = acceptTypes(ast.getArguments());
    TypeSymbol returnType;
    if(ast.getReturnType() != null){
      returnType = phase.resolve(ast.getReturnType());
    }else{
      returnType = BasicSymbol.VOID;
    }
    if(args == null || returnType == null) return null;
    
    int modifier = Modifier.PUBLIC | Modifier.ABSTRACT;
    ClassNode classType = phase.getContextClass();
    String name = ast.getName();    
    MethodNode node = 
      new MethodNode(modifier, classType, name, args, returnType, null);
    phase.put(ast, node);
    classType.addMethod(node);
    return null;
  }
  
  public Object visit(FunctionDeclaration ast, Object context) {
    TypeSymbol[] args = acceptTypes(ast.getArguments());
    TypeSymbol returnType;
    if(ast.getReturnType() != null){
      returnType = phase.resolve(ast.getReturnType());
    }else{
      returnType = BasicSymbol.VOID;
    }
    if(args == null || returnType == null) return null;
    
    ClassNode classType = (ClassNode) phase.loadTopClass();
    int modifier = ast.getModifier() | Modifier.PUBLIC;
    String name = ast.getName();
    
    MethodNode node = 
      new MethodNode(modifier, classType, name, args, returnType, null);
    phase.put(ast, node);
    classType.addMethod(node);
    return null;
  }
  
  public Object visit(GlobalVariableDeclaration ast, Object context) {
    TypeSymbol type = phase.resolve(ast.getType());
    if(type == null) return null;
    
    int modifier = ast.getModifier() | Modifier.PUBLIC;
    ClassNode classType = (ClassNode)phase.loadTopClass();
    String name = ast.getName();
    
    FieldNode node = new FieldNode(modifier, classType, name, type);
    phase.put(ast, node);
    classType.addField(node);
    return null;
  }
    
  public Object visit(AccessSection section, Object context){
    if(section == null) return null;
    
    phase.setAccess(section.getID());
    MemberDeclaration[] members = section.getMembers();
    for(int i = 0; i < members.length; i++){
      accept(members[i], context);
    }
    return null;
  }
  
  public boolean hasCyclicity(ClassNode start){
    return hasCylicitySub(start, new HashSet());
  }
  
  private boolean hasCylicitySub(ClassSymbol symbol, HashSet visit){
    if(symbol == null) return false;
    if(visit.contains(symbol)){
      return true;      
    }
    visit.add(symbol);
    if(hasCylicitySub(symbol.getSuperClass(), (HashSet)visit.clone())){
      return true;      
    }
    ClassSymbol[] interfaces = symbol.getInterfaces();
    for (int i = 0; i < interfaces.length; i++) {
      if(hasCylicitySub(interfaces[i], (HashSet)visit.clone())){
        return true;        
      }
    }
    return false;
  }

  private void constructTypeHierarchy(ClassSymbol symbol, List visit) {
    if(symbol == null || visit.indexOf(symbol) >= 0) return;
    visit.add(symbol);
    if(symbol instanceof ClassNode){
      ClassNode node = (ClassNode) symbol;
      if(node.isResolutionComplete()) return;
      ClassSymbol superClass = null;
      List interfaces = new ArrayList();
      NameResolution resolver = phase.getResolver(node.getName());
      if(node.isInterface()){
        InterfaceDeclaration ast = (InterfaceDeclaration) phase.lookupAST(node);
        superClass = phase.rootClass();
        TypeSpecifier[] typeSpecifiers = ast.getInterfaces();
        for(int i = 0; i < typeSpecifiers.length; i++){
          ClassSymbol superType = validateSuperType(typeSpecifiers[i], true, resolver);
          if(superType != null){
            interfaces.add(superType);
          }
        }
      }else{
        ClassDeclaration ast = (ClassDeclaration) phase.lookupAST(node);
        superClass = 
          validateSuperType(ast.getSuperClass(), false, resolver);
        TypeSpecifier[] typeSpecifiers = ast.getInterfaces();
        for(int i = 0; i < typeSpecifiers.length; i++){
          ClassSymbol superType = validateSuperType(typeSpecifiers[i], true, resolver);
          if(superType != null){
            interfaces.add(superType);
          }
        }
      }
      constructTypeHierarchy(superClass, visit);
      for(Iterator i = interfaces.iterator(); i.hasNext();){
        ClassSymbol superType = (ClassSymbol) i.next();
        constructTypeHierarchy(superType, visit);
      }
      node.setSuperClass(superClass);
      node.setInterfaces((ClassSymbol[]) interfaces.toArray(new ClassSymbol[0]));
      node.setResolutionComplete(true);
    }else{
      constructTypeHierarchy(symbol.getSuperClass(), visit);
      ClassSymbol[] interfaces = symbol.getInterfaces();
      for(int i = 0; i < interfaces.length; i++){
        constructTypeHierarchy(interfaces[i], visit);
      }
    }
  }
  
  private ClassSymbol validateSuperType(
    TypeSpecifier ast, boolean shouldInterface, NameResolution resolver){
    assert ast.getDimension() == 0;
    assert ast.getComponentKind() != TypeComponent.BASIC;
    
    ClassSymbol symbol = null;
    if(ast == null){
      symbol = phase.table().rootClass();
    }else{
      symbol = (ClassSymbol) phase.resolve(ast, resolver);
    }
    if(symbol == null) return null;
    boolean isInterface = symbol.isInterface();
    if(((!isInterface) && shouldInterface) || (isInterface && (!shouldInterface))){
      AbstractSyntaxTreeNode astNode = null;
      if(symbol instanceof ClassNode){
        astNode = phase.lookupAST((ClassNode)symbol);
      }
      report(ILLEGAL_INHERITANCE, astNode, new Object[]{symbol.getName()});
    }
    return symbol;
  }
  
  private void report(int error, AbstractSyntaxTreeNode ast, Object[] items){
    phase.report(error, ast, items);
  }
}
