/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.compiler.problem;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import org.onion_lang.onion.compiler.utility.Messages;
import org.onion_lang.onion.lang.kernel.type.*;
import org.onion_lang.onion.lang.syntax.SourcePosition;

/**
 * @author Kota Mizushima
 * Date: 2005/06/23
 */
public class SemanticErrorReporter {
  private List problems;
  private String currentSourceFile;
  int errorCount;
  int threshold;  
  public interface Constants {
    int INCOMPATIBLE_TYPE = 0;
    int INCOMPATIBLE_OPERAND_TYPE = 1;
    int VARIABLE_NOT_FOUND = 2;
    int CLASS_NOT_FOUND = 3;
    int FIELD_NOT_FOUND = 4;
    int METHOD_NOT_FOUND = 5;
    int AMBIGUOUS_METHOD = 6;
    int DUPLICATE_LOCAL_VARIABLE = 7;
    int DUPLICATE_CLASS = 8;
    int DUPLICATE_FIELD = 9;
    int DUPLICATE_METHOD = 10;
    int DUPLICATE_GLOBAL_VARIABLE = 11;
    int DUPLICATE_FUNCTION = 12;
    int METHOD_NOT_ACCESSIBLE = 13;
    int FIELD_NOT_ACCESSIBLE = 14;
    int CLASS_NOT_ACCESSIBLE = 15;
    int CYCLIC_INHERITANCE = 16;
    int CYCLIC_DELEGATION = 17;
    int ILLEGAL_INHERITANCE = 18;
    int ILLEGAL_METHOD_CALL = 19;
    int CANNOT_RETURN_VALUE = 20;
    int CONSTRUCTOR_NOT_FOUND = 21;
    int AMBIGUOUS_CONSTRUCTOR= 22;
    int INTERFACE_REQUIRED = 23;
    int UNIMPLEMENTED_FEATURE = 24;
    int DUPLICATE_CONSTRUCTOR = 25;
    int DUPLICATE_GENERATED_METHOD = 26;
  }

  public SemanticErrorReporter(int threshold) {
    this.threshold = threshold;
    problems = new ArrayList();
  }
  
  private String format(String string, String arg){
    return MessageFormat.format(string, new Object[]{arg});
  }
  
  private String format(String string, String arg1, String arg2){
    return MessageFormat.format(string, new Object[]{arg1, arg2});
  }
  
  private String format(String string, String arg1, String arg2, String arg3){
    return MessageFormat.format(string, new Object[]{arg1, arg2, arg3});
  }
  
  private String format(
    String string, String arg1, String arg2, String arg3, String arg4){
    return MessageFormat.format(string, new Object[]{arg1, arg2, arg3, arg4});
  }
  
  private String format(String string, String[] args){
    return MessageFormat.format(string, args);
  }
  
  private String message(String property){
    return Messages.get(property);
  }

  private void reportIncompatibleType(
    SourcePosition position, Object[] items){
    TypeSymbol expected = (TypeSymbol) items[0];
    TypeSymbol detected = (TypeSymbol) items[1];
    problem(
      position,
      format(
        message("error.semantic.incompatibleType"),
        expected.getName(), detected.getName()));  
  }
  
  private String names(TypeSymbol[] types){
    StringBuffer buffer = new StringBuffer();
    if(types.length > 0){
      buffer.append(types[0].getName());
      for(int i = 1; i < types.length; i++){
        buffer.append(", ");
        buffer.append(types[i].getName());
      }
    }
    return new String(buffer);
  }
    
  private void reportIncompatibleOperandType(
    SourcePosition position, Object[] items){
    String operator = (String) items[0];
    TypeSymbol[] operands = (TypeSymbol[]) items[1];
    problem(
      position,
      format(message("error.semantic.incompatibleOperandType"), 
      (String)items[0], names(operands)));
  }
  
  private void reportVariableNotFound(
    SourcePosition position, Object[] items){
    problem(
      position,
      format(message("error.semantic.variableNotFound"), (String)items[0]));
  }
  
  private void reportClassNotFound(
    SourcePosition position, Object[] items){
    problem(
      position,
      format(message("error.semantic.classNotFound"), (String)items[0]));
  }
  
  private void reportFieldNotFound(
    SourcePosition position, Object[] items){
    problem(
      position,
      format(
        message("error.semantic.fieldNotFound"), 
        ((TypeSymbol)items[0]).getName(), (String)items[1]));
  }
  
  private void reportMethodNotFound(
    SourcePosition position, Object[] items){
    problem(
      position,
      format(
        message("error.semantic.methodNotFound"),
        ((TypeSymbol)items[0]).getName(), (String)items[1],
        names(((TypeSymbol[])items[2]))));
  }
  
  private void reportAmbiguousMethod(
    SourcePosition position, Object[] items){
    Object[] item1 = (Object[])items[0];
    Object[] item2 = (Object[])items[1];
    String target1 = ((ObjectSymbol)item1[0]).getName();
    String name1 = (String)item1[1];
    String args1 = names((TypeSymbol[])item1[2]);
    String target2 = ((ObjectSymbol)item2[0]).getName();
    String name2 = (String)item2[1];
    String args2 = names((TypeSymbol[])item2[2]);
    problem(
      position,
      format(
        message("error.semantic.ambiguousMethod"),
        new String[]{target1, name1, args2, target2, name2, args2}));
  }
  
  private void reportDuplicateLocalVariable(
    SourcePosition position, Object[] items){
    problem(
      position,
      format(message("error.semantic.duplicatedVariable"), (String)items[0]));
  }
  
  private void reportDuplicateClass(
    SourcePosition position, Object[] items){
    problem(
      position, 
      format(message("error.semantic.duplicatedClass"), (String)items[0]));
  }
  
  private void reportDuplicateField(
    SourcePosition position, Object[] items){
    problem(
      position,
      format(
        message("error.semantic.duplicatedField"),
        ((TypeSymbol)items[0]).getName(), (String)items[1]));
  }
  
  private void reportDuplicateMethod(
    SourcePosition position, Object[] items){
    problem(
      position,
      format(
        message("error.semantic.duplicatedMethod"),
        ((TypeSymbol)items[0]).getName(), (String)items[1],
        names((TypeSymbol[])items[2])));
  }
  
  private void reportDuplicateGlobalVariable(
    SourcePosition position, Object[] items){
    problem(
      position,
      format(
        message("error.semantic.duplicatedGlobalVariable"),
        (String)items[0]));
  }
  
  private void reportDuplicateFunction(
    SourcePosition position, Object[] items){
    problem(
      position,
      format(
        message("error.semantic.duplicatedGlobalVariable"),
        (String)items[0], names((TypeSymbol[])items[1])));
  }
  
  private void reportDuplicateConstructor(
    SourcePosition position, Object[] items){
    problem(
      position,
      format(
        message("error.semantic.duplicatedConstructor"),
        ((TypeSymbol)items[0]).getName(), names((TypeSymbol[])items[1])));
  }
  
  private void reportMethodNotAccessible(
    SourcePosition position, Object[] items){
    problem(
      position, 
      format(
        message("error.semantic.methodNotAccessible"),
        ((ObjectSymbol)items[0]).getName(),
        (String)items[1],
        names(((TypeSymbol[])items[2])),
        ((ClassSymbol)items[3]).getName()));
  }
  
  private void reportFieldNotAccessible(
    SourcePosition position, Object[] items){
    problem(
      position, 
      format(
        message("error.semantic.fieldNotAccessible"), 
        ((ClassSymbol)items[0]).getName(),
        (String)items[1],
        ((ClassSymbol)items[2]).getName()));
  }
  
  private void reportClassNotAccessible(
    SourcePosition position, Object[] items){
    problem(
      position, 
      format(
        message("error.semantic.classNotAccessible"), 
        ((ClassSymbol)items[0]).getName(),
        ((ClassSymbol)items[1]).getName()));
  }
  
  private void reportCyclicInheritance(
    SourcePosition position, Object[] items){
    problem(
      position, format(message("error.semantic.cyclicInheritance"),
      (String)items[0]));
  }

  private void reportCyclicDelegation(
    SourcePosition position, Object[] items){
    problem(position, message("error.semantic.cyclicDelegation"));
  }
  
  private void reportIllegalInheritance(
    SourcePosition position, Object[] items){
  }
  
  private void reportCannotReturnValue(
    SourcePosition position, Object[] items){
    problem(position, message("error.semantic.cannotReturnValue"));
  }
  
  private void reportConstructorNotFound(
    SourcePosition position, Object[] items){
    String type = ((TypeSymbol)items[0]).getName();
    String args = names(((TypeSymbol[])items[1]));
    problem(
      position, 
      format(message("error.semantic.constructorNotFound"), type, args));
  }
  
  private void reportAmbiguousConstructor(SourcePosition position, Object[] items){
    Object[] item1 = (Object[])items[0];
    Object[] item2 = (Object[])items[1];
    String target1 = ((ObjectSymbol)item1[0]).getName();
    String args1 = names((TypeSymbol[])item1[1]);
    String target2 = ((ObjectSymbol)item2[0]).getName();
    String args2 = names((TypeSymbol[])item2[1]);
    problem(
      position,
      format(
        message("error.semantic.ambiguousMethod"),
        target1, args2, target2, args2));
  }
  
  private void reportInterfaceRequied(SourcePosition position, Object[] items){
    TypeSymbol type = (TypeSymbol)items[0];
    problem(
      position, 
      format(message("error.semantic.interfaceRequired"), type.getName()));
  }
  
  private void reportUnimplementedFeature(SourcePosition position, Object[] items){
    problem(
      position, message("error.semantic.unimplementedFeature"));
  }
  
  private void reportDuplicateGeneratedMethod(SourcePosition position, Object[] items){
    problem(
      position,
      format(
        message("error.semantic.duplicateGeneratedMethod"),
        ((TypeSymbol)items[0]).getName(), (String)items[1],
        names((TypeSymbol[])items[2])));
  }
  
  private void problem(
    SourcePosition position, String message){
    problems.add(
      new CompilationProblem(currentSourceFile, position, message));
  }
  
  public void report(
    int error, SourcePosition position, Object[] items){
    errorCount++;    
    switch(error){
      case Constants.INCOMPATIBLE_TYPE:
        reportIncompatibleType(position, items);
      	break;
      case Constants.INCOMPATIBLE_OPERAND_TYPE:
        reportIncompatibleOperandType(position, items);
      	break;
      case Constants.VARIABLE_NOT_FOUND:
        reportVariableNotFound(position, items);
      	break;
      case Constants.CLASS_NOT_FOUND:
        reportClassNotFound(position, items);
      	break;
      case Constants.FIELD_NOT_FOUND:
        reportFieldNotFound(position, items);
      	break;
      case Constants.METHOD_NOT_FOUND:
        reportMethodNotFound(position, items);
      	break;
      case Constants.AMBIGUOUS_METHOD:
        reportAmbiguousMethod(position, items);
      	break;
      case Constants.DUPLICATE_LOCAL_VARIABLE:
        reportDuplicateLocalVariable(position, items);
      	break;
      case Constants.DUPLICATE_CLASS:
        reportDuplicateClass(position, items);
      	break;
      case Constants.DUPLICATE_FIELD:
        reportDuplicateField(position, items);
      	break;
      case Constants.DUPLICATE_METHOD:
        reportDuplicateMethod(position, items);
      	break;
      case Constants.DUPLICATE_GLOBAL_VARIABLE:
        reportDuplicateGlobalVariable(position, items);
      	break;
      case Constants.DUPLICATE_FUNCTION:
        reportDuplicateFunction(position, items);
      	break;
      case Constants.METHOD_NOT_ACCESSIBLE:
        reportMethodNotAccessible(position, items);
      	break;
      case Constants.FIELD_NOT_ACCESSIBLE:
        reportFieldNotAccessible(position, items);
      	break;
      case Constants.CLASS_NOT_ACCESSIBLE:
        reportClassNotAccessible(position, items);
      	break;
      case Constants.CYCLIC_INHERITANCE:
        reportCyclicInheritance(position, items);
      	break;
      case Constants.CYCLIC_DELEGATION:
        reportCyclicDelegation(position, items);
      	break;
      case Constants.ILLEGAL_INHERITANCE:
        reportIllegalInheritance(position, items);
      	break;
      case Constants.CANNOT_RETURN_VALUE:
        reportCannotReturnValue(position, items);
      	break;
      case Constants.CONSTRUCTOR_NOT_FOUND:
        reportConstructorNotFound(position, items);
      	break;
      case Constants.AMBIGUOUS_CONSTRUCTOR:
        reportAmbiguousConstructor(position, items);
      	break;
      case Constants.INTERFACE_REQUIRED:
        reportInterfaceRequied(position, items);
      	break;
      case Constants.UNIMPLEMENTED_FEATURE:
        reportUnimplementedFeature(position, items);
      	break;
      case Constants.DUPLICATE_CONSTRUCTOR:
        reportDuplicateConstructor(position ,items);
      	break;
      case Constants.DUPLICATE_GENERATED_METHOD:
        reportDuplicateGeneratedMethod(position, items);
    }
    if(errorCount > threshold){
      throw new CompilationFailureException(
        problems);
    }
  }
  
  public CompilationProblem[] getProblems(){
    return (CompilationProblem[]) problems.toArray(
      new CompilationProblem[0]);
  }
  
  public void setCurrentSourceFile(String currentSourceFile) {
    this.currentSourceFile = currentSourceFile;
  }
}
