/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.compiler;

import java.io.BufferedReader;
import java.io.IOException;
import java.text.MessageFormat;

import org.onion_lang.onion.compiler.phase.*;
import org.onion_lang.onion.compiler.phase.CodeAnalysisPhase;
import org.onion_lang.onion.compiler.phase.ParsingPhase;
import org.onion_lang.onion.compiler.problem.CompilationFailureException;
import org.onion_lang.onion.compiler.problem.CompilationProblem;
import org.onion_lang.onion.compiler.utility.*;
import org.onion_lang.onion.compiler.utility.Inputs;
import org.onion_lang.onion.compiler.utility.Systems;
import org.onion_lang.onion.lang.kernel.ClassNode;
import org.onion_lang.onion.lang.syntax.CompilationUnit;
import org.onion_lang.onion.lang.syntax.SourcePosition;

/**
 * @author Kota Mizushima
 * Date: 2005/04/08
 */
public class OnionCompiler {
  private CompilerConfig configuration;
  
  public OnionCompiler(CompilerConfig configuration){
    this.configuration = configuration;
  }
  
  public void setConfiguration(CompilerConfig configuration){
    this.configuration = configuration;
  }
  
  public CompilerConfig getConfiguration(){
    return configuration;
  }
  
  public CompiledClass[] compile(String[] fileNames){
    try{
      ParsingPhase parsing = new ParsingPhase(configuration);
      CompilationUnit[] units = parsing.process(fileNames);      
      CodeAnalysisPhase analysis = new CodeAnalysisPhase(configuration);
      ClassNode[] classes = analysis.process(units);
      CodeGenerationPhase generation = new CodeGenerationPhase(configuration);
      CompiledClass[] classFiles = generation.process(classes);
      return classFiles;
    }catch(CompilationFailureException ex){
      for(int i = 0; i < ex.getSizeOfProblems(); i++){
        CompilationProblem problem = ex.getProblem(i);
        SourcePosition position;
        String sourceFile;
        StringBuffer message;
        printCompilationError(problem);
      }
      return null;
    }
  }
  
  private void printCompilationError(CompilationProblem problem) {
    SourcePosition position = problem.getSourcePosition();
    String sourceFile = problem.getSourceFile();
    StringBuffer message = new StringBuffer();
    if(sourceFile == null){
      message.append(
        MessageFormat.format("{0}", new Object[]{problem.getMessage()}));
    }else{
      String line = null;
      try {
        line = 
          position != null ? findLine(sourceFile, position.getLine()) : "";
      } catch (IOException e) {
        e.printStackTrace();
      }
      line = line != null ? line : "";
      message.append(
        MessageFormat.format(
          "{0}: {1},{2}: {3}",
          new Object[]{
            sourceFile,
            position != null ? Integer.toString(position.getLine()) : "",
            position != null ? Integer.toString(position.getColumn()) : "",
            problem.getMessage()
          }));
      
      message.append(Systems.getLineSeparator());
      message.append("\t\t");
      message.append(line);
      message.append(Systems.getLineSeparator());
      message.append("\t\t");
      if(position != null){
        message.append(createCursor(position.getColumn()));
      }
    }
    System.err.println(new String(message));
  }

  private String createCursor(int column){
    return Strings.repeat(" ", column - 1) + "^";
  }
  
  private String findLine(String sourceFile, int lineNumber) throws IOException {
    BufferedReader reader = Inputs.newReader(sourceFile);
    try {
      int countLineNumber = 1;
      String line = null;
      while((line = reader.readLine()) != null){
        if(countLineNumber == lineNumber) break;
        countLineNumber++;
      }
      return line;
    } finally {
      reader.close();
    }
  }
}