/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.compiler.phase;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.onion_lang.onion.compiler.CompilerConfig;
import org.onion_lang.onion.compiler.problem.CompilationFailureException;
import org.onion_lang.onion.compiler.problem.CompilationProblem;
import org.onion_lang.onion.compiler.utility.*;
import org.onion_lang.onion.compiler.utility.Inputs;
import org.onion_lang.onion.lang.syntax.CompilationUnit;
import org.onion_lang.onion.lang.syntax.SourcePosition;
import org.onion_lang.onion.lang.syntax.parser.*;
import org.onion_lang.onion.lang.syntax.parser.JavaccOnionParser;
import org.onion_lang.onion.lang.syntax.parser.ParseException;

/**
 * parsing phase
 * @author Kota Mizushima
 * Date: 2005/04/19
 */
public class ParsingPhase {
  private CompilerConfig config;

  public ParsingPhase(CompilerConfig config) {
    this.config = config;
  }
  
  /**
   * parses files indicated by paths.
   * @param paths an array of path which indicate file which will be parsed
   * @return
   */
  public CompilationUnit[] process(String[] paths){
    CompilationUnit[] units = new CompilationUnit[paths.length];
    List problems = new ArrayList();
    for(int i = 0; i < paths.length; i++){
      try{
        units[i] = parse(paths[i]);
      }catch(IOException e){
        problems.add(createFileCannotRead(paths[i]));
      }catch(ParseException e){
        problems.add(createParseFailure(paths[i], e));
      }
    }
    if(problems.size() > 0) throw new CompilationFailureException(problems);
    return units;
  }
  
  /**
   * creates a compilation problem for cannot reading file.
   * @param path a file path which cannot be read
   * @return compilation problem
   */
  static CompilationProblem createFileCannotRead(String path){
    return new CompilationProblem(null, null, Messages.get("error.fileRead", path));
  }
  
  /**
   * creates a compilation problem for cannot parsing file.
   * @param path a file path which cannot be parsed
   * @param ex
   * @return
   */
  static CompilationProblem createParseFailure(String path, ParseException ex){
    Token errorToken = ex.currentToken.next;
    return new CompilationProblem(
      path, new SourcePosition(errorToken.beginColumn, errorToken.beginColumn),
      Messages.get("error.parsing.syntax", ex.getLocalizedMessage()));
  }
        
  static String toString(int value){ return Integer.toString(value); }
  
  JavaccOnionParser parser(String path) throws IOException {
    return new JavaccOnionParser(Inputs.newReader(path, config.getEncoding()));
  }
  
  CompilationUnit parse(String path) throws IOException, ParseException {
    CompilationUnit unit = parser(path).CompilationUnit();
    unit.setSourceFileName(path);
    return unit;
  }
}