# -*- coding: utf-8 -*-
# app/models/config_output.rb のクラス ConfigOutput をテストする。

require 'test_helper'

class ConfigOutputTest < ActiveSupport::TestCase
  fixtures(:domains, :users, :permissions,
           :products, :displays, :items,
           :config_outputs, :config_output_items)

  def setup
    CacheEachRequest.clear
    User.current = User.find_by_login("admin")
  end

  def teardown
    CacheEachRequest.clear
  end

  EOF = [
    ["none", "abc\r\ndef", "abc\r\ndef"],
    ["disabled", "あいう\r\nえお", "あいうえお"],
    ["specified", "abc\r\nde\nf", "abc de f"],
    ["disabled", 12, "12"],
  ]

  # ConfigOutput#process_eol() が動作することをテストする。
  def test_process_eol
    EOF.each do |eol_pattern, raw, expected|
      config_output = ConfigOutput.new
      config_output.eol_pattern = eol_pattern
      config_output.eol_parameter = " "
      assert_equal expected, config_output.process_eol(raw)
    end
  end

  SEPARATOR = [
    [1, "\t"],
  ]

  # ConfigOutput#separator() が動作することをテストする。
  def test_separator
    SEPARATOR.each do |id, expected|
      assert_equal expected, ConfigOutput.find(id).separator
    end
  end

  QUOTATION = [
    [1, ""],
  ]

  # ConfigOutput#quotation() が動作することをテストする。
  def test_quotation
    QUOTATION.each do |id, expected|
      assert_equal expected, ConfigOutput.find(id).quotation
    end
  end

  # ConfigOutput#clear() を設定を初期化することをテストする。
  def test_clear
    (1..3).each do |id|
      c = ConfigOutput.find(id)
      c.clear
      assert_equal c.display_to_list.name, c.name
      assert_equal "csv", c.file_format
      assert_equal "none", c.eol_pattern
      assert_equal "", c.eol_parameter
      assert_equal "tab", c.separator_pattern
      assert_equal "", c.separator_parameter
      assert_equal "none", c.quotation_pattern
      assert_equal "", c.quotation_parameter
      assert_equal "utf-8", c.encoding
    end
  end

  # ConfigOutput#iconv() がエンコーディングを変換することをテストする。
  def test_iconv
    config_output = ConfigOutput.find(1)
    [
      "開発環境",
      ["abc", "日本語"],
      ["ひらがな", [["456", [8, "深いネスト", {}]], "アカサタナ"]],
    ].each do |s|
      assert_equal s, config_output.iconv(s)
    end
  end

  MIME_TYPE = [
    [1, "text/csv; charset=utf-8; header=present"],
    [2, "application/pdf"],
    [3, "text/xml; charset=utf-8"],
  ]

  # ConfigOutput#mime_type() が動作することをテストする。
  def test_mime_type
    MIME_TYPE.each do |id, expected|
      assert_equal expected, ConfigOutput.find(id).mime_type
    end
  end

  FILE_NAME = [
    [1, "保存版.csv"],
    [2, "top_secret.pdf"],
    [3, "試験\\的 名称.xml"],
  ]

  # ConfigOutput#file_name() が動作することをテストする。
  def test_file_name
    FILE_NAME.each do |id, expected|
      assert_equal expected, ConfigOutput.find(id).file_name('OUT USER AGENT')
    end
  end

  HEADERS = [
    [1, ["foo"]],
    [2, ["項目1", "項目2"]],
    [3, ["名前", "コード"]],
  ]

  # ConfigOutput#headers() が動作することをテストする。
  def test_headers
    HEADERS.each do |id, expected|
      assert_equal expected, ConfigOutput.find(id).headers
    end
  end

  # ConfigOutput#rows() が動作することをテストする。
  def test_rows
    session_readonly = mock('session_readonly')
    assert ConfigOutput.find(3).rows({}, session_readonly).is_a?(Array)
  end

  # 関連する display_to_list が output_module_name を指定している場合の動作をテストする。
  def test_rows__output_module_name
    session_readonly = mock('session_readonly')
    config_output = ConfigOutput.new(:display_to_list_id => 9996)
    rows = config_output.rows({}, session_readonly)
    assert_kind_of Array, rows
    assert_equal 4, rows.size
    assert_equal 9996, rows[0][1]
  end
end
