# -*- coding: utf-8 -*-
# Methods added to this helper will be available to all templates in the application.

require "fragment_hash"

# フレームワークのどの view でも利用できる helper を定義する。
module ApplicationHelper
  include FragmentHash::UrlHelper
  include RandomId
  include DisplayNarrowingHelper::Helper
  include Rfw::UserAgent

  # 「閉じる」アイコン(タグ無し)
  def tree_icon_close_without_tag
    "&#x25BC;"
  end

  # 「開く」アイコン(タグ無し)
  def tree_icon_open_without_tag
    "&#x25B2;"
  end

  # 「末端」アイコン(タグ無し)
  def tree_icon_leaf_without_tag
    "&#x3000;"
  end

  # 「閉じる」アイコン
  def tree_icon_close
    '<img src="/images/close_menu.png" width=26 height=16 class="iepngfix" />'
  end

  # 「開く」アイコン
  def tree_icon_open
    '<img src="/images/open_menu.png" width=26 height=16 class="iepngfix" />'
  end

  # 「末端」アイコン
  def tree_icon_leaf
    '<img src="/images/leaf_menu.png" width=26 height=16 class="iepngfix" />'
  end

  # ラベルつきの <tt>check_box</tt>
  def check_box_with_label(object_name, method, label, options = {}, checked_value = "1", unchecked_value = "0")
    with_label(label, options, object_name, method) do |o|
      check_box(object_name, method, o, checked_value, unchecked_value)
    end
  end

  # ラベルつきの radio box
  def radio_box_with_label(object_name, method)
    "#{object_name.classify}::LABEL_#{method.upcase}".constantize.map do |pair|
      radio_button_with_label(object_name, method, pair.last, s_(pair.first))
    end.join(" ")
  end

  # ラベルつきの <tt>radio_button</tt>
  def radio_button_with_label(object_name, method, tag_value, label, options = {})
    with_label(label, options, object_name, method, tag_value) do |o|
      radio_button(object_name, method, tag_value, o)
    end
  end

  # ラベルつきの <tt>radio_button_tag</tt>
  def radio_button_tag_with_label(name, value, label_body, checked = false, options = {})
    options[:id] ||= "#{name}_#{value}_#{random_id}"
    return radio_button_tag(name, value, checked, options) + content_tag(:label, label_body, {:for => options[:id]})
  end

  # 「閉じる」リンク
  # リンクの html 要素の ID は "link_#{@current_view}_close*" という形になる。
  # * motion_params - link_to_view_motion に第4引数として渡される。
  # * options - link_to_view_motion に第5引数として渡される。
  # * html_options - link_to_view_motion に第6引数として渡される。
  def link_to_close(motion_params, options, html_options = {})
    return link_to_view_motion(h(s_("rfw|button|Close")), @current_view, "close", motion_params, options, html_options)
  end

  # ページ部品
  def pagination(locals)
    {
      :html_class    => "",
      :left_columns  => "",
      :options       => {},
      :right_columns => "",
      :suffix        => "",
    }.each do |key, value|
      locals[key] ||= value
    end
    tds = []
    tds << locals[:left_columns]
    tds << content_tag(:td, first_page_link(locals), :class => "first_page")
    tds << content_tag(:td, previous_page_link(locals[:pages].current.previous, locals), :class => "previous_page")
    current = locals[:pages].current_page.to_i
    ((current-2)..(current+2)).each do |n|
      tds << content_tag(:td, current_page_link(current, n, locals), :class => "each_page")
    end
    tds << content_tag(:td, next_page_link(locals[:pages].current.next, locals), :class => "next_page")
    tds << content_tag(:td, last_page_link(locals), :class => "last_page")
    first_item = locals[:pages].item_count == 0 ? 0 : locals[:pages].current.first_item
    d = ns_("rfw|pagination|%{first_item}-%{last_item} of %{item_count} item",
            "%{first_item}-%{last_item} of %{item_count} items",
            locals[:pages].item_count) % {
      :first_item => first_item,
      :last_item  => locals[:pages].current.last_item,
      :item_count => locals[:pages].item_count
    }
    tds << content_tag(:td, d)
    tds << locals[:right_columns]
    content = content_tag(:tr, tds.join)
    return content_tag(:table, content, :class => "pagination")
  end

  def previous_page_link(page, locals)
    page_link(:prev, page, locals)
  end

  def next_page_link(page, locals)
    page_link(:next, page, locals)
  end

  def page_link(mode, page, locals)
    alt = {
      :prev => s_("rfw|image|alt|Previous page"),
      :next => s_("rfw|image|alt|Next page"),
    }
    if page
      imgs = {
        :prev => '<input type="image" class="iepngfix" src="/images/previous_page_found.png" width="20" height="20" alt="'+s_("rfw|image|alt|Previous page")+'" />',
        :next => '<input type="image" class="iepngfix" src="/images/next_page_found.png" width="20" height="20" alt="'+s_("rfw|image|alt|Next page")+'" />',
      }
      img = imgs[mode]
      if locals[:update]
        result = link_to_remote(img,
                                :update   => locals[:update],
                                :complete => visual_effect(:highlight, locals[:update]),
                                :url      => locals[:options].merge(:action => locals[:action],
                                                                    :page   => page.to_i,
                                                                    :per    => locals[:per]))
      else
        suffix = {
          :prev => '_prev',
          :next => '_next'
        }
        result = link_to_view_motion(img,
                                     locals[:view],
                                     nil,
                                     {"page" => page.to_i, "_" => locals[:suffix] + suffix[mode]},
                                     locals[:options].merge(:page => page.to_i),
                                     :class => locals[:html_class])
      end
    else
      imgs = {
        :prev => '<input type="image" class="iepngfix" src="/images/previous_page_not_found.png" width="20" height="20" alt="'+s_("rfw|image|alt|Previous page")+'" />',
        :next => '<input type="image" class="iepngfix" src="/images/next_page_not_found.png" width="20" height="20" alt="'+s_("rfw|image|alt|Next page")+'" />',
      }
      result = imgs[mode]
    end
    return result
  end

  def current_page_link(current_page, page, locals)
    if locals[:pages].has_page_number?(page)
      if current_page == page
        result = page
      elsif locals[:update]
        result = link_to_remote(page.to_s,
                                :update   => locals[:update],
                                :complete => visual_effect(:highlight, locals[:update]),
                                :url      => locals[:options].merge(:action => locals[:action],
                                                                    :page   => page,
                                                                    :per    => locals[:per]))
      else
        result = link_to_view_motion(page.to_s,
                                     locals[:view],
                                     nil,
                                     {"page" => page, "_" => locals[:suffix]},
                                     locals[:options].merge({ :page => page }),
                                     :class => locals[:html_class])
      end
    else
      result = "&nbsp;"
    end
    return result
  end

  def first_page_link(locals)
    edge_page_link(:first, locals)
  end

  def last_page_link(locals)
    edge_page_link(:last, locals)
  end

  def edge_page_link(mode, locals)
    alt = {
      :first => s_("rfw|image|alt|First page"),
      :last  => s_("rfw|image|alt|Last page")
    }
    file_name = {
      :first => "first_page.png",
      :last  => "last_page.png"
    }
    page = {
      :first => locals[:pages].first.to_i,
      :last  => locals[:pages].last.to_i,
    }
    img = image_tag(file_name[mode], :alt => alt[mode])
    if locals[:update]
      result = link_to_remote(img,
                              :update   => locals[:update],
                              :complete => visual_effect(:highlight, locals[:update]),
                              :url      => locals[:options].merge(:action => locals[:action],
                                                                  :page   => page[mode],
                                                                  :per    => locals[:per]))
    else
      suffix = {
        :first => '_first',
        :last  => '_last'
      }
      result = link_to_view_motion(img,
                                   locals[:view],
                                   nil,
                                   { "page" => page[mode], "_" => locals[:suffix] + suffix[mode]},
                                   locals[:options].merge(:page => page[mode]),
                                   :class => locals[:html_class])
    end
    return result
  end

  CHECK_BUGS = true

  # サブ画面へのリンク
  # リンクの html 要素の ID の形は "link_#{part of view}_#{motion}_#{join of motion_params 'key_value' with '_'}" となる。
  # * name - link_to に第1引数として渡される。
  # * view - "view_*" という形の html 要素の ID を表す文字列。この要素に Ajax によってリンク先の画面が表示される。
  # * motion - リンクの html 要素の ID を構成する。値が nil なら @fragment_hash から計算される。
  # * motion_params - リンクの html 要素の ID を構成する。
  #                   十分なエントリがなければ @fragment_hash から計算される。
  #                   値が nil なエントリは無視される。
  #                   "_" というキーでリンクの html 要素の ID に suffix を与えることができる。
  # * options - link_to に第2引数として渡される。
  #             ハッシュで指定された場合、十分なエントリがなければ @fragment_hash から計算される。
  # * html_options - link_to に第3引数として渡される。
  #                  ただし :id などいくつかのエントリは上書きされる。
  def link_to_view_motion(name, view, motion, motion_params, options, html_options = {})
    view ||= @current_view
    if /\Aview_([a-z0-9]+)(?:_[0-9]+)?\z/ =~ view
      view = $1
    end
    if CHECK_BUGS
      if @current_view.nil?
        raise "[bug] @current_view is nil"
      end
      if /[^a-z0-9]/ =~ view
        raise "[bug] invalid view #{view.inspect}"
      end
      if motion.nil? && "view_#{view}" != @current_view
        raise "[bug] missing motion"
      end
    end
    motion ||= @fragment_hash.motion(view)
    link_params = ["link", view, motion]
    motion_params = motion_params.with_indifferent_access
    if options.is_a?(Hash)
      link_to_options = options.with_indifferent_access
    end
    @fragment_hash[view].each do |key, value|
      if value
        motion_params[key] = value unless motion_params.key?(key)
        if link_to_options
          link_to_options[key] = value unless link_to_options.key?(key)
        end
      end
    end
    @fragment_hash.default_params(view).each do |key, default_value|
      if motion_params[key] == default_value
        motion_params.delete(key)
      end
      if link_to_options
        if link_to_options[key] == default_value
          link_to_options.delete(key)
        end
      end
    end
    suffix = motion_params.delete("_")
    motion_params.keys.sort.each do |key|
      if motion_params[key]
        link_params << key << motion_params[key]
      end
    end
    link_params << suffix
    html_options[:id] = link_params.compact.join("_")

    if html_options[:confirm] && html_options[:method] == :post
      html_options[:onclick] = "if (confirm('#{h html_options[:confirm]}')) { onAjaxLink(this); }; return false"
      html_options.delete(:confirm)
      html_options.delete(:method)
    elsif html_options[:onclick]
      html_options[:onclick] = "#{html_options[:onclick]}return onAjaxLink(this)"
    else
      html_options[:onclick] = "return onAjaxLink(this)"
    end
    link_to_options ||= options
    return link_to(name, link_to_options, html_options)
  end

  # ノードとして表示するとともに、タイトルを設定する。
  def set_title(title, hidden=request.xhr?)
    @title = title
    if hidden
      return "<h1 class='title' style='display: none'>#{h(title)}</h1>"
    else
      return "<h1 class='title'>#{h(title)}</h1>"
    end
  end

  # アプリケーションメニューのタイトルとして表示するとともに設定する。
  def application_menu_title_only(title, view_handle = true)
    @title = title
    return <<-HTML
<div class="application_menu_base">
<table class="application_menu">
  <tr class="application_menu_line1">
    <td>
      <span class="#{view_handle ? 'view_handle draggable ' : ''}title">#{image_tag("mark.png")}#{h title}</span>
    </td>
  </tr>
</table>
</div>
    HTML
  end

  # <em>fragment</em> を埋め込む
  def set_fragment(fragment)
    source, fragment = fragment
    return <<-HTML
<form>
<input type="hidden" class="fragment_setter" name="#{h source}" value="#{h fragment}">
</form>
    HTML
  end

  # Ajax に対応したフォームを構成する。
  def x_form_tag(url={}, options={}, *parameters_for_url, &block)
    view = @current_view
    url = url_for(url) if url.is_a?(Hash)
    if /\Aview_[a-z0-9]+\z/ !~ view
      raise "invalid view #{view.inspect}"
    end
    options[:url] ||= url
    options[:update] = view
    options[:form] = true
    options[:html] ||= {}
    options[:html][:id] ||= random_id
    form_id = options[:html][:id]

    function = remote_function(options)
    if view != "view_main"
      function.sub!(/Updater\('#{view}',/) do
        "Updater($('#{view}')||$('view_main'),"
      end
    end
    function.sub!(/asynchronous:true,/) do
      "app_view:'#{view}',#{$&}"
    end

    options[:html][:onsubmit] = (options[:html][:onsubmit] ? options[:html][:onsubmit] + "; " : "")
    if view != "view_main"
      options[:html][:onsubmit] << "if(!$('#{view}')&&!$('view_main')){return true};"
    else
      options[:html][:onsubmit] << "if(!$('#{view}')){return true};"
    end
    options[:html][:onsubmit] << "#{function};return false;"

    tags = form_tag(options[:html].delete(:action) || url_for(options[:url]),
                    options[:html],
                    *parameters_for_url,
                    &block)
    tags << javascript_tag("(function() {App.block_enter('#{form_id}');}).defer();")
    tags
  end

  # 状態遷移時の効果を加えた <tt>submit_tag</tt>。
  def submit_tag(value, options={})
    options = options.with_indifferent_access
    disable_with = options.delete("disable_with") || s_("rfw|submit_tag|please wait ...")
    onclick = options.delete("onclick")
    confirm = options.delete("confirm")
    if onclick
      onclick = "App.onSubmitTag(this,'#{disable_with}');#{onclick};return false;"
    else
      onclick = "return App.onSubmitTag(this,'#{disable_with}');"
    end
    if confirm
      onclick = "if (confirm('#{confirm}')){ #{onclick} };return false;"
    end
    options = {
      "class" => "button",
      "onclick" => onclick,
    }.merge(options)
    super(value, options)
  end

  # <em>params</em> に対応するエラーメッセージのノードを返す。
  def error_messages_for(*params)
    if @stale_object_error
      err_icon = image_tag("error.png")
      message = [ content_tag(:div, err_icon + s_("rfw|message|Attempted to update a stale object"), :class => "errorExplanation_head"),
                  content_tag(:div, s_("rfw|message|Someone has updated the object. Please go back to show the new version of it."), :class => "errorExplanation_detail")
                ]
      return content_tag(:div, message.join, :class => "errorExplanation_base")
    end
    options = params.last.is_a?(Hash) ? params.pop.symbolize_keys : {}
    objects = params.collect {|object_name| instance_variable_get("@#{object_name}") }.compact
    object_names = params.dup
    count   = objects.inject(0) {|sum, object| sum + object.errors.count }
    if count.zero?
      ''
    else
      return [
        '<div class="errorExplanation_base">',
        '<div class="errorExplanation_head">',
        image_tag("error.png"),
        ns_("rfw|message|%{num} error prohibited this data from being saved",
          "%{num} errors prohibited this data from being saved",
          count) % { :num => count },
        '</div>',
        '<div class="errorExplanation_detail">',
        ns_("rfw|message|There was a problem with the following field.",
          "There were problems with the following fields.",
          count),
        '</div>',
        '</div>',
      ].join("")
    end
  end

  # 選択部品へのリンクを構成する。
  def search_to_pick(singular, options = {})
    plural = singular.to_s.pluralize
    url = {:action => "search_#{plural}"}
    url = url.update(options)
    {
      :url => url,
      :update => "area_#{plural}",
      :complete => visual_effect(:highlight, "area_#{plural}"),
      :html => {:action => url_for(url)}
    }
  end

  # 選択部品へのリンク
  def open_to_pick(x, options = {})
    c = x.class.to_s
    singular = c.underscore
    plural = singular.pluralize
    dom_id = "search_#{plural}_#{x.id}"
    url = {:action => "open_#{singular}", :id => x.id}
    url = url.update(options)
    link_to_remote(tree_icon_open, {
                     :url => url,
                     :success => "$$('#search_#{plural} a').each(function(x){x.style.fontWeight='normal';});$('#{dom_id}').style.fontWeight='bold';",
                     :update => "area_#{plural}",
                   }, {
                     :id => dom_id,
                     :class => "dropdown"
                   })
  end

  def narrowed_to_pick(selected, several) #:nodoc: 
    if selected
      "<li>#{open_to_pick selected} #{h(selected.name)}</li>"
    else
      several.map {|s| "<li>#{open_to_pick s} #{h(s.name)}</li>"}.join
    end
  end

  def link_to_pick(x) #:nodoc:
    c = x.class.to_s.underscore
    link_to_remote h(x.name), :url => {:action => "pick", :id => x.id}, :update => "#{c}_picker"
  end

  def language_selector() #:nodoc:
    select_tag "lang", options_from_collection_for_select(Language.all, "code", "code", GetText.locale.to_s), :onchange => "this.form.onsubmit()"
  end

  # 固定のオプションの配列 <em>c</em> から構成される select box を返す。
  def constant_select_tag(c, name, x, k, blank=nil)
    options = c.map {|pair| [s_(pair.first), pair.last]}
    options.unshift([blank, ""]) if blank
    select_tag(name, options_for_select(options, x.attributes[k]))
  end

  # 左|中央|右にレイアウトされたボタン列を返す。
  def lcr_button_box(buttons=@lcr_buttons, width_class="w_lcr_button_box_full")
    return <<-HTML
    <table class="#{width_class}">
     <tr>
      <td class="lcr_button_box_left">#{buttons[:left] || '&nbsp;'}</td>
      <td class="lcr_button_box_center">#{buttons[:center] || '&nbsp;'}</td>
      <td class="lcr_button_box_right">#{buttons[:right] || '&nbsp;'}</td>
     </tr>
    </table>
    HTML
  end

  # text fields に入力された値を消去するボタンを返す。
  def button_to_erase_fields(*names)
    js = names.map {|name| "$(this.form).getInputs('text', '#{name}').invoke('clear');"}.join
    button_to_function(s_("rfw|button|Erase"), js, {:class => 'button'})
  end

  # 新規／編集／詳細画面でのレコードに関する情報（登録日・登録者・更新日・更新者）を表示
  def record_history_to_td(it)
    unless it.created_at.blank?
      month = h(s_("Date|month|#{it.created_at.strftime("%B")}"))
      created_date = h(s_("Item|Date|%2$s %3$s,%1$s") % [it.created_at.year, month, it.created_at.day])
    end
    unless it.updated_at.blank?
      month = h(s_("Date|month|#{it.updated_at.strftime("%B")}"))
      updated_date = h(s_("Item|Date|%2$s %3$s,%1$s") % [it.updated_at.year, month, it.updated_at.day])
    end
    created_name = it.created_by.person.name unless it.created_by.blank?
    updated_name = it.updated_by.person.name unless it.updated_by.blank?

    rec_hist  = h(s_("ApplicationModel|Create at:")) + (created_date || '') + ("&nbsp;"*2)
    rec_hist << h(s_("ApplicationModel|Create by:")) + (created_name || '') + ("&nbsp;"*3)
    rec_hist << h(s_("ApplicationModel|Update at:")) + (updated_date || '') + ("&nbsp;"*2)
    rec_hist << h(s_("ApplicationModel|Update by:")) + (updated_name || '')

    return content_tag(:td, rec_hist, :colspan=>"2", :class => "record_history")
  end

  def rfw_link_to_open(name, options, seed, stub = 'ancestor', num = 0, html_options = {})
    id = "rfw_link_to_open_#{random_id}"
    url = url_for(options)
    html_options.update(:id => id)
    link = link_to_function name, "", html_options
    link += javascript_tag(<<-JS)
    if ($('#{id}')) {
      $('#{id}').observe('click', Rfw.open.bindAsEventListener($('#{id}'), '#{url}', '#{seed}', '#{stub}', #{num}));
    }
    JS
    link
  end

  def rfw_link_to_close(name, options = nil, seed = nil, stub = 'ancestor', num = 0, html_options = {:class => "button"})
    id = "rfw_link_to_close_#{random_id}"
    html_options.update(:id => id)
    link = link_to_function name, "", html_options
    if block_given?
      link += yield(id)
    end
    if options
      url = url_for(options)
      if seed
        link += javascript_tag(<<-JS)
        if ($('#{id}')) {
          $('#{id}').observe('click', Rfw.open.bindAsEventListener($('#{id}'), '#{url}', '#{seed}', '#{stub}', #{num}));
        }
        JS
      else
        link += javascript_tag(<<-JS)
        if ($('#{id}')) {
          $('#{id}').observe('click', Rfw.open.bindAsEventListener($('#{id}'), '#{url}'));
        }
        JS
      end
    end
    link += javascript_tag(<<-JS)
    if ($('#{id}')) {
      $('#{id}').observe('click', Rfw.close.bindAsEventListener($('#{id}')));
    }
    JS
    link
  end

  def rfw_event_observe(id, event_name, url_options = nil, seed = nil, stub = 'ancestor', num = 0, mark = nil)
    result = ''
    if url_options
      url = url_for(url_options)
      if seed
        if mark
          result << javascript_tag(<<-JS)
          if ($('#{id}')) {
            $('#{id}').observe('#{event_name}', Rfw.submit.bindAsEventListener($('#{id}'), '#{url}', '#{seed}', '#{stub}', #{num}, true));
          }
          JS
        else
          result << javascript_tag(<<-JS)
          if ($('#{id}')) {
            $('#{id}').observe('#{event_name}', Rfw.submit.bindAsEventListener($('#{id}'), '#{url}', '#{seed}', '#{stub}', #{num}));
          }
          JS
        end
      else
        result << javascript_tag(<<-JS)
        if ($('#{id}')) {
          $('#{id}').observe('#{event_name}', Rfw.submit.bindAsEventListener($('#{id}'), '#{url}'));
        }
        JS
      end
    else
      result << javascript_tag(<<-JS)
      if ($('#{id}')) {
        $('#{id}').observe('#{event_name}', Rfw.submit.bindAsEventListener($('#{id}')));
      }
      JS
    end
    result
  end

  def rfw_button_to_submit(name, options = nil, seed = nil, stub = 'ancestor', num = 0, mark = nil, html_options = {:class => "button"}, picker = ::Rfw::Picker.new)
    id = "rfw_button_to_submit_#{random_id}"
    html_options.update(:id => id)
    button = button_to_function name, "", html_options
    picker.html_tag << button
    if block_given?
      picker.javascript << yield(id)
    else
      picker.html_tag << javascript_tag(<<-JS)
      if ($('#{id}')) {
        $('#{id}').form.onsubmit = function() {return false;};
      }
      JS
    end
    # TODO rfw_event_observe を使用する
    # button << rfw_event_observe(id, 'click', options, seed, stub, num, mark)
    # return button
    if options
      url = url_for(options)
      if seed
        if mark
          picker.javascript << "Rfw.submit.bindAsEventListener($('#{id}'), '#{url}', '#{seed}', '#{stub}', #{num}, true)"
        else
          picker.javascript << "Rfw.submit.bindAsEventListener($('#{id}'), '#{url}', '#{seed}', '#{stub}', #{num})"
        end
      else
        picker.javascript << "Rfw.submit.bindAsEventListener($('#{id}'), '#{url}')"
      end
    else
      picker.javascript << "Rfw.submit.bindAsEventListener($('#{id}'))"
    end
    picker.to_html do |html_tag, javascript|
      html_tag.join + javascript_tag(<<-JS)
      if ($('#{id}')) {
        $('#{id}').observe('click', Rfw.combine(#{javascript.join(',')}));
      }
      JS
    end
  end

  def rfw_button_to_pick(name, options, html_options = {:class => "button"})
    id = "rfw_button_to_pick_#{random_id}"
    url = url_for(options)
    html_options.update(:id => id)
    button = button_to_function name, "", html_options
    button << javascript_tag(<<-JS)
    if ($('#{id}')) {
      $('#{id}').form.onsubmit = function() {return false;};
      $('#{id}').observe('click', Rfw.pick.bindAsEventListener($('#{id}'), '#{url}'));
    }
    JS
    button
  end

  def rfw_picker(inputs, options, pick_label = s_("rfw|button|Pick"), seed = '*', stub = 'ancestor', num = 1, mark = true)
    tags = inputs.map do |x|
      if x.is_a?(String)
        x
      else
        case x[:type].to_s
        when "text"
          html_options = x[:html] || {}
          html_options[:id] = nil unless html_options.key?(:id)
          text_field_tag(x[:name], x[:value], html_options)
        when "hidden"
          html_options = x[:html] || {}
          html_options[:id] = nil unless html_options.key?(:id)
          hidden_field_tag(x[:name], x[:value], html_options)
        else
          nil
        end
      end
    end.compact
    result = ''
    picker = ::Rfw::Picker.new(tags)
    button_id = nil
    rfw_button_to_submit(pick_label, options, seed, stub, num, mark, { :class => 'button' }, picker) do |id|
      button_id = id
      extra = block_given? ? yield(id) : ""
      if /\A</ =~ extra
        picker.html_tag << extra
      else
        picker.javascript << extra
      end
      spec = {}
      inputs.each do |x|
        spec[x[:key]] = x[:name] unless x[:key].blank?
      end
      picker.javascript << <<-JS
        Rfw.embedHiddenParameters.bindAsEventListener($('#{id}'), #{spec.to_json})
      JS
      nil
    end
    result << picker.to_html do |html_tags, javascript|
      html_tags.join + javascript_tag(<<-JS)
      if ($('#{button_id}')) {
        $('#{button_id}').observe('click', Rfw.combine(#{javascript.join(',')}));
      }
      JS
    end
    result
  end

  def rfw_link_to_pick(name, values, html_options = {})
    id = "rfw_link_to_pick_#{random_id}"
    html_options.update(:id => id)
    confirm = html_options.delete(:confirm)
    link = link_to_function name, "", html_options
    spec = {}
    values.each do |k, v|
      spec[params[:rfw][k]] = v unless params[:rfw][k].blank?
    end
    if confirm.blank?
      function = "Rfw.pick.bindAsEventListener($('#{id}'), #{spec.to_json})"
    else
      function = "Rfw.pick.bindAsEventListener($('#{id}'), #{spec.to_json}, '#{confirm}')"
    end
    link << javascript_tag(<<-JS)
      (function(){
        if ($('#{id}')) {
          $('#{id}').observe('click', #{function});
        }
      }).defer();
    JS
    link
  end

  def rfw_link_to_cancel_picker(name, html_options = {:class => "button"})
    id = "rfw_link_to_cancel_picker_#{random_id}"
    html_options.update(:id => id)
    link = link_to_function name, "", html_options
    link += javascript_tag(<<-JS);
    if ($('#{id}')) {
      $('#{id}').observe('click', Rfw.cancelPicker.bindAsEventListener($('#{id}')));
    }
    JS
    link
  end

  def rfw_name_for(object_name, method)
    "#{object_name}[#{method}]"
  end

  def rfw_value_for(object_name, method)
    v = instance_variable_get("@#{object_name}")
    v.__send__(method)
  end

  private

  def with_label(label, options, object_name, *rest, &block)
    o = options.dup
    rest = rest.join("_")
    if object_name = object_name.to_s.sub(/\[\]$/, "")
      o[:id] ||= "#{object_name}_#{instance_variable_get("@#{object_name}").id}_#{rest}"
    else
      o[:id] ||= "#{object_name}_#{rest}"
    end
    return yield(o) + content_tag(:label, label, {:for => o[:id]})
  end

  def extra_tags_for_form(html_options)
    super + submit_tag("", :name => "dummy_button", :style => "display:none")
  end
end
