# -*- coding: utf-8 -*-
module Rfw
  module YearMonth

    # 年月の書式を扱う。
    module Format

      # 年月の表現として妥当であればその朔日を返す。
      # さもなくば false を返す。
      def self.valid_year_month(x)
        x = x.to_s
        if /\A(\d{4})(\d{2})\z/ =~ x || /\A(\d{4})[\/-](\d{1,2})\z/ =~ x
          year = $1.to_i
          return false unless (1..9999).include?(year)
          begin
            return ::Date.new(year, $2.to_i)
          rescue ArgumentError
            return false
          end
        end
        return false
      end

    end

    # 項目の年月に関する振る舞いを扱う。
    # 以下のメッセージに respond_to? であることが前提:
    # * validates_year_month?
    module Item

      def year_month?
        validates_year_month?
      end

      def fit_into_internal_year_month_format(x)
        with_year_month_format(x) do |date|
          return x unless respond_to?(:model_class)
          return x unless model_class.respond_to?(:columns_hash)
          column = model_class.columns_hash[column_name]
          return x if column.nil?
          format = (column.limit.nil? || App::INTERNAL_YEAR_MONTH_FORMAT.length <= column.limit) ? App::INTERNAL_YEAR_MONTH_FORMAT : "yyyymm"
          CustomFormatTranslator.format_date(date, format)
        end
      end

      def fit_into_external_year_month_format(x)
        with_year_month_format(x) do |date|
          CustomFormatTranslator.format_date(date, App::EXTERNAL_YEAR_MONTH_FORMAT)
        end
      end

      def fit_into_internal_year_month_sub(x)
        case App::INTERNAL_YEAR_MONTH_FORMAT
        when /\//
          x.to_s.gsub("-", "/")
        when /-/
          x.to_s.gsub("/", "-")
        else
          x.to_s.gsub(/[\/\-]/, "")
        end
      end

      private

      def with_year_month_format(x, &block)
        return x if x.blank?
        case x
        when ::Date
          date = x
        else
          date = Format.valid_year_month(x)
          return x unless date
        end
        block.call(date)
      end

    end
  end
end
