# -*- coding: utf-8 -*-
require 'test_helper'

class MenuTest < ActiveSupport::TestCase
  fixtures :users, :people, :permissions, :grant_ons, :products, :menus

  def setup
    CacheEachRequest.clear
  end

  def teardown
    CacheEachRequest.clear
  end

  # Menu#root がルートになるメニューを返すことをテストする。
  def test_root
    assert_equal menus(:root), Menu.root
  end

  # Menu#visible? が対応する product の権限で決まる。
  def test_visible?
    User.current = users(:demo)
    assert menus(:demo).visible?
  end

  # Menu#root が(ドメイン ID が3の場合に)ルートになるメニューを返すことをテストする。
  def test_root__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_equal menus(:root_for_domain_3), Menu.root
  end

  # Menu#level がメニューの深さを返すことをテストする。
  def test_level
    assert_equal 0, menus(:root).level
    assert_equal 1, menus(:top_1).level
    assert_equal 4, menus(:about).level
  end

  # Menu#level が(ドメイン ID が3の場合に)メニューの深さを返すことをテストする。
  def test_level__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_equal 0, menus(:root_for_domain_3).level
    assert_equal 1, menus(:menu_for_domain_3).level
  end

  # Menu#ancestors が祖先のメニューを返すことをテストする。
  def test_ancestors
    assert_array_as_nested_set(:ancestors,
                               :root => [],
                               :top_2 => [:root],
                               :about => [:root, :top_1, :sub_1_1, :sub_1_1_1])
  end

  # Menu#ancestors が(ドメイン ID が3の場合に)祖先のメニューを返すことをテストする。
  def test_ancestors__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:ancestors,
                               :root_for_domain_3 => [],
                               :menu_for_domain_3 => [:root_for_domain_3])
  end

  # Menu#self_and_ancestors が自分自身および祖先のメニューを返すことをテストする。
  def test_self_and_ancestors
    assert_array_as_nested_set(:self_and_ancestors,
                               :root => [:root],
                               :top_2 => [:root, :top_2],
                               :about => [:root, :top_1, :sub_1_1, :sub_1_1_1, :about])
  end

  # Menu#self_and_ancestors が(ドメイン ID が3の場合に)自分自身および祖先のメニューを返すことをテストする。
  def test_self_and_ancestors__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:self_and_ancestors,
                               :root_for_domain_3 => [:root_for_domain_3],
                               :menu_for_domain_3 => [:root_for_domain_3, :menu_for_domain_3])
  end

  # Menu#siblings が兄弟関係のメニューを返すことをテストする。
  def test_siblings
    assert_array_as_nested_set(:siblings,
                               :root => [],
                               :top_1 => [:top_2, :top_3, :portal, :grant_on_for_dev, :product_detailed_0, :workshop, :demo, :demo2, :demo3, :grant_menu, :grant_item],
                               :sub_1_1 => [:sub_1_2],
                               :about => [])
  end

  # Menu#siblings が(ドメイン ID が3の場合に)兄弟関係のメニューを返すことをテストする。
  def test_siblings__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:siblings,
                               :root_for_domain_3 => [],
                               :menu_for_domain_3 => [])
  end

  # Menu#self_and_siblings が自分自身および兄弟関係のメニューを返すことをテストする。
  def test_self_and_siblings
    assert_array_as_nested_set(:self_and_siblings,
                               :root => [:root],
                               :top_1 => [:top_1, :top_2, :top_3, :portal, :grant_on_for_dev, :product_detailed_0, :workshop, :demo, :demo2, :demo3, :grant_menu, :grant_item],
                               :sub_1_1 => [:sub_1_1, :sub_1_2],
                               :about => [:about])
  end

  # Menu#self_and_siblings が(ドメイン ID が3の場合に)自分自身および兄弟関係のメニューを返すことをテストする。
  def test_self_and_siblings__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:self_and_siblings,
                               :root_for_domain_3 => [:root_for_domain_3],
                               :menu_for_domain_3 => [:menu_for_domain_3])
  end

  # Menu#parent が親メニューを返すことをテストする。
  def test_parent
    assert_nil menus(:root).parent
    assert_equal menus(:root), menus(:top_1).parent
    assert_equal menus(:sub_1_1_1), menus(:about).parent
  end

  # Menu#parent が(ドメイン ID が3の場合に)親メニューを返すことをテストする。
  def test_parent__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_equal menus(:root_for_domain_3), menus(:menu_for_domain_3).parent
  end

  # Menu#children_count が子メニューの個数を返すことをテストする。
  def test_children_count
    assert_equal 12, menus(:root).children_count
    assert_equal 2, menus(:top_1).children_count
    assert_equal 0, menus(:about).children_count
  end

  # Menu#children_count が(ドメイン ID が3の場合に)子メニューの個数を返すことをテストする。
  def test_children_count__for_domain3
    User.current = User.find_by_login("geeko")
    assert_equal 1, menus(:root_for_domain_3).children_count
    assert_equal 0, menus(:menu_for_domain_3).children_count
  end

  # Menu#children が子メニューを返すことをテストする。
  def test_children
    assert_array_as_nested_set(:children,
                               :root => [:top_1, :top_2, :top_3, :portal, :grant_on_for_dev, :product_detailed_0, :workshop, :demo, :demo2, :demo3, :grant_menu, :grant_item],
                               :top_1 => [:sub_1_1, :sub_1_2],
                               :about => [])
  end

  # Menu#children が(ドメイン ID が3の場合に)子メニューを返すことをテストする。
  def test_children__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:children,
                               :root_for_domain_3 => [:menu_for_domain_3],
                               :menu_for_domain_3 => [])
  end

  # Menu#all_children が子メニューを返すことをテストする。
  def test_all_children
    assert_array_as_nested_set(:all_children,
                               :top_1 => [:sub_1_1, :sub_1_1_1, :about, :sub_1_2],
                               :about => [])
  end

  # Menu#all_children が(ドメイン ID が3の場合に)子メニューを返すことをテストする。
  def test_all_children__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:all_children,
                               :root_for_domain_3 => [:menu_for_domain_3])
  end

  # Menu#fullset が自分自身および子孫のメニューを返すことをテストする。
  def test_full_set
    assert_array_as_nested_set(:full_set,
                               :top_1 => [:top_1, :sub_1_1, :sub_1_1_1, :about, :sub_1_2],
                               :about => [:about])
  end

  # Menu#fullset が(ドメイン ID が3の場合に)自分自身および子孫のメニューを返すことをテストする。
  def test_full_set__for_domain_3
    User.current = User.find_by_login("geeko")
    assert_array_as_nested_set(:full_set,
                               :root_for_domain_3 => [:root_for_domain_3, :menu_for_domain_3])
  end

  # Menu#url_options の動作をテストする。
  def test_url_options_without_controller
    top_1 = menus(:top_1)
    expected = {:controller => "menu", :action => "mock", :id => "2"}
    assert_equal expected, top_1.url_options
    assert top_1.with_ajax?
  end

  # Menu#url_options の動作をテストする。
  def test_url_options_with_controller
    portal = menus(:portal)
    expected = {:controller => "portal", :action => "index"}
    assert_equal expected, portal.url_options
    assert portal.with_ajax?
  end

  # Menu#url_options の動作をテストする。
  def test_url_options_without_ajax
    grant_item = menus(:grant_item)
    expected = {:controller => "product", :action => "list", :product_id => "24"}
    assert_equal expected, grant_item.url_options
    assert !grant_item.with_ajax?
  end

  private

  def assert_array_as_nested_set(x, h)
    h.each_pair do |k, v|
      expected = v.map {|s| menus(s).id}
      value    = menus(k).__send__(x).map {|m| m.id}
      assert_equal expected, value
    end
  end
end
