# -*- coding: utf-8 -*-
# app/models/user.rb のクラス User をテストする。

require 'test_helper'

class UserTest < ActiveSupport::TestCase
  fixtures(:domains, :people, :users, :config_passwords, :grant_ons, :permissions,
           :companies, :company_members, :organizations, :organization_members, :groups, :group_members)
  self.use_transactional_fixtures = false

  def setup
    CacheEachRequest.clear
    Clock.time = Clock.now
  end

  def teardown
    CacheEachRequest.clear
    Clock.reset
  end

  # User#authenticate で認証状態を判定できることをテストする。
  def test_authenticate
    assert_equal users(:tesla), User.authenticate(users(:tesla).login, "atest")
    assert_nil User.authenticate("nontesla", "atest")
    assert_nil User.authenticate(users(:tesla).login, "wrong password")
  end

  # User#authenticate でログイン回数が調節されることをテストする。
  def test_authenticate__failed_login_count
    user = users(:tesla)
    assert_equal 0, user.failed_login_count
    assert_nil User.authenticate(user.login, "wrong password")
    user.reload
    assert_equal 0, user.failed_login_count
    (1..5).each do |n|
      assert_nil User.authenticate(user.login, "wrong password", true)
      user.reload
      assert_equal n, user.failed_login_count
    end
    assert_nil User.authenticate(user.login, "wrong password")
    user.reload
    assert_equal 5, user.failed_login_count
    assert_equal user, User.authenticate(user.login, "atest")
    user.reload
    assert_equal 5, user.failed_login_count
    assert_equal user, User.authenticate(user.login, "atest", true)
    user.reload
    assert_equal 0, user.failed_login_count
    assert_equal user, User.authenticate(user.login, "atest", true)
    user.reload
    assert_equal 0, user.failed_login_count
  end

  # User#authenticate_by_token でトークンによる認証が行われることをテストする。
  def test_authenticate_by_token
    user = users(:unverified_user)
    assert_equal user, User.authenticate_by_token(user.id, user.security_token)
  end

  # User#authenticate_by_token でトークンによる認証が有効期間が満了した場合に
  # 失敗することをテストする。
  def test_authenticate_by_token__fails_if_expired
    user = users(:unverified_user)
    Clock.time = Clock.now + 2.days
    assert_nil User.authenticate_by_token(user.id, user.security_token)
  end

  # User#authenticate_by_token でトークンによる認証が間違ったトークンによって
  # 失敗することをテストする。
  def test_authenticate_by_token__fails_if_bad_token
    user = users(:unverified_user)
    assert_nil User.authenticate_by_token(user.id, 'bad_token')
  end

  # User#authenticate_by_token でトークンによる認証が間違った ID によって
  # 失敗することをテストする。
  def test_authenticate_by_token__fails_if_bad_id
    user = users(:unverified_user)
    assert_nil User.authenticate_by_token(-1, user.security_token)
  end

  # User#change_password によるパスワード変更をテストする。
  def test_change_password
    user = users(:long_user)
    user.change_password("a new password")
    user.save
    assert_equal user, User.authenticate(user.login, "a new password")
    assert_nil User.authenticate(user.login, "alongtest")
  end

  # User#generate_security_token でトークンが生成されることをテストする。
  def test_generate_security_token
    user = User.new :domain_id => 1, :person_id => 0, :login => 'user', :salt => 'salt', :salted_password => 'tlas'
    user.save
    token = user.generate_security_token
    assert_not_nil token
    user.reload
    assert_equal token, user.security_token
    assert_equal (Clock.now + user.token_lifetime).to_i, user.token_expiry.to_i
  end

  # User#generate_security_token でトークンを可能ならば再利用することをテストする。
  def test_generate_security_token__reuses_token_when_not_stale
    user = users(:unverified_user)
    Clock.time = Clock.now + user.token_lifetime/2 - 60
    assert_equal user.security_token, user.generate_security_token
  end

  # User#generate_security_token でトークンが古くなったなら新たに生成することをテストする。
  def test_generate_security_token__generates_new_token_when_getting_stale
    user = users(:unverified_user)
    Clock.time = Clock.now + user.token_lifetime/2
    assert_not_equal user.security_token, user.generate_security_token
  end

  # User#change_password で許可されていないパスワードを受け付けないことをテストする。
  def test_change_password__disallowed_passwords
    u = User.new
    u.domain_id = 1
    u.person_id = 0
    u.login = "test_user"

    u.change_password("tiny")
    assert !u.save
    assert u.errors.invalid?('password')

    u.change_password("hugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehugehuge")
    assert !u.save
    assert u.errors.invalid?('password')

    u.change_password("")
    assert !u.save
    assert u.errors.invalid?('password')

    u.change_password("a_s3cure_p4ssword")
    assert u.save
    assert u.errors.empty?
  end

  # User#login に適切なログイン名が格納されているか検証する動作をテストする。
  def test_validates_login
    u = User.new
    u.domain_id = 1
    u.person_id = 0
    u.change_password("teslas_secure_password")

    u.login = "x"
    assert !u.save
    assert u.errors.invalid?(:login)

    u.login = "hugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahugeteslahug"
    assert !u.save
    assert u.errors.invalid?(:login)

    u.login = ""
    assert !u.save
    assert u.errors.invalid?(:login)

    u.login = "oktesla"
    assert u.save
    assert u.errors.empty?
  end

  # User の新規作成をテストする。
  def test_create
    u = User.new
    u.domain_id = 1
    u.person_id = 0
    u.login = "nonexisting_user"
    u.change_password("password")
    assert u.save
  end

  # User のログイン名が重複する場合には登録できないことをテストする。
  def test_create__validates_unique_login
    u = User.new
    u.domain_id = 1
    u.person_id = 0
    u.login = users(:tesla).login
    u.change_password("password")
    assert !u.save
  end

  # User のログイン名が重複する場合には登録できないことをテストする。
  def test_password_expire?
    u = User.find(1000001)
    assert !u.password_expire?
  end

  # User#locktou? が動作することをテストする。
  def test_lockout?
    u = User.find(1000001)
    assert !u.lockout?
  end

  # User#initialize_permissions によって permissions が作成されることをテストする。
  def test_initialize_permissions
    assert_difference('Permission.count', 17) do
      users(:tesla).instance_eval do
        initialize_permissions
      end
    end
  end
end
