# -*- coding: utf-8 -*-
module ProductHelper
  include NarrowingHelper::Helper

  # validation のエラーが存在する場合にエラーメッセージの窓を返す。
  def error_window
    if @product.is_a?(ProductDetailed)
      error_messages_for :it, *(symbols_for(@details.map(&:id) - @old_details, "details") + symbols_for(@new_details, "new_details"))
    else
      error_messages_for :it
    end
  end

  # 新規作成画面へのボタンを返す。
  def td_button_new
    if @display.button_new? && @product.modifiable?
      b = link_to_view_motion(h(s_("rfw|button|New")),
                              @sub_view, "create",
                              {"_" => "&suffix;"},
                              {:action => "new"},
                              :class => "button reloadable_link")
    else
      b = "&nbsp;"
    end
    content_tag(:td, b, :class => "button_new")
  end

  # (指定されていれば)CSV出力画面へのリンクを返す。
  def td_to_csv
    return "" unless @display.csv?
    link = link_to_view_motion(h(s_("rfw|link_to|CSV")),
                               "view_output",
                               "output",
                               {:controller => "output", :action => "edit", :id => @display.id},
                               {:controller => "output", :action => "edit", :id => @display.id},
                               {:id => "output", :class => "button"})
    content_tag(:td, link, :class => "right")
  end

  # (指定されていれば)絞り込みの窓を返す。
  def td_to_narrow
    return "" unless @display.narrowing?
    inputs = @display.narrowing_value_keys.map do |key|
      {:type => "hidden", :name => "narrowing_value_#{key}", :value => session[key], :key => key}
    end
    inputs.unshift({:type => "hidden", :name => "display_id", :value => @display.id, :key => "display_id"})
    inputs.unshift({
                     :type => "hidden",
                     :name => "display_narrowing_id",
                     :value => @display.active_display_narrowing(session).id,
                     :key => "display_narrowing_id"
                   })
    inputs.push({
                  :type => "text",
                  :name => "narrowing_text",
                  :value => narrowing_text,
                  :key => "narrowing_text",
                  :html => {:readonly => true, :size => "45"},
                })
    title = s_("rfw|submit_tag|Pick #{@display.active_display_narrowing(session).name}")
    picker = rfw_picker(inputs, {:controller => @display.narrowing_controller_path(session), :action => @display.narrowing_action_name(session)}, title) do |id|
      "function() {new Form.Element.Observer($('#{id}').previous(), 0.3, function(element, value) {element.form.onsubmit();});}"
    end
    content_tag(:td, picker)
  end

  # 権限で許可されていないカラムの部品を返す。
  def not_permitted_column
      filtered_message = s_("rfw|not_permitted_column|[FILTERED]")
      return MergeableTag.new(:td, filtered_message, :class => "not_permitted_column")
  end

  include ItemTableLayout

  # 詳細画面および編集画面で詳細情報を表示するための table を返す。
  def with_table(&block)
    table = to_table_layout(@items) do |item|
      if item.is_a?(Item)
        yield item
      else
        nil
      end
    end

    dummy_tag  = '<th class="dummy w_label"></th>'
    dummy_tag += '<th class="dummy w_span1"></th>'
    dummy_tag += '<th class="dummy w_label"></th>'
    dummy_tag += '<th class="dummy w_span1"></th>'
    rows = [ content_tag(:tr, dummy_tag, :class=>"dummy") ]

    rows << table.map do |row|
      if row[0].nil?
        content_tag(:tr, content_tag(:td, "&nbsp;", :class => "blank", :colspan => 4))
      else

        cols = row.map do |col|
          if col
            th = MergeableTag.new(:th, "")
            if row[1].nil?
              td = MergeableTag.new(:td, "", :colspan => 3)
            else
              td = MergeableTag.new(:td, "")
            end
            col.each do |c|
              th.merge(c[0])
              td.merge(c[1])
            end
            [th.to_s, td.to_s].join("")
          else
            [
              content_tag(:th, "", :class => "blank"),
              content_tag(:td, "", :class => "blank", :colspan => 3),
            ].join("")
          end
        end
        content_tag(:tr, cols.join(""))
      end
    end
    return content_tag(:table, rows.join(""), :class => "header w_detail")
  end

  # 項目の一覧を含む table を返す。
  def items_to_table
    with_table do |item|
      [
        MergeableTag.new(:th, h(item.human_name)),
        item_to_td(item, @it)
      ]
    end
  end

  # 項目の一覧に対応する input の table を返す。
  def items_to_table_input
    table = with_table do |item|
      options = {}
      options[:class] = "w_detail"
      options[:class] += " required" if item.required?
      label = content_tag(:label, h(item.human_name), :for => "it_#{item.column_name}")
      [
        MergeableTag.new(:th, label, options),
        item_to_td_input(item, :it)
      ]
    end
    table + hidden_field(:it, "lock_version")
  end

  # 項目 <em>i</em> からヘッダを返す。
  def item_to_th(i)
    options = {}
    options[:style] = i.style if i.style
    options[:class] = "required" if i.required?
    MergeableTag.new(:th, h(i.human_name), options)
  end

  # 項目から td を返す。
  def item_to_td(i, it, &block)
    return not_permitted_column unless i.readable?
    begin
      data = i.to_data(it, &block)
      if data.is_a? Proc
        data = data.call(binding())
      end
    rescue Item::Error => e
      logger.error("ERROR: Item::Error: #{e.inspect}")
      return MergeableTag.new(:td, h(e.to_s), :class => "formError")
    rescue NoMethodError => e
      logger.error("ERROR: column not found: #{@model_class}\##{i.column_name} #{e.inspect}\n#{e.backtrace.join("\n")}")
      message = s_("rfw|formError|column not found")
      return MergeableTag.new(:td, h(message), :class => "formError")
    end

    # input_type is not available in display_to_show and display_to_list
=begin
    case i.input_type
    when "text"
    when "textarea"
    when "radio", "select"
      data = data.to_s
      i.input_parameter.split(/,/).map do |pair|
        value, text = pair.split(/:/, 2)
        if value == data
          data = text
          break
        end
      end
    when "checkbox"
      data = i.input_parameter.to_s if data
    else
      logger.warn("WARN: invalid input type: #{i.input_type} (#{@model_class}\##{i.column_name})")
    end
=end

    case i.decorator
    when "front"
      content = h("#{i.decorator_parameter}#{data}")
    when "back"
      content = h("#{data}#{i.decorator_parameter}")
    when "currency"
      content = number_to_currency(data, :unit => "", :precision => 0)
    when "size" # for test
      content = h(number_to_human_size(data))
    when "nl2br"
      content = h(data).gsub(/\r?\n/) { "<br />" }
    when "raw"
      content = data
    else
      content = h(data)
    end

    m, x, label = i.link(it)
    case m
    when :link_to
      if label.blank?
        content = link_to(content, x, i.link_options)
      else
        content = link_to(label, x, i.link_options)
      end
    when :mail_to
      if label.blank?
        content = mail_to(x, content)
      else
        content = mail_to(x, label)
      end
    end

    options = {
      :align => i.align,
      :class => "item",
    }
    options[:style] = i.style if i.style
    content = content_tag(:div, content)
    return MergeableTag.new(:td, content, options)
  end

  # 項目から td にラップされた input を返す。
  def item_to_td_input(i, object_name)
    it = instance_variable_get("@#{object_name}")
    return item_to_td(i, it) {object_name} if i.is_a?(ItemPseudo)
    begin
      data = it.__send__(i.column_name)
    rescue NoMethodError
      logger.error("ERROR: column not found: #{@model_class}\##{i.column_name}")
      message = s_("rfw|formError|column not found")
      return MergeableTag.new(:td, h(message), :class => "formError")
    end

    field_options = {}
    field_options[:readonly] = "readonly" unless i.writable?
    field_options[:readonly] = "readonly" if i.write_protected?
    text_options = {
      :size  => i.size_for_input_field,
      :style => i.style_for_input_field,
    }
    text_options.update(field_options)
    case i.input_type
    when "picker"
      e = picker_tags(i, object_name, it)
    when "text"
      e = field_with_errors(i, object_name) do
        val = if i.date?
                fit_into_external_date_format(data)
              elsif i.year_month?
                fit_into_external_year_month_format(data)
              elsif it.respond_to?("#{i.column_name}_before_type_cast")
                it.__send__("#{i.column_name}_before_type_cast")
              else
                data
              end
        text_field_tag(rfw_name_for(object_name, i.column_name), val, text_options)
      end
    when "textarea"
      e = text_area object_name, i.column_name, text_options
    when "radio"
      choices = i.split_into_input_options(it, session) do |name, value|
        field_options[:id] = "#{object_name}_#{i.column_name}_#{value}_#{random_id}"
        radio_button_tag_with_label(rfw_name_for(object_name, i.column_name), value, h(name), i.type_cast(value) == data, field_options)
      end
      e = field_with_errors(i, object_name) {choices.join}
    when "checkbox"
      e = check_box(object_name, i.column_name, field_options) + i.input_parameter.to_s
    when "select"
      choices = i.split_into_input_options(it, session) {|name, value| [h(name), value]}
      choices.unshift([h(s_("rfw|InputOption|(blank)")), ""]) unless i.required?
      select_options = choices.map do |name, value|
        if i.type_cast(value) == data
          "<option value=\"#{value}\" selected=\"selected\">#{name}</option>"
        else
          "<option value=\"#{value}\">#{name}</option>"
        end
      end
      e = field_with_errors(i, object_name) do
        select_tag(rfw_name_for(object_name, i.column_name), select_options, field_options)
      end
    else
      logger.error("ERROR: invalid input type: #{i.input_type} (#{@model_class}\##{i.column_name})")
      message = s_("rfw|formError|invalid input type")
      return MergeableTag.new(:td, h(message), :class => "formError")
    end

    a = {:class => "input_item"}
    a[:style] = i.style if i.style
    return MergeableTag.new(:td, e, a)
  end

  #
  def detail_to_tr(d, script=nil, options={}, &block)
    tds = @detail_items.map {|i| yield(i, d)}
    if script
      tds.unshift(content_tag(:td, '<img src="/images/drag_handle.png" width=22 height=22 class="iepngfix" />', :class => "handle"))
      if script
        img = '<img src="/images/delete.png" width=16 height=16 class="iepngfix" alt="' + h(s_("rfw|button|Destroy")) + '" />'
        link = link_to_function(img, script)
        tds.push(content_tag(:td, link))
      end
    end
    options.update({:class => cycle("even", "odd", :name => "details_to_table")})
    content_tag(:tr, tds, options)
  end

  #
  def with_details(seed = random_id(), tail=nil, &block)
    col_width_dragg = 24                # column width dragg_handle icon
    col_width_delete = 18               # column width delete icon

    ths = @detail_items.map {|i| item_to_th(i)}
    ths.unshift(content_tag(:th, "&nbsp;", :width => col_width_dragg)) unless tail.blank?
    ths.push(content_tag(:th, "&nbsp;", :width => col_width_delete)) if tail
    head = content_tag(:thead, content_tag(:tr, ths))
    foot = content_tag(:tfoot, "")
    body = content_tag(:tbody, yield, {:id => "details_table_body_#{seed}"})

    table_width = 0
    @detail_items.each do |i|
      if i.layout > 0 && !(i.width.blank?)
        table_width = table_width + i.width.to_i
      end
    end

    if table_width > 0 then
      table_width = table_width + col_width_dragg + col_width_delete
      table = content_tag(:table, [head, foot, body],{:id => "details_table_#{seed}", :class => "details hoverable", :width => table_width})
    else
      table = content_tag(:table, [head, foot, body],{:id => "details_table_#{seed}", :class => "details hoverable"})
    end

    return table + (tail || "")
  end

  # 
  def details_to_table
    return "" unless @product.is_a?(ProductDetailed)
    html_tag = with_details do
      @details.map {|d| detail_to_tr(d) {|i,d| item_to_td(i,d)}}
    end
    rows  = content_tag(:div, html_tag, :class => "detail_scroll_base")
  end

  #
  def details_to_table_input
    return "" unless @product.is_a?(ProductDetailed)
    seed = random_id()
    space_details = "space_details_#{seed}"
    tail = sortable_details(seed)
    tail += hidden_field_tag("new_details", @new_details.to_json, :id => "new_details_#{seed}")
    tail += hidden_field_tag("old_details", @old_details.to_json, :id => "old_details_#{seed}")
    tail += hidden_field_tag("order_details", @order_details.to_json, :id => "order_details_#{seed}")
    d = @product.detail_class.new
    @detail_items.each do |i|
      i.initialize_column_of(d)
    end
    temp = "new_details_#{rand(65536)}#{Time.now.to_i}"
    instance_variable_set("@#{temp}", d)
    tr = detail_to_tr(d, "App.remove_new_details(this,\#{n},'#{seed}')", :id => temp) {|i,d| item_to_td_input(i, temp)}
    tr = tr.gsub(temp, "new_details_\#{n}")
    tr = tr.gsub(/(rfw_button_to_submit)_(\d+)/, '\1_\2_#{random}')
    button = button_to_function(h(s_("rfw|button|New")),
                                "App.add_new_details('#{escape_javascript(tr)}', '#{seed}')",
                                { :class => "button" })
    tail += content_tag(:div, "", :id => space_details)
    html_tag = with_details(seed, tail) do
      pairs = []
      @details.each do |d|
        unless @old_details.include?(d.id)
          html_id = "details_#{d.id}"
          pairs << [html_id, detail_to_tr(d, "App.remove_old_details(this,#{d.id},'#{seed}')", :id => html_id) {|i,d| item_to_td_input(i, html_id)}]
        end
      end
      pairs += @new_details.map do |n|
        d = instance_variable_get("@new_details_#{n}")
        html_id = "new_details_#{n}"
        [html_id, detail_to_tr(d, "App.remove_new_details(this,#{n},'#{seed}')", :id => html_id) {|i,d| item_to_td_input(i, html_id)}]
      end
      if @order_details.empty?
        pairs.map(&:last)
      else
        pairs.sort_by {|pair| @order_details.index(pair.first) || 0}.map(&:last)
      end
    end
    
    rows  = content_tag(:div, html_tag, :class => "detail_scroll_base w_detail")
    rows += content_tag(:div, button, :class => "right")
  end

  def sortable_details(seed)
    javascript_tag(<<-JS)
      jQuery("#details_table_#{seed}").tableDnD({
        onDrop: function(table, row){
          var a = jQuery.map(jQuery(table).find("tbody > tr"), function(e, i){
             return jQuery(e).attr("id");
          });
          var order = "["+jQuery.makeArray(a).join(",")+"]";
          jQuery("#order_details_#{seed}").val(order);
          App.update_tablednd(#{seed});
        },
        dragHandle: "handle"
      });
      App.update_tablednd(#{seed});
    JS
  end

  # <em>method</em> に関するエラーをまとめて返す。
  # cf. ActionView::Helpers::ActiveRecordHelper#error_message_on
  def error_messages_on(object, method)
    if (obj = (object.respond_to?(:errors) ? object : instance_variable_get("@#{object}"))) &&
        (errors = obj.errors.on(method))
      content_tag("div", errors.is_a?(Array) ? errors.join('<br />') : errors, :class => "formError")
    else
      ''
    end
  end

  # エラーメッセージがある場合には fieldWithErrors という class を持つ div でくるんで返す。
  def field_with_errors(i, object_name, &block)
    tags = block.call
    em = error_messages_on(object_name, i.column_name)
    unless em.blank?
      tags = content_tag(:div, tags, :class => "fieldWithErrors") + em
    end
    tags
  end

  def picker_tags(i, object_name, it)
    case App::PICKER_MODE
    when :ajax
      ajax_picker_tags(i, object_name, it)
    when :dom
      field_with_errors(i, object_name) do
        rfw_picker_tags(i, object_name, it)
      end
    else
      raise "unknown picker mode: #{App::PICKER_MODE}"
    end
  end

  # カレンダー選択部品による入力部品を返す。
  def ajax_calendar_picker_tags(i, object_name, it)
    picker_field = i.picker_field(it, object_name)

    if flash[:pick] && picker_field === flash[:pick][:field] && flash[:pick][:return_value]
      it[i.column_name] = flash[:pick][:return_value]
    end

    e = field_with_errors(i, object_name) do
      text_field_tag(rfw_name_for(object_name, i.column_name), i.to_data(it), :size => i.size_for_input_field)
    end
    e << " "
    e << submit_tag(s_("rfw|submit_tag|Pick Date"), :name => "picker[#{picker_field}]", :onclick => "$('#{object_name}_#{i.column_name}').addClassName('selected')", :disabled => !i.writable?)
    # hidden link for ajax
    e << link_to_view_motion("",
      @sub_view, picker_field, {},
      {
        :controller => "calendar",
        :action => "calendar",
        :return_field => picker_field,
        :view => @sub_view,
        :return_format => i.calendar_format,
        :alert_holiday => i.calendar_alert_holiday?,
      }, :style => "display:none")
    return e
  end

  # 選択部品による入力部品を返す。
  def ajax_picker_tags(i, object_name, it)
    return ajax_calendar_picker_tags(i, object_name, it) if i.calendar?
    case i
    when ItemProper
      case i.picked_atom
      when "company"
        submit_text = s_("rfw|submit_tag|Pick Company")
      when "organization"
        submit_text = s_("rfw|submit_tag|Pick Organization")
      when "person"
        submit_text = s_("rfw|submit_tag|Pick Person")
      when "post"
        submit_text = s_("rfw|submit_tag|Pick Post")
      when "group"
        submit_text = s_("rfw|submit_tag|Pick Group")
      else
        raise ArgumentError, "invalid picked atom: #{i.picked_atom}"
      end
      picker_field = i.picker_field(it, object_name)
      picked_class = i.picked_atom.classify.constantize
      if flash[:pick] && picker_field == flash[:pick][:field]
        values = i.picked_values(flash[:pick])
        v = values.find {|x| x.is_a?(picked_class)}
        it[i.column_name] = v.id
      elsif v = picked_class.find_by_id(it[i.column_name])
        values = i.picked_references(v)
      else
        values = i.picked_keys.map {|k| k.classify.constantize.new}
      end
      e = ""
      i.picked_keys.each_with_index do |k,n|
        e << text_field_tag("picker_#{picker_field}_#{k}", values[n].name, :readonly => true, :size => i.size_for_input_field)
        e << " "
      end
      onclick = i.picked_keys.map do |k|
        "$('picker_#{picker_field}_#{k}').addClassName('selected');"
      end.join("")
      e << submit_tag(submit_text, :name => "picker[#{picker_field}]", :onclick => onclick, :disabled => (i.readonly? || !i.writable?))
      unless error_messages_on(object_name, i.column_name).blank?
        e = content_tag(:div, e, :class => "fieldWithErrors")
      end
      e << '<br />'
      # hidden link for ajax
      e << link_to_view_motion("",
                               @sub_view, picker_field, {},
                               {
                                 :controller   => "picker",
                                 :action       => i.picked_atom,
                                 :return_field => picker_field,
                                 :view         => @sub_view,
                               }, :style       => "display:none")
      e << hidden_field(object_name, i.column_name)
      return e
    when ItemPolymorphic
      picker_field = i.picker_field(it, object_name)
      if flash[:pick] && picker_field == flash[:pick][:field]
        if ( k = flash[:pick][:lump_pick] ) && ( v = k.classify.constantize.find_by_id(flash[:pick][k.to_sym]) )
          it.__send__("#{i.column_name}=", v)
        end
      elsif v = it.__send__(i.column_name)
      else
        v = nil
      end
      e = ""
      e << text_field_tag("picker_#{picker_field}", v ? v.name : "", :readonly => true, :size => i.size_for_input_field)
      onclick = "$('picker_#{picker_field}').addClassName('selected');"
      e << submit_tag(s_("rfw|submit_tag|Pick"), :name => "picker[#{picker_field}]", :onclick => onclick, :disabled => (i.readonly? || !i.writable?))
      unless error_messages_on(object_name, i.column_name).blank?
        e = content_tag(:div, e, :class => "fieldWithErrors")
      end
      e << '<br />'
      # hidden link for ajax
      e << link_to_view_motion("",
                               @sub_view, picker_field, {},
                               {
                                 :controller   => "picker",
                                 :action       => "lump",
                                 :return_field => picker_field,
                                 :view         => @sub_view,
                               }, :style       => "display:none")
      e << hidden_field(object_name, "#{i.column_name}_id")
      e << hidden_field(object_name, "#{i.column_name}_type")
      return e
    when ItemPlural
      picker_field = i.picker_field(it, object_name)
      if flash[:pick] && picker_field == flash[:pick][:field]
        if ( k = flash[:pick][:lump_pick] ) && ( picked_one = k.classify.constantize.find_by_id(flash[:pick][k.to_sym]) )
          values = picked_one.__send__(i.data_method_name)
          it[i.column_name] = values.map(&:id).join(",")
        else
          values = []
        end
      else
        v = it.__send__(i.column_name)
        values = i.datum_class.joints(v)
      end
      e = ""
      e << text_field_tag("picker_#{picker_field}", values.map(&:name).join(","), :readonly => true, :size => i.size_for_input_field)
      onclick = "$('picker_#{picker_field}').addClassName('selected');"
      e << submit_tag(s_("rfw|submit_tag|Pick"), :name => "picker[#{picker_field}]", :onclick => onclick, :disabled => (i.readonly? || !i.writable?))
      unless error_messages_on(object_name, i.column_name).blank?
        e = content_tag(:div, e, :class => "fieldWithErrors")
      end
      e << '<br />'
      # hidden link for ajax
      e << link_to_view_motion("",
                               @sub_view, picker_field, {},
                               {
                                 :controller   => "picker",
                                 :action       => "lump",
                                 :return_field => picker_field,
                                 :view         => @sub_view,
                               }, :style       => "display:none")
      e << hidden_field(object_name, i.column_name)
      return e
    else
      raise ArgumentError, "invalid item type: #{i.class}"
    end
  end

  def rfw_calendar_tags(i, object_name, it)
    picker_field = i.picker_field(it, object_name)
    rfw_picker([{
                  :type => "text",
                  :name => rfw_name_for(object_name, i.column_name),
                  :value => i.to_data(it),
                  :key => "return_value",
                  :html => { :size => i.size_for_input_field }
                }],
               {
                 :controller => 'pick/calendar',
                 :action => "search",
                 :return_format => i.calendar_format,
                 :alert_holiday => i.calendar_alert_holiday?,
                 :escape => false,
               },
               s_('rfw|submit_tag|Pick Date'))
  end

  # 選択部品によって選択された内容を表示する際の名前は picked_name が優先される。
  def rfw_picked_name(picked_one)
    return picked_one.picked_name if picked_one.respond_to?(:picked_name)
    picked_one.name
  end

  def rfw_picker_tags(i, object_name, it)
    # pjc3用拡張Picker処理 add 3Lines 2009/2/27 h.nakamura
    if ["destination", "bizcardcompany", "bizcardorganization", "bizcardbranch", "bizcardperson"].include?(i.input_parameter) then
      return rfw_exstra_picker_tags(i, object_name, it)
    end

    return rfw_calendar_tags(i, object_name, it) if i.calendar?

    case i
    when ItemProper
      options = i.rfw_picker_options(it, object_name)
      id = rfw_value_for(object_name, i.column_name)
      unless id.blank?
        found = i.picked_atom.classify.constantize.find_by_id(id)
        name = rfw_picked_name(found) if found
        name = found.to_text if i.input_parameter =~ /\A(?:project)/ && found
      end
      rfw_picker([{
                    :type => 'text',
                    :name => "#{rfw_name_for(object_name, i.column_name)}_name",
                    :value => name,
                    :key => options[:return_field_name],
                    :html => { :readonly => true, :size => i.size_for_input_field }
                  },
                  {
                    :type => 'hidden',
                    :name => rfw_name_for(object_name, i.column_name),
                    :value => id,
                    :key => options[:return_field_id]
                  },
                  '<br />',
                 ],
                 { :controller => options[:controller], :action => 'search' },
                 s_(options[:submit_text])) do |x_id|
        # * プロジェクト
        # * 業務
        # * 案件
        # に関する選択部品の場合には選択が完了した時点で変化があれば一度サーバに送信して編集画面を再描画する。
        if i.input_parameter =~ /\A(?:matter|project)/
          "function() {new Form.Element.Observer($('#{x_id}').previous('input'), 0.3, function(element, value) {element.form.onsubmit();});}"
        elsif i.picking_record?
          # rfw[record_model_name] を動的に指定する。
          "Rfw.embedHiddenParameters.bindAsEventListener($('#{x_id}'), {record_model_name: '#{i.picked_atom.classify}', record_setting_id: '#{i.id}'})"
        else
          ''
        end
      end
    when ItemPolymorphic
      picker_field = i.picker_field(it, object_name)
      id = rfw_value_for(object_name, "#{i.column_name}_id")
      model_class = rfw_value_for(object_name, "#{i.column_name}_type")
      value = nil
      unless id.blank? || model_class.blank?
        found = model_class.constantize.find_by_id(id)
        value = rfw_picked_name(found) if found
      end
      rfw_picker([{
                    :type => 'text',
                    :name => "picker_#{picker_field}",
                    :value => value,
                    :key => 'model_value',
                    :html => { :readonly => true, :size => i.size_for_input_field }
                  },
                  {
                    :type => 'hidden',
                    :name => rfw_name_for(object_name, "#{i.column_name}_type"),
                    :value => model_class,
                    :key => 'model_name'
                  },
                  {
                    :type => 'hidden',
                    :name => rfw_name_for(object_name, "#{i.column_name}_id"),
                    :value => id,
                    :key => 'model_id'
                  },
                  '<br />',
                 ],
                 { :controller => 'pick/lump', :action => 'search', :mode => 'company' },
                 s_('rfw|submit_tag|Pick'))
    when ItemPlural
      picker_field = i.picker_field(it, object_name)
      ids = rfw_value_for(object_name, i.column_name)
      values = Person.joints(ids).map(&:name)
      rfw_picker([{
                    :type => 'text',
                    :name => "picker_#{picker_field}",
                    :value => values.join(','),
                    :key => 'people_names',
                    :html => { :readonly => true, :size => i.size_for_input_field  } },
                  {
                    :type => 'hidden',
                    :name => rfw_name_for(object_name, i.column_name),
                    :value => ids,
                    :key => 'people_ids'
                  },
                  '<br />',
                 ],
                 { :controller => 'pick/lump_people', :action => 'search', :mode => 'company' },
                 s_('rfw|submit_tag|Pick'))
    else
      raise ArgumentError, "invalid item type: #{i.class}"
    end
  end

  # メール送信部品による選択の情報を返す。
  def mailto_info
    mail = params[:mail] || {}
    recipients = Person.joints(mail[:recipients])
    if recipients.size > 0
      info = select_tag "_mail_info", options_from_collection_for_select(recipients, "id", "name_with_in", 0)
    else
      info = ""
    end
    num = text_field_tag "_mail_num", recipients.size, :readonly => true, :class => "mail_num", :id => nil
    info << ns_("rfw|mailto_info|%{num} person", "%{num} people", recipients.size) % { :num => num }
    mail.keys.each do |key|
      info << hidden_field_tag("mail[#{key}]", mail[key], :id => nil)
    end
    return info
  end

  # 詳細画面へのリンクを返す。
  def display_aspect(controller, view)
    url = {:controller => controller, :action => "show", :id => @it.id, :type => @model_class, :product_id => @product.id}
    if /\Ashow/ =~ params[:action] || request.xhr?
      javascript_tag(remote_function(:update => view, :url => url))
    else
      link = link_to_view_motion(s_(controller), view, "show", {}, url, :class => "autoload button")
      content_tag(:div, link, :class => "left_buttons")
    end
  end

  # タブとして表示する詳細画面へのリンクを返す。
  def display_aspect_as_tab(controller, view)
    url = {:controller => controller, :action => "show_as_tab", :id => @it.id, :type => @model_class, :product_id => @product.id}
    if /\Ashow/ =~ params[:action] || request.xhr?
      javascript_tag(remote_function(:update => view, :url => url))
    else
      link = link_to_view_motion(s_(controller), view, "show_as_tab", {}, url, :class => "autoload button")
      content_tag(:div, link, :class => "left_buttons")
    end
  end

  # サマリへのリンクを返す。
  def summary_aspect(controller)
    url = {:controller => controller, :action => "summary", :id => @it.id, :type => @model_class, :product_id => @product.id}
    javascript_tag(remote_function(:update => "summary_#{controller.pluralize}", :url => url))
  end

  # 関連文書を選択するためのリンクを返す。
  def link_to_select_document(it)
    return "" if @relatable && (it == @relatable || @product.document_nodes_for(@relatable).include?(it))
    link_to_remote(tree_icon_open,
                   :update => "document_table_base",
                   :complete => visual_effect(:highlight, "document_table", :queue => {:scope => "document", :limit => 2, :position => "end"}),
                   :url => {:controller => "document", :action => "select", :id => params[:id], :type => params[:type], :product_id => params[:relatable_product_id], :target_id => it.id, :target_type => it.class.to_s, :target_product_id => @product.id})
  end

  # 一覧検索(汎用検索)の設定画面へのリンクを返す。
  def link_to_search
    return "" unless @product.search?
    style = "button uppx"
    if @display.searching?
      caption = s_("rfw|Product|Looking for")
      style += " searching"
    else
      caption = s_("rfw|Product|Search")
    end
    link_to_remote(h(caption),
                   { :update => "view_search",
                     :url => {:controller => "search", :action => "index"},
                     :with => "'display_id=' + $F('list')",
                   },
                   {:class => style})
  end

  # 一覧の設定画面へのリンクを返す。
  def link_to_list
    link_to_view_motion(h(s_("rfw|Product|List")), "view_list", @product.id,
                        {},
                        {:controller => "list", :action => "index", :id => @product.id},
                        {:class => "button reloadable_link"})
  end

  # 画面上下に配置されるボタン群を返す。
  def button_box(suffix)
    buttons = {}
    if params[:id] && @display.button_back?
      buttons[:left] = link_to_view_motion(h(s_("rfw|button|Back")),
                                           @current_view, "show",
                                           {:id => params[:id], "_" => suffix},
                                           {:action => "show", :id => params[:id]},
                                           :class => "button reloadable_link")
    elsif @display.button_close?
      buttons[:left] = link_to_close({:id => nil, "_" => suffix}, {:action => "list", :id => nil}, :class => "button")
    end
    buttons[:center] = ""
    if @display.button_edit? && @product.modifiable? && (!@it.respond_to?(:writable?) || @it.writable?)
      buttons[:center] += link_to_view_motion(h(s_("rfw|button|Edit")),
                                              @current_view, "edit",
                                              {:id => @it.id, "_" => suffix},
                                              {:action => "edit", :id => @it },
                                              :class => "button reloadable_link")
    end
    if @display.button_copy? && @product.modifiable? && (!@it.respond_to?(:writable?) || @it.writable?)
      buttons[:center] += link_to_view_motion(h(s_("rfw|button|Copy")),
                                              @current_view, "copy",
                                              {:id => @it.id, "_" => suffix},
                                              {:action => "new", :id => @it },
                                              :class => "button reloadable_link")
    end
    if @submit_tag
      buttons[:center] += @submit_tag
    end
    if @display.button_delete? && @product.modifiable? && (!@it.respond_to?(:writable?) || @it.writable?)
      url = {:action => "destroy", :id => @it }
      if display_workflow? && @wf_config.issue_id
        if User.current.admin?
          # keep url
        elsif @wf_config.issueable? # TODO: issueable? is neither neccesary nor sufficient
          url = url.merge(:workflow => {:issue_id => @wf_config.issue_id})
        else
          url = nil
        end
      end
      if url
        buttons[:right] = link_to_view_motion(h(s_("rfw|button|Destroy")),
                                              @current_view, "destroy",
                                              {:id => @it.id, "_" => suffix},
                                              url,
                                              :confirm => s_('rfw|confirm|Are you sure?'),
                                              :method => :post,
                                              :class => "button reloadable_link")
      end
    end
    return lcr_button_box(buttons)
  end

  def display_workflow?
    #return false if %w[new create_document].include?(params[:action])
    return @display.workflow_enabled?
  end

  def symbols_for(arr, prefix)
    return [] if arr.nil?
    arr.map{|v| "#{prefix}_#{v}".to_sym }
  end

  # ワークフローで使用するボタン群を返す
  def workflow_buttons(config, dom_id)
    html_buttons = []
    buttons = config.buttons || []
    if config.issueable?
      buttons.each do |v|
        action, label = v.key, v.value
        html_buttons << workflow_button(label, "wf_#{action}", dom_id)
      end
    else
      possible_responses = config.possible_responses
      buttons.each do |v|
        action, label = v.key, v.value
        if config.configurable? && action == 'configure'
          html_buttons << workflow_button(label, "wf_#{action}", dom_id)
        else
          # config 以外
          if possible_responses && possible_responses.include?(action)
            html_buttons << workflow_button(label, "wf_#{action}", dom_id)
          else
            html_buttons << workflow_button(label, "wf_#{action}", dom_id, { :disabled => 'true', :onclick => 'return false' })
          end
        end
      end
    end
    html_buttons.join("\n")
  end

  # ワークフローで使用するボタン(リンク)を一つ返す
  def workflow_button(label, action, dom_id, options = {})
    html_options = { :class => 'button' }
    html_options.merge!(options)
    link_to_remote(label,
                   { :update => dom_id,
                     :url => {
                       :id => @it.id,
                       :product_id => @product.id,
                       :workflow_dom_id => dom_id,
                       :action => action,
                       :method => :post } },
                   html_options)
  end

  # ワークフローの編集画面で使用するラジオボタン群を返す
  def workflow_actions(config)
    checked_value = params[:workflow][:action]
    html_actions = []
    buttons = config.buttons || []
    checked = nil
    if config.issueable?
      buttons.each do |v|
        action, label = v.key, v.value
        next if /config/ =~ action
        checked ||= (action == checked_value)
        html_actions << workflow_action(label, action, (action == checked_value))
      end
    else
      possible_responses = config.possible_responses
      buttons.each do |v|
        action, label = v.key, v.value
        next if /config/ =~ action
        checked ||= (action == checked_value)
        if possible_responses && possible_responses.include?(action)
          html_actions << workflow_action(label, action, (action == checked_value))
        else
          html_actions << workflow_action(label, action, (action == checked_value), { :disabled => 'true'})
        end
      end
    end
    html_actions << workflow_action(_('Do Nothing'), 'do_nothing', !checked)
    html_actions.join("\n")
  end

  # ワークフローで使用するラジオボタンを一つ返す
  def workflow_action(label, action, checked = false, options = {})
    tag =  radio_button_tag("workflow[action]", action, checked, options)
    tag << label_tag("workflow_action_#{action}", label)
  end

  # ワークフローで使用するコンボボックスの選択肢を返す
  def workflow_options(container, key = :id, option_name = nil, selected = nil)
    return nil unless container
    option_values = { }
    container.each do |d|
      option_values[d.name] = d.__send__(key).to_s
    end
    if option_name
      options_for_select(option_values, params[:workflow][option_name][selected])
    else
      options_for_select(option_values, params[:workflow][selected])
    end
  end

  # ワークフローで使用するラジオボタンを返す
  # FIXME: ラベルと値を分ける
  def workflow_radio_buttons(labels)
    checked_value = params[:workflow][:option][:cache] || labels.last
    html = []
    labels.each do |label|
      html << radio_button_tag("workflow[option][cache]", label, (label == checked_value))
      html << label_tag("workflow_option_cache_#{label}", _(label))
    end
    html.join("\n")
  end

  include RandomId

  # ワークフローで使用する picker を返す
  def workflow_picker_tags(i, object_name, it, assignment_id, index)
    submit_text = s_("rfw|submit_tag|Pick Person")
    option = params[:workflow][:option]
    stored_value = option[object_name.to_sym] ? option[object_name.to_sym][assignment_id.to_s].values[index] : nil
    picker_field = "workflow_person_#{object_name}_#{assignment_id}_#{index}"
    picked_class = Person
    if flash[:pick] && picker_field == flash[:pick][:field]
      values = i.picked_values(flash[:pick])
      stored_value = flash[:pick][:person]
    elsif v = picked_class.find_by_id(it[i.column_name])
      values = i.picked_references(v)
    else
      values = i.picked_keys.map {|k| k.classify.constantize.new}
    end
    e = ""
    i.picked_keys.each_with_index do |k,n|
      name = nil
      if stored_value.blank?
        stored_value = values[n].id
        name = values[n].name
      elsif picked_one = picked_class.find_by_id(stored_value.to_i)
        name = picked_one.name
      end
      e << text_field_tag("picker_#{picker_field}_#{k}", name, :readonly => true, :size => i.size_for_input_field)
      e << " "
    end
    onclick = i.picked_keys.map do |k|
      "$('picker_#{picker_field}_#{k}').addClassName('selected');"
    end.join("")
    e << submit_tag(submit_text, :name => "picker[#{picker_field}]", :onclick => onclick, :disabled => (i.readonly? || !i.writable?))
    unless error_messages_on(object_name, i.column_name).blank?
      e = content_tag(:div, e, :class => "fieldWithErrors")
    end
    # hidden link for ajax
    e << link_to_view_motion("",
                             @sub_view, picker_field, {},
                             {
                               :controller   => "picker",
                               :action       => i.picked_atom,
                               :return_field => picker_field,
                               :view         => @sub_view,
                             }, :style       => "display:none")
    e << hidden_field_tag("workflow[option][#{object_name}][#{assignment_id}][#{index}]", stored_value)
    return e
  end

  # ワークフローで使用する picker を複数出力する
  # item_options = :input_parameter => 'person', :model_name => 'MatterReport'
  def workflow_picker_lines(config, object_name, it, item_options = { })
    lines = []
    item = ItemPseudo.new(item_options)
    case object_name
    when 'assignment'
      return if config.assignments.blank?
      max = workflow_assignee_maximum(config.assignments)
      c = config.assignments.first
      max.times do |n|
        lines << content_tag(:tr) do
          content_tag(:th, "#{object_name.capitalize} #{c.name}") +
            content_tag(:td, workflow_picker_tags(item, object_name, it, c.id, n))
        end
      end
    when 'candidates'
      return if config.candidateable.blank?
      config.candidateable.each do |c|
        c.assignee_maximum.times do |n|
          lines << content_tag(:tr) do
            content_tag(:th, "#{object_name.capitalize} #{c.name}") +
              content_tag(:td, workflow_picker_tags(item, object_name, it, c.id, n))
          end
        end
      end
    else
      raise "must not happen. object_name = #{object_name}"
    end
    lines.join("\n")
  end

  def workflow_assignee_maximum(assignments)
    return 0 if assignments.blank?
    assignments.map{|a| a.assignee_maximum }.max
  end

  def workflow_error_messages_for(response, *params)
    if @workflow_error
      # FIXME correct message
      err_icon = image_tag("error.png", :alt => "error")
      message = [ content_tag(:div, err_icon+s_("rfw|message|workflow web service returns an error"), :class => "errorExplanation_head"),
                  content_tag(:div, response.information, :class => "errorExplanation_detail"),
                ]
      return content_tag(:div, message.join, :class => "errorExplanation_base")
    end
  end

  # DOM ベースの picker
  def rfw_mail_form(picker_id, mode)
    seed = picker_id
    tags = ''
    tags << '<img src="/images/mail.png" width=32 height=32 class="iepngfix" alt="'+s_("rfw|image|alt|mail")+'" />'
    tags << select_tag('_mail_info', '', :id => "mail_info_#{seed}", :style => 'display:none')
    tags << text_field_tag('_mail_num', 0, :readonly => true, :class => 'mail_num', :id => "mail_num_#{seed}")
    case mode
    when :show
      tags << submit_tag(h(s_("rfw|submit_tag|mail|Send")),
                         :id => "mail_send_#{seed}",
                         :name => "mailsend",
                         :confirm => s_('rfw|submit_tag|mail|Are you sure?'),
                         :disabled => true)
    when :edit
      # do nothing
    else
      raise "must not happen! mode:#{mode}"
    end
    if params[:mail]
      names      = params[:mail][:names]
      recipients = params[:mail][:recipients]
      field_type = params[:mail][:field_type]
      comment    = params[:mail][:comment]
      attachment = params[:mail][:attachment]
    end
    tags << rfw_picker([{
                          :type  => 'hidden',
                          :name  => 'mail[names]',
                          :value => names,
                          :key   => 'mailto_names'
                        },
                        {
                          :type  => 'hidden',
                          :name  => 'mail[recipients]',
                          :value => recipients,
                          :key   => 'mailto_ids'
                        },
                        {
                          :type  => 'hidden',
                          :name  => 'mail[field_type]',
                          :value => field_type,
                          :key   => 'mailto_type'
                        },
                        {
                          :type  => 'hidden',
                          :name  => 'mail[comment]',
                          :value => comment,
                          :key   => 'mail_comment'
                        },
                        {
                          :type  => 'hidden',
                          :name  => 'mail[attachment]',
                          :value => attachment,
                          :key   => 'mail_attachment'
                        }],
                       { :controller => 'pick/mailto', :action => 'search', :mode => 'company'},
                       h(s_('rfw|submit_tag|mail|To'))) do |link_id|
      javascript_tag(<<-JS)
        Rfw.showMailInfo.defer('#{seed}');
        $('#{link_id}').observe('click', function(e1){
          setTimeout(function(){
            $$('div.rfw_tabs_detail_#{seed}').first().observe('click', Rfw.showMailInfo.curry('#{seed}'));
          }, 1000);
        });
      JS
    end
    tags
  end

  # pjc3用拡張選択部品による入力部品を返す。
  def rfw_exstra_picker_tags(i, object_name, it)
    case i.input_parameter
    when "destination"
      return rfw_destination_picker_tags(i, object_name, it)

    when "bizcardcompany", "bizcardorganization", "bizcardbranch", "bizcardperson"
      action = i.input_parameter.sub("bizcard","")
      return rfw_bizcard_picker_tags(i, object_name, it, action)
    end

    raise ArgumentError, "invalid extra picked name: #{i.input_parameter}"
    return false
  end

  # 行き先ピッカー
  def ajax_destination_picker_tags(i, object_name, it)
    picker_field = i.picker_field(it, object_name)
    if flash[:pick] && flash[:pick] && flash[:pick][:field] == "destination"
      it[i.column_name] = flash[:pick][:return_value]
      args =i.input_initial_value.split(',')
      args.each do |arg|
        it[arg.to_s] = flash[:pick][arg.to_sym]
      end
    end

    e = text_field_tag("#{object_name}[#{i.column_name}]",it[i.column_name.to_sym],:id=>"#{object_name}[#{i.column_name}]", :size => i.size_for_input_field,:readonly => "readonly")
    e << " "
    e << submit_tag(s_("rfw|submit_tag|Pick Destination"), :name => "picker[#{picker_field}]", :onclick => "$('#{object_name}_#{i.column_name}').addClassName('selected')", :disabled => !i.writable?)
    # hidden link for ajax
    e << link_to_view_motion("",
      @sub_view, picker_field, {},
      {
        :controller => "destination",
        :action => "destination",
        :retun_field => picker_field,
        :view => @sub_view,
      }, :style => "display:none")
    return e
  end

  def rfw_destination_picker_tags(i, object_name, it)
    inputs =[]
    inputs << {:type => "text",
               :name => rfw_name_for(object_name, i.column_name),
               :value => rfw_value_for(object_name, i.column_name),
               :key => "return_value",
               :html => {:readonly => true,:size => i.size_for_input_field }
              }
    args =i.input_initial_value.split(',')
    args.each do |arg|
      if params[:rfw]
        value =params[:rfw][arg.to_sym]
      end
      inputs << {
                  :name  => rfw_name_for(object_name, "#{arg}"),
                  :value => value,
                  :key   => "#{arg}",
                }
    end
    rfw_picker(inputs,
               {
                 :controller => 'pick/destination',
                 :action => "search"
               },
               s_("rfw|submit_tag|Pick Destination"))
  end

  # 名刺企業選択ピッカー
  def rfw_bizcard_picker_tags(i, object_name, it, action)

    # 操作対象のモデルを設定
    case action
    when "company"
      title = "PjcBusinessCard|BusinessCards CompanyPicker"
      model_class = PjcBusinessCardCompany
    when "organization"
      title = "PjcBusinessCard|BusinessCards OrganizationPicker"
      model_class = PjcBusinessCardOrg
    when "branch"
      title = "PjcBusinessCard|BusinessCards BranchPicker"
      model_class = PjcBusinessCardBranch
    when "person"
      title = "PjcBusinessCard|BusinessCards PersonPicker"
      model_class = PjcBusinessCard
    else
      title = ""
      model_class = nil
    end

    # 自身のカラム設定をパース
    self_col_name   = nil
    add_on_col_name = nil
    list_condition  = nil
    @picker_field_values ||= {}

    self_col_info = i.input_initializer.split(",")
    self_col_info.each do |val|
      val.strip!
      case
      when !(val.index("+").blank?)
        temp = val.split("+")
        self_col_name   = temp[0]
        add_on_col_name = temp[1] 
      when !(val.index("=").blank?)
        temp = val.split("=")
        value = it[temp[1].to_sym] if !(temp[1].blank?)
        # ピッカー選択時
        if flash[:pick]
          if flash[:pick][:col_name] == temp[1] then
            value = flash[:pick][temp[0].to_sym] if !(temp[0].blank?)
          end
        end
        list_condition = "#{temp[0]}=#{value}" if !(temp[0].blank?) and !(value.blank?)
      else
        self_col_name  = val
      end
    end

    return "" if self_col_name.blank?

    picker_field = i.picker_field(it, object_name)
    if flash[:pick] && !(picker_field.index(flash[:pick][:field]).blank?)

      # 自身に関するデータ取得
      id = flash[:pick]["#{action}_id".to_sym]
      unless id.blank?
        data = model_class.find(:first, :conditions=>["id=?", id])
        it[i.column_name] = data.__send__(self_col_name)
        if (self_col_name=="id") and !(add_on_col_name.blank?)
          add_on_value = data.__send__(add_on_col_name)
        else
          add_on_value = ""
        end
      else
        it[i.column_name] = ""
      end

      # その他のカラムへ設定するデータ取得
      others = i.input_initial_value.split(",")
      others.each do |other|
        values = other.split("=")
        if values.size == 2
          unless values[0].blank? and values[1].blank?
            dest_column = values[0]
            it[dest_column] = ""
            unless values[1].index(":").blank?
              temp = values[1].split(":")    # ［テーブル：カラム］指定を分解
              unless temp[0].blank? and temp[1].blank?
                table_name  = temp[0]
                col_name    = temp[1]
                case table_name
                when "company"
                  id = flash[:pick][:company_id]
                  data = PjcBusinessCardCompany.find(:first, :conditions=>["id=?", id]) unless id.blank?
                when "organization"
                  id = flash[:pick][:organization_id]
                  data = PjcBusinessCardOrg.find(:first, :conditions=>["id=?", id]) unless id.blank?
                when "branch"
                  id = flash[:pick][:branch_id]
                  data = PjcBusinessCardBranch.find(:first, :conditions=>["id=?", id]) unless id.blank?
                when "person"
                  id = flash[:pick][:person_id]
                  data = PjcBusinessCard.find(:first, :conditions=>["id=?", id]) unless id.blank?
                end
                it[dest_column] = data.__send__(col_name) unless data.blank?
              end
            else
              unless id.blank?
                data = model_class.find(:first, :conditions=>["id=?", id])
                it[dest_column] = data.__send__(values[1])
              end
            end
          end
        end
      end
    else
      if (self_col_name=="id") and !(add_on_col_name.blank?)
        item_data = instance_variable_get("@#{object_name}")
        id = item_data.__send__(i.column_name)
        data = model_class.find(:first, :conditions=>["id=?", id])  unless id.blank?
        add_on_value = data.__send__(add_on_col_name) unless data.blank?
      end
    end

    if add_on_col_name.blank?
      #e = text_field object_name, i.column_name, :size => i.size_for_input_field
      e = text_field_tag("#{i.column_name}", it[i.column_name.to_sym], :size => i.size_for_input_field, :readonly => "readonly")
    else
      #e = hidden_field object_name, i.column_name
      e = hidden_field_tag("#{object_name}[#{ i.column_name}]", it[i.column_name.to_sym], :id=>"it_#{i.column_name}" )
      e << text_field_tag("#{i.column_name}_#{add_on_col_name}", add_on_value, :size => i.size_for_input_field, :readonly => "readonly")
    end
    e << "<br />"
    e << submit_tag(s_(title), :name => "picker[#{picker_field}]", :onclick => "$('#{object_name}_#{i.column_name}').addClassName('selected')", :disabled => !i.writable?)
    # hidden link for ajax
    e << link_to_view_motion("",
      @sub_view, picker_field, {},
      {
        :controller     => "business_card",
        :action         => action,
        :view           => @sub_view,
        :return_field   => picker_field,
        :col_name       => i.column_name,
        :list_condition => list_condition
      }, :style => "display:none")

    return e
  end
end
