# -*- coding: utf-8 -*-
# == Schema Information
# Schema version: 20090304040015
#
# Table name: design_tickets
#
#  id           :integer       not null, primary key
#  domain_id    :integer       not null
#  user_id      :integer       not null
#  token        :string(255)   not null
#  created_at   :string(14)
#  updated_at   :string(14)
#  created_by   :integer
#  updated_by   :integer
#  created_in   :integer
#  updated_in   :integer
#  lock_version :integer       default(0), not null
#

# Design Web サービスのログイン情報のモデル
class DesignTicket < ActiveRecord::Base
  MAX_LIFETIME = 60 * 60

  untranslate_all
  timestamps_as_string
  user_monitor
  before_validation :gc
  validates_presence_of :domain_id, :user_id, :token
  validates_uniqueness_of :token

  # 管理者権限のあるユーザ <em>user</em> のログイン情報を記録し有効な ticket を発行する。
  # 失敗した場合には false を返す。
  def self.give(user)
    if user.admin?
      token = (0...4).map {sprintf("%#xd", rand(65536))}.join('-')
      ticket = new(:domain_id => user.domain_id, :user_id => user.id, :token => token)
      return ticket if ticket.save
    end
    return false
  end

  # ハッシュの情報に該当する有効な ticket を返す。
  # <em>args[:invalidate]</em> が真の場合にはその ticket を無効にする。
  # ticket が存在しない場合には単に false を返す。
  def self.take(args)
    ticket = find_by_token(args[:token])
    if (ticket &&
        ticket.domain_id == args[:domain_id] &&
        ticket.user_id == args[:user_id])
      if !args[:invalidate] || ticket.destroy
        return ticket 
      end
    end
    return false
  end

  # ハッシュの情報に該当する有効な ticket があるかどうかを判定する。
  # さらに <em>args[:invalidate]</em> が真の場合にはその ticket を無効にする。
  def self.valid?(args)
    return !!take(args)
  end

  private

  # 古くなったレコードを削除する。
  def gc
    DesignTicket.find(:all).each do |ticket|
      t = ticket.created_at || ticket.updated_at
      if t && Time.now - t > MAX_LIFETIME
        ticket.destroy 
      end
    end
  end
end
