# -*- coding: utf-8 -*-
# == Schema Information
# Schema version: 20090304040015
#
# Table name: documents
#
#  id                   :integer       not null, primary key
#  domain_id            :integer       not null
#  relatable_id         :integer
#  relatable_type       :string(255)
#  relatable_product_id :integer
#  target_id            :integer
#  target_type          :string(255)
#  target_product_id    :integer
#  mutual               :boolean
#  created_at           :string(14)
#  updated_at           :string(14)
#  created_by           :integer
#  updated_by           :integer
#  created_in           :integer
#  updated_in           :integer
#  lock_version         :integer       default(0), not null
#

# 関連文書のモデル。
class Document < ActiveRecord::Base
  untranslate_all
  timestamps_as_string
  user_monitor
  belongs_to :domain
  belongs_to :relatable, :polymorphic => true
  belongs_to :relatable_product, :class_name => "Product", :foreign_key => "relatable_product_id"
  belongs_to :target, :polymorphic => true
  belongs_to :target_product, :class_name => "Product", :foreign_key => "target_product_id"
  has_many :document_paths, :dependent => :destroy
  has_many :document_edges, :through => :document_paths
  after_save :update_graph

  # <em>key</em> に対応する機能を返す。
  def product(key)
    product = __send__("#{key}_product")
    if product && product.visible? && product.document?
      return product
    end
    raise ArgumentError, "product #{product.inspect} is not tuned for document"
  end

  # <em>key</em> に対応する機能の ID を返す。
  def product_id(key)
    product(key).id
  end

  # 文書の URL のためのオプションを返す。
  def url_options_for(x)
    value_for(x) {|key| {:product_id => product_id(key), :id => __send__("#{key}_id")}}
  end

  # 文書の fragment のためのオプションを返す。
  def fragment_options_for(x)
    value_for(x) {|key| {:product => product(key), :document => __send__(key)}}
  end

  # 名前を返すメソッド名を返す。
  def name_for(x)
    method_for(x, "name")
  end

  # 番号を返すメソッド名を返す。
  def number_for(x)
    method_for(x, "number")
  end

  # 番号を返す用いるメソッド名を返す。
  def content_for(x)
    method_for(x, "content")
  end

  # 複製機能のためのコピーを返す。
  def copy
    copied = self.class.new
    copied.attributes = attributes
    copied.relatable_id = nil
    return copied
  end

  private

  def value_for(x, &block)
    yield((x == target) ? :relatable : :target)
  end

  def method_for(x, name)
    value_for(x) do |key|
      method = product(key).__send__("document_#{name}_method")
      __send__(key).__send__(method)
    end
  end

  def update_graph
    # remove old edges if exists
    document_paths.each(&:destroy)
    # accumulate transitions.
    if mutual?
      proc = lambda do |edges, x|
        sets = edges.map(&:documents)
        edges.zip(sets).each do |edge, docs|
          a = DocumentEdge.new
          a.from = x
          a.to = edge.to
          a.from_product_id = x
          a.to_product_id = edge.to_product_id
          a.save!
          b = DocumentEdge.new
          b.from = edge.to
          b.to = x
          b.from_product_id = edge.from_product_id
          b.to_product_id = x
          b.save!
          ids = docs.map(&:id)
          ids.unshift(id)
          ids.each do |document_id|
            DocumentPath.create!(:domain_id => domain_id, :document_id => document_id, :document_edge_id => a.id)
            DocumentPath.create!(:domain_id => domain_id, :document_id => document_id, :document_edge_id => b.id)
          end
        end
      end
      edges_x = DocumentEdge.all_from_but_not_to(relatable, target)
      edges_y = DocumentEdge.all_from_but_not_to(target, relatable)
      proc.call(edges_x, target)
      proc.call(edges_y, relatable)
    end
    # draw the direct edge.
    [{
       :domain_id => domain_id,
       :from_id   => relatable_id,
       :from_type => relatable_type,
       :from_product_id => relatable_product_id,
       :to_id     => target_id,
       :to_type   => target_type,
       :to_product_id => target_product_id,
     }, {
       :domain_id => domain_id,
       :from_id   => target_id,
       :from_type => target_type,
       :from_product_id => target_product_id,
       :to_id     => relatable_id,
       :to_type   => relatable_type,
       :to_product_id => relatable_product_id,
     },
    ].each do |h|
      e = DocumentEdge.create!(h)
      DocumentPath.create!(:domain_id => domain_id, :document_id => id, :document_edge_id => e.id)
    end
  end
end
