/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.SystemParameter;
import org.opengion.fukurou.util.LogWriter;

import org.opengion.fukurou.util.HybsEntry ;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.model.Formatter;
import org.opengion.fukurou.db.ConnectionFactory;

import java.util.Map ;
import java.util.LinkedHashMap ;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;

/**
 * Process_DBMerge は、UPDATE と INSERT を指定し データベースを追加更新
 * する、ChainProcess インターフェースの実装クラスです。
 * 上流（プロセスチェインのデータは上流から下流へと渡されます。）から
 * 受け取った LineModel を元に、DBTableModel 形式ファイルを出力します。
 *
 * データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に
 * 設定された接続(Connection)を使用します。
 * -url_XXXX で指定された XXXX が、-dbid=XXXX に対応します。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * SQL文には、{&#064;SYS.YMDH}等のシステム変数が使用できます。
 * 現時点では、{&#064;SYS.YMD}、{&#064;SYS.YMDH}、{&#064;SYS.HMS} が指定可能です。
 *
 * @og.formSample
 *  Process_DBMerge -dbid=DBGE -insertTable=GE41
 *
 *     -dbid=DB接続ID             ： -dbid=DBGE (例: ParamProcess の -url_XXXX で指定された XXXX)
 *   [ -update=検索SQL文        ] ： -update="UPDATE GE41 SET NAME_JA = [NAME_JA],LABEL_NAME = [LABEL_NAME]
 *                                         WHERE SYSTEM_ID = [SYSTEM_ID] AND CLM = [CLM]"
 *   [ -updateFile=登録SQLﾌｧｲﾙ  ] ： -updateFile=update.sql
 *                                ：   -update や -updateFile が指定されない場合は、エラーです。
 *   [ -update_XXXX=固定値      ] ： -update_SYSTEM_ID=GE
 *                                     SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。
 *                                     WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'
 *   [ -insertTable=登録ﾃｰﾌﾞﾙID ] ： INSERT文を指定する場合は不要。INSERT する場合のテーブルID
 *   [ -insert=検索SQL文        ] ： -insert="INSERT INTO GE41 (SYSTEM_ID,CLM,NAME_JA,LABEL_NAME)
 *                                         VALUES ([SYSTEM_ID],[CLM],[NAME_JA],[LABEL_NAME])"
 *   [ -insertFile=登録SQLﾌｧｲﾙ  ] ： -insertFile=insert.sql
 *                                ：   -insert や -insertFile や、-table が指定されない場合は、エラーです。
 *   [ -insert_XXXX=固定値      ] ： -insert_SYSTEM_ID=GE
 *                                     SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。
 *                                     WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'
 *   [ -const_XXXX=固定値       ] ： -const_FGJ=1
 *                                     LineModel のキー（const_ に続く文字列)の値に、固定値を設定します。
 *                                     キーが異なれば、複数のカラム名を指定できます。
 *   [ -commitCnt=commit処理指定] ： 指定数毎にコミットを発行します。0 の場合は、終了までコミットしません。
 *   [ -display=false|true      ] ：結果を標準出力に表示する(true)かしない(false)か（初期値 false:表示しない)
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_DBMerge extends AbstractProcess implements ChainProcess {
	private static final String UPDATE_KEY	= "update_" ;
	private static final String INSERT_KEY	= "insert_" ;
	private static final String CNST_KEY	= "const_" ;

	private Connection	connection	= null;
	private PreparedStatement insPstmt	= null ;
	private PreparedStatement updPstmt	= null ;

	private String		dbid		= null;
	private String		insert		= null;
	private String		update		= null;
	private String		insertTable	= null;
	private int[]		insClmNos	= null;		// insert 時のファイルのヘッダーのカラム番号
	private int[]		updClmNos	= null;		// update 時のファイルのヘッダーのカラム番号
	private int			commitCnt	= 0;		// コミットするまとめ件数
	private boolean		display		= false;	// 表示しない

	private String[]	cnstClm		= null;		// 固定値を設定するカラム名
	private int[]		cnstClmNos	= null;		// 固定値を設定するカラム番号
	private String[]	constVal	= null;		// カラム番号に対応した固定値

	private boolean		firstRow	= true;		// 最初の一行目
	private int			count		= 0;
	private int			insCount	= 0;
	private int			updCount	= 0;

	private final static Map<String,String> mustProparty   ;		// ［プロパティ］必須チェック用 Map
	private final static Map<String,String> usableProparty ;		// ［プロパティ］整合性チェック Map

	static {
		mustProparty = new LinkedHashMap<String,String>();
		mustProparty.put( "dbid",	"DB接続ID(必須) 例: ParamProcess の -url_XXXX で指定された XXXX" );

		usableProparty = new LinkedHashMap<String,String>();
		usableProparty.put( "update",	"更新SQL文（sql or sqlFile 必須）" +
									CR + "例: \"UPDATE GE41 " +
									CR + "SET NAME_JA = [NAME_JA],LABEL_NAME = [LABEL_NAME] " +
									CR + "WHERE SYSTEM_ID = [SYSTEM_ID] AND CLM = [CLM]\"" );
		usableProparty.put( "updateFile",	"更新SQLファイル（sql or sqlFile 必須）例: update.sql" );
		usableProparty.put( "update_",		"SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。" +
									CR + "WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'" );
		usableProparty.put( "insert",			"登録SQL文（sql or sqlFile 必須）" +
									CR + "例: \"INSERT INTO GE41 " +
									CR + "(SYSTEM_ID,CLM,NAME_JA,LABEL_NAME) " +
									CR + "VALUES ([SYSTEM_ID],[CLM],[NAME_JA],[LABEL_NAME])\"" );
		usableProparty.put( "insertFile",		"登録SQLファイル（sql or sqlFile 必須）例: insert.sql" );
		usableProparty.put( "insertTable",	"INSERT する場合のテーブルID SQL文を指定する場合は不要。" );
		usableProparty.put( "insert_",		"SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。" +
									CR + "WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'" );
		usableProparty.put( "const_",	"LineModel のキー（const_ に続く文字列)の値に、固定値を" +
									CR + "設定します。キーが異なれば、複数のカラム名を指定できます。" +
									CR + "例: -sql_SYSTEM_ID=GE" );
		usableProparty.put( "commitCnt",	"指定数毎にコミットを発行します。" +
									CR + "0 の場合は、終了までコミットしません。(初期値: 0)" );
		usableProparty.put( "display",	"結果を標準出力に表示する(true)かしない(false)か" +
										CR + "(初期値 false:表示しない)" );
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_DBMerge() {
		super( "org.opengion.fukurou.process.Process_DBMerge",mustProparty,usableProparty );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理（ファイルオープン、ＤＢオープン等）に使用します。
	 *
	 * @param   paramProcess ParamProcess
	 */
	public void init( final ParamProcess paramProcess ) {
		Argument arg = getArgument();

		insertTable		= arg.getProparty("insertTable");
		update			= arg.getFileProparty("update","updateFile",false);
		insert			= arg.getFileProparty("insert","insertFile",false);
		commitCnt		= arg.getProparty("commitCnt",commitCnt);
		display			= arg.getProparty("display",display);

		dbid		= arg.getProparty("dbid");
		connection	= paramProcess.getConnection( dbid );

		if( insert == null && insertTable == null ) {
			String errMsg = "insert または、insertFile を指定しない場合は、insertTable を必ず指定してください。";
			throw new RuntimeException( errMsg );
		}

		if( insert != null && insertTable != null ) {
			String errMsg = "insert または、insertFile と、insertTable は、両方同時に指定できません。["
								 + insert + "],[" + insertTable + "]";
			throw new RuntimeException( errMsg );
		}

		// 3.8.0.1 (2005/06/17) {@SYS.XXXX} 変換処理の追加
		// {@SYS.YMDH} などの文字列を、yyyyMMddHHmmss 型の日付に置き換えます。
		// SQL文の {@XXXX} 文字列の固定値への置き換え
		HybsEntry[] entry	=arg.getEntrys(UPDATE_KEY);		// 配列
		SystemParameter sysParam = new SystemParameter( update );
		update = sysParam.replace( entry );

		if( insert != null ) {
			entry	=arg.getEntrys(INSERT_KEY);		// 配列
			sysParam = new SystemParameter( insert );
			insert = sysParam.replace( entry );
		}

		HybsEntry[] cnstKey = arg.getEntrys( CNST_KEY );		// 配列
		int csize	= cnstKey.length;
		cnstClm		= new String[csize];
		constVal	= new String[csize];
		for( int i=0; i<csize; i++ ) {
			cnstClm[i]  = cnstKey[i].getKey();
			constVal[i] = cnstKey[i].getValue();
		}
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理（ファイルクローズ、ＤＢクローズ等）に使用します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/27) commit,rollback,remove 処理を追加
	 *
	 * @param   isOK トータルで、OKだったかどうか(true:成功/false:失敗）
	 */
	public void end( final boolean isOK ) {
		boolean flag1 = Closer.stmtClose( updPstmt );
		updPstmt = null;
		boolean flag2 = Closer.stmtClose( insPstmt );
		insPstmt = null;

		// close に失敗しているのに commit しても良いのか？
		if( isOK ) {
			Closer.commit( connection );
		}
		else {
			Closer.rollback( connection );
		}
		ConnectionFactory.remove( connection,dbid );

		if( ! flag1 ) {
			String errMsg = "update ステートメントをクローズ出来ません。" + CR
							+ " update=[" + update + "] , commit=[" + isOK + "]" ;
			throw new RuntimeException( errMsg );
		}

		if( ! flag2 ) {
			String errMsg = "insert ステートメントをクローズ出来ません。" + CR
							+ " insert=[" + insert + "] , commit=[" + isOK + "]" ;
			throw new RuntimeException( errMsg );
		}
	}

	/**
	 * 引数の LineModel を処理するメソッドです。
	 * 変換処理後の LineModel を返します。
	 * 後続処理を行わない場合（データのフィルタリングを行う場合）は、
	 * null データを返します。つまり、null データは、後続処理を行わない
	 * フラグの代わりにも使用しています。
	 * なお、変換処理後の LineModel と、オリジナルの LineModel が、
	 * 同一か、コピー（クローン）かは、各処理メソッド内で決めています。
	 * ドキュメントに明記されていない場合は、副作用が問題になる場合は、
	 * 各処理ごとに自分でコピー（クローン）して下さい。
	 *
	 * @param   data LineModel オリジナルのLineModel
	 * @return  LineModel  処理変換後のLineModel
	 */
	public LineModel action( final LineModel data ) {
		count++ ;
		int updCnt = 0;
		try {
			if( firstRow ) {
				makePrepareStatement( insertTable,data );

				int size   = cnstClm.length;
				cnstClmNos = new int[size];
				for( int i=0; i<size; i++ ) {
					cnstClmNos[i] = data.getColumnNo( cnstClm[i] );
				}

				firstRow = false;
			}

			// 固定値置き換え処理
			for( int j=0; j<cnstClmNos.length; j++ ) {
				data.setValue( cnstClmNos[j],constVal[j] );
			}

			for( int i=0; i<updClmNos.length; i++ ) {
				updPstmt.setObject( i+1,data.getValue(updClmNos[i]) );
			}

			updCnt = updPstmt.executeUpdate();
			if( updCnt == 0 ) {
				for( int i=0; i<insClmNos.length; i++ ) {
					insPstmt.setObject( i+1,data.getValue(insClmNos[i]) );
				}
				int insCnt = insPstmt.executeUpdate();
				if( insCnt == 0 ) {
					String errMsg = "１件も追加されませんでした。" + CR
								+ " insert=[" + insert + "]" + CR
								+ "[" + data.getRowNo() + "]件目" + CR ;
					throw new RuntimeException( errMsg );
				}
				insCount++ ;
			}
			else if( updCnt > 1 ) {
				String errMsg = "複数行が同時に更新されました。[" + updCnt + "]件" + CR
							+ " update=[" + update + "]" + CR
							+ "[" + data.getRowNo() + "]件目" + CR ;
				throw new RuntimeException( errMsg );
			}
			else {
				updCount ++ ;
			}

			if( commitCnt > 0 && ( count%commitCnt == 0 ) ) {
				Closer.commit( connection );
			}
			if( display ) { printKey( count,updCnt,data ); }
		}
		catch (SQLException ex) {
			String errMsg = "登録処理でエラーが発生しました。" + CR
						+ ((updCnt == 1) ?
								" update=[" + update + "]"
							:	" insert=[" + insert + "]" + CR
								+ " insertTable=[" + insertTable + "]" )
						+ CR
						+ "[" + data.getRowNo() + "]件目" + CR
						+ "errorCode=[" + ex.getErrorCode() + "] State=["
						+ ex.getSQLState() + "]" + CR ;
			throw new RuntimeException( errMsg,ex );
		}
		return data;
	}

	/**
	 * 内部で使用する PreparedStatement を作成します。
	 * 引数指定の SQL または、LineModel から作成した SQL より構築します。
	 *
	 * @param   table  String    処理対象のテーブルID
	 * @param   data LineModel 処理対象のLineModel
	 */
	private void makePrepareStatement( final String table,final LineModel data ) {
		if( insert == null ) {
			StringBuilder buf = new StringBuilder();
			String[] names = data.getNames();
			int size = names.length;

			buf.append( "INSERT INTO " ).append( table ).append( " (" );
			buf.append( names[0] );
			for( int i=1; i<size; i++ ) {
				buf.append( "," ).append( names[i] );
			}
			buf.append( " ) VALUES ( ?" );
			for( int i=1; i<size; i++ ) {
				buf.append( ",?" );
			}
			buf.append( " )" );
			insert = buf.toString();

			// カラム番号を設定します。
			insClmNos = new int[size];
			for( int i=0; i<size; i++ ) {
				insClmNos[i] = i;
			}
		}
		else {
			Formatter format = new Formatter( data );
			format.setFormat( insert );
			insert = format.getQueryFormatString();
			insClmNos = format.getClmNos();
		}

		Formatter format = new Formatter( data );
		format.setFormat( update );
		update = format.getQueryFormatString();
		updClmNos = format.getClmNos();

		try {
			insPstmt = connection.prepareStatement( insert );
			updPstmt = connection.prepareStatement( update );
		}
		catch (SQLException ex) {
			String errMsg = "PreparedStatement を取得できませんでした。" + CR
						+ "insert=[" + insert + "]" + CR
						+ "update=[" + update + "]" + CR
						+ "table=[" + table + "]" + CR
						+ "nameLine=[" + data.nameLine() + "]" ;
			throw new RuntimeException( errMsg,ex );
		}
	}

	/**
	 * 画面出力用のフォーマットを作成します。
	 *
	 * @param	rowNo  int データ読み取り件数
	 * @param	updCnt int 更新件数
	 * @param	data   LineModel
	 */
	private void printKey( final int rowNo , final int updCnt , final LineModel data ) {
		StringBuilder buf = new StringBuilder();

		if( updCnt > 0 ) { buf.append( "UPDATE " ); }
		else 			 { buf.append( "INSERT " ); }

		buf.append( "row=[" ).append( rowNo ).append( "] : " );
		for( int i=0; i < updClmNos.length; i++ ) {
			if( i == 0 ) { buf.append( "key: " ); }
			else         { buf.append( " and " );  }
			buf.append( data.getName( updClmNos[i] ) );
			buf.append( " = " );
			buf.append( data.getValue( updClmNos[i] ) );
		}

		println( buf.toString() );
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "DBID         : " + dbid + CR
				+ TAB + "Input  Count : " + count + CR
				+ TAB + "Update Count : " + updCount + CR
				+ TAB + "Insert Count : " + insCount ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	String
	 */
	public String usage() {
		StringBuilder buf = new StringBuilder();

		buf.append( "Process_DBMerge は、UPDATE と INSERT を指定し データベースを追加更新"			).append( CR );
		buf.append( "する、ChainProcess インターフェースの実装クラスです。"							).append( CR );
		buf.append( "上流（プロセスチェインのデータは上流から下流へと渡されます。）から"			).append( CR );
		buf.append( "受け取った LineModel を元に、データベースの存在チェックを行い、"				).append( CR );
		buf.append( "下流への処理を振り分けます。"													).append( CR );
		buf.append( CR );
		buf.append( "データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に"			).append( CR );
		buf.append( "設定された接続(Connection)を使用します。"										).append( CR );
		buf.append( "-url_XXXX で指定された XXXX が、-dbid=XXXX に対応します。"						).append( CR );
		buf.append( CR );
		buf.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR );
		buf.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR );
		buf.append( "繋げてください。"																).append( CR );
		buf.append( CR );
		buf.append( "SQL文には、{@SYS.YMDH}等のシステム変数が使用できます。"						).append( CR );
		buf.append( "現時点では、{@SYS.YMD}、{@SYS.YMDH}、{@SYS.HMS} が指定可能です。"				).append( CR );
		buf.append( CR ).append( CR );
		buf.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args String[]
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_DBMerge().usage() );
	}
}
