/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.util.Map ;
import java.util.HashMap ;
import java.util.Date ;
import java.util.StringTokenizer;
import java.util.Locale;
import java.text.DateFormat;
import java.text.SimpleDateFormat;

/**
 * SystemParameter は、{&#064;XXXX} 文字列を処理するクラスです。
 * このクラスでは、{&#064;XXXX} 文字列を別の文字列と置き換えることや、
 * 予め予約されている予約語 {&#064;SYS.XXXX} 文字列を置き換えます。
 * 通常の {&#064;XXXX} 文字列の置き換えは、キーと値のペアを、HybsEntry オブジェクトに
 * セットして、その配列を受け取って処理します。
 *
 * 以下の値はあらかじめ、動的に作成されます。
 * ・SYS.YMD       ８byte の今日のシステム日付(yyyyMMdd)
 * ・SYS.YMDH    １４byte の今日のシステム日時(yyyyMMddHHmmss)
 * ・SYS.HMS       ６byte の今日のシステム時間(HHmmss)
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class SystemParameter {

	private final Map<String,String> sysMap = new HashMap<String,String>();

	private final String orginal ;

	/**
	 *  {&#064;XXXX} の特殊文字を含む文字列を、置き換えます。
	 * 対象外の文字列は、そのまま、残されます。
	 *
	 * @param orginal String  変換する文字列(オリジナル)
	 */
	public SystemParameter( final String orginal ) {
		this.orginal = orginal;

		DateFormat formatter = new SimpleDateFormat( "yyyyMMddHHmmss",Locale.JAPAN );
		String ymdh = formatter.format( new Date() );

		sysMap.put( "SYS.YMD"  , ymdh.substring( 0,8 ) );	// "yyyyMMdd"
		sysMap.put( "SYS.YMDH" , ymdh );					// "yyyyMMddHHmmss"
		sysMap.put( "SYS.HMS"  , ymdh.substring( 8 ) );		// "HHmmss"
	}

	/**
	 *  {&#064;XXXX} の特殊文字を含む文字列を、置き換えます。
	 * 対象外の文字列は、そのまま、残されます。
	 *
	 * @param  entry   HybsEntry 置換文字列のキーと値のペアを管理しているEntryオブジェクトの配列
	 * @return String  置換後の文字列
	 */
	public String replace( final HybsEntry[] entry ) {

		if( orginal == null ) { return null; }
		int index = orginal.indexOf( "{@" );
		if( index < 0 ) { return orginal; }

		// HybsEntry[] データを、Mapにマッピングします。
		if( entry != null ) {
			int size   = entry.length;
			for( int i=0; i<size; i++ ) {
				sysMap.put( entry[i].getKey(),entry[i].getValue() );
			}
		}

		StringTokenizer st = new StringTokenizer( orginal,"{}" );
		StringBuilder rtn = new StringBuilder( orginal.length() );
		while( st.hasMoreTokens() ) {
			String parts = st.nextToken();
			if( parts.charAt(0) == '@' ) {
				String param = sysMap.get( parts.substring(1) );
				if( param == null ) { param = ""; }
				rtn.append( param );
			}
			else {
				rtn.append( parts );
			}
		}
		return rtn.toString();
	}
}
