/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import java.util.List;

import org.jfree.data.general.Dataset;
import org.jfree.data.general.PieDataset;
import org.jfree.data.general.ValueDataset;
import org.jfree.data.category.CategoryDataset;
import org.jfree.chart.plot.Plot;
import org.jfree.chart.plot.MultiplePiePlot;
import org.jfree.chart.plot.MeterPlot;
import org.jfree.chart.plot.PiePlot;
import org.jfree.chart.plot.PiePlot3D;
import org.jfree.chart.plot.RingPlot;
import org.jfree.chart.plot.SpiderWebPlot;
import org.jfree.chart.plot.ThermometerPlot;
import org.jfree.chart.labels.StandardCategoryToolTipGenerator;
import org.jfree.chart.labels.StandardPieToolTipGenerator;

/**
 * <p>ChartPlot_Pie は、ChartPlot インターフェースを継承した実体クラスです。
 * JFreeChart では、各種オブジェクトの組み合わせで、色々なグラフを作成できます。
 * チャートタイプが、複数種類存在するため、ここでは、特殊な方法として、各タイプ毎に
 * オブジェクトを構築しています。(ファクトリメソッド的な処理)
 *
 * @version  0.9.0	2007/06/21
 * @author	 Kazuhiko Hasegawa
 * @since	 JDK1.1,
 */
public class ChartPlot_Pie implements ChartPlot {

	/**
	 * Plot オブジェクトを取得します。
	 *
	 * Plot オブジェクト には、その種類の応じた、データセットやレンデラーを
	 * 設定する必要があります。
	 * また、複数のデータセットや、それに関係する属性情報も、設定する必要が
	 * あります。
	 * Plot は、JFreeChart オブジェクトにつき、一つ用意しなければなりません。
	 * チャート合成時でも、Plot は一つです。
	 *
	 * @param	create ChartCreate オブジェクト
	 * @return	Plot オブジェクト
	 */
	public Plot getPlot( final ChartCreate create ) {

		List<ChartDataset> datasetList = create.getDatasetList();
		ChartDataset chDataset = datasetList.get(0);

		Dataset dtset = chDataset.getDataset();

		// クリッカブル・マップ
		HybsURLGenerator urlGen = create.getURLGenerator();
//		boolean useToolTip = create.getUseToolTip();	// 4.3.1.0 (2008/08/09) ツールチップス利用フラグ
		boolean useToolTip = create.isUseToolTip();		// 4.9.9.9 (2009/08/07) メソッド名変更

		Plot plot = null;
		String type = chDataset.getChartType();
		if( "MultiplePie".equalsIgnoreCase( type ) ) {
			plot = new MultiplePiePlot();
			((MultiplePiePlot)plot).setDataset( (CategoryDataset)dtset );
		}
		else if( "Meter".equalsIgnoreCase( type ) ) {
			plot = new MeterPlot();
			((MeterPlot)plot).setDataset( (ValueDataset)dtset );
		}
		else if( "Pie".equalsIgnoreCase( type ) ) {
			plot = new PiePlot();
			((PiePlot)plot).setDataset( (PieDataset)dtset );
			if( urlGen != null ) {
				((PiePlot)plot).setURLGenerator( urlGen );
			}
			if( useToolTip ){	// 4.3.1.0 (2008/08/09) ツールチップスの利用
				((PiePlot)plot).setToolTipGenerator( new StandardPieToolTipGenerator() );
			}
		}
		else if( "Pie3D".equalsIgnoreCase( type ) ) {
			plot = new PiePlot3D();
			((PiePlot3D)plot).setDataset( (PieDataset)dtset );
			if( urlGen != null ) {
				((PiePlot)plot).setURLGenerator( urlGen );
			}
			if( useToolTip ){	// 4.3.1.0 (2008/08/09) ツールチップスの利用
				((PiePlot)plot).setToolTipGenerator( new StandardPieToolTipGenerator() );
			}
		}
		else if( "Ring".equalsIgnoreCase( type ) ) {
			plot = new RingPlot();
			((RingPlot)plot).setDataset( (PieDataset)dtset );
			if( urlGen != null ) {
				((PiePlot)plot).setURLGenerator( urlGen );
			}
			if( useToolTip ){	// 4.3.1.0 (2008/08/09) ツールチップスの利用
				((RingPlot)plot).setToolTipGenerator( new StandardPieToolTipGenerator() );
			}
		}
		else if( "SpiderWeb".equalsIgnoreCase( type ) ) {
			plot = new SpiderWebPlot();
			((SpiderWebPlot)plot).setDataset( (CategoryDataset)dtset );
			if( urlGen != null ) {
				((SpiderWebPlot)plot).setURLGenerator( urlGen );
			}
			if( useToolTip ){	// 4.3.1.0 (2008/08/09) ツールチップスの利用
				((SpiderWebPlot)plot).setToolTipGenerator( new StandardCategoryToolTipGenerator() );
			}
		}
		else if( "Thermometer".equalsIgnoreCase( type ) ) {
			plot = new ThermometerPlot();
			((ThermometerPlot)plot).setDataset( (ValueDataset)dtset );
		}

		return plot;
	}
}
