/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.TreeSet;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBColumnConfig;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelSorter;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.html.CrossMap;
import org.opengion.hayabusa.html.ViewCrossTableParam;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * クロス集計テーブル作成クラスです。
 *
 *   select dept.dname,emp.deptno,substrb(job,1,2) as X,job,mgr,sum(sal),count(*)
 *   from emp,dept
 *   where emp.deptno = dept.deptno
 *   group by dept.dname,emp.deptno,cube(job,mgr)
 *   order by emp.deptno,job,mgr;
 *
 *   HEAD1   ：ヘッダー。前段と同じデータは表示させない。
 *   HEAD2   ：キーブレイクさせるカラム。また、前段と同じデータは表示させない。
 *   HEAD3   ：キーブレイクさせないカラム。また、前段と同じデータでも表示させる。
 *   ROW     ：行データのヘッダーになるカラム
 *   COL     ：列データのヘッダーになるカラム。下記のSUM1,SUM2の両方のヘッダーになる。
 *   SUM1    ：列データの値になるカラム。
 *   SUM2    ：列データの値になるカラム。
 *
 *   SUMカラムの数、キーブレイクのカラム名、グループ化するカラム名を
 *   指定することで、これらのクロス集計結果の表示方法を指定します。
 *
 *   breakColumn    = "DEPTNO"             キーブレイクのカラム名
 *   noGroupColumns = "X"                  グループ化するカラム名
 *   sumNumber      = "2"                  SUMカラムの数
 *   cubeXColumn    = "JOB"                CUBE計算の１つ目（X）カラムを指定
 *   cubeYColumn    = "MGR"                CUBE計算の２つ目（Y）カラムを指定
 *   cubeSortType   = "NUMBER"             CUBE Y の列ヘッダーのソート方法を指定
 *   gokeiSortDir   = "false"              合計カラムのソート方向を指定(初期値：ソートしない)
 *   shokeiLabel    = "SHOKEI"             列小計のカラムに表示するラベルID
 *   gokeiLabel     = "GOKEI"              列合計のカラムに表示するラベルID
 *
 *   各カラムの属性（HEAD,SUM等）を認識する方法
 *
 *     HEAD1 HEAD2 HEAD3 ROW COL SUM1 SUM2 という並びを認識する方法は、
 *     多数の前提条件を利用して、出来るだけ少ないパラメータで自動認識
 *     させています。
 *     若干理解しにくいかもしれませんが、慣れてください。
 *
 *     前提条件：
 *       ROW,COL は、必ず１個ずつ存在する。
 *       HEAD群、ROW,COL,SUM群 という並びになっている。
 *       SUM群の数は、パラメータで指定する。
 *     計算方法：
 *       HEAD数=カラム数(7)-SUM数(2)-1(ROW,COL分） ＝ ４ 個 （0 ～ 3）
 *       ROWアドレス＝cubeXColumn 設定                       (3)      ※ アドレスは０から始まる為
 *       COLアドレス＝cubeYColumn 設定                       (4)
 *       SUMアドレス＝HEAD数＋１ ～ カラム数(7)-1            (5 ～ 6）
 *
 * @og.rev 3.5.4.0 (2003/11/25) 新規作成
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_HTMLCrossTable extends ViewForm_HTMLTable {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private String[] groupByData = null;
	private String[] groupByCls  = null;

	// 3.5.4.8 (2004/02/23) 機能改善
	private int		rowClmNo	= -1;			// ROWカラムのカラム番号
	private int		colClmNo	= -1;			// CLMカラムのカラム番号
	private int		headCount	= 0;			// HEADカラムの数
	private int		sumCount	= 1;			// 合計カラムの数
	private int		breakClmNo	= -1;			// ブレークするカラムのカラム番号
	private boolean[]	noGroupClm = null;		// グループ化する/しないのフラグ配列
	private String		shokeiLabel  = "小計";	// 列小計のカラムに表示するラベルID
	private String		gokeiLabel   = "合計";	// 列合計のカラムに表示するラベルID
	private String		gokeiSortDir = null;	// 列合計のカラムをソートする方向

	// 3.5.6.3 (2004/07/12) ソート方式[STRING,NUMBER,LOAD]
	private String		cubeSortType = "LOAD";

	private DBTableModel table2    = null;
	private boolean		 firstStep = true;

	private String[]	clmKeys = null;

	private String		noDisplayKeys	= null;	// 3.7.0.4 (2005/03/18)

	/**
	 * 初期化します。
	 * ここでは、内部で使用されているキャッシュをクリアし、
	 * 新しいモデル（DBTableModel）と言語（lang） を元に内部データを再構築します。
	 * ただし、設定情報は、以前の状態がそのままキープされています。
	 *
	 * @og.rev 3.5.4.8 (2004/02/23) paramInit メソッドで、初期化を行います。
	 * @og.rev 3.5.6.1 (2004/06/25) lang 言語コード 属性を削除します。
	 *
	 * @param	table DBTableModel
	 */
	public void init( final DBTableModel table ) {
		table2		= table;
		firstStep	= true;
	}

	/**
	 * 内容をクリア（初期化）します。
	 *
	 * @og.rev 3.5.6.3 (2004/07/12) cubeSortType , gokeiSortDir 属性を追加します。
	 * @og.rev 3.7.0.4 (2005/03/18) noDisplayKeys 属性を追加します。
	 * @og.rev 3.7.1.1 (2005/05/31) shokeiLabel,gokeiLabel の初期値変更
	 */
	public void clear() {
		super.clear();
		groupByData = null;
		groupByCls  = null;
		rowClmNo	= -1;			// ROWカラムのカラム番号
		colClmNo	= -1;			// CLMカラムのカラム番号
		headCount	= 0;			// HEADカラムの数
		sumCount	= 1;			// 合計カラムの数
		breakClmNo	= -1;			// ブレークするカラムのカラム番号
		noGroupClm	= null;			// グループ化する/しないのフラグ配列
		table2		= null;
		firstStep	= true;
		clmKeys		= null;
		shokeiLabel		= "小計";		// 列小計のカラムに表示するラベルID
		gokeiLabel		= "合計";		// 列合計のカラムに表示するラベルID
		cubeSortType = "LOAD";		// 3.5.6.3 (2004/07/12)
		gokeiSortDir = null;		// 3.5.6.3 (2004/07/12) 列合計のカラムをソートする方向
		noDisplayKeys	= null;			// 3.7.0.4 (2005/03/18)
	}

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo（表示開始位置）から、pageSize（表示件数）までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) No 欄そのものの作成判断ロジックを追加
	 * @og.rev 3.5.6.1 (2004/06/25) lang 言語コード 属性を削除します。
	 * @og.rev 3.5.6.4 (2004/07/16) ヘッダーとボディー部をJavaScriptで分離
	 * @og.rev 3.7.0.4 (2005/03/18) setNoDisplay メソッドを追加
	 * @og.rev 4.3.1.0 (2008/09/08) 編集行のみを表示する属性(isSkipNoEdit)追加
	 *
	 * @param  startNo	  表示開始位置
	 * @param  pageSize   表示件数
	 * @return	DBTableModel から作成された HTML文字列
	 */
	public String create( final int startNo, final int pageSize )  {
		if( firstStep ) {
			paramInit( table2 );
			super.init( makeCrossTable(table2) );
			super.setNoDisplay( noDisplayKeys ) ;		// 3.7.0.4 (2005/03/18)
			markerSet( this );		// 3.5.6.4 (2004/07/16)
			firstStep = false;
		}

		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)

		headerLine	 = null;

		int lastNo = getLastNo( startNo, pageSize );
		int blc = getBackLinkCount();
		String backData = null;

		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_LARGE );

		out.append( getCountForm( startNo,pageSize ) );
		out.append( getHeader() );

		String ckboxTD = "  <td class=\"" + ViewCrossTableParam.HEADER1 + "\">";

		out.append("<tbody>").append( HybsSystem.CR );
		int bgClrCnt = 0;
		boolean shokei;
		for( int row=startNo; row<lastNo; row++ ) {
//			if( isSkip( row ) ) { continue; }		// 3.5.3.1 (2003/10/31)
			if( isSkip( row ) || isSkipNoEdit( row ) ) { continue; } // 4.3.1.0 (2008/09/08)
			// キーブレイク時のヘッダー設定
			if( breakClmNo >= 0 ) {
				String val = getValue( row,breakClmNo );
				if( backData == null ) {	// キーブレイクの初期データ設定。
					backData = val;
				}
				else {
					if( ! backData.equals( val ) ) {
						backData = val;
						out.append( getHeadLine() );
					}
				}
			}
			// 小計ヘッダー時のクラス設定
			String val2 = getValue( row,rowClmNo );
			if( val2.length() == 0 ) {
				shokei = true;
				out.append("<tr class=\"").append( ViewCrossTableParam.SHOKEI ).append("\">");
			}
			else {
				shokei = false;
				out.append("<tr").append( getBgColorCycleClass( bgClrCnt++ ) ).append(">");
			}
			out.append( HybsSystem.CR );
			// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
			if( isNumberDisplay() ) {
				out.append( makeCheckbox( ckboxTD, row, blc ) ).append( HybsSystem.CR );
			}
			for(int column = 0; column < clmCnt; column++) {
				if( isColumnDisplay( column ) ) {
					if( column < headCount-1 ) {		// CUBEではない行ヘッダー部
						String val = getGroupData( column,getRendererValue(row,column) );
						out.append("  <td class=\"").append( groupByCls[column] ).append("\">");
						out.append( val );
					}
					else if( column == headCount-1 ) {	// ヘッダーの最後尾
						if( shokei ) {
							out.append("  <td class=\"").append( ViewCrossTableParam.SHOKEI ).append("\">");
							out.append( shokeiLabel );
						}
						else {
							if( breakClmNo > 0 ) {	// ヘッダーがある場合
								out.append("  <td class=\"").append( groupByCls[column-1] ).append("\">");
							}
							else {
								out.append("  <td class=\"").append( ViewCrossTableParam.HEADER1 ).append("\">");
							}
							out.append( getRendererValue(row,column) );
						}
					}
					else if( column >= clmCnt-sumCount ) {	// CUBEの最終カラム（列合計）
						out.append("  <td class=\"").append( ViewCrossTableParam.SHOKEI ).append("\">");
						out.append( getRendererValue(row,column) );
					}
					else {		// カラム SUM列
						out.append("  <td>");
						out.append( getRendererValue(row,column) );
					}
					out.append("</td>").append( HybsSystem.CR );
				}
			}
			out.append("</tr>").append( HybsSystem.CR );
		}
		out.append("</tbody>").append( HybsSystem.CR );
		out.append("</table>").append( HybsSystem.CR );

		out.append( getScrollBarEndDiv() );	// 3.8.0.3 (2005/07/15)
		return out.toString();
	}

	/**
	 * パラメータ内容を初期化します。
	 *
	 * @og.rev 3.5.4.8 (2004/02/23) 新規作成
	 * @og.rev 3.5.6.3 (2004/07/12) 列ヘッダーのソート方法を指定
	 *
	 * @param  table DBTableModel 入力もとの DBTableModel オブジェクト
	 */
	private void paramInit( final DBTableModel table ) {
		String breakColumn		= getParam( ViewCrossTableParam.BREAK_COLUMN_KEY     , null );
		String noGroupColumns	= getParam( ViewCrossTableParam.NO_GROUP_COLUMNS_KEY , null );
		String sumNumber		= getParam( ViewCrossTableParam.SUM_NUMBER_KEY       , null );
		shokeiLabel				= getParam( ViewCrossTableParam.SHOKEI_LABEL_KEY     , shokeiLabel );
		gokeiLabel				= getParam( ViewCrossTableParam.GOKEI_LABEL_KEY      , gokeiLabel );
		String cubeXColumn		= getParam( ViewCrossTableParam.CUBE_X_COLUMN_KEY    , null );	// CUBE計算の１つ目（X）カラムを指定
		String cubeYColumn		= getParam( ViewCrossTableParam.CUBE_Y_COLUMN_KEY    , null );	// CUBE計算の２つ目（Y）カラムを指定
		cubeSortType			= getParam( ViewCrossTableParam.CUBE_SORT_TYPE_KEY   , "LOAD" );	// 3.5.6.3 (2004/07/12)
		gokeiSortDir			= getParam( ViewCrossTableParam.GOKEI_SORT_DIR_KEY   , null );	// 3.5.6.3 (2004/07/12)

		if( sumNumber != null ) {
			sumCount = Integer.parseInt( sumNumber );
		}

		// HEAD数=カラム数-SUM数-1(COL分） ROW は、HEADに含みます。
		headCount = table.getColumnCount() - sumCount - 1;

		// 3.5.5.9 (2004/06/07)
		if( cubeXColumn != null ) {
			rowClmNo = table.getColumnNo( cubeXColumn );
		}
		else {
			rowClmNo = headCount-1;			// ROWカラムのカラム番号
		}

		// 3.5.5.9 (2004/06/07)
		if( cubeYColumn != null ) {
			colClmNo = table.getColumnNo( cubeYColumn );
		}
		else {
			colClmNo = headCount;		// CLMカラムのカラム番号
		}

		if( breakColumn != null ) {
			breakClmNo = table.getColumnNo( breakColumn );
		}

		groupByData = new String[headCount];
		groupByCls  = new String[headCount];
		Arrays.fill( groupByCls,ViewCrossTableParam.HEADER2 );		// 変であるが、最初に入れ替えが発生する為。

		noGroupClm    = new boolean[headCount];		// グループ化する/しないのフラグ配列
		Arrays.fill( noGroupClm,false );

		if( noGroupColumns != null ) {
			String[] gClms = StringUtil.csv2Array( noGroupColumns );
			for( int i=0; i<gClms.length; i++ ) {
				noGroupClm[table.getColumnNo( gClms[i] )] = true;
			}
		}

		if( ! "true".equalsIgnoreCase( gokeiSortDir ) &&
			! "false".equalsIgnoreCase( gokeiSortDir ) ) {
				gokeiSortDir = null;
		}
	}

	/**
	 * CUBEではない行ヘッダー部の値が前と同じならば、ゼロ文字列を返します。
	 *
	 * @param  clm int カラム番号
	 * @param  val String 比較する値
	 * @return	前と同じなら,""を、異なる場合は、引数の val を返します。
	 */
	private String getGroupData( final int clm,final String val ) {
		if( noGroupClm[clm] ) { return val; }

		if( val.equals( groupByData[clm] )) {
			return "";
		}
		else {
			groupByData[clm] = val;
			groupByCls[clm] = ( groupByCls[clm].equals( ViewCrossTableParam.HEADER1 )) ? ViewCrossTableParam.HEADER2 : ViewCrossTableParam.HEADER1 ;
			return val;
		}
	}

	/**
	 * 選択用のチェックボックスと行番号と変更タイプ（A,C,D)を表示します。
	 *
	 * @param  ckboxTD チェックボックスのタグ（マルチカラム時のrowspan対応）
	 * @param  row	 行番号
	 * @param  blc	 バックラインカウント（先頭へ戻るリンク間隔）
	 * @return	td タグで囲まれたチェックボックスのHTML文字列
	 */
	protected String makeCheckbox( final String ckboxTD,final int row,final int blc ) {
		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		out.append( ckboxTD ).append("</td>");
		out.append( ckboxTD ).append("</td>");
		out.append( ckboxTD );
		// 3.5.1.0 (2003/10/03) Noカラムに、numberType 属性を追加
		if( blc != 0 && (row+1) % blc == 0 ) {
			out.append( "<a href=\"#top\">" ).append( (row+1) ).append(  "</a>" );
		} else {
			out.append( (row+1) );
		}
		out.append("</td>");

		return out.toString();
	}

	/**
	 * ヘッダー繰り返し部を、getTableHead()メソッドから分離。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 実装をgetHeadLine( String thTag )に移動
	 * @og.rev 3.5.5.0 (2004/03/12) No 欄そのものの作成判断ロジックを追加
	 *
	 * @return	テーブルのタグ文字列
	 */
	protected String getHeadLine() {
		if( headerLine != null ) { return headerLine; }		// キャッシュを返す。

		String rowspan = "";
		if( sumCount > 1 ) { rowspan = " rowspan=\"2\""; }

		String thTag = "<th" + rowspan;

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		buf.append("<tr").append( rowspan ).append(" class=\"row_h\"").append(" >").append( HybsSystem.CR );

		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
			buf.append( thTag ).append(" colspan='3'>").append( getNumberHeader() ).append("</th>");
		}

		buf.append( HybsSystem.CR );
		// ヘッダー部分は、そのまま表示します。
		for(int column = 0; column < headCount; column++) {
			if( isColumnDisplay( column ) ) {
				buf.append( thTag ).append(">");
				buf.append( getColumnLabel(column) );
				buf.append("</th>").append( HybsSystem.CR );
			}
		}

		// ヘッダー部分（上段）は、カラム配列を利用します。
		String colspan = "";
		if( sumCount > 1 ) { colspan = " colspan='" + sumCount + "'"; }
		// 3.7.0.4 (2005/03/18) カラム配列は、カラム番号と別物
		for( int keyNo = 0; keyNo < clmKeys.length-1; keyNo++ ) {
			buf.append( "<th").append( colspan ).append( ">" );
			buf.append( clmKeys[keyNo] );
			buf.append("</th>").append( HybsSystem.CR );
		}

		// 3.5.6.2 (2004/07/05) makeCrossTable( DBTableModel table ) に移動
		// 最終カラム(列）がNULL(空文字列）の場合、列合計です。
		String temp = clmKeys[clmKeys.length-1];
		if( temp == null || temp.length() == 0 ) {
			temp = gokeiLabel;
		}
		buf.append( "<th").append( colspan ).append( ">" );
		buf.append( temp );
		buf.append("</th>").append( HybsSystem.CR );

		buf.append("</tr>").append( HybsSystem.CR );

		if( sumCount > 1 ) {
			buf.append("<tr").append(" class=\"row_h\"").append(" >").append( HybsSystem.CR );
			int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)
			for(int column = headCount; column < clmCnt; column++) {
				if( isColumnDisplay( column ) ) {
					buf.append( "<th>");
					buf.append( getColumnLabel(column) );
					buf.append("</th>").append( HybsSystem.CR );
				}
			}
			buf.append("</tr>").append( HybsSystem.CR );
		}

		headerLine = buf.toString();
		return headerLine;
	}

	/**
	 * クロス集計結果の DBTableModel オブジェクトを作成します。
	 *
	 * @og.rev 3.5.4.8 (2004/02/23) paramInit メソッドで、初期化を行います。
	 * @og.rev 3.5.6.3 (2004/07/12) 列ヘッダーのソート可否の指定を追加
	 * @og.rev 4.0.0.0 (2007/11/27) ヘッダーカラムのエディター、レンデラー適用対応
	 * @og.rev 4.3.5.7 (2008/03/22) ↑リソースが存在しない場合は、ラベルのみ入れ替え
	 *
	 * @param  table DBTableModel 入力もとの DBTableModel オブジェクト
	 * @return	DBTableModel
	 */
	private DBTableModel makeCrossTable( final DBTableModel table ) {
		Set<String> clmData = gatSortAlgorithmSet();

		// 列のキーとなるカラムの値を取得します。
		int rowCnt = table.getRowCount();		// 3.5.5.7 (2004/05/10)
		for( int row=0; row<rowCnt; row++ ) {
			String clm = table.getValue( row,colClmNo );
			if( clm.length() > 0 ) { clmData.add( clm ); }
		}
		// ゼロストリングは、合計行になりますので、最後に追加します。

		// 3.5.6.3 (2004/07/12) ゼロストリングは、合計行になりますので、最後に追加します。
		clmKeys = new String[ clmData.size() + 1 ];
		clmKeys = clmData.toArray( clmKeys ) ;

		clmKeys[clmKeys.length-1] = "" ;

		int numberOfColumns =  headCount + clmKeys.length * sumCount ;

		DBTableModel tableImpl = DBTableModelUtil.newDBTable();
		tableImpl.init( numberOfColumns );

		// ヘッダーカラム(ROWデータ含む）は、そのまま、設定します。
		for(int column=0; column<headCount; column++) {
			tableImpl.setDBColumn( column,table.getDBColumn(column) );
		}

		// 列情報は、合計値のカラム定義を使用します。
		DBColumn[] dbColumn = new DBColumn[sumCount];
		for( int i=0; i<sumCount; i++ ) {
			dbColumn[i] = table.getDBColumn(headCount + 1 + i);
		}

		// 列情報は、列の名前をカラムの値に変えて、合計カラム列のコピー情報を設定します。

		int sumId = 0;
		ResourceManager resource = getResourceManager();

		// 列情報カラムは、ヘッダー分に割り当てられる為、開始が、headCount からになります。
		for(int column=headCount; column<numberOfColumns; column++) {
			// 4.0.0.0 (2007/11/27)
			if( resource != null && "true".equalsIgnoreCase( getParam( "useHeaderColumn", null ) ) ) {
				String clmKey = clmKeys[ (column-headCount)/sumCount ];
				DBColumn dbClm = resource.getDBColumn( clmKey );
				if( dbClm == null ) {
//					tableImpl.setDBColumn( column,dbColumn[sumId] ); }
					DBColumnConfig dbCfg2 = dbColumn[sumId].getConfig();
					dbCfg2.setLabelData( resource.getLabelData( clmKey ) );
					tableImpl.setDBColumn( column, new DBColumn( dbCfg2 ) );
				}
				else { tableImpl.setDBColumn( column,dbClm ); }
			}
			else {
				tableImpl.setDBColumn( column,dbColumn[sumId] );
			}
			sumId++;
			if( sumId % sumCount == 0 ) {
				sumId = 0;
			}
		}

		// クロス集計データの作成
		CrossMap cross = new CrossMap( clmKeys,headCount,sumCount );
		for( int row=0; row<rowCnt; row++ ) {
			String[] data = table.getValues( row );
			cross.add( data );
		}

		// データ部の設定
		int size = cross.getSize();
		for( int row=0; row<size; row++ ) {
			tableImpl.addValues( cross.get( row ), row );
		}

		tableImpl.resetModify();

		final DBTableModel model ;
		if( gokeiSortDir != null ) {
			DBTableModelSorter temp = new DBTableModelSorter();
			temp.setModel( tableImpl );

			boolean direction = Boolean.valueOf( gokeiSortDir ).booleanValue();
			temp.sortByColumn( numberOfColumns-1,direction );
			model = temp ;
		}
		else {
			model = tableImpl;
		}
		return model ;
	}

	/**
	 * 列ヘッダーのソート方法に応じた、Set オブジェクトを返します。
	 * ここでは、NUMBER , STRING , LOAD の３種類用意しています。
	 *
	 * @og.rev 3.5.6.3 (2004/07/12) 新規作成
	 *
	 * @return	Set<String>
	 */
	private Set<String> gatSortAlgorithmSet() {
		final Set<String> rtnSet ;

		if( "LOAD".equalsIgnoreCase( cubeSortType ) ) {
			rtnSet = new LinkedHashSet<String>();
		}
		else if( "NUMBER".equalsIgnoreCase( cubeSortType ) ) {
			rtnSet = new TreeSet<String>( numberSort );
		}
		else if( "STRING".equalsIgnoreCase( cubeSortType ) ) {
			rtnSet = new TreeSet<String>();
		}
		else {
			String errMsg = "cubeSortType は、NUMBER,STRING,LOAD 以外指定できません。" +
							"  cubeSortType=[" + cubeSortType + "]";
			throw new HybsSystemException( errMsg );
		}

		return rtnSet ;
	}

	/**
	 * 表示不可カラム名を、カンマ区切りで与えます。
	 * 例："OYA,KO,HJO,SU,DYSET,DYUPD"
	 * null を与えた場合は,なにもしません。
	 *
	 * 注意：このクラスでは、DBTableModel を作り直すタイミングが、
	 * create メソッド実行時です。(パラメータの初期化が必要な為)
	 * よって、このメソッドは、初期が終了後に、再セットします。
	 *
	 * @og.rev 3.7.0.4 (2005/03/18) 新規作成
	 *
	 * @param	columnName String
	 */
	public void setNoDisplay( final String columnName ) {
		noDisplayKeys = columnName;
	}

	/**
	 * NUMBER ソート機能（整数限定） 内部クラス
	 * これは通常のソートではなく、ヘッダーに使うラベルのソートなので、
	 * 整数のみと限定します。実数の場合は、桁合わせ（小数点以下の桁数）
	 * されているという前提です。
	 *
	 * @og.rev 3.5.6.3 (2004/07/12) 新規作成
	 *
	 */
	private static final Comparator<String> numberSort = new NumberComparator();

	private static class NumberComparator implements Comparator<String>,Serializable {
		private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

		public int compare( final String s1, final String s2 ) {
			if( s1.length() > s2.length() )      { return 1;  }
			else if( s1.length() < s2.length() ) { return -1; }
			else {
				return s1.compareTo( s2 );
			}
		}
	}
}
