/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.mail;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.mail.MailTX;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;

public class MailManager_DB extends AbstractMailManager {

/**
 * パッチによるメール送信の実装クラスです。
 * 送信デーモンはパラメータテーブル（GE30）を監視して、新規のデータが登録されたら、
 * そのデータをパラメータとしてメール合成処理メソッドに渡して合成を行って送信します。
 * 最後に、処理結果を受取って、パラメータテーブルの状況フラグを送信済/送信エラーに更新します。
 * エラーが発生した場合、エラーテーブルにエラーメッセージを書き込みます。
 * 
 * @og.group メールモジュール
 *
 * @version  4.0
 * @author   Sen.Li
 * @since    JDK1.6
 */

	private final static String	selGE30	= "select  UNIQ,PTN_ID,FROM_ID,TO_ID,CC_ID,BCC_ID,HEADER,FOOTER"
										+",PARAM0,PARAM1,PARAM2,PARAM3,PARAM4,PARAM5,PARAM6,PARAM7,PARAM8,PARAM9"
										+",ATTACH1,ATTACH2,ATTACH3,ATTACH4,ATTACH5"
										+ " from GE30" 
										+ " where SYSTEM_ID =? and FGJ='1'";
	private final static String	insGE36	= "insert into GE36(PARA_KEY,ERRMSG,DYSET,USRSET,PRGUPD,SYSTEM_ID,FGJ)"
										+ " values(?,?,?,?,?,?,'1')";
	private final static String	updGE30	= "update GE30 set FGJ= ? where uniq = ? ";
	private final static String	SNED_OK	= "2";
	private final static String	SNED_NG	= "8";
	private final static int GE30_UNIQ		= 0 ;
	private final static int GE30_PTN_ID	= 1 ;
	private final static int GE30_FROM_ID	= 2 ;
	private final static int GE30_TO_ID		= 3 ;
	private final static int GE30_CC_ID		= 4 ;
	private final static int GE30_BCC_ID	= 5 ;
	private final static int GE30_HEADER	= 6 ;
	private final static int GE30_FOOTER	= 7 ;
	private final static int GE30_PARAM0	= 8 ;
	private final static int GE30_PARAM1	= 9 ;
	private final static int GE30_PARAM2	= 10 ;
	private final static int GE30_PARAM3	= 11 ;
	private final static int GE30_PARAM4	= 12 ;
	private final static int GE30_PARAM5	= 13 ;
	private final static int GE30_PARAM6	= 14 ;
	private final static int GE30_PARAM7	= 15 ;
	private final static int GE30_PARAM8	= 16 ;
	private final static int GE30_PARAM9	= 17 ;
	private final static int GE30_ATTACH1	= 18 ;
	private final static int GE30_ATTACH2	= 19 ;
	private final static int GE30_ATTACH3	= 20 ;
	private final static int GE30_ATTACH4	= 21 ;
	private final static int GE30_ATTACH5	= 22 ;
	private final static int GE36_PARA_KEY	= 0 ;
	private final static int GE36_ERRMSG	= 1 ;
	private final static int GE36_DYSET		= 2 ;
	private final static int GE36_USRSET	= 3 ;
	private final static int GE36_PRGUPD	= 4 ;
	private final static int GE36_SYSTEM_ID	= 5 ;
	private final List<String>	 errMsgList 	= new ArrayList<String>();

	/**
	 * バッチより呼出のメインメソッドです。
	 * パラメータテーブル（GE30）を監視します。
	 * 新規のデータが登録されたら、メール文を合成して送信を行います。
	 * エラーが発生した場合、エラーテーブルにエラーメッセージを書き込みます。
	 * 
	 * @param systemId String
	 */
	public void sendDBMail( final String systemId ){
		// パラメータテーブルよりバッチでセットしたデータを取得します。
		String[][] ge30datas = DBUtil.dbExecute( selGE30, new String[]{ systemId }, appInfo );

		int ge30Len = ge30datas.length;
		for( int i=0; i < ge30Len; i++ ) {
			String fgj = SNED_OK; 
			try {
				Map<String, String> initParam = makeParamMap( systemId, ge30datas[i] ); 
				create( initParam );
				send();								// 合成されたメール文書、宛先で送信処理を行います。
				errMsgList.addAll( getErrList() );
			}
			catch( RuntimeException rex ) {
				fgj = SNED_NG;
				errMsgList.add( "メール送信失敗しました。パラメータキー：" + ge30datas[i][GE30_UNIQ] + " " + rex.getMessage() );
			}
			finally {
				commitParamTable( ge30datas[i][GE30_UNIQ], fgj );
//				if ( errMsgList.size() > 0 ) {
				if ( ! errMsgList.isEmpty() ) {
					writeErrorTable( ge30datas[i][GE30_UNIQ], systemId, errMsgList );
					errMsgList.clear();
				}
			}
		}
	}

	/**
	 * パラメータテーブルに登録したデータをパラメータマップにセットします。
	 * 
	 * @param systemId String
	 * @param ge30Data String[]
	 * 
	 */
	private Map<String, String> makeParamMap( final String systemId, final String[] ge30Data ){
		Map<String,String>   paramMap = null;
		if( ( ge30Data != null ) && ( ge30Data.length > 0 ) ) {
			paramMap = new HashMap<String,String>();
			paramMap.put( "SYSTEM_ID", systemId    );
			paramMap.put( "PARAKEY", ge30Data[GE30_UNIQ]    );
			paramMap.put( "PTN_ID" , ge30Data[GE30_PTN_ID]  );
			paramMap.put( "FROM"   , ge30Data[GE30_FROM_ID] );
			paramMap.put( "TO"     , ge30Data[GE30_TO_ID]   );
			paramMap.put( "CC"     , ge30Data[GE30_CC_ID]   );
			paramMap.put( "BCC"    , ge30Data[GE30_BCC_ID]  );
			paramMap.put( "HEADER" , ge30Data[GE30_HEADER]  );
			paramMap.put( "FOOTER" , ge30Data[GE30_FOOTER]  );
			paramMap.put( "PARAM0" , ge30Data[GE30_PARAM0]  );
			paramMap.put( "PARAM1" , ge30Data[GE30_PARAM1]  );
			paramMap.put( "PARAM2" , ge30Data[GE30_PARAM2]  );
			paramMap.put( "PARAM3" , ge30Data[GE30_PARAM3]  );
			paramMap.put( "PARAM4" , ge30Data[GE30_PARAM4]  );
			paramMap.put( "PARAM5" , ge30Data[GE30_PARAM5]  );
			paramMap.put( "PARAM6" , ge30Data[GE30_PARAM6]  );
			paramMap.put( "PARAM7" , ge30Data[GE30_PARAM7]  );
			paramMap.put( "PARAM8" , ge30Data[GE30_PARAM8]  );
			paramMap.put( "PARAM9" , ge30Data[GE30_PARAM9]  );
			paramMap.put( "ATTACH1", ge30Data[GE30_ATTACH1] );
			paramMap.put( "ATTACH2", ge30Data[GE30_ATTACH2] );
			paramMap.put( "ATTACH3", ge30Data[GE30_ATTACH3] );
			paramMap.put( "ATTACH4", ge30Data[GE30_ATTACH4] );
			paramMap.put( "ATTACH5", ge30Data[GE30_ATTACH5] );
			paramMap.put( "DATE", HybsSystem.getDate("yyyy/MM/dd") );
			paramMap.put( "TIME", HybsSystem.getDate("HH:mm:ss") );
			paramMap.put( "LOGIN_USERID", "DAEMON" );
			paramMap.put( "LOGIN_USERNAME", "DAEMON" );
			paramMap.put( "PGID", "DAEMON" );
		}
		return paramMap;
	}

	/**
	 * 送信後、パラメータテーブルの状況フラグを更新します。
	 * 送信エラーなしの場合はフラグを’送信済(2)’、エラーの場合’送信エラー（8）’に更新します。
	 * 
	 * @param ykno String
	 * @param fgj String
	 */
	private void commitParamTable( final String uniq, final String fgj ){
		String[] updGE30Args = { fgj, uniq };
		DBUtil.dbExecute( updGE30, updGE30Args, appInfo );
	}

	/**
	 * エラーテーブルにエラーメッセージを登録します。
	 * 
	 * @og.rev 4.4.0.1 (2009/08/08) メール送信追加
	 * 
	 * @param paraKey String
	 * @param systemId String
	 * @param emList List<String> エラーメッセージリスト
	 *  
	 */
//	private void writeErrorTable( final String paraKey, final String systemId, final List<String> errMsgList ){
	private void writeErrorTable( final String paraKey, final String systemId, final List<String> emList ){
		String[] insGE36Args = new String[6];
		insGE36Args[GE36_PARA_KEY]	= paraKey;
		insGE36Args[GE36_DYSET] 	= HybsSystem.getDate( "yyyyMMddHHmmss" );
		insGE36Args[GE36_USRSET] 	= "DAEMON";
		insGE36Args[GE36_PRGUPD] 	= "DAEMON";
		insGE36Args[GE36_SYSTEM_ID] = systemId;
		for( int i=0; i< emList.size(); i++ ){
			insGE36Args[GE36_ERRMSG] = trim( emList.get( i ), 4000);
			DBUtil.dbExecute( insGE36, insGE36Args,appInfo );
		}

		sendMail( paraKey, systemId, emList ); // 4.4.0.1 (2009/08/08)
	}

	/**
	 * エラー情報のメール送信を行います。
	 * エラーメールは、システムパラメータ の COMMON_MAIL_SERVER(メールサーバー)と
	 * ERROR_MAIL_FROM_USER(エラーメール発信元)と、ERROR_MAIL_TO_USERS(エラーメール受信者)
	 * がすべて設定されている場合に、送信されます。
	 *
	 * @og.rev 4.4.0.1 (2009/08/08) 追加
	 *
	 * @param paraKey String メールキー
	 * @param systemId String システムID
	 * @param emList List<String> エラーメッセージリスト
	 */
//	private void sendMail( final String paraKey, final String systemId, final List<String> errMsgList ) {
	private void sendMail( final String paraKey, final String systemId, final List<String> emList ) {

		String   host = HybsSystem.sys( "COMMON_MAIL_SERVER" );
		String   from = HybsSystem.sys( "ERROR_MAIL_FROM_USER" );
		String[] to = StringUtil.csv2Array( HybsSystem.sys( "ERROR_MAIL_TO_USERS" ) );
		if( host != null && from != null && to.length > 0 ) {
			String subject = "SYSTEM_ID=[" + systemId + "] , PARA_KEY=[" + paraKey + "] , "
						   + "DMN_HOST=[" + HybsSystem.HOST_NAME + "]" ;
			StringBuilder inErrMsg = new StringBuilder();
			inErrMsg.append( emList.size() + "件のエラーがありました。" );
			inErrMsg.append( HybsSystem.CR );
			for( int i=0; i< emList.size(); i++ ){
				inErrMsg.append( i+1 );
				inErrMsg.append( "-----" );
				inErrMsg.append( HybsSystem.CR );
				inErrMsg.append( emList.get( i ) );
				inErrMsg.append( HybsSystem.CR );
			}
			try {
				MailTX tx = new MailTX( host );
				tx.setFrom( from );
				tx.setTo( to );
				tx.setSubject( "メールモジュール送信エラー：" + subject );
				tx.setMessage( inErrMsg.toString() );
				tx.sendmail();
			}
			catch( Throwable ex ) {
				String errMsg = "エラー時メール送信に失敗しました。" + HybsSystem.CR
							+ subject + HybsSystem.CR
							+ ex.getMessage();
				LogWriter.log( errMsg );
				LogWriter.log( ex );
			}
		}
	}
}
