/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.Closer ;

import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.apache.poi.hssf.record.RecordFormatException;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;
import org.apache.poi.hssf.usermodel.HSSFRichTextString;

import java.io.File;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.IOException;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.text.NumberFormat ;
import java.text.DecimalFormat ;
import java.util.Locale ;
import java.util.Iterator ;

/**
 * 【EXCEL取込】雛形EXCELシートの解析処理を行う為の、HSSFListener 拡張クラスです。
 * このオブジェクトは、HSSFRequest クラスの addListenerForAllRecords メソッドに渡す
 * HSSFListener インターフェースを実装しています。また、雛形EXCEL を処理後、ExcelLayout
 * 管理クラスを取得することが出来ます。
 *
 * @og.rev 3.8.0.0 (2005/06/07) 新規追加
 * @og.group 帳票システム
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ExcelDataPickup {

	private final ExcelLayout	layout	;
//	private final File			filename;

	private final InputStream	in;
	private final HSSFWorkbook	wb;
	private DateFormat   dateFormat	= null;
	private NumberFormat numFormat	= null;

	private final boolean	 debug;

	/**
	 * 雛形EXCELを処理済みのExcelLayoutオブジェクトと、
	 * データEXCELファイル名よりオブジェクトを構築します。
	 *
	 * 内部で、HSSFWorkbook を構築します。
	 *
	 * @param layout ExcelLayout 雛形EXCELを処理済みのExcelLayoutオブジェクト
	 * @param filename File データEXCELファイル名
	 * @param debug デバッグフラグ
	 */
//	public ExcelDataPickup( final ExcelLayout layout,final File fname, final boolean debug ) {
	public ExcelDataPickup( final ExcelLayout layout,final File filename, final boolean debug ) {
		this.layout	= layout;
//		filename	= fname;
		this.debug	= debug;

		try {
			in = new FileInputStream( filename );

			POIFSFileSystem fs = new POIFSFileSystem( in );
			wb = new HSSFWorkbook( fs );
		}
		catch (IOException ex) {
			String errMsg = "ファイル読込みエラー[" + filename.getName() + "]"  ;
			throw new HybsSystemException( errMsg,ex );
		}
		catch (RecordFormatException ex) {
			String errMsg = "無効の形式/データが使用されています。[" + filename.getName() + "]"
					+ "現バージョンのPOIでは読み取ることが出来ません。"
					+ "例えば、自動フィルタの設定されたシートが含まれる場合などです。" ;
			throw new HybsSystemException( errMsg,ex );
		}
	}

	/**
	 * データEXCELファイル名のシート数を返します。
	 *
	 * 内部で、HSSFWorkbook を構築します。
	 *
	 * @return int シート数
	 */
	public int getSheetSize() {
		return wb.getNumberOfSheets();
	}

	/**
	 * データEXCELファイル名のデータをピックアップします。
	 *
	 * この処理を行うと、ExcelLayout オブジェクトにデータEXCEL解析結果を
	 * 保存しますので、処理側でその結果を取り出して使用します。
	 *
	 * @og.rev 3.8.1.1 (2005/11/21) デバッグ用コメント修正
	 *
	 * @param modelSheetNo int 雛形シート番号
	 * @param sheetNo int データシート番号
	 * @param loopClm String 繰返必須カラム(なければ通常の１対１処理)
	 */
	public void execute( final int modelSheetNo, final int sheetNo, final String loopClm ) {

		HSSFSheet sheet = wb.getSheetAt( sheetNo );
		layout.dataClear();

		if( debug ) { System.out.println( sheetNo + ":" + wb.getSheetName( sheetNo ) ); }

		Iterator ite = layout.getLayoutDataIterator( modelSheetNo,loopClm ) ;
		while( ite.hasNext() ) {
			ExcelLayoutData data = (ExcelLayoutData)ite.next();
			String clm  = data.getClm();
			int    edbn = data.getEdbn();
			int    row  = data.getRowNo();
			short  col  = data.getColNo();

	//		if( clm.indexOf( debugClm ) >= 0 ) { debug = true; }
	//		else { debug = false; }

			String val = getValue( sheet,row,col );
			layout.addData( clm,edbn,val );
			if( debug ) { System.out.println( data.toString() + "=[" + val + "]" ); }
		}
	}

	/**
	 * シートオブジェクト（HSSFSheet）から行列番号を指定して値を取り出します。
	 * 行オブジェクト(HSSFRow)や列オブジェクト(HSSFCell)が存在しない場合は、nullを返します。
	 * それ以外は、各セルタイプに応じて、処理を行います。
	 * 関数タイプ(HSSFCell.CELL_TYPE_FORMULA)の場合は、まず、文字列として取り出し、ゼロ文字列の
	 * 場合は、数字タイプとして取り出します。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) 結果を rtrim(右スペース削除)します。
	 * @og.rev 3.8.1.1 (2005/11/21) デバッグ用コメント修正
	 * @og.rev 3.9.0.5 (2008/11/27) POI3.2対応 引数の型変更(short->int)
	 *
	 * @param oCell HSSFCell EXCELのセルオブジェクト
	 * @return String セルの値
	 */
//	private String getValue( final HSSFSheet sheet,final int row, final short col ) {
	private String getValue( final HSSFSheet sheet,final int row, final int col ) {

		HSSFRow  oRow  = sheet.getRow(row);
		if( oRow == null ) { return null; }
		HSSFCell oCell = oRow.getCell(col);
		if( oCell == null ) { return null; }

		String strText = "";
		HSSFRichTextString richText ;
		int nCellType = oCell.getCellType();
		switch( nCellType ) {
			case HSSFCell.CELL_TYPE_NUMERIC:
					strText = getNumericTypeString( oCell );
					break;
			case HSSFCell.CELL_TYPE_STRING:
	// POI3.0		strText = oCell.getStringCellValue();
					richText = oCell.getRichStringCellValue();
					strText =  richText.getString();
					if( debug ) { System.out.print( "String :" ); }
					break;
			case HSSFCell.CELL_TYPE_FORMULA:
	//				strText = oCell.getCellFormula();
	// POI3.0		strText = oCell.getStringCellValue();
					richText = oCell.getRichStringCellValue();
					strText =  richText.getString();
					if( strText == null || strText.length() == 0 ) {
						strText = getNumericTypeString( oCell );
					}
					else {
						if( debug ) { System.out.print( "Formula:" ); }
					}
					break;
			case HSSFCell.CELL_TYPE_BOOLEAN:
					strText = String.valueOf(oCell.getBooleanCellValue());
					if( debug ) { System.out.print( "Boolean:" ); }
					break;
			case HSSFCell.CELL_TYPE_BLANK :
					if( debug ) { System.out.print( "Blank  :" ); }
					break;
			case HSSFCell.CELL_TYPE_ERROR :
					if( debug ) { System.out.print( "Error  :" ); }
					break;
			default :
					if( debug ) { System.out.print( "Other " + nCellType + ":" ); }
				break;
		}

		return StringUtil.rTrim( strText );		// 3.8.0.9 (2005/10/17)
	}

	/**
	 * セル値が数字の場合に、数字か日付かを判断して、対応する文字列を返します。
	 *
	 * @og.rev 3.8.1.1 (2005/11/21) デバッグ用コメント修正
	 *
	 * @param oCell HSSFCell
	 * @return String 数字の場合は、文字列に変換した結果を、日付の場合は、"yyyyMMddHHmmss" 形式で返します。
	 */
	private String getNumericTypeString( final HSSFCell oCell ) {
		final String strText ;

		double dval = oCell.getNumericCellValue() ;
		if( HSSFDateUtil.isCellDateFormatted( oCell ) ) {
			if( dateFormat == null ) {
				dateFormat = new SimpleDateFormat( "yyyyMMddHHmmss",Locale.JAPAN );
			}
			strText = dateFormat.format( HSSFDateUtil.getJavaDate( dval ) );
			if( debug ) { System.out.print( "Date   :" ); }
		}
		else {
			// 3.8.0.9 (2005/10/17) 数字について、NumberFormat を行います。
			if( numFormat == null ) {
				numFormat = NumberFormat.getInstance();
				if( numFormat instanceof DecimalFormat ) {
					((DecimalFormat)numFormat).applyPattern( "#.####" );
				}
			}
			strText = numFormat.format( dval );
			if( debug ) { System.out.print( "Numeric:" ); }
		}
		return strText ;
	}

	/**
	 * EXCEL をオープンした InputStream を閉じます。
	 */
	public void close() {
		Closer.ioClose( in );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
	}
}
