/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.hayabusa.db.DBErrMsg;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.common.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.Closer;

import oracle.jdbc.OracleTypes;
import oracle.jdbc.OracleCallableStatement;

import oracle.sql.ARRAY;
import oracle.sql.ArrayDescriptor;

import java.sql.Types;
import java.sql.Connection;
import java.sql.CallableStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import java.util.Map;

/**
 * エントリ系 PL/SQL をコールして、結果カーソルから、DBTableModel を作成します。
 *
 * java.sql.CallableStatement を用いて、データベース検索処理を行います。
 * 引数を配列指定で渡すことが出来，エラー時には、DBErrMsg オブジェクトにエラー情報を
 * 格納して返すことが可能です。
 * 内部変数の受け渡しのデフォルト実装は、AbstractQuery クラスを継承している
 * ため,ここでは、execute() メソッドを実装しています。
 * このクラスでは、ステートメント文を execute() する事により,データベースを
 * 検索した結果を DBTableModel に割り当てます。
 *
 * @og.formSample
 * 例：jsp/TYPESB/result.jsp  （検索系：カーソル返し）
 * 例：jsp/TYPE3B/entry.jsp   （エントリ系）
 *     names には、ARG_ARRAY 配列に順次セットされます。
 *     使用する場合は、一旦わかり易い変数に受けて利用してください。
 *     呼び出す PL/SQL では、検索系PL/SQL です。（下記の例は、エントリ系）
 *
 * &lt;h:query
 *     command    = &quot;NEW&quot;
 *     names      = &quot;SYSTEM_ID,LANG,CLM,NAME_JA,LABEL_NAME,KBSAKU,USER.ID&quot;
 *     queryType  = &quot;JDBCErrMsg&quot;
 *     displayMsg = &quot;&quot; &gt;
 *    &lt;jsp:text&gt;
 *        {call TYPE3B01.TYPE3B01(?,?,?,?)}
 *     &lt;/jsp:text&gt;
 * &lt;/h:query&gt;
 *
 *    CREATE OR REPLACE PACKAGE TYPE3B01 AS
 *        TYPE CUST_CURSOR IS REF CURSOR;
 *        PROCEDURE TYPE3B01(
 *            P_KEKKA      OUT    NUMBER,
 *            P_ERRMSGS    OUT    ERR_MSG_ARRAY,
 *            P_RC1        OUT    CUST_CURSOR,
 *            P_ARGS       IN     ARG_ARRAY     );
 *    END;
 *
 *     P_SYSTEM_ID     GEA08.SYSTEM_ID%TYPE     := P_ARGS(1);    --システムＩＤ
 *     P_LANG          GEA08.LANG%TYPE          := P_ARGS(2);    --言語
 *     P_CLM           GEA08.CLM%TYPE           := P_ARGS(3);    --項目
 *     P_NAME_JA       GEA08.NAME_JA%TYPE       := P_ARGS(4);    --名称（漢字）
 *     P_LABEL_NAME    GEA08.LABEL_NAME%TYPE    := P_ARGS(5);    --表示名称
 *     P_KBSAKU        GEA08.KBSAKU%TYPE        := P_ARGS(6);    --作成区分
 *     P_USRSET        GEA08.USRSET%TYPE        := P_ARGS(7);    --登録者
 *
 * @og.group データ表示
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Query_JDBCErrMsg extends AbstractQuery {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * クエリーを実行します。
	 * 実行方法等は各サブクラスの実装に依存します。
	 * セットされているステートメント文字列とそのタイプが合っていない場合は,
	 * エラーになります。
	 * 実行結果は、DBTableModel にセットされます。
	 * 実行結果の件数は #getExecuteCount() で取得できます。
	 *
	 */
	public void execute() {
		execute( null );
	}

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、CallableStatement を用いて、データベース検索処理を行います。
	 * {call TYPE3B01.TYPE3B01(?,?,?,?)} で、４番目の引数には、
	 * names で指定したリクエスト情報が、ARG_ARRAY 配列に順次セットされます。
	 * 使用する場合は、一旦わかり易い変数に受けて利用してください。
	 * 呼び出す PL/SQL では、検索系PL/SQL です。
	 *
	 * @og.rev 2.3.1.3 (2003/01/28) Open Cursor が、大量に残る件の対応。ResultSet を close()
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 3.3.3.1 (2003/07/18) ＤＢ登録時の後ろスペースを削除する。
	 * @og.rev 3.5.2.0 (2003/10/20) 内部オブジェクトタイプ名を システムパラメータ で定義します。
	 * @og.rev 3.5.6.0 (2004/06/18) nullに対する無駄な比較を削除します。
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 *
	 * @param   args オブジェクトの引数配列
	 */
	public void execute( final String[] args ) {
		CallableStatement callStmt = null ;
		ResultSet resultSet = null ;
		try {
			Connection conn = getConnection();
			callStmt  = getConnection().prepareCall( getStatement() );
			callStmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
			Map<String,Class<?>> map = conn.getTypeMap();
			map.put( ERR_MSG,DBErrMsg.class );	// 4.0.0 (2005/01/31)

			ArrayDescriptor sd = ArrayDescriptor.createDescriptor( ARG_ARRAY, conn );
			ARRAY newArray = new ARRAY( sd,conn,StringUtil.rTrims( args ) );

			callStmt.registerOutParameter(1, Types.INTEGER);
			callStmt.registerOutParameter(2, OracleTypes.ARRAY,ERR_MSG_ARRAY);
			callStmt.registerOutParameter(3, OracleTypes.CURSOR);
			((OracleCallableStatement)callStmt).setARRAY( 4,newArray );

			callStmt.execute();

			int rtnCode = callStmt.getInt(1);
			setErrorCode( rtnCode );
			if( rtnCode < ErrorMessage.NG ) {		// 異常以外の場合
				resultSet = ((OracleCallableStatement)callStmt).getCursor(3);
				createTableModel( resultSet );
			}
			if( rtnCode > ErrorMessage.OK ) {		// 正常以外の場合
				ARRAY rtn3 = ((OracleCallableStatement)callStmt).getARRAY(2);
				Object[] rtnval3 = (Object[])rtn3.getArray();
				ErrorMessage errMessage = new ErrorMessage( "Query_JDBCErrMsg Error!!" );
				for( int i=0; i<rtnval3.length; i++ ) {
					DBErrMsg er = (DBErrMsg)rtnval3[i];
					if( er == null ) { break; }
					errMessage.addMessage( er.getErrMsg() );
				}
				setErrorMessage( errMessage );
			}
		}
		catch (SQLException ex) {
			setErrorCode( ErrorMessage.EXCEPTION );
			String errMsg = ex.getMessage() + ":" + ex.getSQLState() + HybsSystem.CR
						+ getStatement() + HybsSystem.CR;
			rollback();
			realClose();
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			Closer.resultClose( resultSet );
			Closer.stmtClose( callStmt );
		}
	}
}
