/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.fukurou.util.HybsEntry;

import java.io.PrintWriter;
import java.util.List;

/**
 * DBTableModel インターフェース のオブジェクトをPrintWriter を用いて出力する為の，共通インターフェースです。
 *
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface TableWriter {

	/** タブ項目区切り文字	 */
	String TAB_SEPARATOR = "\t";		// タブ項目区切り文字

	/** カンマ項目区切り文字	 */
	String CSV_SEPARATOR = ",";			// カンマ項目区切り文字  3.5.6.0 (2004/06/18)

	/**
	 * 数字タイプのタイプ判定用リスト
	 */
	String NUMBER_TYPE_LIST = "|S9|R|";				// 3.5.6.0 (2004/06/18)

	/**
	 * DBTableModel から 各形式のデータを作成して,PrintWriter に書き出します。
	 * このメソッドは、EXCEL 書き出し時に使用します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @see #isExcel()
	 */
	void writeDBTable() ;

	/**
	 * DBTableModel から 各形式のデータを作成して,PrintWriter に書き出します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に PrintWriter を受け取るように変更します。
	 *
	 * @param	writer PrintWriter
	 */
	void writeDBTable( final PrintWriter writer ) ;

	/**
	 * DBTableModel をセットします。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) lang 引数も同時に設定します。
	 *
	 * @param   table DBTableModel
	 */
	void setDBTableModel( final DBTableModel table, final String lang ) ;

	/**
	 * 内部の DBTableModel を返します。
	 *
	 * @return  DBTableModel
	 */
	DBTableModel getDBTableModel() ;

	/**
	 * DBTableModelの出力順をセットします。
	 * Label,Name,Size,Class,Data の各フィールドの頭文字のアルファベットで
	 * 出力順を設定します。
	 *
	 * @param   headerSequence 出力順 (LNSCD など）
	 */
	void setHeaderSequence( final String headerSequence ) ;

	/**
	 * DBTableModelの出力順を返します。
	 * Label,Name,Size,Class,Data の各フィールドの頭文字のアルファベットで
	 * 出力順を設定します。
	 *
	 * @return  出力順 (LNSCD など）
	 */
	String getHeaderSequence() ;

	/**
	 * データを書き込む場合の,区切り文字をセットします。
	 * なお，このメソッドは,サブクラスによっては,使用しない場合があります。
	 * もし，使用しないサブクラスを作成する場合は, UnsupportedOperationException
	 * を throw するように,サブクラスで実装して下さい。
	 *
	 * @param   separator 区切り文字
	 */
	void setSeparator( final String separator ) ;

	/**
	 * DBTableModelのデータとして書き込むときに、追加モードで書き込むかどうかを設定します。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   flag boolean
	 */
	void setAppend( final boolean flag ) ;

	/**
	 * DBTableModelのデータとして書き込むときに、追加モードで書き込むかどうかを取得します。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @return boolean
	 */
	boolean isAppend() ;

	/**
	 * DBTableModelのデータとして書き込むときのシート名を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheetName String
	 */
	void setSheetName( final String sheetName ) ;

	/**
	 * EXCEL雛型参考ファイルのシート名を設定します。
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * EXCELファイルを書き出す時に、雛型として参照するシート名を指定します。
	 * これにより、複数の形式の異なるデータを順次書き出したり(appendモードを併用)する
	 * ことや、シートを指定して新規にEXCELを作成する場合にフォームを設定する事が可能になります。
	 * デフォルトは、null（第一シート） です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @param   sheetName String
	 */
	void setRefSheetName( final String sheetName ) ;

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、シート名のセット、雛型参照ファイル名のセット、
	 * 書き込み元ファイルのFileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return boolean
	 */
	boolean isExcel() ;

	/**
	 * 出力先ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename 出力先ファイル名
	 */
	void setFilename( final String filename ) ;

	/**
	 * EXCEL雛型参考ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename EXCEL雛型参考ファイル名
	 */
	void setRefFilename( final String filename ) ;

	/**
	 * 読み取り元ファイルのエンコード文字列を指定します。
	 * ファイルは、BufferedReader で受け取る為、本来は、エンコードは不要ですが、
	 * 固定長ファイルの読み取り時のバイトコード分割時に、指定のエンコードで
	 * 分割する必要があります。（例えば、半角文字は、Shift_JIS では、１バイト）
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @param   enc ファイルのエンコード文字列
	 */
	void setEncode( final String enc ) ;

	/**
	 * 行番号情報を、出力する(true)/しない(false)を指定します。
	 *
	 * @og.tag
	 * 通常のフォーマットでは、各行の先頭に行番号を出力します。
	 * これは、#NAME 属性を使用する場合には、必ず出力する必要があります。
	 * (#NAME 属性は、読み取り時には、必須です。)
	 * この、先頭の行番号が不要な場合（つまり、他のシステムへのデータ出力、
	 * このシステムでは、#NAME 属性が出力されないため、読み込みできません。)
	 * この行番号を出力しないようにできます。
	 * デフォルトは、true（出力する） です。
	 *
	 * @og.rev 3.7.0.2 (2005/02/14) 新規追加
	 *
	 * @param   useNumber 行番号情報を、出力する(true)/しない(false)を指定
	 */
	void setUseNumber( final boolean useNumber ) ;

	/**
	 * パラメーターリストをセットします。
	 * 内部は、HybsEntry クラスを持っています。
	 * 引数が、null の場合は、何もしません。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @param   listParam List パラメーターリスト
	 */
	void setParam( final List<HybsEntry> listParam ) ;

	/**
	 * 出力先ファイルのカラム列を、外部（タグ）より指定します。
	 * ただし、指定のカラム名は、DBTableModel上に存在している必要があります。
	 *
	 * @og.rev 4.0.0 (2005/11/30) 新規追加
	 *
	 * @param   clms 出力先ファイルのカラム列（カンマ区切り文字）
	 */
	void setColumns( final String clms ) ;

	/**
	 * EXCEL出力時のデフォルトフォント名を設定します。<br />
	 * これは、EXCEL追加機能として実装されています。<br />
	 *
	 * EXCELファイルを書き出す時に、デフォルトフォント名を指定します。
	 * フォント名は、EXCELのフォント名をそのまま使用してください。<br />
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontName( String )
	 * に設定されます。
	 * デフォルトは、システムリソース の TABLE_WRITER_DEFAULT_FONT_NAME です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   fontName String
	 */
	void setFontName( String fontName ) ;

	/**
	 * EXCEL出力時のデフォルトフォントポイント数を設定します。<br />
	 * これは、EXCEL追加機能として実装されています。<br />
	 *
	 * EXCELファイルを書き出す時に、デフォルトポイント数を指定します。
	 * 内部的に、POI の org.apache.poi.hssf.usermodel.HSSFFont#setFontHeightInPoints( short )
	 * に設定されます。
	 * デフォルトは、システムリソース の TABLE_WRITER_DEFAULT_FONT_POINTS です。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規追加
	 *
	 * @param   point short
	 */
	void setFontPoint( short point ) ;
}
