/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;

/** 【表示】タブ形式のリンクを表示するタグです。
 *
 * このタグ形式の実態は、リンクのリストであるため、実の画面の表示はターゲットを指定して
 * 別フレームで行う必要があります。
 * 
 * タブの指定方法については、listType属性の指定により、クエリ発行(DB)により動的に生成する
 * パターンと、タグ指定(TAG)により、静的に生成するパターンがあります。
 * listType属性に何も指定されていない場合は、Body部分に記述された内容により、自動判定されます。
 * ("SELECT"で始まっている場合はDB、それ以外はTAGとして処理)
 *
 * ①listType属性が"DB"の場合
 *  検索された各カラムは、その順番により次の意味を持ちます。
 *  [第1カラム] タブの名前        : リンク時のキー情報、後述のopenTabName属性のキーとしても使用 ※必須
 *  [第2カラム] タブの表示名称    : タブの表示名称 指定がない場合は、第1カラムが表示名称となります。
 *  [第3カラム] タブのリンク先URL : タブのリンク先URL 指定がない場合は、href属性の値が適用されます。
 *  [第4カラム] タブのクラス属性  : 個別のタブに付加されるクラス属性 指定がない場合は、unselClass属性の値が適用されます。
 *  [第5カラム] タブのロールズ    : タブのロールズを指定します。ユーザーロールズと一致しない指定した場合は、タブが表示されなくなります。
 *  [第6カラム] タブの選択可否    : タブの選択可否を'true'or'false'で指定します。falseを指定した場合は、タブが表示されなくなります。
 *                                  (ロールズで選択不可になっている場合は、この値は無視されます)
 *  各カラムの値は[カラム名]=[値]の形で、リンク先のJSPに引数として渡されます。
 *  また、リンク先のJSPについては、href属性で指定します。
 *  
 * ②listType属性が"TAG"の場合
 *  tabListタグを記述し、個別にタブを定義します。
 *  制御可能な項目は、①DBの場合と同等です。
 *  タブの名前を定義するname属性は、tabListタグで必ず定義する必要があります。
 *  lbl属性が指定されていない場合は、name属性のラベル名称を取得します。
 *  タブのリンク先JSP及び、クラス属性については、tabListタグで指定がない場合、tabListタグの値が適用されます。
 * 
 * [共通設定]
 * 初期設定では、第1番目の"有効な"タブが自動的に開かれます。(="true")
 * 各タブの表示方法で、選択不可能なタブが存在している場合は、それらを読み飛ばした上で、"有効な"タブを
 * 検索します。
 * また、自動で開くタブは、openTabName属性で指定可能であり、これに変数を定義することで、
 * 画面リロード時も、開いていたタブを再度選択された状態で表示することが可能です。
 *
 * 選択したタブ及び非選択のタブの枠線や、背景色等を変更する場合は、custom.cssでクラスを定義し、
 * 変更して下さい。
 * 
 * タブの表示方向(水平方向 or 垂直方向)については、orientation属性で変更することが可能です。
 * (初期値は、水平方向)
 * 水平方向にした場合は、listCount属性により強制的に一定数のタブを表示する毎に、改行を挿入することができます。
 *
 * このタグを使用する場合は、headタグで必ずuseTabLink="true"を指定してJavaScriptをロードして下さい。
 *
 * <p>各属性は、{&#064;XXXX} 変数が使用できます。<br />
 * これは、ServletRequest から、XXXX をキーに値を取り出し,この変数に割り当てます。
 * つまり、このXXXXをキーにリクエストすれば、この変数に値をセットすることができます。</p>
 *
 * @og.formSample
 * ●形式：&lt;og:tabList href="…" … /&gt;
 * ●body：あり
 *
 * ●使用例
 *   ①DBからタブリストを取得する場合
 *    
 *    Body部分に記述されたクエリよりタブ一覧を生成します。
 *    
 *      &lt;og:tabLink
 *          listType        = "DB"                      タブの一覧をどこから取得するか
 *          href            = "result.jsp"              リンク先のJSP
 *          target          = "RESULT"                  リンクターゲット
 *          openTab         = "[true/false]"            タブ表示後にタブを自動で開く
 *          openTabName     = "{&#064;PN}               自動で開くタブの名前
 *          constKeys       = "KEY1"                    次画面に固定で渡すキー一覧
 *          constVals       = "{&#064;VAL1}"            次画面に固定で渡す値一覧
 *          listCount       = "10"                      1行辺りに表示するタブの数
 *          selClass        = "selTab"                  選択タブのクラス
 *          unselClass      = "unselTab"                非選択タブのクラス
 *          width           = "100px"                   タブリンクの幅
 *          height          = "50px"                    タブリンクの高さ
 *       &gt;
 *          &lt;jsp:text&gt;
 *               select PN,HINM,'tabClass','query.jsp','ADMIN','false' from XX01 where PN = '{&#064;PN}' order by PN
 *          &lt;/jsp:text&gt;
 *      &lt;/og:tabLink&gt;
 *      
 *   ②tabListタグからタブリストを生成する場合
 *    
 *    tabListタグよりタブ一覧を生成します。
 *    
 *      &lt;og:tabLink
 *          listType        = "DB"                      タブの一覧をどこから取得するか
 *          href            = "result.jsp"              リンク先のJSP
 *          target          = "RESULT"                  リンクターゲット
 *          openTab         = "[true/false]"            タブ表示後にタブを自動で開く
 *          openTabName     = "{&#064;PN}               自動で開くタブの名前
 *          constKeys       = "KEY1"                    次画面に固定で渡すキー一覧
 *          constVals       = "{&#064;VAL1}"            次画面に固定で渡す値一覧
 *          listCount       = "10"                      1行辺りに表示するタブの数
 *          selClass        = "selTab"                  選択タブのクラス
 *          unselClass      = "unselTab"                非選択タブのクラス
 *          width           = "100px"                   タブリンクの幅
 *          height          = "50px"                    タブリンクの高さ
 *       &gt;
 *          &lt;og:tabList name="TAB1" href="result1.jsp" keys="PN,CDK" vals="ABC,V" /&gt;
 *          &lt;og:tabList name="TAB2" href="result2.jsp" keys="PN,CDK" vals="BCD,W" /&gt;
 *          &lt;og:tabList name="TAB3" href="result3.jsp" keys="PN,CDK" vals="CDE,X" /&gt;
 *      &lt;/og:tabLink&gt;
 *
 * @version  0.9.0	2008/09/26
 * @author	 Nakamura
 * @since	 JDK1.4,
 */
public class TabLinkTag extends CommonTagSupport {
	private static final String		VERSION				= "4.3.3.0 (2008/09/26)";
	private static final long		serialVersionUID	= 4330;

	/** リストのulタグのclass属性 */
	private static final String		UL_TAG_START		= "<ul class=\"tabList\">";
	private static final String		UL_TAG_END			= "</ul>";
	
	/** タブ表示を入れ替えるためのJavaScript関数 */
	private static final String		CHANGE_TAB_SCRIPT	= "changeTab";
	private static final String		INITIAL_TAB_SCRIPT	= "initialTabSelect";
//		"<script type=\"text/javascript\">addEvent(window,\"load\", initialTabSelect);</script>";
	
	/** 自動で開くタブに付加されるID */
	private static final String		FIRST_TAB_ID		= "firstTab";
	
	/** リスト取得タイプのEnum */
	private static enum LIST_TYPE { AUTO, DB, TAG }; 
	
	/** 内部変数 */
	private String		query			= null;
	private transient List<TabData>		tabData		= new ArrayList<TabData>();

	/** タグで設定する属性 */
	private LIST_TYPE	type			= LIST_TYPE.AUTO;
	private String		href			= "result.jsp";
	private String		target			= "RESULT";
	private boolean		openTab			= true;
	private String		openTabName		= null;
	private String[]	constKeys		= null;
	private String[]	constVals		= null;
	private int			listCount		= 10;
	private String		selClass		= "selTab";
	private String		unselClass		= "unselTab";
	private boolean		isHorizontal	= true;
	private String		width			= "auto";
	private String		height			= "auto";
	
	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	int
	 */
	@Override
	public int doStartTag() {
		return( EVAL_BODY_BUFFERED );	// Body を評価する
	}
				
	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 * 
	 * @return int
	 */
	@Override
	public int doAfterBody() {
		query = getBodyString().trim();
		return ( SKIP_BODY );
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 * 
	 * @og.rev 4.3.5.0 (2008/02/01) 処理及び内部構造を大幅に見直し
	 *
	 * @return	int
	 */
	@Override
	public int doEndTag() {
		debugPrint();
		int rtnCode = EVAL_PAGE;

		// 種別の自動判定処理
		if( type == LIST_TYPE.AUTO ) {
			if( query == null || query.length() == 0 ) {
				type = LIST_TYPE.TAG;				
			}
			else {
				if( query.toUpperCase( Locale.JAPAN ).indexOf( "SELECT" ) >= 0 ) {
					type = LIST_TYPE.DB;
				}
				else {
					type = LIST_TYPE.TAG;
				}
			}
		}

		if( type == LIST_TYPE.DB ) {
			makeTabsFromQuery();
		}
		else if( type == LIST_TYPE.TAG ) {
			makeTabsFromTag();
		}

		// リンク一覧が何も設定されていない場合は、処理しない
//		if( tabData.size() > 0 ) {
		if( ! tabData.isEmpty() ) {
			makeTag();
		}

		return( rtnCode );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 */
	@Override
	protected void release2() {
		super.release2();
		query			= null;
		tabData			= new ArrayList<TabData>();
		type			= LIST_TYPE.AUTO;
		href			= "result.jsp";
		target			= "RESULT";
		openTab			= true;
		openTabName		= null;
		constKeys		= null;
		constVals		= null;
		listCount		= 10;
		selClass		= "selTab";
		unselClass		= "unselTab";
		isHorizontal	= true;
		width			= "auto";
		height			= "auto";
	}

	/**
	 * DBからタブリンクの一覧を作成します。
	 * DBTableModelが作成されない(行数が0の場合)は、リンク一覧は生成されません。
	 */
	private void makeTabsFromQuery() {
		DBTableModel table = DBTableModelUtil.makeDBTable( query, new String[0], getResource(), getApplicationInfo() );
		if( table == null || table.getRowCount() == 0 ) {
			return;
		}

		boolean isSetLabel = false;
		boolean isSetHref = false;
		boolean isSetClazz = false;
		boolean isSetRoles = false;
		boolean isSetVisible = false;
		if( table.getColumnCount() > 1 ) { isSetLabel = true; }
		if( table.getColumnCount() > 2 ) { isSetHref = true; }
		if( table.getColumnCount() > 3 ) { isSetClazz = true; }
		if( table.getColumnCount() > 4 ) { isSetRoles = true; }
		if( table.getColumnCount() > 5 ) { isSetVisible = true; }

		int rowCount = table.getRowCount();
		String key = table.getColumnName( 0 );
		for( int row=0; row<rowCount; row++ ) {
			String value	= table.getValue( row, 0 );
			String label	= ( isSetLabel ? StringUtil.nval( table.getValue( row, 1 ), value ) : value );
			String newHref	= ( isSetHref ? StringUtil.nval( table.getValue( row, 2 ), href ) : href );
			String clazz	= ( isSetClazz ? StringUtil.nval( table.getValue( row, 3 ), unselClass ) : unselClass );
			boolean visible	= ( isSetRoles ? getUser().isAccess( table.getValue( row, 4 ) ) : true );
			if( visible ) {
				visible		= ( isSetVisible ? Boolean.valueOf( table.getValue( row, 5 ) ) : true );
			}

			// 第1カラムのカラム名とその値はリンクの引数に含める
			newHref	= XHTMLTag.addUrlEncode( newHref, XHTMLTag.urlEncode( key, value ) );
			
			if( visible ) { // visible=falseの場合は表示しない
				tabData.add( new TabData( newHref, value, label, clazz, visible ) );
			}
		}
	}

	/**
	 * タブリストからタブリンクの一覧を作成します。
	 * (予めaddTagメソッドにより、リンク一覧が登録されているため、ここでは何も処理しません)
	 * 
	 * @see #addTag( String, String, String, String, boolean, String[], String[] )
	 */
	private void makeTabsFromTag() {
		// 何もありません。(PMD エラー回避)
	}

	/**
	 * 子タグであるタブリストタグからタブ情報をセットします。
	 * 
	 * @param hr 画面URL
	 * @param name タブの名前
	 * @param label タブの表示名称
	 * @param clz 非選択状態のタブに付加するclass名
	 * @param visible タブが選択可能(中身を表示できるかどうか)
	 * @param keys リンク先のJSPに渡すキー一覧
	 * @param vals リンク先のJSPに渡す値一覧
	 */
	protected void addTag( final String hr, final String name, final String label, final String clz
							,final boolean visible, final String[] keys, final String[] vals ) {
		String newHref = StringUtil.nval( hr, href );
		if( keys != null && keys.length > 0 ) {
			newHref = XHTMLTag.addUrlEncode( newHref, XHTMLTag.urlEncode( keys, vals ) );
		}

		if( visible ) { // visible=falseの場合は表示しない
			tabData.add( new TabData( newHref, name, StringUtil.nval( label, getLabel( name ) ), StringUtil.nval( clz, unselClass ), visible ) );
		}
	}

	/**
	 * リンク一覧からHTMLタグを作成します。
	 *
	 * @og.rev 5.0.2.0 (2009/11/01) openTab属性がfalseの場合でも、openTabNameに指定されたタブに色付けする。
	 */
	private void makeTag() {
		StringBuilder buf = new StringBuilder();
		buf.append( HybsSystem.CR );

		for( int idx=0; idx<tabData.size(); idx++ ) {
			TabData tab = tabData.get( idx );

			if( idx % listCount == 0 ) {
				if( idx > 0 ) {
					buf.append( UL_TAG_END ).append( HybsSystem.CR );
				}
				buf.append( UL_TAG_START ).append( HybsSystem.CR );
			}

			// openTabNameが定義されていない場合は、1番目の有効なタブを開く
			boolean isFirst = false;
			if( !isFirst && tab.visible ) {
				if( openTabName != null && openTabName.length() > 0 ) {
					if( openTabName.equals( tab.name ) ) {
						isFirst = true;
					}
				}
				else {
					isFirst = true;
				}
			}
			buf.append( tab.makeLiTag( idx, isFirst ) );
		}
		buf.append( UL_TAG_END ).append( HybsSystem.CR );

		// タブを自動で開くためのJavaScriptタグを発行
		// 5.0.2.0 (2009/11/01)
		if( openTab || ( openTabName != null && openTabName.length() > 0 ) ) {
			buf.append( "<script type=\"text/javascript\">addEvent(window,\"load\", function() { " + INITIAL_TAB_SCRIPT + "(" );
			if( openTab ) {
				buf.append( "true, ''" );
			}
			else {
				buf.append( "false, \'" +  selClass + "'" );
			}
			buf.append( "); } );</script>" );
		}

		jspPrint( buf.toString() );
	}

	/**
	 * Tabデータ を管理している 内部クラス
	 *
	 * タブの情報を管理するための簡易的な、内部クラスです。
	 */
	private final class TabData {
		// 引数として初期設定される変数
		private final String href;
		private final String name;
		private final String label;
		private final String clazz;
		// 現状の実装では、visible=falseは渡ってきませんが、将来的にdisableの状態で
		// 表示したい場合等に対応するため残しておきます。
		private final boolean visible;
		
		/**
		 * コンストラクタ
		 * 
		 * @param hr 画面URL
		 * @param nm タブの名前
		 * @param lbl タブの表示名称
		 * @param clz 非選択状態のタブに付加するclass名
		 * @param vsb タブが選択可能(中身を表示できるかどうか)
		 */
		public TabData( final String hr, final String nm, final String lbl, final String clz, final boolean vsb ) {
			href	= hr;
			name	= nm;
			label	= lbl;
			clazz	= clz;
			visible	= vsb;
		}

		/**
		 * liタグの部分の文字列を生成します。
		 * 
		 * @param idx 生成したタブのインデックス番号
		 * @param isFirst 始めの有効なタブかどうか
		 * @return liタグ文字列
		 */
		private String makeLiTag( final int idx, final boolean isFirst ) {
			StringBuilder buf = new StringBuilder();
			buf.append( "<li class=\"" ).append( clazz ).append( "\"" );
			buf.append( " style=\"" );
			buf.append( " width: " ).append( width ).append( ";" );
			buf.append( " height: " ).append( height ).append( ";" );
			// 水平方向の場合のみfloat:leftを付加し、回り込み(+解除)を行う。
			if( isHorizontal ) {
				buf.append( " float: left;" );
				if( idx % listCount == 0 ) {
					// 行が変わったタイミングで、テキストの折り返しをクリア＆左マージン(+10px)を空ける
					buf.append( " clear: left; margin-left: " );
//					buf.append( Math.round( idx/listCount ) * 10 ).append( "px;" );
					// 4.3.4.4 (2009/01/01) Math.roundを呼び出す意味がないため削除
					buf.append( (idx/listCount) * 10 ).append( "px;" );
				}
			}
			buf.append( " \"" );
			buf.append( ">" );
			buf.append( makeLinkTag( isFirst ) );
			buf.append( "</li>" ).append( HybsSystem.CR );

			return buf.toString();
		}

		/**
		 * aタグの部分の文字列を生成します。
		 * タブが選択不可能な状態の場合は、タブの表示文字列をそのまま返します。
		 * 
		 * @og.rev 4.3.6.4 戻るボタンがでない問題への対応
		 * 
		 * @param isFirst 始めの有効なタブかどうか
		 * @return liタグ文字列
		 */
		private String makeLinkTag( final boolean isFirst ) {
//			if( !visible ) { return label; }

			String newHref = XHTMLTag.addUrlEncode( href, XHTMLTag.urlEncode( constKeys, constVals ) );
			// 4.3.6.4 (2009/05/01)
			// タブ画面から遷移した時に、タブの読込により、画面IDが消えてしまい
			// 戻るボタンがでない不具合への対応
			newHref = XHTMLTag.addUrlEncode( newHref, "GAMENID=" + getGUIInfoAttri( "KEY" ) );
			TagBuffer tag = new TagBuffer( "a" );
			tag.add( "href", newHref );
			tag.add( "name", name );
			tag.add( "target", target );
			tag.add( "onClick", CHANGE_TAB_SCRIPT + "( this, \"" + selClass + "\" );" );
			if ( isFirst ) {
				tag.add( "id", FIRST_TAB_ID );
			}
			tag.setBody( label );
			
			return tag.makeTag();
		}
	}

	/**
	 * 【TAG】タブの一覧をどこから取得するかを指定します(初期値:AUTO)
	 *
	 * @og.tag
	 * タブの一覧をどこから取得するかを指定します。
	 * 現状の実装では、クエリを発行して一覧を生成する「DB」と、子タグである
	 * tabListタグを列挙してタブを定義する「TAG」が実装されています。
	 * 
	 * また、「AUTO」と指定した場合は、Body部分の内容に応じて自動的に判定されます。
	 * 初期値は、｢AUTO」です。
	 *
	 * @param	tp タブ一覧取得方法(「AUTO」)
	 */
	public void setListType( final String tp ) {
		String typeStr = nval( getRequestParameter( tp ), null );
		try {
			type = LIST_TYPE.valueOf( typeStr );
		}
		catch ( IllegalArgumentException ex ) {
			StringBuilder errBuf = new StringBuilder( 100 );
			errBuf.append( "listType は" );
			for ( LIST_TYPE obj : LIST_TYPE.values() ) {
				errBuf.append( ',' );
				errBuf.append( obj.name() );
			}
			errBuf.append( "から選んでください。" );
			throw new HybsSystemException( errBuf.toString(), ex );
		}
	}

	/**
	 * 【TAG】リンク先のJSPを指定します(初期値:result.jsp)。
	 *
	 * @og.tag
	 * リンク先のJSPを指定します。
	 * このタブリンクは、あくまで「タブの形をしたリンク」なので、
	 * target属性と合わせてセットする必要があります。
	 * 初期値は、｢result.jsp」です。
	 *
	 * @param	hr リンク先のJSP
	 */
	public void setHref( final String hr ) {
		href = nval( getRequestParameter( hr ), href );
	}

	/**
	 * 【TAG】リンクのターゲットを指定します(初期値:RESULT)。
	 *
	 * @og.tag
	 * リンクのターゲットを指定します。
	 * このタブリンクは、あくまで「タブの形をしたリンク」なので、
	 * target属性を設定し、別のフレームに実画面を表示するようにします。
	 * 初期値は、｢RESULT」です。
	 *
	 * @param	tgt リンクターゲット
	 */
	public void setTarget( final String tgt ) {
		target = nval( getRequestParameter( tgt ), target );
	}

	/**
	 * 【TAG】リンク表示にタブリンクを自動で開くかを指定します(初期値:true(開く))
	 *
	 * @og.tag
	 * リンク表示にタブリンクを自動で開くかを指定します。
	 * openTabName属性が指定されていない場合、自動で開くタブは
	 * 「1番目に表示されたタブリンク」です。
	 * 指定されている場合は、その名前を持つ「1番目」のタブが自動で開かれます。
	 * タブが選択不可能な状態の場合は、「1番目」の条件から除外されます。
	 * 初期値は、「true(開く)」です。
	 *
	 * @param	flag タブ表示後に自動でタブを開くか
	 */
	public void setOpenTab( final String flag ) {
		openTab = nval( getRequestParameter( flag ), openTab );
	}

	/**
	 * 【TAG】最初に開くタブリンクの名前を指定します。
	 *
	 * @og.tag
	 * 最初に開くタブリンクのキーを指定します。
	 *
	 * @param	name 最初に開くタブリンクの名前
	 */
	public void setOpenTabName( final String name ) {
		openTabName = nval( getRequestParameter( name ), openTabName );
	}

	/**
	 * 【TAG】次画面に渡す定数パラメーターのキーを指定します
	 *
	 * @og.tag
	 * 次画面に渡す定数パラメーターのキーを指定します。
	 * キーはカンマ区切りで複数指定が可能です。
	 * パラメーターの値は、constVals属性の数と一致している必要があります。
	 *
	 * @param	keys 定数パラメーターのキー
	 * @see		#setConstVals( String )
	 */
	public void setConstKeys( final String keys ) {
		constKeys = getCSVParameter( keys );
	}
	
	/**
	 * 【TAG】次画面に渡す定数パラメーターの値を指定します
	 *
	 * @og.tag
	 * 次画面に渡す定数パラメーターの値を指定します。
	 * 値はカンマ区切りで複数指定が可能です。
	 * パラメーターの値は、constKeys属性の数と一致している必要があります。
	 *
	 * @param	vals 定数パラメーターのキー
	 * @see		#setConstKeys( String )
	 */
	public void setConstVals( final String vals ) {
		constVals = getCSVParameter( vals );
	}
	
	/**
	 * 【TAG】1行辺りに表示するタブの数を指定します(初期値:10)
	 *
	 * @og.tag
	 * 1行辺りに表示するタブの数を指定します。
	 * 1行辺りのタブの数がこの設定を超えると、自動的に折り返します。
	 * また、折り返し毎に、左に10pxのマージンを設けます。
	 * 初期値は、10です。
	 * この属性は、orientationがHorizontal(水平方向)の場合のみ有効です。
	 *
	 * @param	cnt 1行辺りに表示するタブの数
	 */
	public void setListCount( final String cnt ) {
		listCount = nval( getRequestParameter( cnt ), listCount );
	}
	
	/**
	 * 【TAG】選択タブのクラスを指定します(初期値:selTab)
	 *
	 * @og.tag
	 * タブが選択されている状態にある場合の、タブ部分のクラス名を指定します。
	 * このクラス名を変更する場合は、そのクラスをcustom/custom.css等で再定義して下さい。
	 * 初期値は、selTabです。
	 *
	 * @param	cls 選択タブのクラス名
	 */
	public void setSelClass( final String cls ) {
		selClass = nval( getRequestParameter( cls ), selClass );
	}
	
	/**
	 * 【TAG】非選択タブのクラスを指定します(初期値:unselTab)
	 *
	 * @og.tag
	 * タブが選択されていない状態にある場合の、タブ部分のクラス名を指定します。
	 * このクラス名を変更する場合は、そのクラスをcustom/custom.css等で再定義して下さい。
	 * 初期値は、unselTabです。
	 *
	 * @param	cls 選択タブのクラス名
	 */
	public void setUnselClass( final String cls ) {
		unselClass = nval( getRequestParameter( cls ), unselClass );
	}

	/**
	 * 【TAG】タブの方向、横型(Horizontal)か縦型(Vertical)を指定します(初期値：横型)。
	 *
	 * @og.tag
	 * タブは、上にタブが並ぶ横型と左にタブが並ぶ縦型があります。
	 * この属性では、横型は、Horizontal 、縦型は、Vertical を指定します。
	 * 指定は、文字列の最初の一文字を見ているだけですので、HかVでも構いません。
	 * 
	 * 縦型(Vertical)にした場合、各タブ要素は、フレームサイズの幅に合わせて
	 * 最大で表示されます。幅を固定する場合は、width属性を指定して下さい。
	 * 
	 * 初期値は、横型(Horizontal) です。
	 *
	 * @param	ori タブの方向、横型(Horizontal)か縦型(Vertical)を指定
	 */
	public void setOrientation( final String ori ) {
		String ori2 = nval( getRequestParameter( ori ),null );
		if( ori2 != null && ori2.length() > 0 ) {
			char ch = ori2.toUpperCase(Locale.JAPAN).charAt( 0 );
			if( ch == 'H' ) { isHorizontal = true; }
			else if( ch == 'V' ) { isHorizontal = false; }
			else {
				String errMsg = "orientation の指定は、horizontal または、vertical です。";
				throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * 【TAG】タブリンクの幅を % 、px 、または "auto" で指定します。
	 *
	 * @og.tag
	 * 初期値は、"auto"(自動設定) です。
	 * autoの場合、横型表示では、文字の幅に合わせて自動的に調整され、
	 * 縦型表示の場合は、フレームサイズに合わせて拡大して表示されます。
	 *
	 * @param	wh String 幅 (% 、px 、または "auto" )
	 */
	public void setWidth( final String wh ) {
		width = nval( getRequestParameter( wh ),width );
	}

	/**
	 * 【TAG】タブの高さを、% 、px 、または "auto" で指定します
	 *
	 * @og.tag
	 * タブの高さを、% 、px 、または "auto" で指定します
	 * 初期値は、"auto"(自動設定) です。
	 *
	 * @param	ht String 高さ (% 、px 、または "auto" )
	 */
	public void setHeight( final String ht ) {
		height = nval( getRequestParameter( ht ),height );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @serialData
	 *
	 * @param strm ObjectOutputStream
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @serialData
	 *
	 * @param strm ObjectInputStream
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException, ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title(this.getClass().getName() )
		.println( "VERSION"       , VERSION )
		.println( "listType"      , type.toString() )
		.println( "href"          , href )
		.println( "target"        , target )
		.println( "openTab"       , openTab )
		.println( "openTabName"   , openTabName )
		.println( "constKeys"     , constKeys )
		.println( "constVals"     , constVals )
		.println( "listCount"     , listCount )
		.println( "selClass"      , selClass )
		.println( "unselClass"    , unselClass )
		.println( "isHorizontal"  , isHorizontal )
		.println( "width"         , width )
		.println( "height"        , height )
		.println( "Other...", getAttributes().getAttribute() ).fixForm().toString();
	}
}
