/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;

/**
 * カラムの編集パラメーターのＳＱＬ文の実行結果より、プルダウンメニューを作成して
 * 編集する場合に使用するエディタークラスです。
 *
 * 編集パラメータには、プルダウンメニューを作成するための、SQL文を記述します。
 * このSQL文は、select KEY,LABEL from xx ･･･ という構文で、KEY部分とLABEL部分が
 * 選択されます。現時点では、それ以降の検索結果は無視されますが、将来的な拡張が
 * あるため、検索は、KEYとLABELの２つにしてください。
 *
 * 各カラムの値(value値)に、AAA:BBB:CCC:DDD という値を設定できます。これは、
 * $1,$2,$3,$4 に割り当てなおして、QUERYを実行します。また、$1 は、本来の値として、
 * メニューの初期値設定等に使用します。上記の例では、AAA が値で、それ以降は、
 * 引数になります。
 * 又、$Cには自分自身のカラム名を割り当てます。
 * この機能を使用すれば、動的メニューを行ごとに条件を変えて作成することが
 * 可能になります。
 * 例：select KEY,LABEL from xx where KUBUN='$2' and CDK='$3'
 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
 * 変数は、""(ゼロ文字列)として、扱われます。<br />
 *
 * 編集パラメータに"SEQ"と記述することで正方向にしか選べないシークメニューを実現できます。
 * これにより、シーケンスにステータスを順に挙げていくような、プルダウンメニュー
 * を作成することが出来ます。（逆に戻れないメニュー）
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.2.3.0 (2003/06/06) 新規作成
 * @og.rev 3.4.0.1 (2003/09/03) ＤＢ検索をリアルタイムに変更。
 * @og.rev 4.3.6.0 (2009/04/01) eventColumn対応
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_DBMENU extends AbstractEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private final String query ;
	private final String dbid ;
	private final String lang ;				// 4.0.0 (2006/11/15)
	private final boolean addNoValue ;		// 3.5.5.7 (2004/05/10)
	private final boolean seqFlag ;			// 3.6.0.6 (2004/10/22)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.4.0.1 (2003/09/03) 初期値でQUERY文をキープする。
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 *
	 */
	public Editor_DBMENU() {
		// 4.3.4.4 (2009/01/01)
//		super();
		query	= null;
		dbid	= null;
		lang	= null;			// 4.0.0 (2006/11/15)
		addNoValue = false;		// 3.5.5.7 (2004/05/10)
		seqFlag    = false;		// 3.6.0.6 (2004/10/22)
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.3.1.1 (2003/07/03) name , attributes 属性を final にする。
	 * @og.rev 3.4.0.1 (2003/09/03) 継承の親元の変更に伴う実装の移動。
	 * @og.rev 3.5.5.7 (2004/05/10) addNoValue 属性を追加します。
	 * @og.rev 3.5.5.9 (2004/06/07) editorParam 属性が null の場合は、エラーとします。
	 * @og.rev 3.5.6.0 (2004/06/18) XHTMLTag の 内部配列 SELECT_KEY を隠蔽します。
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 *
	 * @param   clm DBColumn オブジェクト
	 */
	private Editor_DBMENU( final DBColumn clm ) {
	//	super( clm );
		name		= clm.getName();
		addNoValue	= clm.isAddNoValue() ;		// 3.5.5.7 (2004/05/10)
		query		= clm.getEditorParam();
		dbid		= clm.getDbid();
		lang		= clm.getLang();			// 4.0.0 (2006/11/15)
		seqFlag		= false;		// 3.6.0.6 (2004/10/22)
		eventColumn = clm.getEventColumn(); // 4.3.6.0 (2009/04/01)
		eventURL	= clm.getEventURL();	// 4.3.6.0 (2009/04/01)
		editor		= clm.getEditor();		// 4.3.6.0 (2009/04/01)

		// 3.5.5.9 (2004/06/07)
		if( query == null || query.length() == 0 ) {
			String errMsg = "DBMENU Editor では、編集パラメータは必須です。"
					+ " name=[" + name + "]" + HybsSystem.CR ;
			throw new HybsSystemException( errMsg );
		}

		String	disabled = clm.isWritable() ? null : "disabled" ;

		attributes = new Attributes();
		attributes.set( "disabled"	,disabled );

		attributes.addAttributes( clm.getEditorAttributes() );
		optAttr = attributes.get( "optionAttributes" );
		tagBuffer.add( XHTMLTag.selectAttri( attributes ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param   clm DBColumn オブジェクト
	 * @return  CellEditor オブジェクト
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_DBMENU( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。<br />
	 * 又、$Cには自分自身のカラム名を割り当てます。<br />
	 *
	 * @og.rev 3.4.0.1 (2003/09/03) リアルタイムで値を作成する様に変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 新規に Attributes オブジェクトを作成する方式を止めます。
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 *
	 * @param	value String
	 * @return	データの編集用文字列
	 */
	public String getValue( final String value ) {

		TagBuffer tag = new TagBuffer( "select" );
		tag.add( "name"    , name );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
			tag.add( "id"      , name ); // 4.3.6.0 (2009/04/01)
		}
		tag.add( tagBuffer.makeTag() );
		tag.add( optAttr );		// 3.5.5.8 (2004/05/20)

		tag = getOption( tag,value );
		
		//return tag.makeTag();
		return tag.makeTag() + createEventColumnJS( name, editor, -1, eventURL ); // 4.3.6.0 (2009/04/01)
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。<br />
	 * 又、$Cには自分自身のカラム名を割り当てます。<br />
	 *
	 * @og.rev 2.0.0.3 (2002/09/26) optionAttributes 属性に "$i" を使うとその行数に置き換る機能を追加。
	 * @og.rev 3.4.0.1 (2003/09/03) リアルタイムで値を作成する様に変更。
	 * @og.rev 3.5.5.0 (2004/03/12) 名前と行番号の区切り記号("__")を、HybsSystem.JOINT_STRING  に変更。
	 * @og.rev 3.5.5.5 (2004/04/23) 新規に Attributes オブジェクトを作成する方式を止めます。
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 *
	 * @param	row   int 行番号
	 * @param	value String
	 * @return	データ表示/編集用の文字列
	 */
	public String getValue( final int row,final String value ) {

		TagBuffer tag = new TagBuffer( "select" );
		String newName = name + HybsSystem.JOINT_STRING + row; // 4.3.6.0 (2009/04/01)
		//tag.add( "name"    , name + HybsSystem.JOINT_STRING + row );
		tag.add( "name", newName ); // 4.3.6.0 (2009/04/01)
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) { // 4.3.7.2 (2009/06/15)
			tag.add( "id"    , newName ); // 4.3.6.0 (2009/04/01)
		}
		tag.add( tagBuffer.makeTag() );
		tag.add( optAttr );		// 3.5.5.8 (2004/05/20)

		tag = getOption( tag,value );

		// return tag.makeTag();
		return tag.makeTag() + createEventColumnJS( name, editor, row, eventURL ); // 4.3.6.0 (2009/04/01)
	}

	/**
	 * 初期値が選択済みの 選択肢（オプション）をTagBuffer に反映します。
	 * このオプションは、引数の値を初期値とするオプションタグ作成し、TagBuffer
	 * に値を設定して返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。<br />
	 * 又、$Cには自分自身のカラム名を割り当てます。<br />
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) getOption( String value )の廃止を受けて、新規作成
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 4.0.0 (2006/11/15) SelectionFactory に lang 属性を追加します。
	 * @og.rev 4.3.4.0 (2008/12/01) $Cのカラム名置換えを追加
	 *
	 * @param	value  選択されている値
	 * @return	オプションタグ
	 */
	private TagBuffer getOption( final TagBuffer buf,final String value ) {

		// StringFormat format = new StringFormat( query,value);
		StringFormat format = new StringFormat( query, value, name ); // 4.3.4.0 (2008/12/01)
		String newQuery = format.format();
		String newValue = format.getValue();

		Selection selection = SelectionFactory.newDBSelection( newQuery, dbid, lang );
		boolean useMultiSelect = selection.useMultiSelect();
		if( useMultiSelect ) {
			buf.add( "onkeydown", "setKeySelect(this);" );
		}

		if( addNoValue ) {
			buf.setBody( Selection.NO_VALUE_OPTION + selection.getOption( newValue, seqFlag ) );
		}
		else {
			buf.setBody( selection.getOption( newValue, seqFlag ) );
		}

		return buf;
	}
}
