/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.tomcat;

import org.apache.catalina.servlets.DefaultServlet;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Tomcat の、org.apache.catalina.servlets.DefaultServlet を継承した、サーブレットです。
 * これは、スタティックオブジェクトを制御するサーブレットで、ヘッダーに、キャッシュコントロール
 * の制御を入れることで、クライアントにキャッシュさせるようにします。
 * パラメータ制御しても良いのですが、基本的に簡易版として、固定値で、処理を行うようにします。
 * 設定は、tomcat/conf の web.xml で、行います。
 *
 * 付加するHTTPヘッダーは Cache-Control:private, max-age=[キャッシュ時間] です。
 * キャッシュ時間は、<servlet>タグの<init-param>maxage</init-param>で指定します。(単位は秒)
 * 初期値は、86400秒(24時間)です。
 *
 * ※※ キャッシュ対象となるファイルは、エンジン標準として提供している
 *      common, image, menu フォルダのみです。
 *      その他の custom フォルダや各画面のファイルはキャッシュされません。
 * 
 *【設定】
 *  conf/web.xml の <servlet-name>default</servlet-name> の設定で、
 *  servlet-class クラスを、DefaultServlet から CacheDefaultServlet に
 *  置き換えます。
 *  具体的には、パッケージ名も含めた、
 *  <servlet-class>mis.pdm.hayabusa.servlet.CacheDefaultServlet</servlet-class>
 *  に、書き換えます。
 *
 *  init-param の debug を、0 以上 にすると、デバッグ情報が出力されますので
 *  指定のメソッドを通過したかどうか、判断できます。
 *  また、debug を 9 にすると、キャッシュをしなくなります。
 *
 * @mis.group Tomcatパッチ
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class CacheDefaultServlet extends DefaultServlet {

	private static final long serialVersionUID = 4353;

	private String cacheControlHeader = null;

	/**
	 * 初期処理を行います。
	 */
	public void init() throws ServletException {
		super.init();

		int maxage = 86400;
		if ( getServletConfig().getInitParameter( "maxage" ) != null ) {
			maxage = Integer.parseInt( getServletConfig().getInitParameter( "maxage" ) );
		}
		cacheControlHeader = "private, max-age=" + String.valueOf( maxage );
	}

	/**
	 * DefaultServlet のメソッドを継承しています。
	 *
	 * すべてのスタティックリクエストが、集約されるため、ここで設定します。
	 * 設定値は、直接指定しています。
	 *
	 * プロキシ等にはキャッシュさせず、一定時間のキャッシュ、リロード時は、再読み込みする。
	 *
	 * @param request The servlet request we are processing
	 * @param response The servlet response we are creating
	 * @param content Should the content be included?
	 *
	 * @exception IOException if an input/output error occurs
	 * @exception ServletException if a servlet-specified error occurs
	 */
	protected void serveResource( final HttpServletRequest request,
								  final HttpServletResponse response,
								  final boolean content)
							throws IOException, ServletException {

		String path = getRelativePath( request );
		if ( debug > 0 ) {
			log( "CacheDefaultServlet.serveResource:  " + path );
		}

		if ( debug != 9 ) {
			if ( path.indexOf( "/common/" ) >= 0 || path.indexOf( "/menu/" ) >= 0 || path.indexOf( "/image/" ) >= 0 ) {
				// プロキシ等にはキャッシュさせず、２４時間のキャッシュ、リロード時は、再読み込みする。
				response.setHeader( "Cache-Control", cacheControlHeader );
				response.setHeader( "Expires", null ); // for firefox
				response.setHeader( "Pragma", null ); // for firefox

				// Calendar rightNow = Calendar.getInstance();
				// rightNow.add( Calendar.DAY_OF_MONTH, 1 ); // 一日進める
				// response.setHeader( "Expires",rightNow.getTime().toString()
				// );
			}
		}

		super.serveResource( request,response,content );
	}
}
