/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.TagBuffer;

import static org.opengion.fukurou.util.StringUtil.nval ;
import org.opengion.fukurou.util.EnumType ;

import java.util.Locale ;

/**
 * 戻るボタン/戻るリンクを表示するHTML拡張タグです(通常はcustom/query_info.jspに組込み)。
 *
 * custom/query_info.jsp に組み込むことで、全てのquery.jspに組込む事ができます。
 * 戻るボタン/リンクを表示するかどうかの条件判定は、
 * 　条件1：自分自身へ戻る機能はサポートできません。
 * 　条件2：BACK_GAMENID が存在するか、または、gamenId で指示された場合のみ表示。
 * 　条件3：command="NEW" で、キャッシュに設定しておく必要がある。<br />
 *
 * @og.formSample
 * ●形式：&lt;og:backGamen keys="･･･" vals="･･･" &gt; ･･Body･･ &lt;/og:backGamen&gt;
 * ●body：あり
 *
 * ●使用例
 *    &lt;og:backGamen keys="displayMsg,clear" vals="MSG0065,true" &gt;
 *        &lt;img src="{&#064;SYS.JSP}/image/back.gif" alt="{&#064;BACK_GAMENID}" /&gt;
 *        &lt;og:message msg="MSG0049" /&gt;
 *    &lt;/og:backGamen&gt;
 *
 *  デフォルトの戻るリンクを表示します。
 *    &lt;og:backGamen /&gt;
 *
 *  戻るリンクに使用するラベルに msg 属性で指定のメッセージリソースを使用します。
 *    &lt;og:backGamen msg="MSG0049"  /&gt;
 *
 *  戻るボタンを表示します。
 *    &lt;og:backGamen type="button"  /&gt;
 *
 *  戻るボタンにヒストリーバックをボタンを表示します。
 *    &lt;og:backGamen type="historyBack"  /&gt;
 *
 *  戻るボタンに、ショートカットキー(Alt+R)を指定します。
 *    &lt;og:backGamen type="historyBack" msg="MSG0049" accesskey="R"  /&gt;
 *
 * @og.rev 3.1.8.0 (2003/05/16) 新規作成
 * @og.group 画面制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class BackGamenTag extends HTMLTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	// 4.0.0 (2005/05/31) JDK5.0 enum 対応
	/**
	 * type 属性として指定できる選択肢を定義します。
	 */
	private static final EnumType<String> CHECK_TYPE =
				new EnumType<String>( "ボタンのタイプ","link" )
					.append( "link"   ,"戻るリンクを作成します。" )
					.append( "relLink","戻るリンク(相対パス)を作成します。" )
					.append( "button" ,"戻るボタンを作成します。" )
					.append( "relButton","戻るボタン(相対パス)を作成します。" )
					.append( "historyBack","通常のヒストリバックボタンを作成します。" ) ;

	private String		type		= CHECK_TYPE.getDefault() ;
	private String		command		= "RENEW" ;
	private String		gamenId		= null;
	private String		target		= "CONTENTS";
	private String		accesskey	= "R";
	private String[]	keys		= null;
	private String[]	vals		= null;
	private String		body		= null;
	private String		backAddress	= null; // 4.0.1.0 (2007/12/18)

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	int
	 */
	@Override
	public int doStartTag() {
		return( EVAL_BODY_BUFFERED );	// Body を評価する。（ extends BodyTagSupport 時）
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return	int
	 */
	@Override
	public int doAfterBody() {
		body = getBodyString();

		return(SKIP_BODY);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 *
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		type		= CHECK_TYPE.getDefault() ;
		command		= "RENEW" ;
		gamenId		= null;
		target		= "CONTENTS";
		accesskey	= "R";
		keys		= null;
		vals		= null;
		body		= null;
		backAddress	= null;
	}

	/**
	 * 戻るボタン/戻るリンクを作成します。
	 *
	 * @og.rev 3.1.8.0 (2003/05/16) BACK_GAMENID のキャッシュの取り出し先を変更する。
	 * @og.rev 3.5.4.0 (2003/11/25) 履歴（history)オブジェクトのback() メソッドを利用した戻る機能を追加します。
	 * @og.rev 4.0.1.0 (2007/12/17) BackAddress対応
	 * @og.rev 5.0.2.0 (2009/11/01) 相対パスでの戻るリンク対応
	 * 
	 * @return  戻るボタン/戻るリンク
	 */
	protected String makeTag() {
		String rtn = "" ;

		if( gamenId == null ) {
			gamenId = getBackGamenId() ;
		}
		if( backAddress == null) { // 4.0.1.0 (2007/12/17)
			backAddress = getBackAddress();
		}

		// 4.0.0 (2005/05/31)
		if( "historyBack".equalsIgnoreCase( type ) ) {
			String hb = getRequest().getParameter( "historyBack" );
			if( "1".equals( hb ) ) {
				rtn = makeHistoryBackTag();
			}
		}
		else {
			if( checkCondition( gamenId ) ) {
				set( "target" ,nval( get( "target" ), target ) );

				// 5.0.2.0 (2009/11/01) 相対パスでの戻る対応
				if( "link".equalsIgnoreCase( type ) || "relLink".equalsIgnoreCase( type ) ) {
					rtn = makeLinkTag();
				}
				else if( "button".equalsIgnoreCase( type ) || "relButton".equalsIgnoreCase( type ) ) {
					rtn = makeButtonTag();	// 3.8.1.2 (2005/12/19)
				}
				else {
					String errMsg = "指定の type は、下記の範囲で指定してください。"
									+ "type=" + type + " : "
									+ CHECK_TYPE.toString();
					throw new HybsSystemException( errMsg );
				}
			}
		}

		return rtn ;
	}

	/**
	 * 戻るリンクを作成します。
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) 戻るアドレスを、GUIInfoより取得するように変更。
	 * @og.rev 4.0.0 (2005/01/31) GUIInfoの実アドレスのパラメータを考慮する。
	 * @og.rev 3.7.0.3 (2005/03/01) 戻る行番号を、BACK_ROW_KEYより取得し、SEL_ROW で渡す。
	 * @og.rev 4.0.1.0 (2007/12/17) 戻るアドレスをbackAddressを使うように変更
	 * @og.rev 4.0.2.1 (2007/12/27) リクエスト部からGAMENIDを外していたが、二段階戻る場合に不具合が出たため元に戻す
	 * @og.rev 5.0.2.0 (2009/11/01) 相対パスでの戻るリンク対応
	 *
	 * @return  戻るリンク
	 */
	private String makeLinkTag() {
		set( "body",getMsglbl( gamenId ) );
		String href = get( "href" );
		if( href == null) {
			// 3.5.5.0 (2004/03/12)
			// 5.0.2.0 (2009/11/01) 相対パスでの戻る対応
			if( backAddress == null || "relLink".equalsIgnoreCase( type ) ) {// 4.0.1.0 (2007/12/17) if文追加
				GUIInfo guiInfo = getGUIInfo( gamenId ); // 4.0.0 (2005/01/31)
				if( guiInfo == null ) { return ""; } // 見つからない場合は、アクセス不可
				String address = guiInfo.getRealAddress( "index.jsp" );
				href = getRequestParameter( address );
			}
			else { // 4.0.1.0 (2007/12/17)
				href = backAddress;
			}
		}

		// 3.7.0.3 (2005/03/01) 戻る行番号を、BACK_ROW_KEYより取得し、SEL_ROW で渡す。
		String rowkey = HybsSystem.BACK_ROW_KEY + getGUIInfoAttri( "KEY" );
		String rowVal = (String)getSessionAttribute( rowkey );

		String[] keys2 = new String[] { "command","GAMENID","SEL_ROW" };
		String[] vals2 = new String[] {  command , gamenId , rowVal   };
//		String[] keys2 = new String[] { "command", "SEL_ROW" }; // 4.0.1.0 (2007/12/18)→4.0.2.1(2007/12/27)GAMENIDを付けるように戻す
//		String[] vals2 = new String[] {  command , rowVal   }; // 4.0.1.0 (2007/12/18)→上と同様
		String urlEnc2 = XHTMLTag.urlEncode( keys2,vals2 );
		href = XHTMLTag.addUrlEncode( href,urlEnc2 );

		String urlEnc = XHTMLTag.urlEncode( keys,vals );
		href = XHTMLTag.addUrlEncode( href,urlEnc );

		set( "href",href );

		return XHTMLTag.link( getAttributes() ) ;
	}

	/**
	 * 戻るボタンのフォームを作成します。
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) 戻るアドレスを、GUIInfoより取得するように変更。
	 * @og.rev 3.5.5.5 (2004/04/23) hidden の出力に、XHTMLTag.hidden を使用します。
	 * @og.rev 3.8.1.2 (2005/12/19) メソッド名変更、inputタグ⇒buttonタグ変更
	 * @og.rev 4.0.1.0 (2007/12/17) BACK_ADDRESS対応
	 * @og.rev 5.0.2.0 (2009/11/01) 相対パスでの戻るリンク対応
	 *
	 * @return  戻るボタンのフォーム
	 */
	private String makeButtonTag() {

		String href = get( "href" );
		if( href == null ) {
			// 3.5.5.0 (2004/03/12)
			// 5.0.2.0 (2009/11/01) 相対パスでの戻る対応
			if( backAddress == null || "relButton".equalsIgnoreCase( type ) ) { // 4.0.1.0 (2007/12/18) if文化
				GUIInfo guiInfo = getGUIInfo( gamenId );
				String address = guiInfo.getRealAddress();
				set( "action", address + "index.jsp" );
			}
			else {
				set( "action", backAddress ); // 4.0.1.0 (2007/12/18)
			}
		}
		else {
			set( "action",href ) ;
		}

		StringBuilder body = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

	 	// 3.8.1.2 (2005/12/19) inputタグ⇒buttonタグへ変更。
		TagBuffer tag = new TagBuffer( "button" );
		tag.add( "type","submit" );
		tag.add( "accesskey",get( "accesskey" ) );
		tag.setBody( getMsglbl( gamenId ) );
		body.append( tag.makeTag() );
		body.append( HybsSystem.CR );

		// command を hidden で作成します。
		body.append( XHTMLTag.hidden( "command",command ) );	// hidden(name,value);
		body.append( HybsSystem.CR );

		// GAMENID を hidden で作成します。
		body.append( XHTMLTag.hidden( "GAMENID",gamenId ) );	// hidden(name,value);
		body.append( HybsSystem.CR );


		// keys,vals を hidden で作成します。
		if( keys != null ) {
			if( keys.length != vals.length ) {
				String errMsg = "キーとバリューの個数が異なります。"
							+ HybsSystem.CR
							+ "keys.length=[" + keys.length + "]  vals.length=[" + vals.length + "]";
				throw new HybsSystemException( errMsg );
			}

			for( int i=0; i<keys.length; i++ ) {
				body.append( XHTMLTag.hidden( keys[i],vals[i] ) );	// hidden(name,value);
				body.append( HybsSystem.CR );
			}
		}

		// 3.7.0.3 (2005/03/01) 戻る行番号を、BACK_ROW_KEYより取得し、SEL_ROW で渡す。
		String rowkey = HybsSystem.BACK_ROW_KEY + getGUIInfo( "GUI.KEY" );
		String rowVal = (String)getSessionAttribute( rowkey );
		// 3.7.0.5 (2005/04/11) null の時は、返さないように変更
		if( rowVal != null ) {
			body.append( XHTMLTag.hidden( "SEL_ROW",rowVal ) );	// hidden(name,value);
			body.append( HybsSystem.CR );
		}

		set( "body",body.toString() );

		return XHTMLTag.form( getAttributes() ) ;
	}

	/**
	 * 戻るリンク(historyBack)を作成します。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 履歴（history)オブジェクトのback() メソッドを利用した戻る機能を追加します。
	 *
	 * @return  戻るリンク
	 */
	private String makeHistoryBackTag() {
		StringBuilder link = new StringBuilder();

		link.append( "<a onClick=\"history.back()\">" );
		link.append( getMsglbl( gamenId ) );
		link.append( "</a>" );

		return link.toString() ;
	}

	/**
	 * BACK_GAMENID の値を取り出します。
	 *
	 * 値は、キャッシュではなく、session より HybsSystem.BACK_GAMENID_KEY をキーに
	 * 取り出します。
	 *
	 * @og.rev 3.1.8.0 (2003/05/16) BACK_GAMENID のキャッシュの取り出し先を変更する。
	 *
	 * @return  BACK_GAMENID の値
	 */
	private String getBackGamenId() {
		String key = HybsSystem.BACK_GAMENID_KEY + getGUIInfoAttri( "KEY" );	// 4.0.0 (2005/01/31)
		return (String)getSessionAttribute( key );
	}

	/**
	 * BACK_ADDRESS の値を取り出します。
	 *
	 * 値は、キャッシュではなく、session より HybsSystem.BACK_ADDRESS_KEY をキーに
	 * 取り出します。
	 *
	 * @og.rev 4.0.1.0 (2007/12/17) メソッド追加
	 *
	 * @return  BACK_GAMENID の値
	 */
	private String getBackAddress() {
		String key = HybsSystem.BACK_ADDRESS_KEY + getGUIInfoAttri( "KEY" );	// 4.0.0 (2005/01/31)
		return (String)getSessionAttribute( key );
	}


	/**
	 * 【TAG】ボタンのタイプ(button,link,relButton,relLink,historyBack)を指定します(初期値:link)。
	 *
	 * button と設定すると、戻るボタンに、link と設定すると、戻るリンクになります。
	 * historyBack は、IE等の戻る操作と同じで、JavaScriptのヒストリーバックを行います。
	 * また、relButton、relLinkとすると、それぞれ相対パス(画面IDから飛び先のアドレスを元に生成)
	 * で戻り先のアドレスが生成されます。
	 * 
	 * 初期値は、戻るリンク（link）です。
	 *
	 * @og.tag
	 * ここでは､button,historyBack,link,relButton,relLink のみを許可します。
	 * 
	 * @og.rev 5.0.2.0 (2009/11/01) 相対パスでの戻るリンク対応
	 *
	 * @param   tp ボタンのタイプ(button,link,historyBack)
	 */
	public void setType( final String tp ) {
		type = CHECK_TYPE.nval( tp );
	}

	/**
	 * 【TAG】(通常使いません)戻る時に指定する command を設定できます(初期値:RENEW)。
	 *
	 * @og.tag
	 * 通常は、RENEW で戻ります。
	 * （初期値は、RENEW なので設定不要です。）
	 *
	 * @param	cmd コマンド
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command );
		if( command != null ) { command = command.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】(通常使いません)戻り先の画面をセットします(初期値:BACK_GAMENID)。
	 *
	 * @og.tag
	 * 通常は、自動的に、BACK_GAMENID の値がセットされますが、
	 * 先祖に戻る（画面Ａ⇒画面Ｂ⇒画面Ｃ のときに、画面Ａに戻る）場合や、
	 * 別の画面に、進む場合に、直接指定します。
	 * （初期値は、来た画面：BACK_GAMENID ですので、設定不要です。）
	 * 値は、キャッシュではなく、session より HybsSystem.BACK_GAMENID_KEY をキーに
	 * 取り出します。
	 * これは、command="NEW" で、BACK_GAMENID リクエストが存在し、BACK_GAMENID と
	 * 自分自身の画面IDが異なる場合のみ、RequestCacheTag#backGamenIdSet メソッドで
	 * session に登録されます。
	 *
	 * @param   id 戻り先の画面ID
	 */
	public void setGamenId( final String id ) {
		gamenId = nval( getRequestParameter( id ),gamenId );
	}

	/**
	 * 【TAG】リンク先に渡すキーをCSV形式で複数指定します。
	 *
	 * @og.tag
	 * 戻る時に、検索時のキャッシュに指定した引数以外に指定したり、別の値に置き換えたり
	 * する場合のキーを設定できます。カンマ区切りで複数指定できます。
	 * vals 属性には、キーに対応する値を、設定してください。
	 * 例:<b>keys="displayMsg,clear"</b> vals="MSG0065,true"
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 *
	 * @param	key リンク先に渡すキー
	 * @see		#setVals( String )
	 */
	public void setKeys( final String key ) {
		keys = getCSVParameter( key );
	}

	/**
	 * 【TAG】keys属性に対応する値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * キーに設定した値を、カンマ区切り文字で複数して出来ます。
	 * 指定順序は、キーと同じにしておいて下さい。
	 * 例:keys="displayMsg,clear" <b>vals="MSG0065,true"</b>
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) CommonTagSupport#getCSVParameter を使用
	 *
	 * @param	val keys属性に対応する値
	 * @see		#setKeys( String )
	 */
	public void setVals( final String val ) {
		vals = getCSVParameter( val );
	}

	/**
	 * 【HTML】リンク先の表示ターゲットを指定します(初期値：CONTENTS)。
	 *
	 * @og.tag
	 * リンク先の文書を表示させるフレーム、またはウィンドウの名前を指定します。
	 *
	 * @param	tgt リンク先の表示ターゲット
	 */
	public void setTarget( final String tgt ) {
		set( "target",getRequestParameter( tgt ) );
	}

	/**
	 * 【HTML】リンク先のURLを指定します。
	 *
	 * @og.tag リンク先のURLを指定します。
	 *
	 * @param	href String
	 */
	public void setHref( final String href ) {
		set( "href",getRequestParameter( href ) );
	}

	/**
	 * メッセージラベル（msglbl)をセットします。
	 *
	 * メッセージラベルは、meg属性か、lbl属性で登録された値を、
	 * それぞれのリソースに応じて各言語に変換した結果を格納しています。
	 * meg属性 と lbl属性は、同時登録できません。
	 *
	 *
	 * @param	 gamenId String
	 * @return   メッセージラベル
	 */
	private String getMsglbl( final String gamenId ) {
		StringBuilder rtn = new StringBuilder();
		String msg = getMsglbl();

		if( msg != null ) {
			rtn.append( msg );
			set( "accesskey",nval( get( "accesskey" ),accesskey ) );
			rtn.append( "(" ).append( get( "accesskey" ) ).append( ")" ) ;
		}
		else if( body != null && body.length() > 0 ) {
			rtn.append( body );
		}
		else {
			GUIInfo guiInfo = getGUIInfo( gamenId );		// 4.0.0 (2005/01/31)
			if( guiInfo == null ) { return ""; }	// 見つからない場合は、アクセス不可

			rtn.append( guiInfo.getLongName() );
		}

		return rtn.toString() ;
	}

	/**
	 * 画面に戻るボタン/リンクを表示するかどうかを条件判定します。
	 * 引数の gamenId は、BACK_GAMENID の事で、このタグの属性定義で設定されて
	 * いない場合は、session より、BACK_GAMENID を取り出します。つまり、取り出す為には、
	 * command="NEW" で、キャッシュに設定しておく必要があります。
	 *
	 * 随時、条件を追加していきます。
	 *
	 * 条件1：自分自身へ戻る機能はサポートできません。
	 * <del>条件2：BACK_GAMENID が存在するか、または、gamenId で指示された場合のみ表示。</del>
	 * 条件3：command="NEW" で、キャッシュに設定しておく必要がある。
	 *
	 * @og.rev 3.5.5.0 (2004/03/12) デバッグ情報を出力するように機能追加
	 *
	 * @param	 gmnId String
	 * @return   判定結果
	 */
	private boolean checkCondition( final String gmnId ) {

		String thisGamen = getGUIInfoAttri( "KEY" );	// 4.0.0 (2005/01/31)
		boolean rtn = ( gmnId != null ) &&
						( gmnId.length() > 0 ) &&
						( ! gmnId.equals( thisGamen ) ) ;

		// 3.5.5.0 (2004/03/12)
		if( isDebug() ) {
			String cmd =pageContext.getRequest().getParameter( "command" );

			StringBuilder buf = new StringBuilder();
			buf.append( getDocumentLink() );		// 4.0.0 (2005/02/28)
			buf.append( "<pre>" );
			buf.append( "command  =[" ).append( cmd       ).append( "]" ).append( HybsSystem.CR );
			buf.append( "type     =[" ).append( type      ).append( "]" ).append( HybsSystem.CR );
			buf.append( "gamenId  =[" ).append( gmnId     ).append( "]" ).append( HybsSystem.CR );
			buf.append( "thisGamen=[" ).append( thisGamen ).append( "]" ).append( HybsSystem.CR );
			buf.append( "checkCondition=[" ).append( rtn  ).append( "]" );
			buf.append( "</pre>" );
			jspPrint( buf.toString() );
		}

		return rtn ;
	}
}
