/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.hayabusa.db.DBSysArg;
import org.opengion.hayabusa.db.DBUserArg;
import org.opengion.hayabusa.db.DBErrMsg;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;

import oracle.jdbc.OracleTypes;
import oracle.jdbc.OracleCallableStatement;

import oracle.sql.ARRAY;
import oracle.sql.ArrayDescriptor;

import java.sql.Types;
import java.sql.Connection;
import java.sql.CallableStatement;
import java.sql.SQLException;

import java.util.Map;

/**
 * PL/SQL をコールする 登録系 Queryクラスです。
 *
 * java.sql.CallableStatement を用いて、データベース検索処理を行います。
 * 引数に、SYSARG_ARRAYと、ユーザーARG_ARRAY を配列指定で渡すことが出来，
 * エラー時には、DBErrMsg オブジェクトにエラー情報を格納して返すことが可能です。
 * 内部変数の受け渡しのデフォルト実装は、AbstractQuery クラスを継承している
 * ため,ここでは、execute() メソッドを実装しています。
 * このクラスでは、ステートメント文を execute() する事により,データベースを
 * 検索した結果を DBTableModel に割り当てます。
 *
 * @og.formSample
 * 例：jsp/TYPE1B/result.jsp
 *     names には、GEA08ARG で定義したカラムを指定します。
 *     呼び出す PL/SQL では、登録系PL/SQL です。
 *
 * &lt;h:plsqlUpdate
 *     command    = &quot;{&#064;command}&quot;
 *     names      = &quot;SYSTEM_ID,LANG,CLM,NAME_JA,LABEL_NAME,KBSAKU,FGJ,USRSET&quot;
 *     dbType     = &quot;GEA08ARG&quot;
 *     queryType  = &quot;JDBCPLSQL&quot; &gt;
 *     &lt;jsp:text&gt;
 *            { call TYPE1B01.TYPE1B01( ?,?,?,?,? ) }
 *     &lt;/jsp:text&gt;
 * &lt;/h:plsqlUpdate&gt;
 *
 *	PROCEDURE TYPE1B01 (
 *         P_KEKKA     OUT   NUMBER,          -- エラー結果(0:正常 1:警告 2:異常)
 *         P_ERRMSGS   OUT   ERR_MSG_ARRAY,   -- エラーのあるときのエラーメッセージ配列
 *         P_NAMES     IN    VARCHAR2,
 *         P_SYSARGS   IN    SYSARG_ARRAY,    -- 引数 SYSTEMデータ
 *         P_GE08ARGS  IN    GEA08ARG_ARRAY   -- 引数 USERデータ
 *  );
 *
 * @og.group データ表示
 * @og.group データ編集
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Query_JDBCPLSQL extends AbstractQuery {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、PreparedQuery で使用する引数を配列でセットするものです。
	 * select * from emp where deptno = ? and job = ? などの PreparedQuery の
	 * ? 部分の引数を
	 * 順番にセットしていきます。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 * @og.rev 3.5.2.0 (2003/10/20) 内部オブジェクトタイプ名を システムパラメータ で定義します。
	 * @og.rev 3.5.6.0 (2004/06/18) nullに対する無駄な比較を削除します。
	 * @og.rev 3.8.0.8 (2005/10/03) エラーメッセージの出力順をメッセージ＋Queryに変更します。
	 * @og.rev 4.0.0 (2005/01/31) 引数をすべて受け取って実行するメソッドを標準メソッドとして追加
	 *
	 * @param names       String
	 * @param dbArrayType String
	 * @param sysArg    DBSysArg[]
	 * @param userArg   DBUserArg[]
	 */
	public void execute( final String names,final String dbArrayType,
							final DBSysArg[] sysArg,final DBUserArg[] userArg ) {
		CallableStatement callStmt = null ;
		try {
			Connection conn = getConnection();
			callStmt  = getConnection().prepareCall( getStatement() );
			callStmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
			Map<String,Class<?>> map = conn.getTypeMap();
			map.put( ERR_MSG,DBErrMsg.class );	// 4.0.0 (2005/01/31)

			ArrayDescriptor sd2 = ArrayDescriptor.createDescriptor( SYSARG_ARRAY, conn );
			ARRAY newArray2 = new ARRAY( sd2,conn,sysArg );

			ArrayDescriptor sd = ArrayDescriptor.createDescriptor( dbArrayType, conn );
			ARRAY newArray = new ARRAY( sd,conn,userArg );

			callStmt.registerOutParameter(1, Types.INTEGER);
			callStmt.registerOutParameter(2, OracleTypes.ARRAY,ERR_MSG_ARRAY);
			callStmt.setString( 3,names );
			((OracleCallableStatement)callStmt).setARRAY( 4,newArray2 );
			((OracleCallableStatement)callStmt).setARRAY( 5,newArray );

			callStmt.execute();

			int rtnCode = callStmt.getInt(1);
			setErrorCode( rtnCode );
			if( rtnCode > ErrorMessage.OK ) {		// 正常以外の場合
				ARRAY rtn3 = ((OracleCallableStatement)callStmt).getARRAY(2);
				Object[] rtnval3 = (Object[])rtn3.getArray();
				ErrorMessage errMessage = new ErrorMessage( "Query_JDBCPLSQL Error!!" );
				for( int i=0; i<rtnval3.length; i++ ) {
					DBErrMsg er = (DBErrMsg)rtnval3[i];
					if( er == null ) { break; }
					errMessage.addMessage( er.getErrMsg() );
				}
				setErrorMessage( errMessage );
			}
		}
		catch (SQLException ex) {
			setErrorCode( ErrorMessage.EXCEPTION );
			String errMsg = ex.getMessage() + ":" + ex.getSQLState() + HybsSystem.CR
						+ getStatement() + HybsSystem.CR;
			rollback();
			realClose();
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}
		finally {
			Closer.stmtClose( callStmt );
		}
	}
}
