/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.XHTMLTag;

/**
 * 検索結果から、テキストフィールドタグを自動生成する、テキストフィールド作成クラスです。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_HTMLTextField extends AbstractViewForm {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	// 4.0.0 (2005/01/31) HTML_LABEL_SEPARATOR を boolean 変数として取得します。
	private final String CLM = ( HybsSystem.sysBool( "HTML_LABEL_SEPARATOR" ) ) ? ":" : "" ;

	private String columnSpace = HybsSystem.sys( "HTML_COLUMS_SPACE" ) ;		// 項目間のスペース
	private int maxRowNumber   = HybsSystem.sysInt( "HTML_MAXROW_NUMBER" ) ;	// 縦方向の最大表示件数
	private final static int PAGE_SIZE = 1;

	// 4.3.4.4 (2009/01/01)
//	/**
//	 * デフォルトコンストラクター
//	 *
//	 */
//	public ViewForm_HTMLTextField() {
//		super();
//	}

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo（表示開始位置）から、pageSize（表示件数）までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 2.0.1.0 (2002/10/10) ラベルとフィールドのセパレーターとして、コロン（：）を使用するかどうかを指定できる
	 * @og.rev 3.6.0.5 (2004/10/18) 印刷時の罫線出力関連機能の追加。id 属性を出力します。
	 *
	 * @param  startNo	  表示開始位置
	 * @param  pageSize   表示件数
	 * @return	DBTableModel から作成された HTML文字列
	 */
	public String create( final int startNo, final int pageSize )  {
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		int numberOfColumns = getColumnDisplayCount() ;
		String[] label = new String[numberOfColumns];
		String[] value = new String[numberOfColumns];

		int realCount = getColumnCount();
		int clmNo = 0;
		for( int i=0; i<realCount; i++ ) {
			if( isColumnDisplay(i) ) {
				label[clmNo] = getColumnLabel(i);
				value[clmNo] = getValueLabel(startNo,i);
				clmNo++ ;
			}
		}

		int columnNumber = ( numberOfColumns / maxRowNumber ) + 1 ;

		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_LARGE );

		out.append( getCountForm( startNo,pageSize ) );
		out.append( makeSelectNo( startNo ) ).append( HybsSystem.CR );

		out.append("<table id=\"viewTextField\" border=\"0\" summary=\"ViewForm_HTMLTextField\" >");	// 3.6.0.5 (2004/10/18)
		int rowNumber = (numberOfColumns +1 )/ columnNumber ;
		for( int row=0; row<rowNumber; row++ ) {
			out.append("<tr>").append( HybsSystem.CR );
			for( int clm=0; clm<columnNumber; clm++ ) {
				int realClm = row + clm * rowNumber ;
				out.append("<td id=\"label\">");
				if( realClm < numberOfColumns ) {
					out.append( label[realClm] );
					out.append( CLM );
				}
				out.append("</td>").append( HybsSystem.CR );
				out.append("<td>");
				if( realClm < numberOfColumns ) { out.append( value[realClm] ); }
				out.append("</td>").append( HybsSystem.CR );
				out.append("<td width=\"").append( columnSpace ).append("\">　</td>").append( HybsSystem.CR );
			}
			out.append("</tr>").append( HybsSystem.CR );
		}
		out.append("</table>");

		return out.toString();
	}

	/**
	 * 画面に選択された番号を表示します。
	 * また、書き込み許可がある場合は、選択用の隠しフィールドを作成します。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) hidden の出力に、XHTMLTag.hidden を使用します。
	 *
	 * @param  row	 行番号
	 * @return	隠しフィールドのHTML文字列
	 */
	protected String makeSelectNo( final int row ) {
		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		if( isWritable( row ) ) {
			// XHTMLTag.hidden( name,value ) を使用。
			out.append( XHTMLTag.hidden( HybsSystem.ROW_SEL_KEY,String.valueOf( row ) ) );
		}

		return out.toString();
	}

	/**
	 * 内容をクリア（初期化）します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド（synchronized付き）を非同期に変更する。
	 *
	 */
	public void clear() {
		super.clear();
		columnSpace   = HybsSystem.sys( "HTML_COLUMS_SPACE" ) ;			// 項目間のスペース
		maxRowNumber  = HybsSystem.sysInt( "HTML_MAXROW_NUMBER" ) ;		// 縦方向の最大表示件数
	}

	/**
	 * 表示件数を取得します。
	 *
	 * @return	表示件数
	 */
	public int getPageSize() {
		return PAGE_SIZE;
	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return  使用可能（true)/ 使用不可能 (false)
	 */
	public boolean canUseFormat() {
		return false;
	}

	/**
	 * カラムのラベル名(長)を返します。
	 * カラムの項目名に対して,見える形の文字列を返します。
	 * 一般には,リソースバンドルと組合せて,各国ロケール毎にラベルを
	 * 切替えます。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加( longLabel を返します。)
	 *
	 * @param	column int
	 * @return	カラムのラベル名(長)
	 */
	protected String getColumnLabel( final int column ) {
		return getDBColumn( column ).getLongLabel();
	}
}
