/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * <p>TableFilter は、TableFilterTag で使用される DBTableModelの処理用のインターフェースです。<br />
 * 適切な実装（サブクラス）を TableFilterTag に与えることで，DBTableModelの処理を適宜
 * 切り替えることが出来ます。</p>
 *
 * @og.rev 3.8.5.0 (2006/03/20) 新規作成
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public interface TableFilter {

	/**
	 * DBTableModelの処理を実行します。<br />
	 * 処理自体は, 各サブクラスの実装に依存します。<br />
	 *
	 */
	DBTableModel execute() ;

	/**
	 * DBTableModel をセットします。 <br />
	 *
	 * @param	table DBTableModel
	 */
	void setDBTableModel( DBTableModel table ) ;

	/**
	 * データ処理の方法(A:追加 C:更新 D:削除)を指定します。
	 *
	 * 通常は、DBTableModel に自動設定されている modifyType を元に、データ処理方法を
	 * 選別します。(A:追加 C:更新 D:削除)
	 * この場合、行単位で modifyType の値を取得して判別する必要がありますが、一般には
	 * 処理対象は、全件おなじ modifyType である可能性が高いです。
	 * また、selectedAll などで強制的に全件処理対象とする場合は、modifyType に値が
	 * 設定さていません。その様な場合に外部より modifyType を指定します。
	 * デフォルトは、自動判定 です。</p>
	 *
	 * @param  type データ処理の方法(A:追加 C:更新 D:削除)
	 */
	void setModifyType( String type ) ;

	/**
	 * キーと値のペアの変数配列を受け取ります。
	 *
	 * @param   keys String[]
	 * @param   vals String[]
	 */
	void setKeysVals( String[] keys,String[] vals ) ;

	/**
	 * 表示データの HybsSystem.ROW_SEL_KEY を元に、選ばれた 行を
	 * 処理の対象とします。
	 *
	 * @param   rowNo int[]
	 */
	void setParameterRows( int[] rowNo ) ;

	/**
	 * アクセスログ取得の為,ApplicationInfo オブジェクトを設定します。 <br />
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) 新規追加
	 *
	 * @param   appInfo ApplicationInfo
	 */
	void setApplicationInfo( ApplicationInfo appInfo ) ;
	
	/**
	 * DBIDを指定します。<br />
	 * 
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 * 
	 * @param dbid DBID
	 */
	void setDbid( String dbid ) ;
	
	/**
	 * ボディー部分のSQLを指定します。<br />
	 * 
	 * @og.rev 4.2.4.0 (2008/06/23) 新規追加
	 * 
	 * @param sql ボディー部分のSQL
	 */
	void setSql( String sql ) ;

	
	/**
	 * リソースオブジェクトを指定します。<br />
	 * 
	 * @og.rev 4.3.7.4 (2009/07/01) 新規追加
	 * 
	 * @param resource リソースオブジェクト
	 */
	void setResource( ResourceManager resource ) ;

	/**
	 * デバッグ情報を表示するかどうかを指定します。
	 * true でデバッグ情報を表示します。
	 *
	 * @param   flag boolean
	 */
	void setDebug( boolean flag ) ;

	/**
	 * エラーコード を取得します。 <br />
	 * エラーコード は、ErrorMessage クラスで規定されているコードです。
	 *
	 * @return   エラーコード
	 */
	int getErrorCode() ;

	/**
	 * エラーメッセージオブジェクト を取得します。 <br />
	 *
	 * @return   エラーメッセージオブジェクト
	 */
	ErrorMessage getErrorMessage() ;
}
