/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.util.Argument;
import org.opengion.fukurou.util.SystemParameter;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.HybsEntry ;
import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.model.Formatter;
import org.opengion.fukurou.db.ConnectionFactory;

import java.util.Map ;
import java.util.LinkedHashMap ;
import java.util.Set ;
import java.util.HashSet ;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ParameterMetaData;
import java.sql.SQLException;

/**
 * Process_DBWriter は、上流から受け取ったデータをデータベースに書き込む
 * CainProcess インターフェースの実装クラスです。
 *
 * 上流（プロセスチェインのデータは上流から下流へと渡されます。）から受け取った
 * LineModel を元に、データベースへの書き込みを行います。
 *
 * データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に
 * 設定された接続(Connection)を使用します。
 * -url_XXXX で指定された XXXX が、-dbid=XXXX に対応します。
 *
 * 引数文字列中にスペースを含む場合は、ダブルコーテーション("") で括って下さい。
 * 引数文字列の 『=』の前後には、スペースは挟めません。必ず、-key=value の様に
 * 繋げてください。
 *
 * SQL文には、{&#064;SYS.YMDH}等のシステム変数が使用できます。
 * 現時点では、{&#064;SYS.YMD}、{&#064;SYS.YMDH}、{&#064;SYS.HMS} が指定可能です。
 *
 * @og.formSample
 *  Process_DBWriter -dbid=DBGE -table=GE41
 *
 *     -dbid=DB接続ID             ： -dbid=DBGE (例: ParamProcess の -url_XXXX で指定された XXXX)
 *   [ -table=登録テーブルID    ] ： ＳＱＬ文を指定する場合は不要。INSERT する場合のテーブルID
 *   [ -sql=検索SQL文           ] ： -sql="UPDATE GE41 SET NAME_JA = [NAME_JA],LABEL_NAME = [LABEL_NAME]
 *                                         WHERE SYSTEM_ID = [SYSTEM_ID] AND CLM = [CLM]"
 *   [ -sqlFile=登録SQLファイル ] ： -sqlFile=update.sql
 *                                ：   -sql や -sqlFile が指定されない場合は、-table で指定のテーブルに全カラム insert です。
 *   [ -sql_XXXX=固定値         ] ： -sql_SYSTEM_ID=GE
 *                                     SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。
 *                                     WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'
 *   [ -const_XXXX=固定値       ] ： -const_FGJ=1
 *                                     LineModel のキー（const_ に続く文字列)の値に、固定値を設定します。
 *                                     キーが異なれば、複数のカラム名を指定できます。
 *   [ -omitClms=AAA,BBB,…     ] ： -omitClms=UNIQ,FGJ,DYSET
 *                                     -table 属性でINSERT文を自動作成する場合、取り除くカラム名を
 *                                     カンマ区切りで複数指定できます。
 *   [ -commitCnt=commit処理指定] ： 指定数毎にコミットを発行します。0 の場合は、終了までコミットしません。
 *   [ -display=false|true      ] ：結果を標準出力に表示する(true)かしない(false)か（初期値 false:表示しない)
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Process_DBWriter extends AbstractProcess implements ChainProcess {
	private static final String CNST_KEY = "const_" ;
	private static final String SQL_KEY  = "sql_" ;

	private Connection	connection	= null;
	private PreparedStatement pstmt	= null;
	private ParameterMetaData pMeta = null;		// 5.1.1.0 (2009/11/11) setObject に、Type を渡す。(PostgreSQL対応)
	private boolean useParamMetaData = false;	// 5.1.1.0 (2009/11/11) setObject に、Type を渡す。(PostgreSQL対応)

	private String		dbid		= null;
	private String		sql			= null;
	private String		table		= null;
	private int[]		clmNos		= null;		// ファイルのヘッダーのカラム番号
	private int			commitCnt	= 0;		// コミットするまとめ件数
	private boolean		display		= false;	// 表示しない

	private String[]	cnstClm		= null;		// 固定値を設定するカラム名
	private int[]		cnstClmNos	= null;		// 固定値を設定するカラム番号
	private String[]	constVal	= null;		// カラム番号に対応した固定値

	private boolean		firstRow	= true;		// 最初の一行目
	private int			count		= 0;
	private String[]	omitClms	= null;		// 4.0.0.0 (2007/09/21) table 指定時に取り除くカラム

	private final static Map<String,String> mustProparty   ;		// ［プロパティ］必須チェック用 Map
	private final static Map<String,String> usableProparty ;		// ［プロパティ］整合性チェック Map

	static {
		mustProparty = new LinkedHashMap<String,String>();
		mustProparty.put( "dbid",	"DB接続ID(必須) 例: ParamProcess の -url_XXXX で指定された XXXX" );

		usableProparty = new LinkedHashMap<String,String>();
		usableProparty.put( "table",		"INSERT する場合のテーブルID SQL文を指定する場合は不要。" );
		usableProparty.put( "sql",			"更新SQL文（sql or sqlFile 必須）" +
									CR + "例: \"UPDATE GE41 " +
									CR + "SET NAME_JA = [NAME_JA],LABEL_NAME = [LABEL_NAME] " +
									CR + "WHERE SYSTEM_ID = [SYSTEM_ID] AND CLM = [CLM]\"" );
		usableProparty.put( "sqlFile",		"登録SQLファイル（sql or sqlFile 必須）例: update.sql" );
		usableProparty.put( "sql_",		"SQL文中の{&#064;XXXX}文字列を指定の固定値で置き換えます。" +
									CR + "WHERE SYSTEM_ID='{&#064;SYSTEM_ID}' ⇒ WHERE SYSTEM_ID='GE'" );
		usableProparty.put( "const_",	"LineModel のキー（const_ に続く文字列)の値に、固定値を" +
									CR + "設定します。キーが異なれば、複数のカラム名を指定できます。" +
									CR + "例: -sql_SYSTEM_ID=GE" );
		// 4.0.0.0 (2007/09/21) 属性を追加
		usableProparty.put( "omitClms",	"-table 属性でINSERT文を自動作成する場合、取り除くカラム名を" +
									CR + "カンマ区切りで複数指定できます。" +
									CR + "例: -omitClms=UNIQ,FGJ,DYSET" );
		usableProparty.put( "commitCnt",	"指定数毎にコミットを発行します。" +
									CR + "0 の場合は、終了までコミットしません。(初期値: 0)" );
		usableProparty.put( "display",	"結果を標準出力に表示する(true)かしない(false)か" +
										CR + "(初期値 false:表示しない)" );
	}

	/**
	 * デフォルトコンストラクター。
	 * このクラスは、動的作成されます。デフォルトコンストラクターで、
	 * super クラスに対して、必要な初期化を行っておきます。
	 *
	 */
	public Process_DBWriter() {
		super( "org.opengion.fukurou.process.Process_DBWriter",mustProparty,usableProparty );
	}

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理（ファイルオープン、ＤＢオープン等）に使用します。
	 *
	 * @og.rev 4.0.0.0 (2007/09/21) omitClms 属性を追加
	 * @og.rev 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
	 *
	 * @param   paramProcess ParamProcess
	 */
	public void init( final ParamProcess paramProcess ) {
		Argument arg = getArgument();

		table			= arg.getProparty("table");
		sql				= arg.getFileProparty("sql","sqlFile",false);
		commitCnt		= arg.getProparty("commitCnt",commitCnt);
		display			= arg.getProparty("display",display);

		dbid		= arg.getProparty("dbid");
		connection	= paramProcess.getConnection( dbid );
		// 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
		useParamMetaData = ApplicationInfo.useParameterMetaData( connection );

		// 取り除くカラム名リストを配列に変換します。
		String tempClms	= arg.getProparty("omitClms",null);
		if( tempClms != null ) {
			omitClms = StringUtil.csv2Array( tempClms );
		}

		if( sql == null && table == null ) {
			String errMsg = "sql を指定しない場合は、table を必ず指定してください。";
			throw new RuntimeException( errMsg );
		}

		// 3.8.0.1 (2005/06/17) {@SYS.XXXX} 変換処理の追加
		// {@SYS.YMDH} などの文字列を、yyyyMMddHHmmss 型の日付に置き換えます。
		// SQL文の {@XXXX} 文字列の固定値への置き換え
		HybsEntry[] entry	=arg.getEntrys(SQL_KEY);		// 配列
		SystemParameter sysParam = new SystemParameter( sql );
		sql = sysParam.replace( entry );

		HybsEntry[] cnstKey = arg.getEntrys( CNST_KEY );		// 配列
		int csize	= cnstKey.length;
		cnstClm		= new String[csize];
		constVal	= new String[csize];
		for( int i=0; i<csize; i++ ) {
			cnstClm[i]  = cnstKey[i].getKey();
			constVal[i] = cnstKey[i].getValue();
		}
	}

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理（ファイルクローズ、ＤＢクローズ等）に使用します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/27) commit,rollback,remove 処理を追加
	 * @og.rev 5.1.1.0 (2009/11/11) pMeta のクリア
	 *
	 * @param   isOK トータルで、OKだったかどうか(true:成功/false:失敗）
	 */
	public void end( final boolean isOK ) {
		boolean flag = Closer.stmtClose( pstmt );
		pstmt = null;
		pMeta = null;		// 5.1.1.0 (2009/11/11)

		if( isOK ) {
			Closer.commit( connection );
		}
		else {
			Closer.rollback( connection );
		}
		ConnectionFactory.remove( connection,dbid );

		if( !flag ) {
			String errMsg = "ステートメントをクローズ出来ません。";
			throw new RuntimeException( errMsg );
		}
	}

	/**
	 * 引数の LineModel を処理するメソッドです。
	 * 変換処理後の LineModel を返します。
	 * 後続処理を行わない場合（データのフィルタリングを行う場合）は、
	 * null データを返します。つまり、null データは、後続処理を行わない
	 * フラグの代わりにも使用しています。
	 * なお、変換処理後の LineModel と、オリジナルの LineModel が、
	 * 同一か、コピー（クローン）かは、各処理メソッド内で決めています。
	 * ドキュメントに明記されていない場合は、副作用が問題になる場合は、
	 * 各処理ごとに自分でコピー（クローン）して下さい。
	 *
	 * @og.rev 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
	 *
	 * @param   data LineModel オリジナルのLineModel
	 * @return  LineModel  処理変換後のLineModel
	 */
	public LineModel action( final LineModel data ) {
		count++ ;
//		if( display ) { println( data.dataLine() ); }
		try {
			if( firstRow ) {
				pstmt = makePrepareStatement( table,data );
				// 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
				if( useParamMetaData ) {
					pMeta = pstmt.getParameterMetaData();
				}

				int size   = cnstClm.length;
				cnstClmNos = new int[size];
				for( int i=0; i<size; i++ ) {
					cnstClmNos[i] = data.getColumnNo( cnstClm[i] );
				}

				firstRow = false;
			}

			// 固定値置き換え処理
			for( int j=0; j<cnstClmNos.length; j++ ) {
				data.setValue( cnstClmNos[j],constVal[j] );
			}

			// 5.1.1.0 (2009/11/11) setObject に ParameterMetaData の getParameterType を渡す。(PostgreSQL対応)
			if( useParamMetaData ) {
				for( int i=0; i<clmNos.length; i++ ) {
					int type = pMeta.getParameterType( i+1 );
					pstmt.setObject( i+1,data.getValue(clmNos[i]),type );
				}
			}
			else {
				for( int i=0; i<clmNos.length; i++ ) {
					pstmt.setObject( i+1,data.getValue(clmNos[i]) );
				}
			}

			pstmt.execute();
			if( commitCnt > 0 && ( count%commitCnt == 0 ) ) {
				Closer.commit( connection );
			}
		}
		catch (SQLException ex) {
			String errMsg = "sql=[" + sql + "]" + CR +
						"errorCode=[" + ex.getErrorCode() + "] State=[" +
						ex.getSQLState() + "]" + CR ;
			throw new RuntimeException( errMsg,ex );
		}
		if( display ) { println( data.dataLine() ); }	// 5.1.2.0 (2010/01/01) display の条件変更
		return data;
	}

	/**
	 * 内部で使用する PreparedStatement を作成します。
	 * 引数指定の SQL または、LineModel から作成した SQL より構築します。
	 *
	 * @og.rev 4.0.0.0 (2007/09/21) omitClms 属性を追加
	 *
	 * @param   table  String    処理対象のテーブルID
	 * @param   data LineModel 処理対象のLineModel
	 * @return  PreparedStatement
	 */
	private PreparedStatement makePrepareStatement( final String table,final LineModel data ) {
		if( sql == null ) {
			StringBuilder buf = new StringBuilder();
			String[] names = data.getNames();

			// カラムを取り除く場合
			if( omitClms != null ) {
				Set<String> set = new HashSet<String>();
				for( int i=0; i<names.length; i++ ) {
					set.add( names[i] );
				}
				for( int i=0; i<omitClms.length; i++ ) {
					set.remove( omitClms[i] );
				}
				names = set.toArray( new String[set.size()] );
			}
			int size = names.length;

			buf.append( "INSERT INTO " ).append( table ).append( " (" );
			buf.append( names[0] );
			for( int i=1; i<size; i++ ) {
				buf.append( "," ).append( names[i] );
			}
			buf.append( " ) VALUES ( ?" );
			for( int i=1; i<size; i++ ) {
				buf.append( ",?" );
			}
			buf.append( " )" );
			sql = buf.toString();

			// カラム番号を設定します。
			clmNos = new int[size];
			for( int i=0; i<size; i++ ) {
				clmNos[i] = data.getColumnNo( names[i] );		// 4.0.0.0 (2007/09/21)
			}
		}
		else {
			Formatter format = new Formatter( data );
			format.setFormat( sql );
			sql = format.getQueryFormatString();
			clmNos = format.getClmNos();
		}

		final PreparedStatement ps ;
		try {
			ps = connection.prepareStatement( sql );
		}
		catch (SQLException ex) {
			String errMsg = "PreparedStatement を取得できませんでした。" + CR
						+ "sql=[" + sql + "]" + CR
						+ "table=[" + table + "]" + CR
						+ "nameLine=[" + data.nameLine() + "]" ;
			throw new RuntimeException( errMsg,ex );
		}

		return ps;
	}

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	public String report() {
		String report = "[" + getClass().getName() + "]" + CR
				+ TAB + "DBID         : " + dbid + CR
				+ TAB + "Output Count : " + count ;

		return report ;
	}

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	String
	 */
	public String usage() {
		StringBuilder buf = new StringBuilder();

		buf.append( "Process_DBWriter は、上流から受け取ったデータをデータベースに書き込む"			).append( CR );
		buf.append( "CainProcess インターフェースの実装クラスです。"								).append( CR );
		buf.append( CR );
		buf.append( "上流（プロセスチェインのデータは上流から下流へと渡されます。）から"			).append( CR );
		buf.append( "受け取った LineModel を元に、データベースへの書き込みを行います。"				).append( CR );
		buf.append( CR );
		buf.append( "データベース接続先等は、ParamProcess のサブクラス(Process_DBParam)に"			).append( CR );
		buf.append( "設定された接続(Connection)を使用します。"										).append( CR );
		buf.append( "-url_XXXX で指定された XXXX が、-dbid=XXXX に対応します。"						).append( CR );
		buf.append( CR );
		buf.append( "引数文字列中に空白を含む場合は、ダブルコーテーション(\"\") で括って下さい。"	).append( CR );
		buf.append( "引数文字列の 『=』の前後には、空白は挟めません。必ず、-key=value の様に"		).append( CR );
		buf.append( "繋げてください。"																).append( CR );
		buf.append( CR );
		buf.append( "SQL文には、{@SYS.YMDH}等のシステム変数が使用できます。"						).append( CR );
		buf.append( "現時点では、{@SYS.YMD}、{@SYS.YMDH}、{@SYS.HMS} が指定可能です。"				).append( CR );
		buf.append( CR ).append( CR );
		buf.append( getArgument().usage() ).append( CR );

		return buf.toString();
	}

	/**
	 * このクラスは、main メソッドから実行できません。
	 *
	 * @param	args String[]
	 */
	public static void main( final String[] args ) {
		LogWriter.log( new Process_DBWriter().usage() );
	}
}
