/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.io.FileInputStream;
import java.io.UnsupportedEncodingException;
import java.io.IOException;
import java.util.StringTokenizer;

/**
 * FileString は、ファイルを読み取って、そのまま String 文字列として返すクラスです。
 * SQLを記述したファイルや、コマンドを記述したファイルを読み取って、コマンドラインからの
 * 入力として利用できます。
 *
 * ファイルは、一旦すべて読み取ってメモリ上で加工されます。
 * パラメータ等の設定ファイルをイメージしている為、巨大なファイルの読み込みは出来ません。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileString  {
	private String	filename	= null ;
	private String	encode		= null ;
	private String	value		= null ;

	/**
	 * ファイル名を設定します。
	 *
	 * @param  filename String
	 */
	public void setFilename( final String filename ) {
		if( filename == null ) {
			String errMsg = "ファイル名が指定されていません。" ;
			throw new RuntimeException( errMsg );
		}

		this.filename = filename;
	}

	/**
	 * ファイルのエンコードを設定します。
	 *
	 * @param  encode String
	 */
	public void setEncode( final String encode ) {
		this.encode   = encode;
	}

	/**
	 * ファイルを読み取って、文字列を作成します。
	 *
	 * データの読取に失敗した場合でも、String オブジェクトを返します。
	 *
	 * @return  ファイルを読み取った文字列
	 * @throws RuntimeException 指定のエンコードが存在しなかったとき。
	 */
	public String getValue() {
		if( value != null ) { return value; }

		FileInputStream file = null;
		try {
			file = new FileInputStream( filename );
			byte[] buf = new byte[file.available()];
			int len = file.read(buf);
			if( len != buf.length ) {
				String errMsg = "読み取りファイルのデータが切り捨てられました。" +
								"File=" + filename + " Length=" + len ;
				LogWriter.log( errMsg );
			}
			if( encode != null ) {
				value = new String( buf,encode );
			}
			else {
				value = new String( buf );
			}
		}
		catch( UnsupportedEncodingException ex ) {
			String errMsg = "指定されたエンコーディングがサポートされていません。[" + encode + "]" ;
			throw new RuntimeException( errMsg,ex );
		}
		catch( IOException ex ) {
			String errMsg = "ファイル名がオープン出来ませんでした。[" + filename + "]" ;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( file );
		}

		return value ;
	}

	/**
	 * 指定の区切り文字で分割された文字列配列を取得します。
	 * 区切り文字が連続している場合は、一つの区切り文字と認識します。
	 * 処理は、java.util.StringTokenizer で、処理する方法と同一です。
	 * 例えば、";" を指定すれば、SQL を複数記述しておくことが可能になります。
	 *
	 * @param   delim String ファイルを分割する区切り文字
	 * @return  ファイルを読み取った文字列
	 */
	public String[] getValue( final String delim ) {

		StringTokenizer token = new StringTokenizer( getValue(),delim );

		String[] rtn = new String[token.countTokens()];

		for( int i=0; token.hasMoreTokens(); i++ ) {
			rtn[i] = token.nextToken();
		}

		return rtn ;
	}
}
