/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBMetaData;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Locale ;

/**
 * データベース情報(DBMetaData)より、テーブル、カラム等の情報を取得するタグです。
 *
 * データベースに関する包括的な情報を提供する、DatabaseMetaData の内容を
 * 取得して、DBTableModel にセットするタグです。
 *
 * @og.formSample
 * ●形式：&lt;og:dbMetaDataQuery action="･･･" ･･･ /&gt;
 * ●body：なし
 *
 * ●使用例
 *    command属性 は、columnSetタグのcommand属性と同一の場合のみ、処理します。
 *    [command属性]
 *      NEW       新規
 *      RENEW     再検索
 *
 *    [action属性]
 *      SCHEMAS    このデータベースで使用可能なスキーマ名を取得します。
 *      TABLES     指定されたカタログで使用可能なテーブルに関する記述を取得します。
 *      COLUMNS    指定されたカタログで使用可能なテーブル列の記述を取得します。
 *      INDEXINFO  指定されたテーブルのインデックスと統計情報に関する記述を取得します。
 *      PROCEDURES 指定されたカタログで使用可能なストアドプロシージャに関する記述を取得します。
 *
 * @og.group ＤＢ検索
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBMetaDataQueryTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	public static final String CMD_NEW	 = "NEW" ;
	/** command 引数に渡す事の出来る コマンド  再検索 {@value} */
	public static final String CMD_RENEW = "RENEW" ;
	/** command 引数に渡す事の出来る コマンド リスト  */
	private static final String[] COMMAND_LIST = new String[] { CMD_NEW , CMD_RENEW };

	/** action 引数に渡す事の出来る アクションコマンド  スキーマ名 {@value} */
	public static final String ACT_SCHEMAS     = "SCHEMAS" ;
	/** action 引数に渡す事の出来る アクションコマンド  テーブル {@value} */
	public static final String ACT_TABLES      = "TABLES" ;
	/** action 引数に渡す事の出来る アクションコマンド  テーブル列 {@value} */
	public static final String ACT_COLUMNS     = "COLUMNS" ;
	/** action 引数に渡す事の出来る アクションコマンド  インデックスと統計情報{@value} */
	public static final String ACT_INDEXINFO   = "INDEXINFO" ;
	/** action 引数に渡す事の出来る アクションコマンド  ストアドプロシージャ{@value} */
	public static final String ACT_PROCEDURES  = "PROCEDURES" ;

	/** action 引数に渡す事の出来る コマンド リスト  */
	private static final String[] ACTION_LIST = new String[] {
		ACT_SCHEMAS , ACT_TABLES , ACT_COLUMNS , ACT_INDEXINFO , ACT_PROCEDURES };

	private String	tableId	    = HybsSystem.TBL_MDL_KEY;

	private String	command		= "NEW";
	private String	action		= null;			// taglib で必須属性にします。
	private long	dyStart 	= 0;			// 実行時間測定用のDIV要素
//	private String	displayMsg	= "MSG0033";	// 　件検索しました。
	private String	displayMsg	= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
	private String	notfoundMsg	= "MSG0077";	// 対象データはありませんでした。

	// 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
//	private String	dbid		= "DEFAULT";
	private String	dbid		= null;
	private String	catalog		= null;
	private String	schema		= null;
	private String	tableName	= null;
	private String	procName	= null;
	private String	columnName	= null;
	private boolean	unique		= false;	// true:ユニークのみ / false:非ユニーク含む
	private boolean	approximate	= true;		// true:概数 / false:正確
	private boolean	isMainTrans	= true;		// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 * 
	 * @return	int
	 */
	@Override
	public int doStartTag() {
		dyStart = System.currentTimeMillis();
		if( ! check( command, COMMAND_LIST ) ) { return(SKIP_BODY); }

		useMainTrans( isMainTrans );			// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
		startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)

		// 3.5.6.5 (2004/08/09) 削除するのは、セッションのオブジェクトでよい。
		// 3.6.0.0 (2004/09/24) 削除するのは、scope="session" の場合のみ。
		if( "session".equals( getScope() ) ) {
			removeSessionAttribute( tableId );
			removeSessionAttribute( HybsSystem.VIEWFORM_KEY );
		}

		return(SKIP_BODY);				// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.0.0 (2006/11/14) notfoundMsg 属性を追加。displayMsg は、VIEW_USE_DISPLAY_MSG で制御
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage > getResource().getLabel )
	 *
	 * @return	int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		String label  = "";				// 4.0.0 (2005/11/30) 検索しなかった場合。
		if( check( command, COMMAND_LIST ) ) {
			StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );

			DBTableModel table = actionExec( action );
			int executeCount = table.getRowCount();	// 検索した数

			// 実行件数の表示 command="NEW" のときのみ、displayMsg を表示させます。
			// 4.0.0 (2005/11/30) 出力順の変更。一番最初に出力します。
//			boolean	useStatusBar = HybsSystem.sysBool( "VIEW_USE_DISPLAY_MSG" );
			if(  CMD_NEW.equals( command ) ) {
//				if( useStatusBar && executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
				if( executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
					buf.append( executeCount );
//					buf.append( getResource().getMessage( displayMsg ) );
					buf.append( getResource().getLabel( displayMsg ) );
					buf.append( HybsSystem.BR );
				}
				else if( executeCount == 0 && notfoundMsg != null && notfoundMsg.length() > 0 ) {
//					buf.append( getResource().getMessage( notfoundMsg ) );
					buf.append( getResource().getLabel( notfoundMsg ) );
					buf.append( HybsSystem.BR );
				}
			}

			// 3.3.3.3 (2003/08/06) 検索結果の件数を、"DB.COUNT" キーでリクエストにセットする。
			setRequestAttribute( "DB.COUNT"   , String.valueOf( executeCount ) );
			// 3.3.3.3 (2003/08/06) 検索結果を、"DB.ERR_CODE" キーでリクエストにセットする。
			setRequestAttribute( "DB.ERR_CODE", String.valueOf( ErrorMessage.OK ) );

			// オーバーフロー時のメッセージを表示
	//		if( table != null && table.isOverflow() ) {
	//			buf.append( getResource().getMessage( overflowMsg ) );
	//			buf.append( HybsSystem.BR );
	//		}

			// 実行件数の表示
			// 2.0.0.8 (2002/10/09) command="NEW" のときのみ、displayMsg を表示させます。
	//		if( displayMsg != null && displayMsg.length() > 0 && command.equals( CMD_NEW ) ) {
	//			buf.append( executeCount );
	//			buf.append( getResource().getMessage( displayMsg ) );
	//			buf.append( HybsSystem.BR );
	//		}

			label = buf.toString();

			// 3.6.0.8 (2004/11/19) トランザクションチェックを行います。
			if( ! commitTableObject( tableId, table ) ) {
				jspPrint( "DBMetaDataQueryTag Query処理が割り込まれました。DBTableModel は登録しません。" );
				return (SKIP_PAGE);
			}
		}
		jspPrint( label );

		// 3.5.4.7 (2004/02/06)
		long dyTime = System.currentTimeMillis()-dyStart;
		jspPrint( "<div id=\"queryTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)
		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 */
	@Override
	protected void release2() {
		super.release2();
		tableId	    = HybsSystem.TBL_MDL_KEY;
//		dbid		= "DEFAULT";
		dbid		= null;
		catalog		= null;
		schema		= null;
		tableName	= null;
		procName	= null;
		columnName	= null;
		unique		= false;	// true:ユニークのみ / false:非ユニーク含む
		approximate	= true;		// true:概数 / false:正確
		dyStart		= 0;
//		displayMsg	= "MSG0033";	// 　件検索しました。
		displayMsg	= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
		notfoundMsg	= "MSG0077";	// 対象データはありませんでした。
		isMainTrans	= true;			// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
	}

	/**
	 * 指定のアクションを実行し、結果を、DBTableModel にセットして返します。
	 * アクションは、(SCHEMAS,TABLES,COLUMNS,INDEXINFO,PROCEDURES)を指定します。
	 *
	 * @og.tag
	 *
	 *      SCHEMAS    このデータベースで使用可能なスキーマ名を取得します。
	 *      TABLES     指定されたカタログで使用可能なテーブルに関する記述を取得します。
	 *      COLUMNS    指定されたカタログで使用可能なテーブル列の記述を取得します。
	 *      INDEXINFO  指定されたテーブルのインデックスと統計情報に関する記述を取得します。
	 *      PROCEDURES 指定されたカタログで使用可能なストアドプロシージャに関する記述を取得します。
	 *
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 *
	 * @param	action アクション文字列
	 * @return  DBTableModel
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.DBMetaDataQueryTag.ACT_COLUMNS">アクション定数</a>
	 */
	private DBTableModel actionExec( final String action ) {
		DBMetaData metaData = new DBMetaData();
		metaData.setDbid( dbid );
		metaData.setResourceManager( getResource() );
		metaData.setApplicationInfo( getApplicationInfo() );	// 3.8.7.0 (2006/12/15)

		DBTableModel tbl = null;

		if( ACT_SCHEMAS.equals( action ) ) {
			tbl = metaData.getSchemas() ;
		}
		else if( ACT_TABLES.equals( action ) ) {
			tbl = metaData.getTables( catalog, schema, tableName ) ;
		}
		else if( ACT_COLUMNS.equals( action ) ) {
			tbl = metaData.getColumns(catalog, schema, tableName, columnName) ;
		}
		else if( ACT_INDEXINFO.equals( action ) ) {
			tbl = metaData.getIndexInfo(catalog, schema, tableName, unique, approximate) ;
		}
		else if( ACT_PROCEDURES.equals( action ) ) {
			tbl = metaData.getProcedures(catalog, schema, procName) ;
		}

		return tbl ;
	}

	/**
	 * 【TAG】コマンド(NEW,RENEW)をセットします(初期値:NEW)。
	 *
	 * @og.tag
	 * コマンドは,HTMLから（get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 *
	 * @param	cmd コマンド（public static final 宣言されている文字列)
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.DBMetaDataQueryTag.CMD_NEW">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】アクション(SCHEMAS,TABLES,COLUMNS,INDEXINFO,PROCEDURES)を指定します。
	 *
	 * @og.tag
	 * アクションは,HTMLから（get/post)指定されますので,ACT_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 *
	 *      SCHEMAS    このデータベースで使用可能なスキーマ名を取得します。
	 *      TABLES     指定されたカタログで使用可能なテーブルに関する記述を取得します。
	 *      COLUMNS    指定されたカタログで使用可能なテーブル列の記述を取得します。
	 *      INDEXINFO  指定されたテーブルのインデックスと統計情報に関する記述を取得します。
	 *      PROCEDURES 指定されたカタログで使用可能なストアドプロシージャに関する記述を取得します。
	 *
	 * @param	cmd アクション文字列
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.DBMetaDataQueryTag.ACT_COLUMNS">アクション定数</a>
	 */
	public void setAction( final String cmd ) {
		action = getRequestParameter( cmd );

		if( ! check( action, ACTION_LIST ) ) {

			StringBuilder errMsg = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
			errMsg.append( "指定のアクションは実行できません。アクションエラー" );
			errMsg.append( HybsSystem.CR );
			errMsg.append( "action=[" ).append( action ).append( "] " );
			errMsg.append( HybsSystem.CR );

			for( int i=0; i<ACTION_LIST.length; i++ ) {
				errMsg.append( " | " );
				errMsg.append( ACTION_LIST[i] );
			}
			errMsg.append( " | " );
			throw new HybsSystemException( errMsg.toString() );
		}
	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】カタログ名をセットします(初期値:null)。
	 *
	 * @og.tag データベースに格納されたカタログ名と一致しなければならない。
	 * "" はカタログなしでカタログ名を検索する。
	 * null は、カタログ名を検索の限定に使用してはならないことを意味する
	 *
	 * @param	catalog カタログ名
	 */
	public void setCatalog( final String catalog ) {
		this.catalog = nval( getRequestParameter( catalog ),this.catalog );
	}

	/**
	 * 【TAG】スキーマ名パターンをセットします(初期値:null)。
	 *
	 * @og.tag データベースに格納されたスキーマ名と一致しなければならない。
	 * "" はスキーマなしでスキーマ名を検索する。
	 * null は、スキーマ名を検索の限定に使用してはならないことを意味する
	 *
	 * @param	schema スキーマ名パターン
	 */
	public void setSchema( final String schema ) {
		this.schema = nval( getRequestParameter( schema ),this.schema );
	}

	/**
	 * 【TAG】テーブル名パターンをセットします(初期値:null)。
	 *
	 * @og.tag 。データベースに格納されたテーブル名と一致しなければならない
	 *
	 * @param	tableName テーブル名パターン
	 */
	public void setTableName( final String tableName ) {
		this.tableName = nval( getRequestParameter( tableName ),this.tableName );
	}

	/**
	 * 【TAG】プロシージャ名パターンをセットします(初期値:null)。
	 *
	 * @og.tag 。データベースに格納されたプロシージャ名と一致しなければならない
	 *
	 * @param	procName プロシージャ名パターン
	 */
	public void setProcName( final String procName ) {
		this.procName = nval( getRequestParameter( procName ),this.procName );
	}

	/**
	 * 【TAG】列名パターンをセットします(初期値:null)。
	 *
	 * @og.tag データベースに格納された列名と一致しなければならない
	 *
	 * @param	columnName 列名パターン
	 */
	public void setColumnName( final String columnName ) {
		this.columnName = nval( getRequestParameter( columnName ),this.columnName );
	}

	/**
	 * 【TAG】返すインデックスの種類(true:ユニークのみ / false:非ユニーク含む)を指定します(初期値:false)。
	 *
	 * @og.tag  true の場合は、一意の値のインデックスだけを返す。
	 * false の場合は、一意であるかどうかにかかわらずインデックスを返す
	 * 初期値は、false:非ユニーク含む です。
	 *
	 * @param	uniqFlag 返すインデックスの種類(true:ユニークのみ / false:非ユニーク含む)
	 */
	public void setUnique( final String uniqFlag ) {
		this.unique = nval( getRequestParameter( uniqFlag ),this.unique );
	}

	/**
	 * 【TAG】統計情報の精度(true:概数 / false:正確)を指定します(初期値:true)。
	 *
	 * @og.tag 指定されたテーブルのインデックスと統計情報に関する記述を取得する場合に
	 * 結果の精度を指定します。
	 * true の場合は、結果は概数またはデータ値から外れることもある。
	 * false の場合は、正確であることが要求される
	 * 初期値は、true:概数 です。
	 *
	 * @param	appFlag 統計情報の精度(true:概数 / false:正確)
	 */
	public void setApproximate( final String appFlag ) {
		this.approximate = nval( getRequestParameter( appFlag ),this.approximate );
	}

	/**
	 * 【TAG】(通常は使いません)結果をDBTableModelに書き込んで、sessionに登録するときのキーを指定します。
	 *
	 * @og.tag
	 * 初期値は、HybsSystem.TBL_MDL_KEY です。
	 *
	 * @param	id sessionに登録する時の ID
	 */
	public void setTableId( final String id ) {
		this.tableId = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】検索結果を画面上に表示するメッセージリソースIDを指定します(初期値:MSG0033[　件検索しました])。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したメッセージをリソースから取得して
	 * 表示します。
	 * 表示させたくない場合は, displayMsg = "" をセットしてください。
	 * 初期値は、検索件数を表示します。
	 *
	 * @param	id ディスプレイに表示させるメッセージ ID
	 */
	public void setDisplayMsg( final String id ) {
		String ids = getRequestParameter( id );
		if( ids != null ) { displayMsg = ids; }
	}

	/**
	 * 【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])。
	 *
	 * @og.tag
	 * ここでは、検索結果がゼロ件の場合のみ、特別なメッセージを表示させます。
	 * 従来は、displayMsg と兼用で、『0　件検索しました』という表示でしたが、
	 * displayMsg の初期表示は、OFF になりましたので、ゼロ件の場合のみ別に表示させます。
	 * 表示させたくない場合は, notfoundMsg = "" をセットしてください。
	 * 初期値は、MSG0077[対象データはありませんでした]です。
	 *
	 * @param	id ディスプレイに表示させるメッセージ ID
	 */
	public void setNotfoundMsg( final String id ) {
		String ids = getRequestParameter( id );
		if( ids != null ) { notfoundMsg = ids; }
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 */
	protected String getTagName() {
		return "dbMetaDataQuery" ;
	}

	/**
	 * 【TAG】(通常使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します。(初期値:false)
	 *
	 * @og.tag
	 * (通常使いません)タグで処理される処理が、メインとなるトランザクション処理かどうかを指定します。(初期値:false)
	 * この値は、ファイルダウンロード処理に影響します。この値がtrueに指定された時にcommitされたDBTableModelが
	 * ファイルダウンロードの対象の表になります。
	 * 
	 * このパラメーターは、通常、各タグにより実装され、ユーザーが指定する必要はありません。
	 * 但し、1つのJSP内でDBTableModelが複数生成される場合に、前に処理したDBTableModelについてファイルダウンロードをさせたい
	 * 場合は、後ろでDBTableModelを生成するタグで、明示的にこの値をfalseに指定することで、ファイルダウンロード処理の対象から
	 * 除外することができます。
	 * 
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param  flag メイントランザクションかどうか
	 */
	public void setMainTrans( final String flag ) {
		isMainTrans = nval( getRequestParameter( flag ),isMainTrans );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tableId"	    ,tableId	)
				.println( "command"		,command	)
				.println( "action"		,action		)
				.println( "dyStart" 	,dyStart 	)
				.println( "displayMsg"	,displayMsg	)
				.println( "dbid"		,dbid		)
				.println( "catalog"		,catalog	)
				.println( "schema"		,schema		)
				.println( "tableName"	,tableName	)
				.println( "procName"	,procName	)
				.println( "columnName"	,columnName	)
				.println( "unique"		,unique		)
				.println( "approximate"	,approximate)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
