/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.resource.LDAPSearch;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil ;
import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.List;
import java.util.Locale;

/**
 * LDAPの内容を検索するための、ldapQueryタグです。
 *
 * 検索した結果は、DBTableModel にセットされるため、
 * JDBCQuery と同様に、viewタグで表示させることが可能です。
 *
 * 下記の項目については、src/resource/システムパラメータ に、予め
 * 設定しておくことで、タグごとに指定する必要がなくなります。
 * ・LDAP_INITIAL_CONTEXT_FACTORY
 * ・LDAP_PROVIDER_URL
 * ・LDAP_ENTRYDN
 * ・LDAP_SEARCH_BASE
 * ・LDAP_SEARCH_SCOPE
 *
 * このタグリブでは、検索のみサポートしています。
 *
 * @og.formSample
 * ●形式：
 *     &lt;og:ldapQuery  debug=&quot;false&quot;
 *         command     = &quot;NEW&quot;                                       NEW,RENEW が使用可能です。
 *         initctx     = &quot;com.sun.jndi.ldap.LdapCtxFactory&quot;          初期コンテキストファクトリ
 *         providerURL = &quot;ldap://ldap.opengion.org:389&quot;             サービスプロバイダの構成情報
 *         entrydn     = &quot;cn=inquiry-sys,o=opengion,c=JP&quot;       属性の取得元のオブジェクトの名前
 *         password    = &quot;xxxxxx&quot;                                    属性の取得元のオブジェクトのパスワード
 *         searchbase  = &quot;soouid=employeeuser,o=opengion,c=JP&quot;        検索するコンテキストまたはオブジェクトのベース名
 *         filter      = &quot;(objectClass=person)&quot;                      検索に使用するフィルタ式。null は不可
 *         attributes  = &quot;uid,cn,officeName,ou,mail,belongOUID&quot;      属性を識別する属性 IDをカンマ区切りで指定
 *         searchScope = &quot;SUBTREE&quot;                                   検索範囲を設定(『OBJECT』『ONELEVEL』『SUBTREE』のどれか)
 *         maxRowCount = &quot;0&quot;                                         検索時の最大検索数（0は、無制限：デフォルト）
 *     /&gt;
 * ●body：なし
 *
 * ●使用例
 *     システムパラメータ に、必要なLDAP定数を定義しておいた場合は、下記のタグで検索できます。
 *     例では、filter にあいまい検索(*)とAND条件(&amp;(条件A)(条件B)) を指定した例を挙げておきます。
 *     filter の形式と解釈は、RFC 2254 に準拠しています。
 *     （RFC 2254 で定義されている演算子のすべてをすべての属性に適用できるわけではありません。）
 *
 *      &lt;og:ldapQuery  debug=&quot;false&quot;
 *          command     = &quot;NEW&quot;
 *          filter      = &quot;(&amp;(cnphoneticlangjp=OPENGION*)(businessName=*COMMON*))&quot;
 *          attributes  = &quot;uid,cn,officeName,ou,mail,belongOUID&quot;
 *      /&gt;
 *
 *    ・queryOption タグの内部に記述すれば、プルダウンメニューを作成することが可能です。
 *    &lt;og:select name="UID"&gt;
 *        &lt;og:queryOption&gt;
 *            &lt;og:ldapQuery
 *                filter      = "(&amp;amp;(|(belongOUID=60000)(belongOUID=70000))(uid=C*))"
 *                attributes  = "uid,uid,sn"
 *                orderBy     = "uid"
 *            /&gt;
 *        &lt;/og:queryOption&gt;
 *    &lt;/og:select&gt;
 *
 * @og.rev 3.1.0.0 (2003/03/20) ＬＤＡＰにアクセスできる、LDAPQueryTag.java を新規に作成。
 * @og.group その他入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class LDAPQueryTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	public static final String CMD_NEW	 = "NEW" ;
	/** command 引数に渡す事の出来る コマンド  再検索 {@value} */
	public static final String CMD_RENEW = "RENEW" ;
	/** command 引数に渡す事の出来る コマンド リスト  */
	private static final String[] COMMAND_LIST = new String[] { CMD_NEW , CMD_RENEW };

	private String			initctx 			= HybsSystem.sys( "LDAP_INITIAL_CONTEXT_FACTORY" );
	private String			providerURL 		= HybsSystem.sys( "LDAP_PROVIDER_URL" );
	private String			entrydn 			= HybsSystem.sys( "LDAP_ENTRYDN" );
	private String			password 			= HybsSystem.sys( "LDAP_PASSWORD" );		// 4.2.2.0 (2008/05/10)
	private String			searchbase			= HybsSystem.sys( "LDAP_SEARCH_BASE" );
	private String			filter 				= null;		// "employeeNumber=87019";

	// 検索範囲。OBJECT_SCOPE、ONELEVEL_SCOPE、SUBTREE_SCOPE のどれか 1 つ
	private String			searchScope			= HybsSystem.sys( "LDAP_SEARCH_SCOPE" );
	private static final long	COUNTLIMIT		= 0;			// 返すエントリの最大数。0 の場合、フィルタを満たすエントリをすべて返す
	private int				timeLimit			= 0;			// 結果が返されるまでのミリ秒数。0 の場合、無制限
	private String[]		attrs				= null;			// エントリと一緒に返される属性の識別子。null の場合、すべての属性を返す。空の場合、属性を返さない
	private String[]		columns				= null;			// DBTableModel を作成する場合のカラム名になります。null の場合は、attributes 属性
	private boolean			returningObjFlag	= false;		// true の場合、エントリの名前にバインドされたオブジェクトを返す。false 場合、オブジェクトを返さない
	private boolean			derefLinkFlag		= false;		// true の場合、検索中にリンクを間接参照する

	private String			tableId 			= HybsSystem.TBL_MDL_KEY;	// sessionｷｰ
	private String			orderBy				= null;						// ｿｰﾄ項目（csv）
	private String			command				= CMD_NEW;					// ｺﾏﾝﾄﾞ
	private int				executeCount		= 0;						// 検索/実行件数
	private int				errCode			 	= ErrorMessage.OK;			// Ver4.3.1.1 (2008/09/05) 検索エラーコード
	private int 		 	maxRowCount			= 0;			// 最大検索数(0は無制限)
//	private String			displayMsg   		= "MSG0033";	// 件検索しました。
	private String			displayMsg			= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
	private String			overflowMsg  		= "MSG0007";	// 検索結果が、制限行数を超えましたので、残りはカットされました。
	private String			notfoundMsg			= "MSG0077";	// 対象データはありませんでした。
	private boolean			isMainTrans			= true;			// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.6.0.8 (2004/11/19) DBTableModel をセーブする時に、トランザクションチェックを行います。
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage > getResource().getLabel )
	 * @og.rev 4.3.1.1 (2008/09/03) 検索結果の件数、状況コードをリクエスト変数 "DB.COUNT"、"DB.ERR_CODE" にセットする。
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// QueryOptionTag の内部で記述された場合の処理。
		QueryOptionTag queryOption = (QueryOptionTag)findAncestorWithClass( this,QueryOptionTag.class );
		if( queryOption != null ) {
			queryOption.setTableModel( makeDBTable() );
			return(EVAL_PAGE);
		}

		// 通常の LDAPQuery 処理
		if( check( command, COMMAND_LIST ) ) {
			useMainTrans( isMainTrans );			// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
			startQueryTransaction( tableId );		// 3.6.0.8 (2004/11/19)

			DBTableModel table = makeDBTable();

			// 3.6.0.8 (2004/11/19) トランザクションチェックを行います。
			if( ! commitTableObject( tableId, table ) ) {
				jspPrint( "LDAPQueryTag Query処理が割り込まれました。DBTableModel は登録しません。" );
				return (SKIP_PAGE);
			}
			StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );

			// 実行件数の表示 command="NEW" のときのみ、displayMsg を表示させます。
			// 4.0.0 (2005/11/30) 出力順の変更。一番最初に出力します。
			if( CMD_NEW.equals( command ) ) {
				if( executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
					buf.append( executeCount );
					buf.append( getResource().getLabel( displayMsg ) );
					buf.append( HybsSystem.BR );
				}
				else if( executeCount == 0 && notfoundMsg != null && notfoundMsg.length() > 0 ) {
					buf.append( getResource().getLabel( notfoundMsg ) );
					buf.append( HybsSystem.BR );
				}
			}

			if( maxRowCount > 0 && maxRowCount <= executeCount ) {
				buf.append( getResource().getLabel( overflowMsg ) );
				buf.append( HybsSystem.BR ).append( HybsSystem.CR );
			}
			// 4.3.1.1 (2008/09/03) 検索結果の件数、エラーコードをセットする。
			setRequestAttribute( "DB.COUNT" , String.valueOf( executeCount ) );
			setRequestAttribute( "DB.ERR_CODE", String.valueOf( errCode ) );

			jspPrint( buf.toString() );
		}

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 4.2.2.0 (2008/05/10) password 追加
	 * @og.rev 5.1.6.0 (2010/05/01) DBLastSqlの処理は、DBTableModelが新規作成された処理でのみ行う。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();

		searchScope			= HybsSystem.sys( "LDAP_SEARCH_SCOPE" );	// 検索範囲。OBJECT_SCOPE、ONELEVEL_SCOPE、SUBTREE_SCOPE のどれか 1 つ
		timeLimit			= 0;			// 結果が返されるまでのミリ秒数。0 の場合、無制限
		derefLinkFlag		= false;		// true の場合、検索中にリンクを間接参照する
		returningObjFlag	= false;		// true の場合、エントリの名前にバインドされたオブジェクトを返す。false 場合、オブジェクトを返さない
		attrs				= null;			// エントリと一緒に返される属性の識別子。null の場合、すべての属性を返す。空の場合、属性を返さない
		columns				= null;			// DBTableModel を作成する場合のカラム名になります。null の場合は、attributes 属性

		initctx 			= HybsSystem.sys( "LDAP_INITIAL_CONTEXT_FACTORY" );
		providerURL 		= HybsSystem.sys( "LDAP_PROVIDER_URL" );
		entrydn 			= HybsSystem.sys( "LDAP_ENTRYDN" );
		password 			= HybsSystem.sys( "LDAP_PASSWORD" );		// 4.2.2.0 (2008/05/10)
		searchbase			= HybsSystem.sys( "LDAP_SEARCH_BASE" );
		filter 				= null;		// "employeeNumber=87019";

		tableId 			= HybsSystem.TBL_MDL_KEY;	// sessionｷｰ
		orderBy				= null;			// ｿｰﾄ項目（csv）
		command				= CMD_NEW;		// ｺﾏﾝﾄﾞ
		executeCount		= 0;			// 検索/実行件数
		errCode			 	= ErrorMessage.OK; // Ver4.3.1.1 (2008/09/05) 検索エラーコード
		maxRowCount			= 0;			// 最大検索数(0は無制限)
//		displayMsg   		= "MSG0033";	// 件検索しました。
		displayMsg			= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
		overflowMsg  		= "MSG0007";	// 検索結果が、制限行数を超えましたので、残りはカットされました。		*/
		notfoundMsg			= "MSG0077";	// 対象データはありませんでした。
		isMainTrans			= true;			// 5.1.6.0 (2010/05/01) DBLastSqlの処理の見直し
	}

	/**
	 * LDPA から、値を取り出し、DBTableModel を作成します。
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) entrydn と、password 処理の追加
	 *
	 * @param	DBTableModel
	 */
	private DBTableModel makeDBTable() {
		if( filter == null ) {
			String errMsg = "S検索フィルターに null が指定されました。" ;
			throw new HybsSystemException( errMsg );
		}

		LDAPSearch serch = new LDAPSearch();
		serch.setSearchScope( searchScope ) ;
		serch.setTimeLimit( timeLimit ) ;
		serch.setDerefLinkFlag( derefLinkFlag ) ;
		serch.setReturningObjFlag( returningObjFlag ) ;
		serch.setMaxRowCount( maxRowCount ) ;
		serch.setAttributes( attrs ) ;
		serch.setInitctx( initctx ) ;
		serch.setProviderURL( providerURL ) ;
		serch.setSearchbase( searchbase ) ;
		if( entrydn != null  ) { serch.setEntrydn( entrydn ) ; }	//4 .2.2.0 (2008/05/10)
		if( password != null ) { serch.setPassword( password ) ; }	//4 .2.2.0 (2008/05/10)
		serch.setOrderBy( orderBy ) ;
		serch.init();

		List<String[]> list = serch.search( filter );
		executeCount = list.size();

//		DBTableModel table = DBTableModelUtil.newDBTable();
		String[] ids = serch.getAttributes();	// attrs が null の場合、全件。それ以外は、attrs

		// DBTableModel に使用するカラム名の登録
		if( columns != null && columns.length > 0 ) {
			if( ids.length != columns.length ) {
				String errMsg = "attributes 属性とcolumns属性の個数が異なります。"
							+ "attrs=[" + attrs.length + "]:" + StringUtil.array2csv( attrs )
							+ HybsSystem.CR
							+ " columns=[" + columns.length + "]:" + StringUtil.array2csv( columns )
							+ HybsSystem.CR ;
				throw new HybsSystemException( errMsg );
			}
			for( int i=0; i<columns.length; i++ ) {
				// columns にカラム名が設定されている場合のみ、使用する。
				if( columns[i] != null && columns[i].length() > 0 ) {
					ids[i] = columns[i];
				}
			}
		}

		DBTableModel table = null;
		if( ids.length > 0 ) {
			table = DBTableModelUtil.newDBTable();
			table.init( ids.length );
			for( int clm=0; clm<ids.length; clm++ ) {
				DBColumn dbColumn = getResource().makeDBColumn( ids[clm] );
				table.setDBColumn( clm,dbColumn );
			}

			for( int row=0; row<executeCount; row++ ) {
				table.addColumnValues( list.get( row ) );
			}
		}
		return table ;
	}

	/**
	 * 【TAG】検索範囲(OBJECT/ONELEVEL/SUBTREE)を設定します(初期値:システムパラメータ の LDAP_SEARCH_SCOPE)。
	 *
	 * @og.tag
	 * 検索範囲を OBJECT_SCOPE、ONELEVEL_SCOPE、SUBTREE_SCOPE のどれか 1 つです。
	 * 指定文字列は、それぞれ『OBJECT』『ONELEVEL』『SUBTREE』です。
	 *
	 * @param	scope SearchControls の検索範囲
	 */
	public void setSearchScope( final String scope ) {
		searchScope = nval( getRequestParameter( scope ), searchScope );
	}

	/**
	 * 【TAG】これらの SearchControls の時間制限をミリ秒単位で設定します(初期値:0[無制限])。
	 *
	 * @og.tag
	 * 値が 0 の場合、無制限に待つことを意味します。
	 *
	 * @param	limit ミリ秒単位の時間制限（初期値 無制限）
	 */
	public void setTimeLimit( final String limit ) {
		timeLimit = nval( getRequestParameter( limit ), timeLimit );
	}

	/**
	 * 【TAG】検索中のリンクへの間接参照を有効または無効(true/false)にします(初期値:false)。
	 *
	 * @og.tag 検索中のリンクへの間接参照を有効または無効にします。
	 *
	 * @param	deref リンクを逆参照する場合は true、そうでない場合は false（初期値 false）
	 */
	public void setDerefLinkFlag( final String deref ) {
		derefLinkFlag = nval( getRequestParameter( deref ), derefLinkFlag );
	}

	/**
	 * 【TAG】結果の一部としてオブジェクトを返すことを有効または無効(true/false)にします(初期値:false)。
	 *
	 * @og.tag
	 * 無効にした場合、オブジェクトの名前およびクラスだけが返されます。
	 * 有効にした場合、オブジェクトが返されます。
	 *
	 * @param	pbjflag オブジェクトが返される場合は true、そうでない場合は false（初期値 false）
	 */
	public void setReturningObjFlag( final String pbjflag ) {
		returningObjFlag = nval( getRequestParameter( pbjflag ), returningObjFlag );
	}

	/**
	 * 【TAG】レジストリの最大検索件数をセットします(初期値:0[無制限])。
	 *
	 * @og.tag
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 * 0 は、無制限です。（初期値は、無制限です。）
	 *
	 * @param	count レジストリの最大検索件数
	 */
	public void setMaxRowCount( final String count ) {
		maxRowCount = nval( getRequestParameter( count ),maxRowCount );
	}

	/**
	 * 【TAG】検索の一部として返される属性をCSVで複数指定します。
	 *
	 * @og.tag
	 * null は属性が何も返されないことを示します。
	 * このメソッドからは、空の配列をセットすることは出来ません。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 3.7.0.4 (2005/03/18) リクエスト変数の値を取得後、分解します。
	 *
	 * @param	val 返される属性を識別する属性 ID の羅列（カンマ区切り文字）
	 */
	public void setAttributes( final String val ) {
		String att = nval( getRequestParameter( val ),null );
		if( att != null ) {
			attrs = StringUtil.csv2Array( att );
		}
	}

	/**
	 * 【TAG】作成する DBTableModel のカラム名をCSV形式で指定します。
	 *
	 * @og.tag
	 * なにも指定しない場合は、attributes 属性が使用されます。
	 * LDAP検索結果に、別名をつけるイメージで使用します。
	 *
	 * @og.rev 3.7.1.0 (2005/04/26) 新規追加
	 *
	 * @param	clm 作成する DBTableModel のカラム名（カンマ区切り文字）
	 */
	public void setColumns( final String clm ) {
		String cl = nval( getRequestParameter( clm ),null );
		if( cl != null ) {
			columns = StringUtil.csv2Array( cl );
		}
	}

	/**
	 * 【TAG】初期コンテキストファクトリを指定します(初期値:システムパラメータ の INITIAL_CONTEXT_FACTORY)。
	 *
	 * @og.tag
	 * 初期値は、システムパラメータ の INITIAL_CONTEXT_FACTORY 属性です。
	 * 例）com.sun.jndi.ldap.LdapCtxFactory
	 *
	 * @param	ctx INITIAL_CONTEXT_FACTORY
	 */
	public void setInitctx( final String ctx ) {
		initctx = nval( getRequestParameter( ctx ), initctx );
	}

	/**
	 * 【TAG】サービスプロバイダの構成情報を指定します(初期値:システムパラメータ の LDAP_PROVIDER_URL)。
	 *
	 * @og.tag
	 * プロトコルとサーバーとポートを指定します。
	 * 例）『ldap://ldap.opengion.org:389』
	 *
	 * @param	url PROVIDER_URL
	 */
	public void setProviderURL( final String url ) {
		providerURL = nval( getRequestParameter( url ), providerURL );
	}

	/**
	 * 【TAG】検索するコンテキストまたはオブジェクトの名前を設定します(初期値:システムパラメータ の LDAP_SEARCH_BASE)。
	 *
	 * @og.tag
	 * 例）『soOUID=employeeuser,o=opengion,c=JP』
	 *
	 * @param	base SEARCHBASE
	 */
	public void setSearchbase( final String base ) {
		searchbase = nval( getRequestParameter( base ), searchbase );
	}

	/**
	 * 【TAG】検索に使用するフィルタ式(例：belongOUID=61200)。
	 *
	 * @og.tag
	 * 例）『belongOUID=61200』
	 *
	 * @param	siki FILTER
	 */
	public void setFilter( final String siki ) {
		filter = nval( getRequestParameter( siki ), filter );
	}

	/**
	 * 【TAG】属性の取得元のオブジェクトの名前を設定します(初期値:システムパラメータ の LDAP_ENTRYDN)。
	 *
	 * @og.tag
	 * 初期値:システムパラメータ の LDAP_ENTRYDN が使用されます。
	 * 例）『soOUID=opengion-sys,o=opengion,c=JP』
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) 初期値変更
	 *
	 * @param	dn String
	 */
	public void setEntrydn( final String dn ) {
		entrydn = nval( getRequestParameter( dn ), entrydn );
	}

	/**
	 * 【TAG】属性の取得元のオブジェクトのパスワードを設定します(初期値:システムパラメータ の LDAP_PASSWORD)。
	 *
	 * @og.tag
	 * 初期値:システムパラメータ の LDAP_PASSWORD が使用されます。
	 *
	 * @og.rev 4.2.2.0 (2008/05/10) 新規追加
	 *
	 * @param	pwd String
	 */
	public void setPassword( final String pwd ) {
		password = nval( getRequestParameter( pwd ), password );
	}

	/**
	 * 【TAG】(通常は使いません)結果をDBTableModelに書き込んで、sessionに登録するときのキーを指定します。
	 *
	 * @og.tag
	 * 初期値は、HybsSystem.TBL_MDL_KEY で定義した値です。
	 *
	 * @param	id sessionに登録する時の ID
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ), tableId );
	}

	/**
	 * 【TAG】検索した結果を表示する表示順をファイル属性名で指定します。
	 *
	 * @og.tag
	 * attributes 属性で指定するキー、または、LDAPから返されたキーについて
	 * その属性でソートします。逆順を行う場合は、DESC を指定のカラム名の後ろに
	 * 付けて下さい。
	 *
	 * @param	ordr	ソートキーを指定。
	 */
	public void setOrderBy( final String ordr ) {
		orderBy = nval( getRequestParameter( ordr ), orderBy );
		if( orderBy != null ) { orderBy = orderBy.toUpperCase(Locale.JAPAN) ; }
	}

	/**
	 * 【TAG】コマンド(NEW,RENEW)をセットします(初期値:NEW)。
	 *
	 * @og.tag
	 * コマンドは,HTMLから（get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 * 何も設定されない、または、null の場合は、"NEW" が初期値にセットされます。
	 *
	 * @param	cmd コマンド（public static final 宣言されている文字列)
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.LDAPQueryTag.CMD_NEW">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】検索結果を画面上に表示するメッセージリソースIDを指定します(初期値は検索件数)。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したメッセージをリソースから取得して
	 * 表示します。
	 * 表示させたくない場合は, displayMsg = "" をセットしてください。
	 * 初期値は、検索件数を表示します。
	 *
	 * @param	id ディスプレイに表示させるメッセージ ID
	 */
	public void setDisplayMsg( final String id ) {
		displayMsg = getRequestParameter( id );
	}

	/**
	 * 【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])。
	 *
	 * @og.tag
	 * ここでは、検索結果がゼロ件の場合のみ、特別なメッセージを表示させます。
	 * 従来は、displayMsg と兼用で、『0　件検索しました』という表示でしたが、
	 * displayMsg の初期表示は、OFF になりましたので、ゼロ件の場合のみ別に表示させます。
	 * 表示させたくない場合は, notfoundMsg = "" をセットしてください。
	 * 初期値は、MSG0077[対象データはありませんでした]です。
	 *
	 * @param	id ディスプレイに表示させるメッセージ ID
	 */
	public void setNotfoundMsg( final String id ) {
		String ids = getRequestParameter( id );
		if( ids != null ) { notfoundMsg = ids; }
	}

	/**
	 * 【TAG】オーバーフロー時に画面上に表示するメッセージIDを指定します(初期値:MSG0007)。
	 *
	 * @og.tag
	 * 表示させたくない場合は, overflowMsg = "" をセットしてください。
	 *
	 * @param	id ディスプレイに表示させるメッセージ ID
	 */
	public void setOverflowMsg( final String id ) {
		overflowMsg = getRequestParameter( id );
	}

	/**
	 * 【TAG】(通常使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します。(初期値:false)
	 *
	 * @og.tag
	 * (通常使いません)タグで処理される処理が、メインとなるトランザクション処理かどうかを指定します。(初期値:false)
	 * この値は、ファイルダウンロード処理に影響します。この値がtrueに指定された時にcommitされたDBTableModelが
	 * ファイルダウンロードの対象の表になります。
	 * 
	 * このパラメーターは、通常、各タグにより実装され、ユーザーが指定する必要はありません。
	 * 但し、1つのJSP内でDBTableModelが複数生成される場合に、前に処理したDBTableModelについてファイルダウンロードをさせたい
	 * 場合は、後ろでDBTableModelを生成するタグで、明示的にこの値をfalseに指定することで、ファイルダウンロード処理の対象から
	 * 除外することができます。
	 * 
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param  flag メイントランザクションかどうか
	 */
	public void setMainTrans( final String flag ) {
		isMainTrans = nval( getRequestParameter( flag ),isMainTrans );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"				,VERSION		)
				.println( "initctx" 			,initctx 		)
				.println( "providerURL" 		,providerURL 	)
				.println( "entrydn" 			,entrydn 		)
				.println( "searchbase"			,searchbase		)
				.println( "filter" 				,filter 		)
				.println( "searchScope"			,searchScope	)
				.println( "COUNTLIMIT"			,COUNTLIMIT		)
				.println( "timeLimit"			,timeLimit		)
				.println( "attrs"				,attrs			)
				.println( "columns"				,columns		)
				.println( "returningObjFlag"	,returningObjFlag)
				.println( "derefLinkFlag"		,derefLinkFlag	)
				.println( "tableId" 			,tableId 		)
				.println( "orderBy"				,orderBy		)
				.println( "command"				,command		)
				.println( "executeCount"		,executeCount	)
				.println( "errCode"				,errCode		)
				.println( "maxRowCount"			,maxRowCount	)
				.println( "displayMsg"   		,displayMsg   	)
				.println( "overflowMsg"  		,overflowMsg  	)
				.println( "Other..."			,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
